﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.IO;
using System.Text;

using NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Model;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Converter
{
    internal class Nw4rSoundFileManager
    {
        #region ** 固定値

        private const string CacheDirectoryNameDefault = "cache";

        private const string UserKeyIDLabel = "Label";
        private const string UserKeyIDSourceName = "SourceName";

        private const string ParameterAggregateOutput = "AggregateOutput";
        private const string ParameterCacheDirectoryName = "CacheDirectoryName";

        public const char SourceNameFormatSeparator = ':';
        public const char SourceNameTypeSeparator = '|';

        #endregion

        #region ** フィールド

        private string _dependFilePath = string.Empty;			    // 依存管理ファイルパス
        private string _cachDirectoryName = CacheDirectoryNameDefault;  // キャッシュディレクトリ名
        private bool _aggregateOutput = false;				        // キャッシュディレクトリに出力する

        private DependencyManager _dependencies = null;	// 依存管理クラス
        private DependencyManager _cacheDependencies = null;	// 依存管理クラス（キャッシュ管理用）
        private FileDictionary _extensionFiles = null;    // 外部ファイルパスディクショナリ

        private Dictionary<string, IOutputItem> _outputMap;	// 出力マップ
        private Dictionary<string, Nw4rSoundFile> _fileMap;	    // ファイルマップ

        private Dictionary<Type, INw4rFileFactory> _fileFactories = new Dictionary<Type, INw4rFileFactory>();	// ファイルファクトリ

        #endregion

        public Nw4rSoundFileManager(string dependFilePath, string cacheDirectoryName)
        {
            Debug.Assert(null != dependFilePath);
            Debug.Assert(0 < dependFilePath.Length);
            Debug.Assert(null != cacheDirectoryName);

            _dependFilePath = dependFilePath;
            _cachDirectoryName = cacheDirectoryName;

            _fileFactories.Add(typeof(Nw4rStreamSound), new Nw4rStreamSoundFileFactory(this));
            _fileFactories.Add(typeof(Nw4rWaveSound), new Nw4rWaveSoundFileFactory(this));
            _fileFactories.Add(typeof(Nw4rWaveSoundSet), new Nw4rWaveSoundSetFileFactory(this));
            _fileFactories.Add(typeof(Nw4rSequenceSound), new Nw4rSequenceSoundFileFactory(this));
            _fileFactories.Add(typeof(Nw4rBank), new Nw4rBankFileFactory(this));
            _fileFactories.Add(typeof(Nw4rGroup), new Nw4rGroupFileFactory(this));
            _fileFactories.Add(typeof(Nw4rInstrumentVelocityRegion), new Nw4rWaveSoundFileFactory(this));
        }

        #region ** プロパティ

        public bool AggregateOutput
        {
            get { return _aggregateOutput; }
            set
            {
                if (value == _aggregateOutput) { return; }

                _aggregateOutput = value;

                if (IsInitialized)
                {
                    Initialize();
                }
            }
        }

        public string CacheDirectoryAbsolutePath
        {
            get
            {
                string path = Path.Combine(Path.GetDirectoryName(_dependFilePath), _cachDirectoryName);

                if (null == path || 0 == path.Length) { throw new Nw4rFileFormatInternalException("invalid cache directory path."); }
                if (!path.EndsWith(_cachDirectoryName)) { throw new Nw4rFileFormatInternalException("invalid cache directory path."); }
                return path;
            }
        }

        public IFileDictionary ExtensionFilePathes
        {
            get { return _extensionFiles; }
        }

        private bool IsInitialized
        {
            get { return (null != _dependencies); }
        }

        #endregion

        #region ** メソッド

        public void Load()
        {
            Initialize();

            try
            {

                _cacheDependencies.Load(_dependFilePath);

                // AggregateOutput 設定が異なる場合、キャッシュは無効
                if (_cacheDependencies.UserParameters.ContainsKey(ParameterAggregateOutput))
                {

                    bool aggregateOutput = bool.Parse(_cacheDependencies.UserParameters[ParameterAggregateOutput]);
                    string cacheDirectoryName = (_cacheDependencies.UserParameters.ContainsKey(ParameterCacheDirectoryName)) ?
                                                 _cacheDependencies.UserParameters[ParameterCacheDirectoryName] : null;

                    if (_aggregateOutput != aggregateOutput ||
                        (null == cacheDirectoryName || _cachDirectoryName != cacheDirectoryName))
                    {
                        _cacheDependencies.Clean();
                    }

                }

            }
            catch
            {
                _dependencies.ManagementFilePath = _dependFilePath;
                _cacheDependencies.ManagementFilePath = _dependFilePath;
            }
        }

        public void Save()
        {
            Save(true);
        }

        public void Save(bool performGarbageCollection)
        {
            // ガベージコレクト
            if (performGarbageCollection)
            {
                CollectGarbage();
            }

            // AggregateOutput を依存ファイルに出力
            if (_dependencies.UserParameters.ContainsKey(ParameterAggregateOutput))
            {
                _dependencies.UserParameters[ParameterAggregateOutput] = _aggregateOutput.ToString();
            }
            else
            {
                _dependencies.UserParameters.Add(ParameterAggregateOutput, _aggregateOutput.ToString());
            }

            // CacheDirectoryName を依存ファイルに出力
            if (_dependencies.UserParameters.ContainsKey(ParameterCacheDirectoryName))
            {
                _dependencies.UserParameters[ParameterCacheDirectoryName] = _cachDirectoryName;
            }
            else
            {
                _dependencies.UserParameters.Add(ParameterCacheDirectoryName, _cachDirectoryName);
            }

            if (File.Exists(_dependencies.ManagementFilePath))
            {
                File.SetAttributes(_dependencies.ManagementFilePath, FileAttributes.Normal);
            }

            _dependencies.Save();

            if (File.Exists(_dependencies.ManagementFilePath))
            {
                File.SetAttributes(_dependencies.ManagementFilePath, FileAttributes.Hidden);
            }
        }

        public void Clean()
        {
            Initialize();

            if (File.Exists(_dependFilePath))
            {

                DependencyManager dependencies = new DependencyManager();
                dependencies.Load(_dependFilePath);
                dependencies.Clean();

                File.Delete(_dependFilePath);
            }
        }

        public string MakeRelativePath(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }
            if (0 == filePath.Length) { return string.Empty; }

            return NPath.MakeRelative(filePath, _dependencies.BaseDirectoryAbsolutePath + Path.DirectorySeparatorChar);
        }

        #region ** 中間出力

        public Nw4rSoundOutput GetIntermediateOutput(Nw4rItem item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }

            INw4rFileFactory fileFactory = _fileFactories[item.GetType()];

            IOutputItem output = GetIntermediateOutputItem(item, fileFactory);
            if (null == output) { return null; }

            Nw4rSoundIntermediateFile file = null;

            if (_fileMap.ContainsKey(output.Key))
            {
                file = _fileMap[output.Key] as Nw4rSoundIntermediateFile;
                Debug.Assert(null != file);
            }
            else
            {
                Debug.Assert(0 < output.Outputs.Count);
                file = fileFactory.CreateIntermediateFile(item, output.Outputs[0].AbsoluteFilePath);
                _fileMap.Add(output.Key, file);
            }

            return new Nw4rSoundOutput(file, output);
        }

        private IOutputItem GetIntermediateOutputItem(Nw4rItem item, INw4rFileFactory fileFactory)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            if (null == fileFactory) { throw new ArgumentNullException("fileFactory"); }

            string sourceName = fileFactory.GetSourceName(item) + SourceNameTypeSeparator + "Intermediate";
            if (_outputMap.ContainsKey(sourceName)) { return _outputMap[sourceName]; }

            // 有効な出力キャッシュがない場合、新しく出力を作成する
            // 中間ファイルは依存管理ファイルに登録しない
            IOutputItem output = fileFactory.CreateIntermediateOutput(_dependencies, item);
            if (null == output) { return null; }

            output.ForceDirty = true;

            RegisterOutputMap(output, sourceName);

            return output;
        }

        #endregion

        #region ** バイナリ出力

        public Nw4rSoundOutput GetBinaryOutput(Nw4rItem item)
        {
            try
            {
                return GetBinaryOutput(item, false);
            }
            catch (DependencyFileNotFoundException exception)
            {

                if (item is Nw4rInstrumentVelocityRegion)
                {

                    Nw4rInstrument inst = (item as Nw4rInstrumentVelocityRegion).Instrument;

                    if (null != inst)
                    {
                        throw new Nw4rFileNotFoundException(exception.FileName, inst.AbsoluteKey);
                    }

                }

                throw new Nw4rFileNotFoundException(exception.FileName, item.Label);

            }
        }

        public bool IsValidBinaryOutputCache(Nw4rItem item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }

            try
            {

                INw4rFileFactory fileFactory = _fileFactories[item.GetType()];
                string sourceName = fileFactory.GetSourceName(item);

                IOutputItem cachedOutput = GetCacheItem(sourceName);
                if (null == cachedOutput) { return false; }

                return !cachedOutput.Dirty;

            }
            catch (DependencyFileNotFoundException exception)
            {

                if (item is Nw4rInstrumentVelocityRegion)
                {

                    Nw4rInstrument inst = (item as Nw4rInstrumentVelocityRegion).Instrument;

                    if (null != inst)
                    {
                        throw new Nw4rFileNotFoundException(exception.FileName, inst.AbsoluteKey);
                    }

                }

                throw new Nw4rFileNotFoundException(exception.FileName, item.Label);

            }
        }

        public Nw4rSoundOutput GetBinaryOutputForGroup(Nw4rGroup group, Nw4rItem item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }

            if (!_fileFactories.ContainsKey(item.GetType())) { return null; }

            try
            {

                INw4rFileFactoryExForGroup fileFactory = _fileFactories[item.GetType()] as INw4rFileFactoryExForGroup;
                if (null == fileFactory) { return null; }

                IOutputItem output = GetBinaryOutputItemForGroup(group, item, fileFactory);
                Nw4rSoundBinaryFileForGroup file = null;

                if (_fileMap.ContainsKey(output.Key))
                {
                    file = _fileMap[output.Key] as Nw4rSoundBinaryFileForGroup;
                    Debug.Assert(null != file);
                }
                else
                {
                    Debug.Assert(0 < output.Outputs.Count);
                    file = fileFactory.CreateBinaryFileForGroup(group, item, output.Outputs[0].AbsoluteFilePath);
                    _fileMap.Add(output.Key, file);
                }

                return new Nw4rSoundOutput(file, output);

            }
            catch (DependencyFileNotFoundException exception)
            {

                if (item is Nw4rInstrumentVelocityRegion)
                {

                    Nw4rInstrument inst = (item as Nw4rInstrumentVelocityRegion).Instrument;

                    if (null != inst)
                    {
                        throw new Nw4rFileNotFoundException(exception.FileName, inst.AbsoluteKey);
                    }

                }

                throw new Nw4rFileNotFoundException(exception.FileName, item.Label);

            }
        }

        private Nw4rSoundOutput GetBinaryOutput(Nw4rItem item, bool cached)
        {
            if (null == item) { throw new ArgumentNullException("item"); }

            INw4rFileFactory fileFactory = _fileFactories[item.GetType()];
            IOutputItem output = GetBinaryOutputItem(item, fileFactory, cached);
            if (null == output) { return null; }

            Nw4rSoundBinaryFile file = null;

            if (_fileMap.ContainsKey(output.Key))
            {
                file = _fileMap[output.Key] as Nw4rSoundBinaryFile;
                Debug.Assert(null != file);
            }
            else
            {

                Debug.Assert(0 < output.Outputs.Count);

                file = fileFactory.CreateBinaryFile(item, output.Outputs[0].AbsoluteFilePath);
                _fileMap.Add(output.Key, file);

                // 外部ファイルパス一覧に追加
                if (0 < file.ExtensionFileRelativePath.Length)
                {
                    _extensionFiles.Add(file.ExtensionFileRelativePath, file);
                }

            }

            return new Nw4rSoundOutput(file, output);
        }

        private IOutputItem GetBinaryOutputItem(Nw4rItem item, INw4rFileFactory fileFactory, bool cached)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            if (null == fileFactory) { throw new ArgumentNullException("fileFactory"); }

            string sourceName = fileFactory.GetSourceName(item);
            IOutputItem newOutput = GetNewItem(sourceName);
            IOutputItem targetOutput = null;

            // 一度でも取得した出力は、そのまま返す
            if (null != newOutput)
            {
                if (!_dependencies.OutputItems.ContainsKey(newOutput.Key))
                {
                    throw new Nw4rFileFormatInternalException("unexpected error");
                }
                return newOutput;
            }

            IOutputItem cachedOutput = GetCacheItem(sourceName);

            if (null != cachedOutput && !cachedOutput.Dirty)
            {
                targetOutput = cachedOutput;
            }
            else
            {

                newOutput = fileFactory.CreateBinaryOutput(_dependencies, item);

                if (null == newOutput)
                {
                    throw new Nw4rFileFormatInternalException("unexpected error");
                }

            }

            // 有効な出力キャッシュがない場合、新しく作成した出力を使用する
            if (!cached && null == targetOutput)
            {
                targetOutput = newOutput;
            }

            if (null != targetOutput)
            {
                _dependencies.Add(targetOutput);
                _dependencies.SetUserKey(UserKeyIDLabel, item.AbsoluteKey, targetOutput);
                _dependencies.SetUserKey(UserKeyIDSourceName, sourceName, targetOutput);
            }

            // 出力一覧に登録する
            RegisterOutputMap(targetOutput, sourceName);

            return targetOutput;
        }

        private IOutputItem GetBinaryOutputItemForGroup(Nw4rGroup group, Nw4rItem item, INw4rFileFactoryExForGroup fileFactory)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            if (null == fileFactory) { throw new ArgumentNullException("fileFactory"); }

            string sourceName = fileFactory.GetSourceNameForGroup(group, item);
            IOutputItem newOutput = GetNewItem(sourceName);
            IOutputItem targetOutput = null;

            // 一度でも取得した出力は、そのまま返す
            if (null != newOutput)
            {
                if (!_dependencies.OutputItems.ContainsKey(newOutput.Key))
                {
                    throw new Nw4rFileFormatInternalException("unexpected error");
                }
                return newOutput;
            }

            IOutputItem cachedOutput = GetCacheItem(sourceName);

            if (null != cachedOutput && !cachedOutput.Dirty)
            {
                targetOutput = cachedOutput;
            }
            else
            {

                newOutput = fileFactory.CreateOutputForGroup(_dependencies, group, item);

                if (null == newOutput)
                {
                    throw new Nw4rFileFormatInternalException("unexpected error");
                }

            }

            // 有効な出力キャッシュがない場合、新しく作成した出力を使用する
            if (null == targetOutput)
            {
                targetOutput = newOutput;
            }

            if (null != targetOutput)
            {
                _dependencies.Add(targetOutput);
                _dependencies.SetUserKey(UserKeyIDLabel, item.AbsoluteKey, targetOutput);
                _dependencies.SetUserKey(UserKeyIDSourceName, sourceName, targetOutput);
            }

            // 出力一覧に登録する
            RegisterOutputMap(targetOutput, sourceName);

            return targetOutput;
        }

        private IOutputItem GetNewItem(string sourceName)
        {
            if (!_dependencies.OutputItems.ContainsUserKey(UserKeyIDSourceName, sourceName)) { return null; }
            return _dependencies.OutputItems[UserKeyIDSourceName, sourceName];
        }

        #endregion

        #region ** キャッシュ操作

        private IOutputItem GetCacheItem(string sourceName)
        {
            if (null == sourceName) { throw new ArgumentNullException("sourceName"); }

            if (!_cacheDependencies.OutputItems.ContainsUserKey(UserKeyIDSourceName, sourceName))
            {
                return null;
            }

            return _cacheDependencies.OutputItems[UserKeyIDSourceName, sourceName];
        }

        private void RegisterOutputMap(IOutputItem output, string sourceName)
        {
            if (null == output) { return; }

            if (!_outputMap.ContainsKey(sourceName))
            {
                _outputMap.Add(sourceName, output);
            }

            // セパレータがある場合、「ソース名:フォーマット:サウンドの種類」となっている
            // フォーマットのみが異なる場合、キャッシュが破棄されないように「ソース名:サウンドの種類」として追加登録しておく
            string[] sourceNames = sourceName.Split(SourceNameFormatSeparator);

            if (3 == sourceNames.Length)
            {

                string sourceNameWithoutFormat = sourceNames[0] + SourceNameFormatSeparator + sourceNames[2];

                if (!_outputMap.ContainsKey(sourceNameWithoutFormat))
                {
                    _outputMap.Add(sourceNameWithoutFormat, output);
                }

            }
        }

        private void CollectGarbage()
        {
            Dictionary<string, IOutputInfo> useFiles = new Dictionary<string, IOutputInfo>();
            Collection<string> garbageFiles = new Collection<string>();

            foreach (IOutputItem outputItem in _dependencies.OutputItems.Values)
            {
                foreach (IOutputInfo output in outputItem.Outputs)
                {
                    useFiles.Add(output.AbsoluteFilePath, output);
                }
            }

            if (_cacheDependencies.OutputItems.ContainsUserKeyMap(UserKeyIDSourceName))
            {
                foreach (string sourceName in _cacheDependencies.OutputItems.GetUserKeys(UserKeyIDSourceName))
                {
                    if (_outputMap.ContainsKey(sourceName)) { continue; }

                    // フォーマットセパレータを含む場合、「ソース名:サウンドの種類」を含むかどうかも確認する
                    string[] sourceNames = sourceName.Split(SourceNameFormatSeparator);

                    if (3 == sourceNames.Length)
                    {
                        if (_outputMap.ContainsKey(sourceNames[0] + SourceNameFormatSeparator + sourceNames[2])) { continue; }
                    }

                    foreach (IOutputInfo output in _cacheDependencies.OutputItems[UserKeyIDSourceName, sourceName].Outputs)
                    {
                        if (useFiles.ContainsKey(output.AbsoluteFilePath)) { continue; }
                        garbageFiles.Add(output.AbsoluteFilePath);
                    }
                }
            }

            foreach (string filePath in garbageFiles)
            {
                File.Delete(filePath);
            }
        }

        #endregion

        #region ** 初期化

        private void Initialize()
        {
            Debug.Assert(null != _dependFilePath);
            Debug.Assert(0 < _dependFilePath.Length);

            _dependencies = new DependencyManager();
            _cacheDependencies = new DependencyManager();
            _extensionFiles = new FileDictionary();

            _outputMap = new Dictionary<string, IOutputItem>(NoCaseStringCompare.Instance);
            _fileMap = new Dictionary<string, Nw4rSoundFile>(NoCaseStringCompare.Instance);

            _dependencies.ManagementFilePath = _dependFilePath;
            _cacheDependencies.ManagementFilePath = _dependFilePath;

            if (_aggregateOutput)
            {
                _dependencies.BaseDirectoryPath = _cachDirectoryName;
            }
        }

        #endregion

        #endregion

        #region ** NoCaseStringCompare

        private class NoCaseStringCompare : IEqualityComparer<string>
        {
            public static NoCaseStringCompare Instance = new NoCaseStringCompare();

            private NoCaseStringCompare() { }

            /// <summary>
            /// 指定した string が等しいかどうかを判断します。
            /// </summary>
            /// <param name="x">string1。</param>
            /// <param name="y">string2。</param>
            /// <returns>指定した string が等しい場合は true。それ以外の場合は false。</returns>
            public bool Equals(string x, string y)
            {
                if (null == x) { throw new ArgumentNullException("x"); }
                if (null == y) { throw new ArgumentNullException("y"); }

                return string.Compare(x, y, true) == 0;
            }

            /// <summary>
            /// 指定した string のハッシュ コードを返します。
            /// </summary>
            /// <param name="text">string。</param>
            /// <returns>ハッシュコード</returns>
            public int GetHashCode(string text)
            {
                if (null == text) { throw new ArgumentNullException("text"); }
                return text.ToLower().GetHashCode();
            }
        }

        #endregion

        #region ** ファイルパスディクショナリ

        public interface IFileDictionary : IEnumerable<KeyValuePair<string, Nw4rSoundFile>>
        {
            bool ContainsKey(string key);
        }

        private class FileDictionary : Dictionary<string, Nw4rSoundFile>, IFileDictionary
        {
            public FileDictionary() : base(NoCaseStringCompare.Instance) { }
        }

        #endregion
    }

    #region ** サウンド出力クラス

    internal class Nw4rSoundOutput
    {
        #region ** フィールド

        private Nw4rSoundFile _file = null;
        private IOutputItem _output = null;

        #endregion

        public Nw4rSoundOutput(Nw4rSoundFile file, IOutputItem output)
        {
            Debug.Assert(null != file);
            Debug.Assert(null != output);

            _file = file;
            _output = output;
        }

        #region ** プロパティ

        public Nw4rSoundFile File
        {
            get { return _file; }
        }

        public IOutputItem OutputItem
        {
            get { return _output; }
        }

        public bool Dirty
        {
            get { return _output.Dirty; }
        }

        #endregion
    }

    internal class Nw4rSoundOutputWrapper<FileType>
        where FileType : class
    {
        private Nw4rSoundOutput _soundOutput = null;

        public Nw4rSoundOutputWrapper(Nw4rSoundOutput soundOutput)
        {
            Debug.Assert(null != soundOutput);
            _soundOutput = soundOutput;
        }

        #region ** プロパティ

        public FileType File
        {
            get { return _soundOutput.File as FileType; }
        }

        public IOutputItem OutputItem
        {
            get { return _soundOutput.OutputItem; }
        }

        public bool Dirty
        {
            get { return _soundOutput.Dirty; }
        }

        public Nw4rSoundOutput SoundCache
        {
            get { return _soundOutput; }
        }

        #endregion
    }

    internal class Nw4rSoundBinaryOutput : Nw4rSoundOutputWrapper<Nw4rSoundBinaryFile>
    {
        public Nw4rSoundBinaryOutput(Nw4rSoundOutput soundCache) : base(soundCache) { }
    }

    internal class Nw4rSoundIntermediateOutput : Nw4rSoundOutputWrapper<Nw4rSoundIntermediateFile>
    {
        public Nw4rSoundIntermediateOutput(Nw4rSoundOutput soundCache) : base(soundCache) { }
    }

    internal class Nw4rStreamSoundOutput : Nw4rSoundOutputWrapper<Nw4rStreamSoundBinaryFile>
    {
        public Nw4rStreamSoundOutput(Nw4rSoundOutput soundCache) : base(soundCache) { }
    }

    internal class Nw4rWaveSoundSetOutput : Nw4rSoundOutputWrapper<Nw4rWaveSoundSetBinaryFile>
    {
        public Nw4rWaveSoundSetOutput(Nw4rSoundOutput soundCache) : base(soundCache) { }
    }

    internal class Nw4rSequenceSoundOutput : Nw4rSoundOutputWrapper<Nw4rSequenceSoundBinaryFile>
    {
        public Nw4rSequenceSoundOutput(Nw4rSoundOutput soundCache) : base(soundCache) { }
    }

    internal class Nw4rBankOutput : Nw4rSoundOutputWrapper<Nw4rBankBinaryFile>
    {
        public Nw4rBankOutput(Nw4rSoundOutput soundCache) : base(soundCache) { }
    }

    internal class Nw4rGroupOutput : Nw4rSoundOutputWrapper<Nw4rGroupBinaryFile>
    {
        public Nw4rGroupOutput(Nw4rSoundOutput soundCache) : base(soundCache) { }
    }

    #endregion

    #region ** ファイルファクトリ クラス

    #region ** ファイルファクトリ 基本クラス

    internal interface INw4rFileFactory
    {
        IOutputItem CreateBinaryOutput(DependencyManager outputOwner, Nw4rItem item);
        IOutputItem CreateIntermediateOutput(DependencyManager outputOwner, Nw4rItem item);

        Nw4rSoundBinaryFile CreateBinaryFile(Nw4rItem item);
        Nw4rSoundBinaryFile CreateBinaryFile(Nw4rItem item, string filePath);

        Nw4rSoundIntermediateFile CreateIntermediateFile(Nw4rItem item);
        Nw4rSoundIntermediateFile CreateIntermediateFile(Nw4rItem item, string filePath);

        string GetSourceName(Nw4rItem item);
    }

    internal interface INw4rFileFactoryExForGroup : INw4rFileFactory
    {
        IOutputItem CreateOutputForGroup(DependencyManager outputOwner, Nw4rGroup group, Nw4rItem item);

        Nw4rSoundBinaryFileForGroup CreateBinaryFileForGroup(Nw4rGroup group, Nw4rItem item);
        Nw4rSoundBinaryFileForGroup CreateBinaryFileForGroup(Nw4rGroup group, Nw4rItem item, string filePath);

        string GetSourceNameForGroup(Nw4rGroup group, Nw4rItem item);
    }

    internal abstract class Nw4rFileFactory<_ItemType> : INw4rFileFactory
        where _ItemType : Model.Nw4rSoundSetItem
    {
        private Nw4rSoundFileManager _fileManager = null;

        public Nw4rFileFactory(Nw4rSoundFileManager fileManager)
        {
            if (null == fileManager) { throw new ArgumentNullException("fileManager"); }
            _fileManager = fileManager;
        }

        #region ** プロパティ

        protected Nw4rSoundFileManager FileManager
        {
            get { return _fileManager; }
        }

        #endregion

        #region ** メソッド

        #region ** 出力の作成

        public abstract IOutputItem CreateBinaryOutput(DependencyManager outputOwner, _ItemType item);

        protected virtual IOutputItem CreateBinaryOutputInternal(DependencyManager outputOwner, Nw4rItem item)
        {
            return CreateBinaryOutput(outputOwner, item as _ItemType);
        }

        public virtual IOutputItem CreateIntermediateOutput(DependencyManager outputOwner, _ItemType item)
        {
            return null;
        }

        protected virtual IOutputItem CreateIntermediateOutputInternal(DependencyManager outputOwner, Nw4rItem item)
        {
            return CreateIntermediateOutput(outputOwner, item as _ItemType);
        }

        #endregion

        #region ** ファイルの作成

        public abstract Nw4rSoundBinaryFile CreateBinaryFile(_ItemType item, string filePath);

        protected virtual Nw4rSoundBinaryFile CreateBinaryFileInternal(Nw4rItem item, string filePath)
        {
            Debug.Assert(item is _ItemType);
            return CreateBinaryFile(item as _ItemType, filePath);
        }

        public virtual Nw4rSoundIntermediateFile CreateIntermediateFile(_ItemType item, string filePath)
        {
            return null;
        }

        protected virtual Nw4rSoundIntermediateFile CreateIntermediateFileInternal(Nw4rItem item, string filePath)
        {
            Debug.Assert(item is _ItemType);
            return CreateIntermediateFile(item as _ItemType, filePath);
        }

        #endregion

        #region ** 名前の取得

        public abstract string GetSourceName(_ItemType soundSetItem);

        protected virtual string GetSourceNameInternal(Nw4rItem item)
        {
            Debug.Assert(item is _ItemType);
            return GetSourceName(item as _ItemType);
        }

        #endregion

        #endregion

        #region ** INw4rFileFactory の実装

        IOutputItem INw4rFileFactory.CreateBinaryOutput(DependencyManager outputOwner, Nw4rItem item)
        {
            Debug.Assert(null != outputOwner);
            Debug.Assert(null != item);

            return CreateBinaryOutputInternal(outputOwner, item);
        }

        IOutputItem INw4rFileFactory.CreateIntermediateOutput(DependencyManager outputOwner, Nw4rItem item)
        {
            Debug.Assert(null != outputOwner);
            Debug.Assert(null != item);
            return CreateIntermediateOutputInternal(outputOwner, item);
        }

        Nw4rSoundBinaryFile INw4rFileFactory.CreateBinaryFile(Nw4rItem item)
        {
            Debug.Assert(null != item);
            return CreateBinaryFileInternal(item, null);
        }

        Nw4rSoundBinaryFile INw4rFileFactory.CreateBinaryFile(Nw4rItem item, string filePath)
        {
            Debug.Assert(null != item);
            return CreateBinaryFileInternal(item, filePath);
        }

        Nw4rSoundIntermediateFile INw4rFileFactory.CreateIntermediateFile(Nw4rItem item)
        {
            Debug.Assert(null != item);
            return CreateIntermediateFileInternal(item, null);
        }

        Nw4rSoundIntermediateFile INw4rFileFactory.CreateIntermediateFile(Nw4rItem item, string filePath)
        {
            Debug.Assert(null != item);
            return CreateIntermediateFileInternal(item, filePath);
        }

        string INw4rFileFactory.GetSourceName(Nw4rItem item)
        {
            Debug.Assert(null != item);
            return GetSourceNameInternal(item);
        }

        #endregion
    }

    internal abstract class Nw4rFileFactoryExForGroup<_ItemType> : Nw4rFileFactory<_ItemType>, INw4rFileFactoryExForGroup
        where _ItemType : Model.Nw4rSoundSetItem
    {
        public Nw4rFileFactoryExForGroup(Nw4rSoundFileManager fileManager) : base(fileManager) { }

        #region ** メソッド

        #region ** 出力の作成

        public abstract IOutputItem CreateOutputForGroup(DependencyManager outputOwner, Nw4rGroup group, _ItemType item);

        protected virtual IOutputItem CreateOutputForGroupInternal(DependencyManager outputOwner, Nw4rGroup group, Nw4rItem item)
        {
            return CreateOutputForGroup(outputOwner, group, item as _ItemType);
        }

        #endregion

        #region ** ファイルの作成

        public abstract Nw4rSoundBinaryFileForGroup CreateBinaryFileForGroup(Nw4rGroup group, _ItemType item, string filePath);

        protected virtual Nw4rSoundBinaryFileForGroup CreateBinaryFileForGroupInternal(Nw4rGroup group, Nw4rItem item, string filePath)
        {
            Debug.Assert(item is _ItemType);
            return CreateBinaryFileForGroup(group, item as _ItemType, filePath);
        }

        #endregion

        #region ** 名前の取得

        public abstract string GetSourceNameForGroup(Nw4rGroup group, _ItemType soundSetItem);

        protected virtual string GetSourceNameForGroupInternal(Nw4rGroup group, Nw4rItem item)
        {
            Debug.Assert(item is _ItemType);
            return GetSourceNameForGroup(group, item as _ItemType);
        }

        #endregion

        #endregion

        #region ** INw4rFileFactoryExForGroup の実装

        IOutputItem INw4rFileFactoryExForGroup.CreateOutputForGroup(DependencyManager outputOwner, Nw4rGroup group, Nw4rItem item)
        {
            Debug.Assert(null != outputOwner);
            Debug.Assert(null != group);
            Debug.Assert(null != item);
            return CreateOutputForGroupInternal(outputOwner, group, item);
        }

        Nw4rSoundBinaryFileForGroup INw4rFileFactoryExForGroup.CreateBinaryFileForGroup(Nw4rGroup group, Nw4rItem item)
        {
            Debug.Assert(null != group);
            Debug.Assert(null != item);
            return CreateBinaryFileForGroupInternal(group, item, null);
        }

        Nw4rSoundBinaryFileForGroup INw4rFileFactoryExForGroup.CreateBinaryFileForGroup(Nw4rGroup group, Nw4rItem item, string filePath)
        {
            Debug.Assert(null != group);
            Debug.Assert(null != item);
            return CreateBinaryFileForGroupInternal(group, item, filePath);
        }

        string INw4rFileFactoryExForGroup.GetSourceNameForGroup(Nw4rGroup group, Nw4rItem item)
        {
            Debug.Assert(null != group);
            Debug.Assert(null != item);
            return GetSourceNameForGroupInternal(group, item);
        }

        #endregion
    }

    #endregion

    /// <summary>
    /// ストリームサウンドファイルファクトリ
    /// </summary>
    internal class Nw4rStreamSoundFileFactory : Nw4rFileFactory<Nw4rStreamSound>
    {
        public Nw4rStreamSoundFileFactory(Nw4rSoundFileManager fileManager) : base(fileManager) { }

        #region ** 出力の作成

        public override IOutputItem CreateBinaryOutput(DependencyManager outputOwner, Nw4rStreamSound streamSound)
        {
            if (null == outputOwner) { throw new ArgumentNullException("outputOwner"); }
            if (null == streamSound) { throw new ArgumentNullException("streamSound"); }

            IOutputItem output = outputOwner.CreateOutput();

            // BinaryOutput
            if (FileManager.AggregateOutput)
            {
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryFileName(streamSound)));
            }
            else
            {
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryFilePath(streamSound)));
            }


            // Dependencies
            RegisterDependencies(outputOwner, output, streamSound);

            return output;
        }

        public override IOutputItem CreateIntermediateOutput(DependencyManager outputOwner, Nw4rStreamSound streamSound)
        {
            if (null == outputOwner) { throw new ArgumentNullException("outputOwner"); }
            if (null == streamSound) { throw new ArgumentNullException("streamSound"); }

            string filePath = GetIntermediateFilePath(streamSound);
            if (null == filePath || 0 == filePath.Length) { return null; }

            IOutputItem output = outputOwner.CreateOutput();

            // IntermediateOutput
            output.Outputs.Add(outputOwner.CreateOutputItem(filePath));


            // Dependencies
            RegisterDependencies(outputOwner, output, streamSound);

            return output;
        }

        private void RegisterDependencies(DependencyManager outputOwner, IOutputItem output, Nw4rStreamSound streamSound)
        {
            Debug.Assert(null != outputOwner);
            Debug.Assert(null != output);
            Debug.Assert(null != streamSound);

            // Dependencies
            // マルチトラックの場合はサウンドセットファイルに依存する
            if (1 < streamSound.XmlData.TrackList.Count)
            {
                Model.Nw4rSoundSet soundSet = streamSound.Parent as Model.Nw4rSoundSet;
                output.Dependencies.Add(
                    outputOwner.CreateDependencyFileInfo(FileManager.MakeRelativePath(soundSet.XmlData.FilePath)));
            }

            // 各トラックの波形ファイルに依存する
            foreach (Nw4rSoundSetStrm.Track track in streamSound.XmlData.TrackList)
            {
                output.Dependencies.Add(
                    outputOwner.CreateDependencyFileInfo(FileManager.MakeRelativePath(track.FullPath)));
            }
        }

        #endregion

        #region ** ファイルの作成

        public override Nw4rSoundBinaryFile CreateBinaryFile(Nw4rStreamSound streamSound, string filePath)
        {
            Debug.Assert(null != streamSound);

            string filePathWork = filePath;

            if (null == filePathWork)
            {
                filePathWork = Path.Combine(GetDirectoryName(streamSound), GetBinaryFileName(streamSound));
            }

            string extensionfilePath = GetExtensionFilePath(streamSound, streamSound.Root.ExtensionFileDirectoryPath);
            if (0 >= extensionfilePath.Length)
            {
                throw new Nw4rFileFormatInternalException("failed to create stream sound binary file");
            }

            return new Nw4rStreamSoundBinaryFile(filePathWork, extensionfilePath);
        }

        public override Nw4rSoundIntermediateFile CreateIntermediateFile(Nw4rStreamSound streamSound, string filePath)
        {
            Debug.Assert(null != streamSound);

            string filePathWork = filePath;

            if (null == filePathWork)
            {
                filePathWork = GetIntermediateFileAbsolutePath(streamSound);
                if (null == filePathWork || 0 == filePathWork.Length) { return null; }
            }

            return new Nw4rSoundIntermediateFile(filePathWork);
        }

        #endregion

        #region ** 名前の取得

        public override string GetSourceName(Nw4rStreamSound streamSound)
        {
            if (null == streamSound) { return string.Empty; }

            if (0 == streamSound.XmlData.TrackList.Count)
            {
                throw new Nw4rFileFormatInternalException("invalid stream sound (track not found)");
            }

            string baseSourceName = string.Empty;

            if (1 < streamSound.XmlData.TrackList.Count)
            {
                return streamSound.XmlData.Label;
            }
            else
            {
                baseSourceName = FileManager.MakeRelativePath(streamSound.XmlData.TrackList[0].FullPath);
            }

            switch (streamSound.XmlData.TrackList[0].FileFormat)
            {
                case Nw4rSoundSetStrmFileFormat.Pcm16:
                    return baseSourceName +
                            Nw4rSoundFileManager.SourceNameFormatSeparator + "pcm16" +
                            Nw4rSoundFileManager.SourceNameFormatSeparator + "stream";


                case Nw4rSoundSetStrmFileFormat.Pcm8:
                    return baseSourceName +
                            Nw4rSoundFileManager.SourceNameFormatSeparator + "pcm8" +
                            Nw4rSoundFileManager.SourceNameFormatSeparator + "stream";

                case Nw4rSoundSetStrmFileFormat.Adpcm:
                    return baseSourceName +
                            Nw4rSoundFileManager.SourceNameFormatSeparator + "adpcm" +
                            Nw4rSoundFileManager.SourceNameFormatSeparator + "stream";
            }

            throw new Nw4rFileFormatInternalException("invalid stream sound file format");
        }

        private string GetDirectoryName(Nw4rStreamSound streamSound)
        {
            if (null == streamSound) { return string.Empty; }
            return Path.GetDirectoryName(streamSound.XmlData.TrackList[0].FullPath);
        }

        private string GetFileNameBase(Nw4rStreamSound streamSound)
        {
            if (null == streamSound) { return string.Empty; }
            return Path.GetFileNameWithoutExtension(streamSound.XmlData.TrackList[0].FilePath);
        }

        private string GetBinaryFilePath(Nw4rStreamSound streamSound)
        {
            if (null == streamSound) { return string.Empty; }

            if (0 == streamSound.XmlData.TrackList.Count)
            {
                throw new Nw4rFileFormatInternalException("invalid stream sound (track not found)");
            }

            return FileManager.MakeRelativePath(
                    Path.Combine(GetDirectoryName(streamSound), GetBinaryFileName(streamSound)));
        }

        private string GetBinaryFileName(Nw4rStreamSound streamSound)
        {
            if (null == streamSound) { return string.Empty; }

            if (0 == streamSound.XmlData.TrackList.Count)
            {
                throw new Nw4rFileFormatInternalException("invalid stream sound (track not found)");
            }
            else if (1 < streamSound.XmlData.TrackList.Count)
            {
                return streamSound.XmlData.Label + ".multi.brstm";
            }

            switch (streamSound.XmlData.TrackList[0].FileFormat)
            {
                case Nw4rSoundSetStrmFileFormat.Pcm16:
                    return GetFileNameBase(streamSound) + ".pcm16.brstm";

                case Nw4rSoundSetStrmFileFormat.Pcm8:
                    return GetFileNameBase(streamSound) + ".pcm8.brstm";

                case Nw4rSoundSetStrmFileFormat.Adpcm:
                    return GetFileNameBase(streamSound) + ".adpcm.brstm";
            }

            throw new Nw4rFileFormatInternalException("invalid stream sound format");
        }

        private string GetExtensionFilePath(Nw4rStreamSound streamSound, string extensionFileDirectoryPath)
        {
            if (null == streamSound) { return string.Empty; }
            if (null == extensionFileDirectoryPath)
            {
                throw new Nw4rFileFormatInternalException("extension file directory must not null.");
            }

            if (0 == streamSound.XmlData.TrackList.Count)
            {
                throw new Nw4rFileFormatInternalException("invalid stream sound (track not found).");
            }
            else if (1 < streamSound.XmlData.TrackList.Count)
            {
                return Path.Combine(extensionFileDirectoryPath, streamSound.XmlData.Label + ".brstm").Replace('\\', '/');
            }

            switch (streamSound.XmlData.TrackList[0].FileFormat)
            {
                case Nw4rSoundSetStrmFileFormat.Pcm16:
                case Nw4rSoundSetStrmFileFormat.Pcm8:
                case Nw4rSoundSetStrmFileFormat.Adpcm:

                    int count = 0;
                    string filePath = Path.Combine(extensionFileDirectoryPath,
                                                    GetFileNameBase(streamSound) + ".brstm").Replace('\\', '/');

                    while (FileManager.ExtensionFilePathes.ContainsKey(filePath))
                    {

                        filePath = Path.Combine(extensionFileDirectoryPath,
                                                 string.Format("{0}.{1}.brstm",
                                                                GetFileNameBase(streamSound), count)
                                               ).Replace('\\', '/');
                        count++;

                    }

                    return filePath;
            }

            throw new Nw4rFileFormatInternalException("invalid stream sound format");
        }

        private string GetIntermediateFilePath(Nw4rStreamSound streamSound)
        {
            string absolutePath = GetIntermediateFileAbsolutePath(streamSound);
            if (0 == absolutePath.Length) { return string.Empty; }

            return FileManager.MakeRelativePath(absolutePath);
        }

        private string GetIntermediateFileAbsolutePath(Nw4rStreamSound streamSound)
        {
            string fileName = GetIntermediateFileName(streamSound);
            if (0 == fileName.Length) { return string.Empty; }

            return Path.GetFullPath(Path.Combine(FileManager.CacheDirectoryAbsolutePath, fileName));
        }

        private string GetIntermediateFileName(Nw4rStreamSound streamSound)
        {
            if (null == streamSound) { return string.Empty; }
            if (1 == streamSound.XmlData.TrackList.Count) { return string.Empty; }
            return streamSound.XmlData.Label + ".multi.rstm";
        }

        #endregion
    }

    /// <summary>
    /// ウェーブサウンドファイルファクトリ
    /// </summary>
    internal class Nw4rWaveSoundFileFactory : Nw4rFileFactory<Nw4rWaveSound>
    {
        public Nw4rWaveSoundFileFactory(Nw4rSoundFileManager fileManager) : base(fileManager) { }

        #region ** 出力の作成

        public override IOutputItem CreateBinaryOutput(DependencyManager outputOwner, Nw4rWaveSound waveSound)
        {
            if (null == outputOwner) { throw new ArgumentNullException("outputOwner"); }
            if (null == waveSound) { throw new ArgumentNullException("waveSound"); }

            IOutputItem output = outputOwner.CreateOutput();

            // BinaryFile
            if (FileManager.AggregateOutput)
            {
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryFileName(waveSound)));
            }
            else
            {
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryFilePath(waveSound)));
            }


            // Dependencies
            // 波形ファイルに依存する
            output.Dependencies.Add(
                outputOwner.CreateDependencyFileInfo(FileManager.MakeRelativePath(waveSound.XmlData.FullPath)));

            return output;
        }

        public IOutputItem CreateBinaryOutput(DependencyManager outputOwner, Nw4rInstrumentVelocityRegion velocityRegion)
        {
            if (null == outputOwner) { throw new ArgumentNullException("outputOwner"); }
            if (null == velocityRegion) { throw new ArgumentNullException("velocityRegion"); }

            IOutputItem output = outputOwner.CreateOutput();

            // BinaryFile
            if (FileManager.AggregateOutput)
            {
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryFileName(velocityRegion)));
            }
            else
            {
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryFilePath(velocityRegion)));
            }


            // Dependencies
            // 波形ファイルに依存する
            output.Dependencies.Add(
                outputOwner.CreateDependencyFileInfo(FileManager.MakeRelativePath(velocityRegion.FilePath)));

            return output;
        }

        protected override IOutputItem CreateBinaryOutputInternal(DependencyManager outputOwner, Nw4rItem item)
        {
            if (item is Nw4rWaveSound)
            {
                return CreateBinaryOutput(outputOwner, item as Nw4rWaveSound);
            }
            else if (item is Nw4rInstrumentVelocityRegion)
            {
                return CreateBinaryOutput(outputOwner, item as Nw4rInstrumentVelocityRegion);
            }

            throw new Nw4rFileFormatInternalException("failed to create output. ( invalid item type )");
        }

        #endregion

        #region ** ファイルの作成

        public override Nw4rSoundBinaryFile CreateBinaryFile(Nw4rWaveSound waveSound, string filePath)
        {
            Debug.Assert(null != waveSound);

            string filePathWork = filePath;

            if (null == filePathWork)
            {
                filePathWork = Path.Combine(Path.GetDirectoryName(waveSound.XmlData.FullPath),
                                             GetBinaryFileName(waveSound));
            }

            return new Nw4rSoundBinaryFile(filePathWork);
        }

        public Nw4rSoundBinaryFile CreateBinaryFile(Nw4rInstrumentVelocityRegion velocityRegion, string filePath)
        {
            Debug.Assert(null != velocityRegion);

            string filePathWork = filePath;

            if (null == filePathWork)
            {
                filePathWork = Path.Combine(Path.GetDirectoryName(velocityRegion.FilePath),
                                             GetBinaryFileName(velocityRegion));
            }

            return new Nw4rSoundBinaryFile(filePathWork);
        }

        protected override Nw4rSoundBinaryFile CreateBinaryFileInternal(Nw4rItem item, string filePath)
        {
            if (item is Nw4rWaveSound)
            {
                return CreateBinaryFile(item as Nw4rWaveSound, filePath);
            }
            else if (item is Nw4rInstrumentVelocityRegion)
            {
                return CreateBinaryFile(item as Nw4rInstrumentVelocityRegion, filePath);
            }

            throw new Nw4rFileFormatInternalException("failed to create binary file. ( invalid item type )");
        }

        #endregion

        #region ** 名前の取得

        public override string GetSourceName(Nw4rWaveSound waveSound)
        {
            if (null == waveSound) { return string.Empty; }

            switch (waveSound.XmlData.FileFormat)
            {
                case Nw4rSoundSetWaveSoundFileFormat.Pcm16:
                    return FileManager.MakeRelativePath(waveSound.XmlData.FullPath) +
                            Nw4rSoundFileManager.SourceNameFormatSeparator + "pcm16" +
                            Nw4rSoundFileManager.SourceNameFormatSeparator + "wave";

                case Nw4rSoundSetWaveSoundFileFormat.Pcm8:
                    return FileManager.MakeRelativePath(waveSound.XmlData.FullPath) +
                            Nw4rSoundFileManager.SourceNameFormatSeparator + "pcm8" +
                            Nw4rSoundFileManager.SourceNameFormatSeparator + "wave";

                case Nw4rSoundSetWaveSoundFileFormat.Adpcm:
                    return FileManager.MakeRelativePath(waveSound.XmlData.FullPath) +
                            Nw4rSoundFileManager.SourceNameFormatSeparator + "adpcm" +
                            Nw4rSoundFileManager.SourceNameFormatSeparator + "wave";
            }

            throw new Nw4rFileFormatInternalException("invalid wave sound file format");
        }

        public string GetSourceName(Nw4rInstrumentVelocityRegion velocityRegion)
        {
            if (null == velocityRegion) { return string.Empty; }

            switch (velocityRegion.XmlData.FileFormat)
            {
                case Nw4rXmlBankPcmFileFormat.Pcm16:
                    return FileManager.MakeRelativePath(velocityRegion.FilePath) +
                          Nw4rSoundFileManager.SourceNameFormatSeparator + "pcm16" +
                          Nw4rSoundFileManager.SourceNameFormatSeparator + "wave";

                case Nw4rXmlBankPcmFileFormat.Pcm8:
                    return FileManager.MakeRelativePath(velocityRegion.FilePath) +
                         Nw4rSoundFileManager.SourceNameFormatSeparator + "pcm8" +
                         Nw4rSoundFileManager.SourceNameFormatSeparator + "wave";

                case Nw4rXmlBankPcmFileFormat.Adpcm:
                    return FileManager.MakeRelativePath(velocityRegion.FilePath) +
                         Nw4rSoundFileManager.SourceNameFormatSeparator + "adpcm" +
                         Nw4rSoundFileManager.SourceNameFormatSeparator + "wave";
            }

            throw new Nw4rFileFormatInternalException("invalid bank wave file format");
        }

        protected override string GetSourceNameInternal(Nw4rItem item)
        {
            if (item is Nw4rWaveSound)
            {
                return GetSourceName(item as Nw4rWaveSound);
            }
            else if (item is Nw4rInstrumentVelocityRegion)
            {
                return GetSourceName(item as Nw4rInstrumentVelocityRegion);
            }

            throw new Nw4rFileFormatInternalException("failed to get source name. ( invalid item type )");
        }

        private string GetBinaryFilePath(Nw4rWaveSound waveSound)
        {
            if (null == waveSound) { return string.Empty; }

            return FileManager.MakeRelativePath(
                        Path.Combine(Path.GetDirectoryName(waveSound.XmlData.FullPath), GetBinaryFileName(waveSound)));
        }

        private string GetBinaryFilePath(Nw4rInstrumentVelocityRegion velocityRegion)
        {
            if (null == velocityRegion) { return string.Empty; }

            return FileManager.MakeRelativePath(
                        Path.Combine(Path.GetDirectoryName(velocityRegion.FilePath), GetBinaryFileName(velocityRegion)));
        }

        private string GetBinaryFileName(Nw4rWaveSound waveSound)
        {
            if (null == waveSound) { return string.Empty; }

            switch (waveSound.XmlData.FileFormat)
            {
                case Nw4rSoundSetWaveSoundFileFormat.Pcm16:
                    return Path.GetFileNameWithoutExtension(waveSound.XmlData.FilePath) + ".pcm16.brwav";

                case Nw4rSoundSetWaveSoundFileFormat.Pcm8:
                    return Path.GetFileNameWithoutExtension(waveSound.XmlData.FilePath) + ".pcm8.brwav";

                case Nw4rSoundSetWaveSoundFileFormat.Adpcm:
                    return Path.GetFileNameWithoutExtension(waveSound.XmlData.FilePath) + ".adpcm.brwav";
            }

            throw new Nw4rFileFormatInternalException("invalid wave sound file format");
        }

        private string GetBinaryFileName(Nw4rInstrumentVelocityRegion velocityRegion)
        {
            if (null == velocityRegion) { return string.Empty; }

            switch (velocityRegion.XmlData.FileFormat)
            {
                case Nw4rXmlBankPcmFileFormat.Pcm16:
                    return Path.GetFileNameWithoutExtension(velocityRegion.XmlData.FilePath) + ".pcm16.brwav";

                case Nw4rXmlBankPcmFileFormat.Pcm8:
                    return Path.GetFileNameWithoutExtension(velocityRegion.XmlData.FilePath) + ".pcm8.brwav";

                case Nw4rXmlBankPcmFileFormat.Adpcm:
                    return Path.GetFileNameWithoutExtension(velocityRegion.XmlData.FilePath) + ".adpcm.brwav";
            }

            throw new Nw4rFileFormatInternalException("invalid bank wave file format");
        }

        #endregion
    }

    /// <summary>
    /// ウェーブサウンドセットファイルファクトリ
    /// </summary>
    internal class Nw4rWaveSoundSetFileFactory : Nw4rFileFactoryExForGroup<Nw4rWaveSoundSet>
    {
        public Nw4rWaveSoundSetFileFactory(Nw4rSoundFileManager fileManager) : base(fileManager) { }

        #region ** 出力の作成

        public override IOutputItem CreateBinaryOutput(DependencyManager outputOwner, Nw4rWaveSoundSet waveSoundSet)
        {
            if (null == outputOwner) { throw new ArgumentNullException("outputOwner"); }
            if (null == waveSoundSet) { throw new ArgumentNullException("waveSoundSet"); }

            IOutputItem output = outputOwner.CreateOutput();

            // BinaryFile
            // ARam BinaryFile
            if (FileManager.AggregateOutput)
            {
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryFileName(waveSoundSet)));
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryWaveArchiveFileName(waveSoundSet)));
            }
            else
            {
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryFilePath(waveSoundSet)));
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryWaveArchiveFilePath(waveSoundSet)));
            }

            // Dependencies
            RegisterDependencies(outputOwner, output, waveSoundSet);

            return output;
        }

        public override IOutputItem CreateOutputForGroup(DependencyManager outputOwner, Nw4rGroup group, Nw4rWaveSoundSet waveSoundSet)
        {
            if (null == outputOwner) { throw new ArgumentNullException("outputOwner"); }
            if (null == waveSoundSet) { throw new ArgumentNullException("waveSoundSet"); }

            IOutputItem output = outputOwner.CreateOutput();

            // BinaryFile
            if (FileManager.AggregateOutput)
            {
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryFileNameForGroup(group, waveSoundSet)));
            }
            else
            {
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryFilePathForGroup(group, waveSoundSet)));
            }

            // Dependencies
            RegisterDependencies(outputOwner, output, waveSoundSet);

            return output;
        }

        private void RegisterDependencies(DependencyManager outputOwner, IOutputItem output, Nw4rWaveSoundSet waveSoundSet)
        {
            Debug.Assert(null != outputOwner);
            Debug.Assert(null != output);
            Debug.Assert(null != waveSoundSet);

            // Dependencies
            // サウンドセットファイルに依存する
            output.Dependencies.Add(outputOwner.CreateDependencyFileInfo(
                FileManager.MakeRelativePath((waveSoundSet.Parent as Model.Nw4rSoundSet).XmlData.FilePath)));

            // ウェーブサウンドの出力に依存する
            foreach (Nw4rWaveSound waveSound in waveSoundSet.Components)
            {

                Nw4rSoundOutput itemSoundCache = FileManager.GetBinaryOutput(waveSound);
                if (null == itemSoundCache) { continue; }

                output.Dependencies.Add(outputOwner.CreateDependencyOutputItemInfo(itemSoundCache.OutputItem));

            }
        }

        #endregion

        #region ** ファイルの作成

        public override Nw4rSoundBinaryFile CreateBinaryFile(Nw4rWaveSoundSet waveSoundSet, string filePath)
        {
            Debug.Assert(null != waveSoundSet);

            string filePathWork = filePath;
            string aramFilePathWork = Path.ChangeExtension(filePath, ".brwar");

            if (null == filePathWork || null == aramFilePathWork)
            {

                filePathWork = GetBinaryFileAbsolutePath(waveSoundSet);
                aramFilePathWork = GetBinaryWaveArchiveFileAbsolutePath(waveSoundSet);

            }

            return new Nw4rWaveSoundSetBinaryFile(filePathWork, aramFilePathWork);
        }

        public override Nw4rSoundBinaryFileForGroup CreateBinaryFileForGroup(Nw4rGroup group, Nw4rWaveSoundSet waveSoundSet,
                                                                             string filePath)
        {
            Debug.Assert(null != group);
            Debug.Assert(null != waveSoundSet);

            string filePathWork = filePath;

            if (null == filePathWork)
            {
                filePathWork = GetBinaryFileAbsolutePathForGroup(group, waveSoundSet);
            }

            return new Nw4rSoundBinaryFileForGroup(filePathWork, waveSoundSet.BinaryFile);
        }

        #endregion

        #region ** 名前の取得

        public override string GetSourceName(Nw4rWaveSoundSet waveSoundSet)
        {
            if (null == waveSoundSet) { return string.Empty; }
            return waveSoundSet.Label;
        }

        public override string GetSourceNameForGroup(Nw4rGroup group, Nw4rWaveSoundSet waveSoundSet)
        {
            if (null == group) { return string.Empty; }
            return waveSoundSet.Label + "." + group.Label;
        }

        private string GetBinaryFilePath(Nw4rWaveSoundSet waveSoundSet)
        {
            if (null == waveSoundSet) { return string.Empty; }

            return FileManager.MakeRelativePath(GetBinaryFileAbsolutePath(waveSoundSet));
        }

        private string GetBinaryFilePathForGroup(Nw4rGroup group, Nw4rWaveSoundSet waveSoundSet)
        {
            if (null == waveSoundSet) { return string.Empty; }

            return FileManager.MakeRelativePath(GetBinaryFileAbsolutePathForGroup(group, waveSoundSet));
        }

        private string GetBinaryWaveArchiveFilePath(Nw4rWaveSoundSet waveSoundSet)
        {
            if (null == waveSoundSet) { return string.Empty; }

            return FileManager.MakeRelativePath(GetBinaryWaveArchiveFileAbsolutePath(waveSoundSet));
        }

        private string GetBinaryFileAbsolutePath(Nw4rWaveSoundSet waveSoundSet)
        {
            if (null == waveSoundSet) { return string.Empty; }

            return Path.GetFullPath(
                    Path.Combine(FileManager.CacheDirectoryAbsolutePath, GetBinaryFileName(waveSoundSet)));
        }

        private string GetBinaryFileAbsolutePathForGroup(Nw4rGroup group, Nw4rWaveSoundSet waveSoundSet)
        {
            if (null == waveSoundSet) { return string.Empty; }

            return Path.GetFullPath(
                    Path.Combine(FileManager.CacheDirectoryAbsolutePath, GetBinaryFileNameForGroup(group, waveSoundSet)));
        }

        private string GetBinaryWaveArchiveFileAbsolutePath(Nw4rWaveSoundSet waveSoundSet)
        {
            if (null == waveSoundSet) { return string.Empty; }

            return Path.GetFullPath(
                    Path.Combine(FileManager.CacheDirectoryAbsolutePath, GetBinaryWaveArchiveFileName(waveSoundSet)));
        }

        private string GetBinaryFileName(Nw4rWaveSoundSet waveSoundSet)
        {
            if (null == waveSoundSet) { return string.Empty; }
            return waveSoundSet.Label + ".brwsd";
        }

        private string GetBinaryFileNameForGroup(Nw4rGroup group, Nw4rWaveSoundSet waveSoundSet)
        {
            if (null == group) { return string.Empty; }
            return waveSoundSet.Label + "." + group.Label + ".brwsd";
        }

        private string GetBinaryWaveArchiveFileName(Nw4rWaveSoundSet waveSoundSet)
        {
            if (null == waveSoundSet) { return string.Empty; }
            return waveSoundSet.Label + ".brwar";
        }

        #endregion
    }

    /// <summary>
    /// シーケンスサウンドファイルファクトリ
    /// </summary>
    internal class Nw4rSequenceSoundFileFactory : Nw4rFileFactory<Nw4rSequenceSound>
    {
        public Nw4rSequenceSoundFileFactory(Nw4rSoundFileManager fileManager) : base(fileManager) { }

        #region ** 出力の作成

        public override IOutputItem CreateBinaryOutput(DependencyManager outputOwner, Nw4rSequenceSound sequenceSound)
        {
            if (null == outputOwner) { throw new ArgumentNullException("outputOwner"); }
            if (null == sequenceSound) { throw new ArgumentNullException("sequenceSound"); }

            IOutputItem output = outputOwner.CreateOutput();

            // BinaryFile
            if (FileManager.AggregateOutput)
            {
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryFileName(sequenceSound)));
            }
            else
            {
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryFilePath(sequenceSound)));
            }


            // Dependencies
            // サウンドファイルに依存する
            output.Dependencies.Add(outputOwner.CreateDependencyFileInfo(
                FileManager.MakeRelativePath(sequenceSound.XmlData.FullPath)));

            // テキストシーケンスの参照ファイルに依存する
            string textSequenceFilePath = null;

            switch (sequenceSound.XmlData.FileType)
            {
                case Nw4rSoundSetSeqFileType.Smf:
                    if (null == sequenceSound.XmlIntermediateFile)
                    {
                        throw new Nw4rFileFormatInternalException("sequence intermediate file is not created.");
                    }

                    textSequenceFilePath = sequenceSound.XmlIntermediateFile.FilePath;
                    break;

                case Nw4rSoundSetSeqFileType.Text:
                    textSequenceFilePath = sequenceSound.XmlData.FullPath;
                    break;
            }

            if (null != textSequenceFilePath)
            {

                using (Lexer lexer = new Lexer(textSequenceFilePath))
                {

                    while (lexer.ReadToken() != LexerToken.End)
                    {
                        // 依存ファイル収集のため空回し
                    }

                    foreach (string dependFilePath in lexer.DependFiles)
                    {
                        if (Path.GetFullPath(textSequenceFilePath) == dependFilePath) { continue; }
                        output.Dependencies.Add(
                            outputOwner.CreateDependencyFileInfo(FileManager.MakeRelativePath(dependFilePath)));
                    }

                }

            }

            return output;
        }

        public override IOutputItem CreateIntermediateOutput(DependencyManager outputOwner, Nw4rSequenceSound sequenceSound)
        {
            if (null == outputOwner) { throw new ArgumentNullException("outputOwner"); }
            if (null == sequenceSound) { throw new ArgumentNullException("sequenceSound"); }

            string filePath = GetIntermediateFilePath(sequenceSound);
            if (null == filePath || 0 == filePath.Length) { return null; }

            IOutputItem output = outputOwner.CreateOutput();

            // IntermediateFile
            output.Outputs.Add(outputOwner.CreateOutputItem(filePath));


            // Dependencies
            // サウンドファイルに依存する
            output.Dependencies.Add(outputOwner.CreateDependencyFileInfo(
                FileManager.MakeRelativePath(sequenceSound.XmlData.FullPath)));

            return output;
        }

        #endregion

        #region ** ファイルの作成

        public override Nw4rSoundBinaryFile CreateBinaryFile(Nw4rSequenceSound sequenceSound, string filePath)
        {
            Debug.Assert(null != sequenceSound);
            ValidateSequenceSound(sequenceSound);

            string filePathWork = filePath;

            if (null == filePathWork)
            {
                filePathWork = Path.Combine(sequenceSound.XmlData.FullPath, GetBinaryFileName(sequenceSound));
            }

            return new Nw4rSequenceSoundBinaryFile(filePathWork);
        }

        public override Nw4rSoundIntermediateFile CreateIntermediateFile(Nw4rSequenceSound sequenceSound, string filePath)
        {
            Debug.Assert(null != sequenceSound);
            ValidateSequenceSound(sequenceSound);

            string filePathWork = filePath;

            if (null == filePathWork)
            {

                filePathWork = GetIntermediateFileAbsolutePath(sequenceSound);
                if (null == filePathWork || 0 == filePathWork.Length) { return null; }

            }

            return new Nw4rSoundIntermediateFile(filePathWork);
        }

        private void ValidateSequenceSound(Nw4rSequenceSound sequenceSound)
        {
            if (null == sequenceSound)
            {
                Debug.Assert(false);
                return;
            }

            // SMF なのに拡張子が RSEQ だったらエラー
            if (sequenceSound.XmlData.FileType == Nw4rSoundSetSeqFileType.Smf &&
                Path.GetExtension(sequenceSound.XmlData.FilePath) == ".rseq")
            {
                throw new Nw4rFileFormatException("file extension is invalid", sequenceSound.Label);
            }
        }

        #endregion

        #region ** 名前の取得

        public override string GetSourceName(Nw4rSequenceSound sequenceSound)
        {
            if (null == sequenceSound) { return string.Empty; }
            return FileManager.MakeRelativePath(sequenceSound.XmlData.FullPath);
        }

        private string GetBinaryFilePath(Nw4rSequenceSound sequenceSound)
        {
            if (null == sequenceSound) { return string.Empty; }

            return FileManager.MakeRelativePath(
                    Path.Combine(Path.GetDirectoryName(sequenceSound.XmlData.FullPath), GetBinaryFileName(sequenceSound)));
        }

        private string GetBinaryFileName(Nw4rSequenceSound sequenceSound)
        {
            if (null == sequenceSound) { return string.Empty; }
            return Path.GetFileNameWithoutExtension(sequenceSound.XmlData.FilePath) + ".brseq";
        }

        private string GetIntermediateFilePath(Nw4rSequenceSound sequenceSound)
        {
            string absolutePath = GetIntermediateFileAbsolutePath(sequenceSound);
            if (0 == absolutePath.Length) { return string.Empty; }
            return FileManager.MakeRelativePath(absolutePath);
        }

        private string GetIntermediateFileAbsolutePath(Nw4rSequenceSound sequenceSound)
        {
            if (null == sequenceSound) { return string.Empty; }
            if (Nw4rSoundSetSeqFileType.Smf != sequenceSound.XmlData.FileType) { return string.Empty; }
            return Path.GetFullPath(Path.ChangeExtension(sequenceSound.XmlData.FullPath, ".rseq"));
        }

        #endregion
    }

    /// <summary>
    /// バンクファイルファクトリ
    /// </summary>
    internal class Nw4rBankFileFactory : Nw4rFileFactoryExForGroup<Nw4rBank>
    {
        public Nw4rBankFileFactory(Nw4rSoundFileManager fileManager) : base(fileManager) { }

        #region ** 出力の作成

        public override IOutputItem CreateBinaryOutput(DependencyManager outputOwner, Nw4rBank bank)
        {
            if (null == outputOwner) { throw new ArgumentNullException("outputOwner"); }
            if (null == bank) { throw new ArgumentNullException("bank"); }

            IOutputItem output = outputOwner.CreateOutput();

            // BinaryFile and ARam BinaryFile
            if (FileManager.AggregateOutput)
            {
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryFileName(bank)));
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryWaveArchiveFileName(bank)));
            }
            else
            {
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryFilePath(bank)));
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryWaveArchiveFilePath(bank)));
            }

            // Include File
            output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryIncludeFilePath(bank)));


            // Dependencies
            RegisterDependencies(outputOwner, output, bank);

            return output;
        }

        public override IOutputItem CreateOutputForGroup(DependencyManager outputOwner, Nw4rGroup group, Nw4rBank bank)
        {
            if (null == outputOwner) { throw new ArgumentNullException("outputOwner"); }
            if (null == bank) { throw new ArgumentNullException("bank"); }

            IOutputItem output = outputOwner.CreateOutput();

            // BinaryFile
            if (FileManager.AggregateOutput)
            {
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryFileNameForGroup(group, bank)));
            }
            else
            {
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryFilePathForGroup(group, bank)));
            }


            // Dependencies
            RegisterDependencies(outputOwner, output, bank);

            return output;
        }

        private void RegisterDependencies(DependencyManager outputOwner, IOutputItem output, Nw4rBank bank)
        {
            Debug.Assert(null != outputOwner);
            Debug.Assert(null != output);
            Debug.Assert(null != bank);

            // Dependencies
            // バンクファイルに依存する
            output.Dependencies.Add(
                outputOwner.CreateDependencyFileInfo(FileManager.MakeRelativePath(bank.XmlData.FullPath)));

            // 各ベロシティリージョンの出力に依存する
            foreach (Nw4rInstrument instrument in bank.Components)
            {
                foreach (Nw4rInstrumentKeyRegion keyRegion in instrument.Components)
                {
                    foreach (Nw4rInstrumentVelocityRegion velocityRegion in keyRegion.Components)
                    {

                        Nw4rSoundOutput waveCache = FileManager.GetBinaryOutput(velocityRegion);
                        if (null == waveCache) { continue; }

                        output.Dependencies.Add(outputOwner.CreateDependencyOutputItemInfo(waveCache.OutputItem));

                    }
                }
            }
        }

        #endregion

        #region ** ファイルの作成

        public override Nw4rSoundBinaryFile CreateBinaryFile(Nw4rBank bank, string filePath)
        {
            Debug.Assert(null != bank);

            string filePathWork = filePath;
            string aramFilePathWork = null;
            string includeFilePathWork = null;

            if (null == filePathWork)
            {

                string outputDirectoryWork = Path.GetDirectoryName(bank.XmlData.FullPath);

                filePathWork = Path.Combine(outputDirectoryWork, GetBinaryFileName(bank));
                aramFilePathWork = Path.Combine(outputDirectoryWork, GetBinaryWaveArchiveFileName(bank));
                includeFilePathWork = Path.Combine(outputDirectoryWork, GetBinaryIncludeFilePath(bank));

            }
            else
            {
                aramFilePathWork = Path.ChangeExtension(filePath, ".brwar");
                includeFilePathWork = GetBinaryIncludeFileAbsolutePath(bank);
            }

            return new Nw4rBankBinaryFile(filePathWork, aramFilePathWork, includeFilePathWork);
        }

        public override Nw4rSoundBinaryFileForGroup CreateBinaryFileForGroup(Nw4rGroup group, Nw4rBank bank, string filePath)
        {
            Debug.Assert(null != group);
            Debug.Assert(null != bank);

            string filePathWork = filePath;

            if (null == filePathWork)
            {
                string outputDirectoryWork = Path.GetDirectoryName(bank.XmlData.FullPath);
                filePathWork = Path.Combine(outputDirectoryWork, GetBinaryFileNameForGroup(group, bank));
            }

            return new Nw4rSoundBinaryFileForGroup(filePathWork, bank.BinaryFile);
        }

        #endregion

        #region ** 名前の取得

        public override string GetSourceName(Nw4rBank bank)
        {
            if (null == bank) { return string.Empty; }
            return FileManager.MakeRelativePath(bank.XmlData.FullPath);
        }

        public override string GetSourceNameForGroup(Nw4rGroup group, Nw4rBank bank)
        {
            if (null == group) { return string.Empty; }
            return bank.Label + "." + group.Label;
        }

        private string GetBinaryFilePath(Nw4rBank bank)
        {
            if (null == bank) { return string.Empty; }

            return FileManager.MakeRelativePath(
                    Path.Combine(Path.GetDirectoryName(bank.XmlData.FullPath), GetBinaryFileName(bank)));
        }

        private string GetBinaryFilePathForGroup(Nw4rGroup group, Nw4rBank bank)
        {
            if (null == group) { return string.Empty; }
            if (null == bank) { return string.Empty; }

            return FileManager.MakeRelativePath(
                    Path.Combine(Path.GetDirectoryName(bank.XmlData.FullPath), GetBinaryFileNameForGroup(group, bank)));
        }

        private string GetBinaryWaveArchiveFilePath(Nw4rBank bank)
        {
            if (null == bank) { return string.Empty; }

            return FileManager.MakeRelativePath(
                    Path.Combine(Path.GetDirectoryName(bank.XmlData.FullPath), GetBinaryWaveArchiveFileName(bank)));
        }

        private string GetBinaryFileName(Nw4rBank bank)
        {
            if (null == bank) { return string.Empty; }
            return Path.GetFileNameWithoutExtension(bank.XmlData.FilePath) + ".brbnk";
        }

        private string GetBinaryFileNameForGroup(Nw4rGroup group, Nw4rBank bank)
        {
            if (null == group) { return string.Empty; }
            if (null == bank) { return string.Empty; }
            return Path.GetFileNameWithoutExtension(bank.XmlData.FilePath) + "." + group.Label + ".brbnk";
        }

        private string GetBinaryWaveArchiveFileName(Nw4rBank bank)
        {
            if (null == bank) { return string.Empty; }
            return Path.GetFileNameWithoutExtension(bank.XmlData.FilePath) + ".brwar";
        }

        private string GetBinaryIncludeFilePath(Nw4rBank bank)
        {
            if (null == bank) { return string.Empty; }
            return FileManager.MakeRelativePath(GetBinaryIncludeFileAbsolutePath(bank));
        }

        private string GetBinaryIncludeFileAbsolutePath(Nw4rBank bank)
        {
            if (null == bank) { return string.Empty; }
            return Path.GetFullPath(Path.ChangeExtension(bank.XmlData.FullPath.Replace('/', '\\'), "rinl"));
        }

        #endregion
    }

    /// <summary>
    /// グループファイルファクトリ
    /// </summary>
    internal class Nw4rGroupFileFactory : Nw4rFileFactory<Nw4rGroup>
    {
        public Nw4rGroupFileFactory(Nw4rSoundFileManager fileManager) : base(fileManager) { }

        #region ** 出力の作成

        public override IOutputItem CreateBinaryOutput(DependencyManager outputOwner, Nw4rGroup group)
        {
            IOutputItem output = outputOwner.CreateOutput();

            // BinaryFile
            if (FileManager.AggregateOutput)
            {
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryFileName(group)));
            }
            else
            {
                output.Outputs.Add(outputOwner.CreateOutputItem(GetBinaryFilePath(group)));
            }

            // Dependencies
            // サウンドセットファイルに依存する
            output.Dependencies.Add(outputOwner.CreateDependencyFileInfo(
                FileManager.MakeRelativePath((group.Parent as Model.Nw4rSoundSet).XmlData.FilePath)));

            // グループアイテムの出力に依存する
            foreach (Model.Nw4rSoundSetItem item in group.AllDependItems)
            {

                Nw4rSoundOutput itemSoundCache = FileManager.GetBinaryOutputForGroup(group, item);
                if (null == itemSoundCache) { continue; }

                output.Dependencies.Add(outputOwner.CreateDependencyOutputItemInfo(itemSoundCache.OutputItem));

            }

            return output;
        }

        #endregion

        #region ** ファイルの作成

        public override Nw4rSoundBinaryFile CreateBinaryFile(Nw4rGroup group, string filePath)
        {
            Debug.Assert(null != group);

            string filePathWork = filePath;

            if (null == filePathWork)
            {
                filePathWork = GetBinaryFileAbsolutePath(group);
            }

            return new Nw4rGroupBinaryFile(filePathWork);
        }

        #endregion

        #region ** 名前の取得

        public override string GetSourceName(Nw4rGroup group)
        {
            if (null == group) { return string.Empty; }
            return group.Label;
        }

        private string GetBinaryFilePath(Nw4rGroup group)
        {
            if (null == group) { return string.Empty; }
            return FileManager.MakeRelativePath(GetBinaryFileAbsolutePath(group));
        }

        private string GetBinaryFileAbsolutePath(Nw4rGroup group)
        {
            if (null == group) { return string.Empty; }

            return Path.GetFullPath(Path.Combine(FileManager.CacheDirectoryAbsolutePath, GetBinaryFileName(group)));
        }

        private string GetBinaryFileName(Nw4rGroup group)
        {
            if (null == group) { return string.Empty; }
            return group.Label + ".brwar";
        }

        #endregion
    }

    #endregion
}
