﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Globalization;
using System.Reflection;
using System.IO;
using System.Text;
using System.Threading;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Converter
{
    #region ** コマンド基本クラス

    internal abstract class Nw4rCommand
    {
        private Nw4rCommandProcessor _owner = null;

        public Nw4rCommand(Nw4rCommandProcessor owner)
        {
            //			Debug.Assert( null != owner );
            _owner = owner;
        }

        #region ** プロパティ

        public Nw4rCommandProcessor Owner
        {
            get { return _owner; }
        }

        #endregion

        #region ** イベント

        public event Nw4rCommandEventHandler Completed;
        public event Nw4rCommandOutputLineEventHandler OutputLine;

        #endregion

        #region ** イベントハンドラ

        protected abstract void OnInvoke();

        protected virtual void OnCompleted(Nw4rCommandEventArgs e)
        {
            if (null != Completed)
            {
                Completed(this, e);
            }
        }

        protected virtual void OnOutputLine(Nw4rCommandOutputLineEventArgs e)
        {
            if (null != OutputLine)
            {
                OutputLine(this, e);
            }
        }

        #endregion

        #region ** メソッド

        public virtual void Invoke()
        {
            try
            {

                OnInvoke();

                OnCompleted(new Nw4rCommandEventArgs(this));

            }
            catch (Exception exception)
            {
                OnCompleted(new Nw4rCommandEventArgs(this, exception));
                throw exception;
            }
        }

        protected void OutputLineInternal(OutputLine outputLine)
        {
            if (null == outputLine)
            {
                Debug.Assert(false);
                return;
            }

            OnOutputLine(new Nw4rCommandOutputLineEventArgs(this, outputLine));
        }

        protected void CreateDirectory(string directoryPath)
        {
            CreateDirectory(directoryPath, string.Empty);
        }

        protected void CreateDirectory(string directoryPath, string ownerLabel)
        {
            if (null == directoryPath) { throw new ArgumentNullException("directoryPath"); }
            if (null == ownerLabel) { throw new ArgumentNullException("ownerLabel"); }

            try
            {
                Directory.CreateDirectory(directoryPath);
            }
            catch (FileNotFoundException exception)
            {
                throw new Nw4rFileNotFoundException(exception.FileName, ownerLabel);
            }
            catch (DirectoryNotFoundException exception)
            {
                throw new Nw4rFileFormatException(exception.Message, ownerLabel);
            }
            catch (UnauthorizedAccessException)
            {
                throw new Nw4rFileAccessDeniedException(directoryPath, ownerLabel);
            }
            catch (PathTooLongException)
            {
                throw new Nw4rFilePathTooLongException(directoryPath, ownerLabel);
            }
        }

        #endregion
    }

    #region ** 例外

    internal class Nw4rCommandException : Exception
    {
        private bool _exitProcess = false;

        public Nw4rCommandException() : this(string.Empty, true, null) { }
        public Nw4rCommandException(string message) : this(message, true, null) { }
        public Nw4rCommandException(string message, bool exitProcess) : this(message, exitProcess, null) { }
        public Nw4rCommandException(string message, bool exitProcess, Exception innerException) : base(message, innerException)
        {
            _exitProcess = exitProcess;
        }

        #region ** プロパティ

        /// <summary>
        /// コマンド処理を終了するかどうかを取得します。
        /// </summary>
        public bool ExitProcess { get { return _exitProcess; } }

        #endregion
    }

    #endregion

    internal class Nw4rCommandProcessor
    {
        private Nw4rCommandQueue _commands = new Nw4rCommandQueue();

        private Thread _thread = null;
        private ManualResetEvent _endThreadEvent = new ManualResetEvent(true);
        private bool _endThreadFlag = true;
        private Nw4rCommand _currentCommand = null;
        private Exception _lastException = null;

        #region ** プロパティ

        public Exception LastException
        {
            get { return _lastException; }
        }

        #endregion

        #region ** イベント

        public event Nw4rCommandEventHandler CurrentCommandChanged;
        public event Nw4rCommandEventHandler CommandCompleted;
        public event Nw4rCommandEventHandler ProcessCompleted;

        #endregion

        #region ** イベントハンドラ

        protected virtual void OnCurrentCommandChanged(Nw4rCommandEventArgs e)
        {
            if (null == CurrentCommandChanged) { return; }
            CurrentCommandChanged(this, e);
        }

        protected virtual void OnCommandCompleted(Nw4rCommandEventArgs e)
        {
            if (null == CommandCompleted) { return; }
            CommandCompleted(this, e);
        }

        protected virtual void OnProcessCompleted(Nw4rCommandEventArgs e)
        {
            if (null == ProcessCompleted) { return; }
            ProcessCompleted(this, e);
        }

        #endregion

        #region ** メソッド

        public void Begin()
        {
            Begin(null);
        }

        public void Begin(object userData)
        {
            if (null != _thread) { throw new Exception(); }

            _thread = new Thread(new ParameterizedThreadStart(this.CommandThread));
            Debug.Assert(null != _thread);

            _endThreadFlag = false;
            _endThreadEvent.Reset();

            _thread.Start(userData);
        }

        public void End()
        {
            _endThreadFlag = true;

            if (null != _thread)
            {
                _endThreadEvent.WaitOne(5000, false);
                _thread = null;
            }
        }

        public bool Wait()
        {
            return Wait(Timeout.Infinite);
        }

        public bool Wait(int timeout)
        {
            if (null == _thread) { return true; }
            return _endThreadEvent.WaitOne(timeout, false);
        }

        public void Push(Nw4rCommand command)
        {
            if (null == command) { throw new ArgumentNullException("comamnd"); }

            using (AutoLock commandsLock = new AutoLock(_commands))
            {
                _commands.Enqueue(command);
            }
        }

        private Nw4rCommand Peek()
        {
            using (AutoLock commandsLock = new AutoLock(_commands))
            {

                if (0 == _commands.Count) { return null; }
                return _commands.Peek();

            }
        }

        private Nw4rCommand Pop()
        {
            using (AutoLock commandsLock = new AutoLock(_commands))
            {
                if (0 == _commands.Count) { return null; }
                return _commands.Dequeue();
            }
        }

        #endregion

        #region ** コマンド処理スレッド

        protected virtual void OnBeginCommandThread(object data) { }
        protected virtual void OnEndCommandThread(Exception e, object data) { }

        protected virtual void OnPreInvokeCommand(Nw4rCommand command) { }
        protected virtual void OnPostInvokeCommand(Nw4rCommand command) { }

        protected virtual void OnCommandThread(object data)
        {
            while (true)
            {

                _currentCommand = Peek();
                if (null == _currentCommand) { break; }
                if (_endThreadFlag) { break; }

                try
                {

                    OnPreInvokeCommand(_currentCommand);
                    _currentCommand.Invoke();

                }
                catch (Exception exception)
                {
                    _lastException = exception;
                    break;
                    //if( exception is Nw4rCommandException && ( exception as Nw4rCommandException ).ExitProcess ) { break; }
                }
                finally
                {
                    OnPostInvokeCommand(_currentCommand);
                    OnCommandCompleted(new Nw4rCommandEventArgs(_currentCommand, _lastException));
                }

                Pop();

            }

            _endThreadFlag = true;
            OnProcessCompleted(new Nw4rCommandEventArgs(_currentCommand, _lastException));
        }

        private void CommandThread(object data)
        {
            if (Directory.Exists(
                    Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "en")))
            {
                Thread.CurrentThread.CurrentUICulture = new CultureInfo("en");
            }

            OnBeginCommandThread(data);

            OnCommandThread(data);

            OnEndCommandThread(_lastException, data);

            _endThreadEvent.Set();
        }

        #endregion

        #region ** コマンドキュー

        /// <summary>
        /// コマンドロッククラス
        /// </summary>
        private class AutoLock : IDisposable
        {
            #region ** パラメータ

            private Nw4rCommandQueue _target;
            private bool _disposed = false;

            #endregion

            public AutoLock(Nw4rCommandQueue target)
            {
                Debug.Assert(null != target);

                _target = target;
                _target.Lock();
            }

            #region ** メソッド

            private void Dispose(bool disposing)
            {
                if (_disposed) { return; }

                Debug.Assert(null != _target);
                _target.Unlock();

                _disposed = true;
            }

            #endregion

            #region ** IDisposable の実装

            void IDisposable.Dispose()
            {
                Dispose(true);
                GC.SuppressFinalize(this);
            }

            #endregion
        }

        /// <summary>
        /// コマンドキュー
        /// </summary>
        private class Nw4rCommandQueue : Queue<Nw4rCommand>
        {
            #region ** パラメータ

            private Object _lock = new Object();

            #endregion

            #region ** メソッド

            public void Lock()
            {
                Monitor.Enter(_lock);
            }

            public void Unlock()
            {
                Monitor.Exit(_lock);
            }

            #endregion
        }

        #endregion
    }

    #endregion

    #region ** コンバートコマンド

    internal abstract class Nw4rConvertCommand : Nw4rCommand
    {
        private object _input = null;		// 入力オブジェクト
        private object _output = null;		// 出力オブジェクト
        private bool _silent = false;		// テキストを出力しない
        private bool _forcedConvert = false;		// コンバートの強制実行設定

        private Util.ProcessHandler _processHandler = null;

        public Nw4rConvertCommand(object input, object output) : base(null)
        {
            _input = input;
            _output = output;
        }

        #region ** プロパティ

        public object Input
        {
            get { return _input; }
        }

        public object Output
        {
            get { return _output; }
        }

        public bool Silent
        {
            get { return _silent; }
            set { _silent = value; }
        }

        public bool ForcedConvert
        {
            get { return _forcedConvert; }
            set { _forcedConvert = value; }
        }

        public Util.ProcessHandler ProcessHandler
        {
            get { return _processHandler; }
            set { _processHandler = value; }
        }

        protected virtual bool ShouldConvert
        {
            get { return true; }
        }

        #endregion

        #region ** イベントハンドラ

        protected virtual void OnClean() { }

        #endregion

        #region ** メソッド

        public override void Invoke()
        {
            if (null == _input) { throw new Exception("Input is null."); }
            if (null == _output) { throw new Exception("Output is null."); }

            if (!ForcedConvert)
            {
                if (!ShouldConvert)
                {
                    OnCompleted(new Nw4rCommandEventArgs(this));
                    return;
                }
            }

            base.Invoke();
        }

        public void Clean()
        {
            OnClean();
        }

        #endregion
    }

    internal abstract class Nw4rConvertCommandT<_InputType, _OutputType> : Nw4rConvertCommand
        where _InputType : class
        where _OutputType : class
    {
        public Nw4rConvertCommandT(_InputType input, _OutputType output) : base(input, output) { }

        #region ** プロパティ

        public new _InputType Input
        {
            get { return base.Input as _InputType; }
        }

        public new _OutputType Output
        {
            get { return base.Output as _OutputType; }
        }

        #endregion
    }

    internal class NW4rEmptyCommand : Nw4rConvertCommand
    {
        public NW4rEmptyCommand() : base(null, null) { }

        protected override void OnInvoke() { }
    }

    internal class Nw4rConvertCommandProcessor : Nw4rCommandProcessor
    {
        #region ** イベント

        public event Nw4rCommandOutputLineEventHandler OutputLine;

        #endregion

        #region ** イベントハンドラ

        protected override void OnPreInvokeCommand(Nw4rCommand command)
        {
            base.OnPreInvokeCommand(command);

            command.OutputLine += OnCommandOutputLine;
        }

        protected override void OnPostInvokeCommand(Nw4rCommand command)
        {
            base.OnPostInvokeCommand(command);

            command.OutputLine -= OnCommandOutputLine;
        }

        protected override void OnCommandCompleted(Nw4rCommandEventArgs e)
        {
            if (null != e.Exception)
            {
                (e.command as Nw4rConvertCommand).Clean();
            }

            base.OnCommandCompleted(e);
        }

        protected virtual void OnOutputLine(Nw4rCommandOutputLineEventArgs e)
        {
            if (null == OutputLine) { return; }
            OutputLine(this, e);
        }

        private void OnCommandOutputLine(object sender, Nw4rCommandOutputLineEventArgs e)
        {
            OnOutputLine(e);
        }

        #endregion
    }

    #endregion

    #region ** イベントデリゲートとパラメータ

    internal delegate void Nw4rCommandEventHandler(object sender, Nw4rCommandEventArgs e);
    internal delegate void Nw4rCommandOutputLineEventHandler(object sender, Nw4rCommandOutputLineEventArgs e);

    internal class Nw4rCommandEventArgs : EventArgs
    {
        #region ** パラメータ

        Nw4rCommand _command;
        Exception _exception;

        #endregion

        public Nw4rCommandEventArgs(Nw4rCommand command) : this(command, null) { }

        public Nw4rCommandEventArgs(Nw4rCommand command, Exception e)
        {
            _command = command;
            _exception = e;
        }

        #region ** プロパティ

        public Nw4rCommand command
        {
            get { return _command; }
        }

        public Exception Exception
        {
            get { return _exception; }
        }

        public bool Failed
        {
            get { return (null != _exception); }
        }

        #endregion
    }

    internal class Nw4rCommandOutputLineEventArgs : Nw4rCommandEventArgs
    {
        private OutputLine _information = null;

        public Nw4rCommandOutputLineEventArgs(Nw4rCommand command, OutputLine information) : base(command)
        {
            _information = information;
        }

        #region ** プロパティ

        public OutputLine OutputInformation
        {
            get { return _information; }
        }

        #endregion
    }

    #endregion
}
