﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Collections;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Midi
{

    public class SmfReaderException : MidiException
    {
        public SmfReaderException(string s) : base(s) { }
    }

    /// <summary>
    /// SmfReader の概要の説明です。
    /// </summary>
    public class SmfReader : IDisposable
    {
        FileStream strm;

        class TrackReader
        {
            BinaryReader reader;

            byte runningStatus;
            ulong curTick = 0;

            public TrackReader(byte[] trackChunk)
            {
                MemoryStream strm = new MemoryStream(trackChunk);
                reader = BinaryReaderBigEndian.CreateInstance(strm, new System.Text.ASCIIEncoding());
            }

            ulong ReadSmfLen()
            {
                ulong len = 0;

                for (int i = 0; ; i++)
                {
                    if (i >= 4)
                    {
                        throw new SmfReaderException("Illegal format");
                    }

                    byte b = reader.ReadByte();
                    len <<= 7;
                    len |= (byte)(b & 0x7f);
                    if ((b & 0x80) == 0) break;
                }

                return len;
            }

            SmfMidiEvent ReadMidiEvent(byte status)
            {
                SmfMidiEvent midiEvent = new SmfMidiEvent(curTick);

                return midiEvent;
            }

            SmfMetaEvent ReadMetaEvent(byte status)
            {
                SmfMetaEvent metaEvent = new SmfMetaEvent(curTick);

                return metaEvent;
            }

            SmfSysExEvent ReadSysExEvent(byte status)
            {
                SmfSysExEvent sysExEvent = new SmfSysExEvent(curTick);

                return sysExEvent;
            }

            public SmfEvent ReadEvent()
            {
                ulong delta = ReadSmfLen();
                curTick += delta;

                byte status = reader.ReadByte();

                byte data1;
                if ((status & 0x80) == 0)
                {
                    data1 = status;
                    status = runningStatus;
                }
                else
                {
                    runningStatus = status;
                }

                if ((status & 0xf0) != 0xf0)
                {
                    // MIDIイベント
                    return ReadMidiEvent(status);
                }
                else if (status == 0xff)
                {
                    // メタイベント
                    return ReadMetaEvent(status);
                }
                else if (status == 0xf0)
                {
                    // SysExイベント
                    return ReadSysExEvent(status);
                }
                else if (status == 0xf7)
                {
                    // SysExイベント
                    return ReadSysExEvent(status);
                }
                else
                {
                    throw new SmfReaderException("Unknown status byte " + status);
                }
            }
        }

        void IDisposable.Dispose()
        {
            strm.Close();
        }

        public Smf Open(string filePath)
        {
            Smf smf = new Smf();

            strm = File.OpenRead(filePath);
            BinaryReader reader = BinaryReaderBigEndian.CreateInstance(strm, new System.Text.ASCIIEncoding());

            while (strm.Position < strm.Length)
            {
                string chunkType = new string(reader.ReadChars(4));
                UInt32 chunkSize = reader.ReadUInt32();
                long chunkEnd = strm.Position + chunkSize;
                long nextChunkPos = ((chunkEnd + 0x01) & ~0x01); // 2バイトアライン

                if (chunkType == "MThd")
                {
                    smf.HeaderChunk.Read(reader);
                }
                else if (chunkType == "MTrk")
                {
                    TrackReader trackReader = new TrackReader(reader.ReadBytes((int)chunkSize));

                    SmfTrack track = new SmfTrack();

                    SmfEvent smfEvent;
                    while ((smfEvent = trackReader.ReadEvent()) != null)
                    {
                        track.AppendEvent(smfEvent);
                    }

                    smf.Tracks.Add(track);
                }
                else
                {
                    // unknown chunk
                }

                strm.Position = nextChunkPos;
            }

            return smf;
        }

        public void Close()
        {
            strm.Close();
        }

    }

}
