﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat
{
    /// <summary>
    /// 依存関係管理クラス
    /// </summary>
    public class DependencyManager : IXmlSerializable
    {
        #region ** 固定値

        // 要素名
        private const string RootElementName = "NintendoSoundDependencyRoot";
        private const string ElementName = "DependencyManager";
        private const string OutputItemsElementName = "OutputItems";
        private const string UserKeyMapsElementName = "UserKeyMaps";
        private const string UserKeyMapElementName = "UserKeyMap";
        private const string UserKeyMapItemElementName = "UserKeyMapItem";
        private const string UserParametersElementName = "UserParameters";
        private const string UserParameterElementName = "UserParameter";

        // 属性名
        private const string VersionAttributeName = "Version";
        private const string BaseDirectoryPathAttributeName = "BaseDirectoryPath";
        private const string IDAttributeName = "ID";
        private const string KeyAttributeName = "Key";

        // バージョン
        private static readonly Version Version = new Version(1, 0);

        #endregion

        #region ** フィールド

        // コレクション
        private OutputCollection _outputItems = new OutputCollection();			// 出力コレクション
        private UserParameterDictionary _userParameters = new UserParameterDictionary();	// ユーザパラメータコレクション

        private string _managementFilePath = string.Empty;	// 管理ファイルのパス
        private string _baseDirectoryPath = string.Empty;	// ベースディレクトリパス

        #endregion

        #region ** プロパティ

        /// <summary>
        /// 管理ファイルのパスを取得または設定します。
        /// </summary>
        public string ManagementFilePath
        {
            get { return _managementFilePath; }
            set { _managementFilePath = value; }
        }

        /// <summary>
        /// 出力ファイルのベースディレクトリパスを取得または設定します。
        /// </summary>
        public string BaseDirectoryPath
        {
            get { return _baseDirectoryPath; }
            set { _baseDirectoryPath = value; }
        }

        /// <summary>
        /// 出力ファイルのベースディレクトリパスを取得または設定します。
        /// </summary>
        public string BaseDirectoryAbsolutePath
        {
            get
            {
                if (null == _baseDirectoryPath) { return null; }

                if (Path.IsPathRooted(_baseDirectoryPath)) { return _baseDirectoryPath; }

                if (null == _managementFilePath) { return null; }
                if (0 == _managementFilePath.Length) { return string.Empty; }

                return Path.Combine(Path.GetDirectoryName(_managementFilePath), _baseDirectoryPath);
            }
        }

        /// <summary>
        /// 出力アイテムコレクションを取得します。
        /// </summary>
        public IOutputItemCollection OutputItems
        {
            get { return _outputItems; }
        }

        public UserParameterDictionary UserParameters
        {
            get { return _userParameters; }
        }

        #endregion

        #region ** メソッド

        #region ** 出力の追加と削除

        /// <summary>
        /// 新しい出力を追加します。
        /// </summary>
        /// <param name="item">追加する出力。</param>
        public void Add(IOutputItem output)
        {
            if (null == output) { throw new ArgumentNullException("output"); }
            AddInternal(output);
        }

        /// <summary>
        /// 出力を削除します。
        /// </summary>
        /// <param name="output">削除する出力。</param>
        public void Remove(IOutputItem output)
        {
            if (null == output) { throw new ArgumentNullException("output"); }

            if (!_outputItems.Contains(output)) { return; }

            _outputItems.Remove(output);
        }

        /// <summary>
        /// 既存の出力にユーザキーを設定します。
        /// </summary>
        /// <param name="userKeyMapID">ユーザキーマップID。</param>
        /// <param name="userKey">関連付けるユーザキー。</param>
        /// <param name="output">既存の出力。</param>
        public void SetUserKey(string userKeyMapID, string userKey, IOutputItem output)
        {
            if (null == userKeyMapID) { throw new ArgumentNullException("userKeyMapID"); }
            if (0 == userKeyMapID.Length) { throw new ArgumentException("userKeyMapID"); }
            if (null == userKey) { throw new ArgumentNullException("userKey"); }
            if (0 == userKey.Length) { throw new ArgumentException("userKey"); }
            if (null == output) { throw new ArgumentNullException("output"); }

            if (!_outputItems.Contains(output.Key))
            {
                throw new DependencyOutputNotFound(output.Key);
            }

            SetUserKeyInternal(userKeyMapID, userKey, output);
        }

        /// <summary>
        /// ユーザキーを削除し出力の参照カウントをデクリメントします。
        /// </summary>
        /// <param name="userKeyMapID">ユーザキーマップID。</param>
        /// <param name="userKey">ユーザキー。</param>
        public void RemoveByUserKey(string userKeyMapID, string userKey)
        {
            if (null == userKeyMapID) { throw new ArgumentNullException("userKeyMapID"); }
            if (0 == userKeyMapID.Length) { throw new ArgumentException("userKeyMapID"); }
            if (null == userKey) { throw new ArgumentNullException("userKey"); }
            if (0 == userKey.Length) { throw new ArgumentException("userKey"); }

            RemoveByUserKeyInternal(userKeyMapID, userKey);
        }

        /// <summary>
        /// 新しい出力を追加します。
        /// </summary>
        /// <param name="output">追加する出力。</param>
        private void AddInternal(IOutputItem output)
        {
            Debug.Assert(null != output);
            _outputItems.Add(output);
        }

        /// <summary>
        /// 既存の出力にユーザキーを設定します。
        /// </summary>
        /// <param name="userKeyMapID">ユーザキーマップID。</param>
        /// <param name="userKey">関連付けるユーザキー。</param>
        /// <param name="outputItem">既存の出力。</param>
        private void SetUserKeyInternal(string userKeyMapID, string userKey, IOutputItem output)
        {
            Debug.Assert(null != userKeyMapID);
            Debug.Assert(null != userKey);
            Debug.Assert(null != output);
            _outputItems.SetUserKey(userKeyMapID, userKey, output);
        }

        /// <summary>
        /// ユーザキーを削除し出力の参照カウントをデクリメントします。
        /// </summary>
        /// <param name="userKeyMapID">ユーザキーマップID。</param>
        /// <param name="userKey">ユーザキー。</param>
        private void RemoveByUserKeyInternal(string userKeyMapID, string userKey)
        {
            Debug.Assert(null != userKeyMapID);
            Debug.Assert(null != userKey);
            _outputItems.RemoveByUserKey(userKeyMapID, userKey);
        }

        #endregion

        #region ** クリーン

        /// <summary>
        /// 出力ファイルを全て削除します。
        /// </summary>
        public void Clean()
        {
            // 出力ファイルを削除する
            foreach (OutputItem output in _outputItems.Values)
            {
                output.Clean();
            }

            _outputItems.Clear();

            // DependencyManager管理ファイルを削除する
            try
            {
                File.Delete(_managementFilePath);
            }
            catch (Exception)
            {
            }
        }

        #endregion

        #region ** オブジェクトの生成

        /// <summary>
        /// 出力のインスタンスを作成します。
        /// </summary>
        /// <returns>新しい出力のインスタンス</returns>
        public IOutputItem CreateOutput()
        {
            return new OutputItem(this, Guid.NewGuid().ToString());
        }

        /// <summary>
        /// アイテムのインスタンスを作成します。
        /// </summary>
        /// <param name="baseName">アイテムのベースネーム</param>
        /// <returns>新しいアイテムのインスタンス</returns>
        public IOutputInfo CreateOutputItem(string baseName)
        {
            return new OutputInfo(this, CreateOutputItemName(baseName));
        }

        /// <summary>
        /// アイテム名を作成します。
        /// </summary>
        /// <param name="baseName">アイテムのベースネーム</param>
        /// <returns>新しいアイテム名</returns>
        public string CreateOutputItemName(string baseName)
        {
            if (null == baseName) { throw new ArgumentNullException("baseName"); }
            if (0 == baseName.Length) { throw new ArgumentException("baseName"); }

            string directory = Path.GetDirectoryName(baseName);
            string workName = baseName;
            int count = 0;

            while (_outputItems.ContainsOutputName(workName))
            {

                count++;
                workName = Path.Combine(directory, Path.GetFileNameWithoutExtension(baseName)
                                                     + count.ToString() + Path.GetExtension(baseName));

            }

            return workName;
        }

        public IDependencyFileInfo CreateDependencyFileInfo(string filePath)
        {
            return new DependencyFileInfo(this, filePath);
        }

        public IDependencyFileInfo CreateDependencyOutputItemInfo(IOutputItem outputItem)
        {
            return CreateDependencyOutputItemInfo(outputItem.Key);
        }

        public IDependencyFileInfo CreateDependencyOutputItemInfo(string outputItemKey)
        {
            return new DependencyOutputItemInfo(this, outputItemKey);
        }

        #endregion

        #region ** ファイル入出力

        public static DependencyManager FromFile(string filePath)
        {
            DependencyManager dependencyManager = new DependencyManager();
            dependencyManager.Load(filePath);

            return dependencyManager;
        }

        public void Load(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            try
            {

                _managementFilePath = filePath;

                using (XmlReader reader = XmlTextReader.Create(filePath))
                {
                    ReadXml(reader);
                }

            }
            catch (FileNotFoundException exception)
            {
                _managementFilePath = string.Empty;
                _outputItems.Clear();
                throw exception;
            }
            catch (Exception exception)
            {
                _managementFilePath = string.Empty;
                _outputItems.Clear();
                Debug.WriteLine(exception.ToString());
                throw exception;
            }
        }

        public void Save()
        {
            if (null == _managementFilePath || 0 == _managementFilePath.Length)
            {
                throw new Exception("invalid management file path.");
            }

            Save(_managementFilePath);
        }

        public void Save(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            // ディレクトリを作成する
            string directoryPath = Path.GetDirectoryName(filePath);
            if (null != directoryPath && 0 < directoryPath.Length)
            {
                Directory.CreateDirectory(directoryPath);
            }

            XmlWriterSettings settings = new XmlWriterSettings();
            settings.Encoding = Encoding.UTF8;
            settings.Indent = true;

            using (XmlWriter writer = XmlTextWriter.Create(filePath, settings))
            {
                WriteXml(writer);
            }
        }

        #endregion

        #region ** XML 入出力

        /// <summary>
        /// このプロパティは予約されています。代わりに System.Xml.Serialization.XmlSchemaProviderAttribute をクラスに適用します。
        /// </summary>
        /// <returns>XML 表現を記述する System.Xml.Schema.XmlSchema。</returns>
        public XmlSchema GetSchema() { return null; }

        #region ** 入力

        /// <summary>
        /// オブジェクトの XML 表現からオブジェクトを生成します。
        /// </summary>
        /// <param name="reader">オブジェクトの逆シリアル化元である System.Xml.XmlReader ストリーム。</param>
        public void ReadXml(XmlReader reader)
        {
            if (null == reader) { throw new ArgumentNullException("reader"); }

            _outputItems.Clear();


            XmlDocument document = new XmlDocument();
            document.Load(reader);

            if (RootElementName != document.DocumentElement.Name)
            {
                throw new DependencyManagerInvalidFileFormatException();
            }

            if (!document.DocumentElement.HasAttribute(VersionAttributeName))
            {
                throw new DependencyManagerInvalidFileVersionException();
            }

            // バージョンチェック
            string[] versions = document.DocumentElement.GetAttribute(VersionAttributeName).Split(new char[] { '.' });
            if (2 != versions.Length ||
                Version.Major != int.Parse(versions[0]) || Version.Minor != int.Parse(versions[1]))
            {
                throw new DependencyManagerInvalidFileVersionException();
            }

            // DependencyManager
            XmlElement dependencyManagerElement = document.DocumentElement[ElementName];
            if (null == dependencyManagerElement)
            {
                throw new DependencyManagerInvalidFileFormatException();
            }

            ReadDependencyManagerElement(dependencyManagerElement);
        }

        private void ReadDependencyManagerElement(XmlElement dependencyManagerElement)
        {
            Debug.Assert(null != dependencyManagerElement);

            // BaseDirectoryPath
            if (dependencyManagerElement.HasAttribute(BaseDirectoryPathAttributeName))
            {
                _baseDirectoryPath = dependencyManagerElement.GetAttribute(BaseDirectoryPathAttributeName);
            }

            // OutputItems
            XmlElement outputItemsElement = dependencyManagerElement[OutputItemsElementName];
            if (null == outputItemsElement)
            {
                throw new DependencyManagerInvalidFileFormatException();
            }

            ReadOutputItemsElement(outputItemsElement);


            // UserKeyMaps
            XmlElement userKeyMapsElement = dependencyManagerElement[UserKeyMapsElementName];
            if (null != userKeyMapsElement)
            {
                ReadUserKeyMapsElement(userKeyMapsElement);
            }


            // UserParameters
            XmlElement userParametersElement = dependencyManagerElement[UserParametersElementName];
            if (null != userParametersElement)
            {
                ReadUserParametersElement(userParametersElement);
            }
        }

        private void ReadOutputItemsElement(XmlElement outputItemsElement)
        {
            Debug.Assert(null != outputItemsElement);

            foreach (XmlElement outputItemElement in outputItemsElement.GetElementsByTagName(OutputItem.ElementName))
            {
                OutputItem newOutputItem = OutputItem.FromXml(this, outputItemElement);
                if (null == newOutputItem) { continue; }
                AddInternal(newOutputItem);
            }

        }

        private void ReadUserKeyMapsElement(XmlElement userKeyMapsElement)
        {
            Debug.Assert(null != userKeyMapsElement);

            // UserKeyMap
            foreach (XmlElement userKeyMapElement in userKeyMapsElement.GetElementsByTagName(UserKeyMapElementName))
            {
                ReadUserKeyMapElement(userKeyMapElement);
            }
        }

        private void ReadUserKeyMapElement(XmlElement userKeyMapElement)
        {
            Debug.Assert(null != userKeyMapElement);
            if (!userKeyMapElement.HasAttribute(IDAttributeName)) { return; }

            string userKeyMapID = userKeyMapElement.GetAttribute(IDAttributeName);

            // UserKeyMapItem
            foreach (XmlElement element in userKeyMapElement.GetElementsByTagName(UserKeyMapItemElementName))
            {
                ReadUserKeyMapItemElement(element, userKeyMapID);
            }
        }

        private void ReadUserKeyMapItemElement(XmlElement userKeyMapItemElement, string userKeyMapID)
        {
            Debug.Assert(null != userKeyMapItemElement);
            if (!userKeyMapItemElement.HasAttribute(KeyAttributeName)) { return; }

            try
            {

                string userKey = userKeyMapItemElement.GetAttribute(KeyAttributeName);
                OutputItem outputItem = _outputItems[userKeyMapItemElement.InnerText] as OutputItem;

                SetUserKeyInternal(userKeyMapID, userKey, outputItem);

            }
            catch { }
        }

        private void ReadUserParametersElement(XmlElement userParametersElement)
        {
            Debug.Assert(null != userParametersElement);

            // UserParameter
            foreach (XmlElement element in userParametersElement.GetElementsByTagName(UserParameterElementName))
            {
                ReadUserParameterElement(element);
            }
        }

        private void ReadUserParameterElement(XmlElement userParametersElement)
        {
            Debug.Assert(null != userParametersElement);

            if (!userParametersElement.HasAttribute(KeyAttributeName)) { return; }

            string key = userParametersElement.GetAttribute(KeyAttributeName);
            if (_userParameters.ContainsKey(key)) { return; }

            _userParameters.Add(key, userParametersElement.InnerText);
        }

        #endregion

        #region ** 出力

        /// <summary>
        /// オブジェクトを XML 表現に変換します。
        /// </summary>
        /// <param name="writer">オブジェクトのシリアル化先の System.Xml.XmlWriter ストリーム。</param>
        public void WriteXml(XmlWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            XmlDocument document = new XmlDocument();
            document.AppendChild(ToXml(document));

            document.Save(writer);
        }

        /// <summary>
        /// DependencyManagerをXML出力します。
        /// </summary>
        /// <param name="document">XMLドキュメント</param>
        /// <returns>XML要素</returns>
        private XmlElement ToXml(XmlDocument document)
        {
            // ルート要素
            XmlElement rootElement = document.CreateElement(RootElementName);
            rootElement.SetAttribute(VersionAttributeName, string.Format("{0}.{1}", Version.Major, Version.Minor));

            // DependencyManager 要素
            XmlElement dependencyManagerElement = document.CreateElement(ElementName);
            rootElement.AppendChild(dependencyManagerElement);

            // BaseDirectoryPath 属性
            if (0 < _baseDirectoryPath.Length)
            {
                dependencyManagerElement.SetAttribute(BaseDirectoryPathAttributeName, _baseDirectoryPath);
            }

            // OutputItems 要素
            XmlElement outputItemsElement = OutputItemsToXml(document);
            if (null != outputItemsElement)
            {
                dependencyManagerElement.AppendChild(outputItemsElement);
            }

            // UserKeyMaps 要素
            XmlElement userKeyMapsElement = UserKeyMapsToXml(document);
            if (null != userKeyMapsElement)
            {
                dependencyManagerElement.AppendChild(userKeyMapsElement);
            }

            // UserParameters 要素
            XmlElement userParametersElement = UserParametersToXml(document);
            if (null != userParametersElement)
            {
                dependencyManagerElement.AppendChild(userParametersElement);
            }

            return rootElement;
        }

        /// <summary>
        /// 出力コレクションをXML出力します。
        /// </summary>
        /// <param name="document">XMLドキュメント</param>
        /// <returns>XML要素</returns>
        private XmlElement OutputItemsToXml(XmlDocument document)
        {
            Debug.Assert(null != document);

            if (0 == _outputItems.Values.Count) { return null; }

            XmlElement outputItemsElement = document.CreateElement(OutputItemsElementName);

            foreach (OutputItem outputItem in _outputItems.Values)
            {

                XmlElement outputItemElement = outputItem.ToXml(document);
                if (null == outputItemElement) { continue; }

                outputItemsElement.AppendChild(outputItemElement);

            }

            return outputItemsElement;
        }

        /// <summary>
        /// 全てのユーザキーマップをXML出力します。
        /// </summary>
        /// <param name="document">XMLドキュメント</param>
        /// <returns>XML要素</returns>
        private XmlElement UserKeyMapsToXml(XmlDocument document)
        {
            Debug.Assert(null != document);

            _outputItems.CollectGarbage();

            if (0 == _outputItems.UserKeyMapIDs.Count) { return null; }

            // UserKeyMaps
            XmlElement userKeyMapsElement = document.CreateElement(UserKeyMapsElementName);

            foreach (string userKeyMapID in _outputItems.UserKeyMapIDs)
            {

                // UserKeyMap
                XmlElement userKeyMapElement = UserKeyMapToXml(document, userKeyMapID);
                if (null == userKeyMapElement) { return null; }

                userKeyMapsElement.AppendChild(userKeyMapElement);

            }

            return (0 < userKeyMapsElement.ChildNodes.Count) ? userKeyMapsElement : null;
        }

        /// <summary>
        /// ユーザキーマップをXML出力します。
        /// </summary>
        /// <param name="document">XMLドキュメント</param>
        /// <returns>XML要素</returns>
        private XmlElement UserKeyMapToXml(XmlDocument document, string userKeyMapID)
        {
            Debug.Assert(null != document);
            Debug.Assert(null != userKeyMapID);

            if (0 == _outputItems.GetUserKeys(userKeyMapID).Count) { return null; }

            XmlElement userKeyMapElement = document.CreateElement(UserKeyMapElementName);

            userKeyMapElement.SetAttribute(IDAttributeName, userKeyMapID);

            foreach (string userKey in _outputItems.GetUserKeys(userKeyMapID))
            {

                XmlElement userKeyMapItemElement = UserKeyMapItemToXml(document, userKeyMapID, userKey);
                if (null == userKeyMapItemElement) { continue; }

                userKeyMapElement.AppendChild(userKeyMapItemElement);

            }

            return userKeyMapElement;
        }

        /// <summary>
        /// ユーザキーをXML出力します。
        /// </summary>
        /// <param name="document">XMLドキュメント</param>
        /// <param name="key">キー</param>
        /// <returns>XML要素</returns>
        private XmlElement UserKeyMapItemToXml(XmlDocument document, string userKeyMapID, string userKey)
        {
            Debug.Assert(null != document);
            Debug.Assert(null != userKey);
            Debug.Assert(0 < userKey.Length);

            if (!_outputItems.ContainsUserKey(userKeyMapID, userKey)) { return null; }

            XmlElement userKeyMapItemElement = document.CreateElement(UserKeyMapItemElementName);

            // UserKey
            userKeyMapItemElement.SetAttribute(KeyAttributeName, userKey);

            // Key
            userKeyMapItemElement.InnerText = _outputItems[userKeyMapID, userKey].Key;

            return userKeyMapItemElement;
        }

        /// <summary>
        /// ユーザパラメータをXML出力します。
        /// </summary>
        /// <param name="document">XMLドキュメント</param>
        /// <returns>XML要素</returns>
        private XmlElement UserParametersToXml(XmlDocument document)
        {
            Debug.Assert(null != document);

            if (0 == _userParameters.Count) { return null; }

            XmlElement userParametersElement = document.CreateElement(UserParametersElementName);

            // UserParameter
            foreach (string key in _userParameters.Keys)
            {

                XmlElement userParameterElement = document.CreateElement(UserParameterElementName);

                userParameterElement.SetAttribute(KeyAttributeName, key);	// Key
                userParameterElement.InnerText = _userParameters[key];	// Value

                userParametersElement.AppendChild(userParameterElement);

            }

            return userParametersElement;
        }

        #endregion

        #endregion

        #endregion

        #region ** 出力クラス

        /// <summary>
        /// 出力クラス
        /// </summary>
        private class OutputItem : IOutputItem
        {
            #region ** 固定値

            // 要素名
            public const string ElementName = "OutputItem";
            private const string OutputsElementName = "Outputs";
            private const string DependenciesElementName = "Dependencies";

            // 属性名
            private const string KeyAttributeName = "Key";

            #endregion

            #region ** フィールド

            private DependencyManager _owner = null;
            private int _referenceCount = 0;
            private string _key = string.Empty;
            private bool _forceDirty = false;

            private OutputInfoCollection _outputs = null;
            private DependencyFileInfoCollection _dependencies = null;

            #endregion

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="owner">DependencyOutputItemInfoを所有するDependencyManager</param>
            /// <param name="key">出力を一意に識別するキー</param>
            public OutputItem(DependencyManager owner, string key)
            {
                if (null == owner) { throw new ArgumentNullException("owner"); }
                if (null == key) { throw new ArgumentNullException("key"); }
                if (0 == key.Length) { throw new ArgumentException("key"); }

                _owner = owner;
                _key = key;

                _outputs = new OutputInfoCollection(this);
                _dependencies = new DependencyFileInfoCollection(this);

                _outputs.ItemInserting += OnOutputInfoInserting;
                _outputs.ItemInserted += OnOutputInfoInserted;
                _outputs.ItemRemoved += OnOutputInfoRemoved;
                _outputs.ItemClearing += OnOutputInfoClearing;
            }

            #region ** プロパティ

            /// <summary>
            /// DependencyManagerを取得します。
            /// </summary>
            public DependencyManager Owner
            {
                get { return _owner; }
            }

            /// <summary>
            /// 参照カウントを取得します。
            /// </summary>
            public int ReferenceCount
            {
                get { return _referenceCount; }
            }

            /// <summary>
            /// キーを取得します。
            /// </summary>
            public string Key
            {
                get { return _key; }
            }

            /// <summary>
            /// 出力情報コレクションを取得します。
            /// </summary>
            public IOutputInfoCollection Outputs
            {
                get { return _outputs; }
            }

            /// <summary>
            /// 依存情報コレクションを取得します。
            /// </summary>
            public IDependencyInfoCollection Dependencies
            {
                get { return _dependencies; }
            }

            /// <summary>
            /// 出力ファイルの有効性を確認します。
            /// </summary>
            public bool Dirty
            {
                get
                {
                    if (_forceDirty) { return true; }

                    // 出力ファイルの存在確認
                    foreach (IOutputInfo item in Outputs)
                    {
                        if (!File.Exists(item.AbsoluteFilePath)) { return true; }
                    }

                    // 依存データの有効性確認
                    foreach (IDependencyFileInfo dependencyInfo in _dependencies)
                    {

                        if (dependencyInfo is DependencyOutputItemInfo)
                        {

                            IOutputItem output = (dependencyInfo as DependencyOutputItemInfo).Output;
                            if (null == output || output.Dirty) { return true; }

                        }
                        else
                        {

                            foreach (IOutputInfo item in Outputs)
                            {
                                if (item.LastUpdatedTime.CompareTo(dependencyInfo.LastUpdatedTime) < 0) { return true; }
                            }
                        }

                    }

                    return false;
                }
            }

            /// <summary>
            /// 強制的に無効にする設定を行います。
            /// </summary>
            public bool ForceDirty
            {
                set { _forceDirty = value; }
            }

            /// <summary>
            /// 依存ファイルが存在しているかを確認します。
            /// </summary>
            public bool DependencyFilesExists
            {
                get
                {
                    try
                    {
                        ValidateDependencies();
                    }
                    catch (Exception)
                    {
                        return false;
                    }

                    return true;
                }
            }

            #endregion

            #region ** イベント

            public event OutputInfoEventHandler ItemInserting;
            public event OutputInfoEventHandler ItemInserted;
            public event OutputInfoEventHandler ItemRemoved;
            public event EventHandler ItemClearing;

            #endregion

            #region ** イベントハンドラ

            private void OnOutputInfoInserting(object sender, OutputInfoEventArgs e)
            {
                if (null != ItemInserting)
                {
                    ItemInserting(this, e);
                }
            }

            private void OnOutputInfoInserted(object sender, OutputInfoEventArgs e)
            {
                if (null != ItemInserted)
                {
                    ItemInserted(this, e);
                }
            }

            private void OnOutputInfoRemoved(object sender, OutputInfoEventArgs e)
            {
                if (null != ItemRemoved)
                {
                    ItemRemoved(this, e);
                }
            }

            private void OnOutputInfoClearing(object sender, EventArgs e)
            {
                if (null != ItemClearing)
                {
                    ItemClearing(this, e);
                }
            }

            #endregion

            #region ** メソッド

            #region ** 参照カウントの操作

            /// <summary>
            /// 参照カウントをインクリメントします。
            /// </summary>
            public void AddReference()
            {
                _referenceCount++;
            }

            /// <summary>
            /// 参照カウントをデクリメントします。
            /// </summary>
            public void RemoveReference()
            {
                if (0 < _referenceCount)
                {

                    _referenceCount--;

                    // 参照がなくなったら、出力ファイルを削除する
                    if (0 < _referenceCount) { return; }

                }

                Clean();
            }

            /// <summary>
            /// 参照カウントをクリアします。
            /// </summary>
            public void ResetReference()
            {
                _referenceCount = 0;

                // 出力ファイルを削除する
                Clean();
            }

            #endregion

            #region ** クリーン

            /// <summary>
            /// 出力ファイルを削除します。
            /// </summary>
            public void Clean()
            {
                foreach (IOutputInfo item in Outputs)
                {
                    item.DeleteFile();
                }
            }

            #endregion

            #region ** 依存関係のチェック

            /// <summary>
            /// 依存ファイルが存在しているかを確認します。
            /// </summary>
            public void ValidateDependencies()
            {
                // 依存ファイルの存在を確認する
                foreach (IDependencyFileInfo dependencyInfo in _dependencies)
                {

                    if (dependencyInfo is DependencyOutputItemInfo)
                    {

                        if (null == (dependencyInfo as DependencyOutputItemInfo).Output)
                        {
                            throw new DependencyOutputNotFound((dependencyInfo as DependencyOutputItemInfo).OutputKey);
                        }

                        continue;

                    }

                    if (!File.Exists(dependencyInfo.AbsoluteFilePath))
                    {
                        throw new DependencyFileNotFoundException(dependencyInfo.AbsoluteFilePath);
                    }

                }
            }

            #endregion

            #region ** XML入出力

            /// <summary>
            /// XML要素から出力情報を読み込みます。
            /// </summary>
            /// <param name="owner">DependencyOutputItemInfoを所有するDependencyManager</param>
            /// <param name="xmlElement">XML要素</param>
            /// <returns></returns>
            public static OutputItem FromXml(DependencyManager owner, XmlElement xmlElement)
            {
                if (null == owner) { throw new ArgumentNullException("owner"); }
                if (null == xmlElement) { throw new ArgumentNullException("xmlElement"); }

                if (ElementName != xmlElement.Name) { throw new DependencyManagerInvalidFileFormatException(); }
                if (!xmlElement.HasAttribute(KeyAttributeName)) { throw new DependencyManagerInvalidFileFormatException(); }

                // Outputs
                XmlElement outputsElement = xmlElement[OutputsElementName];
                if (null == outputsElement) { return null; }

                // Dependencies
                XmlElement dependenciesElement = xmlElement[DependenciesElementName];


                OutputItem newOutputItem = new OutputItem(owner, xmlElement.GetAttribute(KeyAttributeName));

                // OutputInfo
                foreach (XmlElement outputInfoElement
                                            in outputsElement.GetElementsByTagName(OutputInfo.ElementName))
                {
                    OutputInfo newOutputInfo = OutputInfo.FromXml(owner, outputInfoElement);
                    if (null == newOutputInfo) { continue; }
                    newOutputItem.Outputs.Add(newOutputInfo);
                }

                if (null != dependenciesElement)
                {

                    // DependencyFileInfo
                    foreach (XmlElement dependeFileInfoElement
                                                    in dependenciesElement.GetElementsByTagName(DependencyFileInfo.ElementName))
                    {
                        DependencyFileInfo newDependencyInfo = DependencyFileInfo.FromXml(owner, dependeFileInfoElement);
                        if (null == newDependencyInfo) { continue; }
                        newOutputItem.Dependencies.Add(newDependencyInfo);
                    }

                    // DependencyOutputItemInfo
                    foreach (XmlElement dependeOutputItemInfoElement
                                                in dependenciesElement.GetElementsByTagName(DependencyOutputItemInfo.ElementName))
                    {
                        DependencyOutputItemInfo newDependencyInfo = DependencyOutputItemInfo.FromXml(owner, dependeOutputItemInfoElement);
                        if (null == newDependencyInfo) { continue; }
                        newOutputItem.Dependencies.Add(newDependencyInfo);
                    }

                }

                return newOutputItem;
            }

            /// <summary>
            /// 出力情報をXML出力します。
            /// </summary>
            /// <param name="document">XMLドキュメント</param>
            /// <returns>XML要素</returns>
            public XmlElement ToXml(XmlDocument document)
            {
                if (null == document) { throw new ArgumentNullException("document"); }

                XmlElement outputItemElement = document.CreateElement(ElementName);

                // Key
                outputItemElement.SetAttribute(KeyAttributeName, Key);

                // Outputs
                if (0 < Outputs.Count)
                {

                    XmlElement outputsElement = document.CreateElement(OutputsElementName);

                    foreach (OutputInfo outputInfo in Outputs)
                    {

                        XmlElement outputInfoElement = outputInfo.ToXml(document);
                        if (null == outputInfoElement) { continue; }

                        outputsElement.AppendChild(outputInfoElement);

                    }

                    if (0 < outputsElement.ChildNodes.Count)
                    {
                        outputItemElement.AppendChild(outputsElement);
                    }

                }

                // Dependencies
                if (0 < _dependencies.Count)
                {

                    XmlElement dependenciesElement = document.CreateElement(DependenciesElementName);

                    foreach (IDependencyFileInfo dependencyFileInfo in _dependencies)
                    {

                        XmlElement dependencyFileInfoElement = dependencyFileInfo.ToXml(document);
                        if (null == dependencyFileInfo) { continue; }

                        dependenciesElement.AppendChild(dependencyFileInfoElement);

                    }

                    if (0 < dependenciesElement.ChildNodes.Count)
                    {
                        outputItemElement.AppendChild(dependenciesElement);
                    }

                }

                return (0 < outputItemElement.ChildNodes.Count) ? outputItemElement : null;
            }

            #endregion

            #endregion

            #region ** コレクション

            private class OutputInfoCollection : Collection<IOutputInfo>, IOutputInfoCollection
            {
                private OutputItem _owner = null;

                public OutputInfoCollection(OutputItem owner)
                {
                    Debug.Assert(null != owner);
                    _owner = owner;
                }

                #region ** イベント

                public event OutputInfoEventHandler ItemInserting;
                public event OutputInfoEventHandler ItemInserted;
                public event OutputInfoEventHandler ItemRemoved;
                public event EventHandler ItemClearing;

                #endregion

                #region ** メソッド

                protected override void InsertItem(int index, IOutputInfo item)
                {
                    if (item.Owner != _owner.Owner) { throw new DependencyManagerInvalidOutputInfo(item.Name); }

                    if (null != ItemInserting)
                    {
                        ItemInserting(this, new OutputInfoEventArgs(item));
                    }

                    base.InsertItem(index, item);

                    if (null != ItemInserted)
                    {
                        ItemInserted(this, new OutputInfoEventArgs(item));
                    }
                }

                protected override void RemoveItem(int index)
                {
                    IOutputInfo item = Items[index];

                    base.RemoveItem(index);

                    if (null != ItemRemoved)
                    {
                        ItemRemoved(this, new OutputInfoEventArgs(item));
                    }
                }

                protected override void ClearItems()
                {
                    if (null != ItemClearing)
                    {
                        ItemClearing(this, new EventArgs());
                    }

                    base.ClearItems();
                }

                #endregion
            }

            public class DependencyFileInfoCollection : Collection<IDependencyFileInfo>, IDependencyInfoCollection
            {
                private OutputItem _owner = null;

                public DependencyFileInfoCollection(OutputItem owner)
                {
                    Debug.Assert(null != owner);
                    _owner = owner;
                }

                protected override void InsertItem(int index, IDependencyFileInfo item)
                {
                    if (item.Owner != _owner.Owner) { throw new DependencyManagerInvalidDependencyInfo(item.AbsoluteFilePath); }
                    base.InsertItem(index, item);
                }
            }

            #endregion
        }

        /// <summary>
        /// 出力情報クラス
        /// </summary>
        private class OutputInfo : IOutputInfo
        {
            #region ** 固定値

            // 要素名
            public const string ElementName = "OutputInfo";

            // 属性名
            private const string KeyAttributeName = "Key";
            private const string NameAttributeName = "Name";

            #endregion

            #region ** フィールド

            private DependencyManager _owner = null;
            private string _key = string.Empty;
            private string _name = string.Empty;

            #endregion

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="owner">DependencyOutputItemInfoを所有するDependencyManager</param>
            /// <param name="name">出力情報名</param>
            public OutputInfo(DependencyManager owner, string name)
            {
                if (null == owner) { throw new ArgumentNullException("owner"); }
                if (null == name) { throw new ArgumentNullException("name"); }
                if (0 == name.Length) { throw new ArgumentException("name"); }

                _owner = owner;
                _name = name.ToString();
            }

            #region ** プロパティ

            /// <summary>
            /// DependencyManagerを取得します。
            /// </summary>
            public DependencyManager Owner
            {
                get { return _owner; }
            }

            /// <summary>
            /// キーを取得します。
            /// </summary>
            public string Key
            {
                get { return Name; }
            }

            /// <summary>
            /// 出力情報名を取得または設定します。
            /// </summary>
            public string Name
            {
                get { return _name; }
                set { _name = value; }
            }

            /// <summary>
            /// 出力ファイルのパスを取得します。
            /// </summary>
            public string FilePath
            {
                get { return _name; }
            }

            /// <summary>
            /// 出力ファイルの絶対パスを取得します。
            /// </summary>
            public string AbsoluteFilePath
            {
                get
                {
                    // 絶対パスの場合はそのまま返す
                    if (Path.IsPathRooted(FilePath)) { return FilePath; }

                    if (null == _owner) { return string.Empty; }
                    return Path.GetFullPath(Path.Combine(_owner.BaseDirectoryAbsolutePath, FilePath));
                }
            }

            /// <summary>
            /// 最終更新時間を取得します。
            /// </summary>
            public DateTime LastUpdatedTime
            {
                get
                {
                    if (!File.Exists(AbsoluteFilePath)) { throw new DependencyFileNotFoundException(AbsoluteFilePath); }
                    return File.GetLastWriteTime(AbsoluteFilePath);
                }
            }

            #endregion

            #region ** メソッド

            /// <summary>
            /// 出力ファイルを削除します。
            /// </summary>
            public void DeleteFile()
            {
                try
                {
                    File.Delete(AbsoluteFilePath);
                }
                catch (Exception)
                {
                }
            }

            #region ** XML入出力

            /// <summary>
            /// XML要素から出力情報を読み込みます。
            /// </summary>
            /// <param name="owner">DependencyOutputItemInfoを所有するDependencyManager</param>
            /// <param name="xmlElement">XML要素</param>
            /// <returns></returns>
            public static OutputInfo FromXml(DependencyManager owner, XmlElement xmlElement)
            {
                if (null == owner) { throw new ArgumentNullException("owner"); }
                if (null == xmlElement) { throw new ArgumentNullException("xmlElement"); }

                if (ElementName != xmlElement.Name) { throw new DependencyManagerInvalidFileFormatException(); }
                if (!xmlElement.HasAttribute(NameAttributeName)) { throw new DependencyManagerInvalidFileFormatException(); }

                return new OutputInfo(owner, xmlElement.GetAttribute(NameAttributeName));
            }

            /// <summary>
            /// 出力情報をXML出力します。
            /// </summary>
            /// <param name="document">XMLドキュメント</param>
            /// <returns>XML要素</returns>
            public XmlElement ToXml(XmlDocument document)
            {
                Debug.Assert(null != document);

                // OutuptInfo
                XmlElement outputInfoElement = document.CreateElement(ElementName);
                outputInfoElement.SetAttribute(NameAttributeName, Name.ToString());

                return outputInfoElement;
            }

            #endregion

            #endregion
        }

        #endregion

        #region ** 依存情報クラス

        /// <summary>
        /// 依存ファイル情報 基本クラス
        /// </summary>
        public abstract class DependencyFileInfoBase : IDependencyFileInfo
        {
            #region ** 固定値

            // 属性名
            private const string FileSizeAttributeName = "FileSize";
            private const string LastUpdatedTimeAttributeName = "LastUpdatedTime";

            #endregion

            #region ** フィールド

            private DependencyManager _owner = null;

            #endregion

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="owner">DependencyOutputItemInfoを所有するDependencyManager</param>
            protected DependencyFileInfoBase(DependencyManager owner)
            {
                if (null == owner) { throw new ArgumentNullException("owner"); }
                _owner = owner;
            }

            #region ** プロパティ

            /// <summary>
            /// DependencyManagerを取得します。
            /// </summary>
            public DependencyManager Owner
            {
                get { return _owner; }
            }

            /// <summary>
            /// 依存ファイルのファイルパスを取得します。
            /// </summary>
            public abstract string FilePath { get; }

            /// <summary>
            /// 依存ファイルの絶対ファイルパスを取得します。
            /// </summary>
            public abstract string AbsoluteFilePath { get; }

            /// <summary>
            /// 最終更新時間を取得します。
            /// </summary>
            public DateTime LastUpdatedTime
            {
                get
                {
                    if (!File.Exists(AbsoluteFilePath)) { throw new DependencyFileNotFoundException(AbsoluteFilePath); }
                    return File.GetLastWriteTime(AbsoluteFilePath);
                }
            }

            #endregion

            #region ** XML入出力

            /// <summary>
            /// 依存ファイル情報をXML出力します。
            /// </summary>
            /// <param name="document">XMLドキュメント</param>
            /// <returns>XML要素</returns>
            protected abstract XmlElement ToXml(XmlDocument document);

            /// <summary>
            /// 依存ファイル情報をXML出力します。
            /// </summary>
            /// <param name="document">XMLドキュメント</param>
            /// <returns>XML要素</returns>
            XmlElement IDependencyFileInfo.ToXml(XmlDocument document)
            {
                return ToXml(document);
            }

            #endregion
        }

        /// <summary>
        /// 依存ファイル情報クラス
        /// </summary>
        public class DependencyFileInfo : DependencyFileInfoBase
        {
            #region ** 固定値

            // 要素名
            public const string ElementName = "DependencyFileInfo";

            // 属性名
            private const string FilePathAttributeName = "FilePath";

            #endregion

            #region ** フィールド

            private string _filePath = string.Empty;

            #endregion

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="owner">DependencyOutputItemInfoを所有するDependencyManager</param>
            /// <param name="filePath">依存ファイルのパス</param>
            public DependencyFileInfo(DependencyManager owner, string filePath) : base(owner)
            {
                if (null == filePath) { throw new ArgumentNullException("filePath"); }
                if (0 == filePath.Length) { throw new ArgumentException("filePath"); }

                _filePath = filePath;
            }

            #region ** プロパティ

            /// <summary>
            /// 依存ファイルのパスを取得します。
            /// </summary>
            public override string FilePath
            {
                get { return _filePath; }
            }

            /// <summary>
            /// 依存ファイルの絶対パスを取得します。
            /// </summary>
            public override string AbsoluteFilePath
            {
                get
                {
                    // 絶対パスの場合はそのまま返す
                    if (Path.IsPathRooted(FilePath)) { return FilePath; }

                    if (null == Owner) { return string.Empty; }
                    return Path.GetFullPath(Path.Combine(Owner.BaseDirectoryAbsolutePath, FilePath));
                }
            }

            #endregion

            #region ** XML入出力

            /// <summary>
            /// XML要素からDependencyFileを生成します。
            /// </summary>
            /// <param name="owner">DependencyOutputItemInfoを所有するDependencyManager</param>
            /// <param name="xmlElement">入力XML要素</param>
            /// <returns>生成したDependencyFile。</returns>
            public static DependencyFileInfo FromXml(DependencyManager owner, XmlElement xmlElement)
            {
                if (null == owner) { throw new ArgumentNullException("owner"); }
                if (null == xmlElement) { throw new ArgumentNullException("xmlElement"); }

                if (ElementName != xmlElement.Name) { throw new DependencyManagerInvalidFileFormatException(); }
                if (!xmlElement.HasAttribute(FilePathAttributeName)) { throw new DependencyManagerInvalidFileFormatException(); }

                DependencyFileInfo newInstance = new DependencyFileInfo(owner, xmlElement.GetAttribute(FilePathAttributeName));

                // FilePath
                newInstance._filePath = xmlElement.GetAttribute(FilePathAttributeName);

                return newInstance;
            }

            /// <summary>
            /// 依存ファイル情報をXML出力します。
            /// </summary>
            /// <param name="document">XMLドキュメント</param>
            /// <returns>XML要素</returns>
            protected override XmlElement ToXml(XmlDocument document)
            {
                Debug.Assert(null != document);

                // DependencyFileInfo
                XmlElement dependencyFileElement = document.CreateElement(ElementName);
                dependencyFileElement.SetAttribute(FilePathAttributeName, FilePath.ToString());

                return dependencyFileElement;
            }

            #endregion
        }

        /// <summary>
        /// 依存出力情報クラス
        /// </summary>
        public class DependencyOutputItemInfo : DependencyFileInfoBase
        {
            #region ** 固定値

            // 要素名
            public const string ElementName = "DependencyOutputItemInfo";

            // 属性名
            private const string OutputItemKeyAttributeName = "OutputItemKey";

            #endregion

            #region ** フィールド

            private string _outputItemKey = string.Empty;

            #endregion

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="owner">DependencyOutputItemInfoを所有するDependencyManager</param>
            /// <param name="fileName">依存する出力情報の名前</param>
            public DependencyOutputItemInfo(DependencyManager owner, string outputItemKey) : base(owner)
            {
                if (null == outputItemKey) { throw new ArgumentNullException("outputKey"); }
                if (0 == outputItemKey.Length) { throw new ArgumentException("outputKey"); }

                _outputItemKey = outputItemKey;
            }

            #region ** プロパティ

            /// <summary>
            /// 依存する出力情報の名前を取得します。
            /// </summary>
            public string OutputKey
            {
                get { return _outputItemKey; }
            }

            /// <summary>
            /// 依存する出力ファイルのパスを取得します。
            /// </summary>
            public override string FilePath
            {
                get
                {
                    if (null == Output) { return string.Empty; }
                    if (0 == Output.Outputs.Count) { return string.Empty; }
                    return Output.Outputs[0].FilePath;
                }
            }

            /// <summary>
            /// 依存する出力ファイルの絶対パスを取得します。
            /// </summary>
            public override string AbsoluteFilePath
            {
                get
                {
                    if (null == Output) { return string.Empty; }
                    if (0 == Output.Outputs.Count) { return string.Empty; }
                    return Output.Outputs[0].AbsoluteFilePath;
                }
            }

            /// <summary>
            /// 依存する出力情報を取得します。
            /// </summary>
            public IOutputItem Output
            {
                get
                {
                    if (!Owner.OutputItems.ContainsKey(_outputItemKey)) { return null; }
                    return Owner.OutputItems[_outputItemKey];
                }
            }

            #endregion

            #region ** XML入出力

            /// <summary>
            /// XML要素からDependencyOutputItemInfoを生成します。
            /// </summary>
            /// <param name="owner">DependencyOutputItemInfoを所有するDependencyManager</param>
            /// <param name="xmlElement">入力XML要素</param>
            /// <returns>生成したDependencyOutputItemInfo。</returns>
            public static DependencyOutputItemInfo FromXml(DependencyManager owner, XmlElement xmlElement)
            {
                if (null == owner) { throw new ArgumentNullException("owner"); }
                if (null == xmlElement) { throw new ArgumentNullException("xmlElement"); }

                if (ElementName != xmlElement.Name) { throw new DependencyManagerInvalidFileFormatException(); }
                if (!xmlElement.HasAttribute(OutputItemKeyAttributeName)) { throw new DependencyManagerInvalidFileFormatException(); }

                DependencyOutputItemInfo newInstance = new DependencyOutputItemInfo(owner, xmlElement.GetAttribute(OutputItemKeyAttributeName));

                // OutputItemKey
                newInstance._outputItemKey = xmlElement.GetAttribute(OutputItemKeyAttributeName);

                return newInstance;
            }

            /// <summary>
            /// 依存出力ファイル情報をXML出力します。
            /// </summary>
            /// <param name="document">XMLドキュメント</param>
            /// <returns>XML要素</returns>
            protected override XmlElement ToXml(XmlDocument document)
            {
                Debug.Assert(null != document);

                // DependencyFileInfo
                XmlElement dependencyOutputItemInfoElement = document.CreateElement(ElementName);
                dependencyOutputItemInfoElement.SetAttribute(OutputItemKeyAttributeName, _outputItemKey.ToString());

                return dependencyOutputItemInfoElement;
            }

            #endregion
        }

        #endregion

        #region ** イベントデリゲートとイベントパラメータ

        public delegate void OutputInfoEventHandler(object sender, OutputInfoEventArgs e);

        public class OutputInfoEventArgs : EventArgs
        {
            private IOutputInfo _outputInfo = null;

            public OutputInfoEventArgs(IOutputInfo outputInfo)
            {
                _outputInfo = outputInfo;
            }

            public IOutputInfo OutputInfo
            {
                get { return _outputInfo; }
            }
        }

        #endregion

        #region ** コレクション

        /// <summary>
        /// アイテムアイテムコレクション クラス
        /// </summary>
        private class OutputCollection : KeyedCollection<string, IOutputItem>, IOutputItemCollection
        {
            private Dictionary<string, Dictionary<string, IOutputItem>> _userKeyMap
                                            = new Dictionary<string, Dictionary<string, IOutputItem>>();

            private Dictionary<string, IOutputInfo> _outputNameMap
                                    = new Dictionary<string, IOutputInfo>(NoCaseStringCompare.Instance);

            #region ** インデクサ

            /// <summary>
            /// 指定したユーザキーを持つ出力を取得します。
            /// </summary>
            /// <param name="userKeyMapID">ユーザキーマップID。</param>
            /// <param name="userKey">取得する出力のユーザキー。</param>
            /// <returns>指定したユーザキーを持つ出力。</returns>
            public IOutputItem this[string userKeyMapID, string userKey]
            {
                get { return GetValueFromUserKey(userKeyMapID, userKey); }
            }

            #endregion

            #region ** イベントハンドラ

            private void OnOutputInfoInserting(object sender, OutputInfoEventArgs e)
            {
                if (_outputNameMap.ContainsKey(e.OutputInfo.Name)) { throw new Exception(); }
            }

            private void OnOutputInfoInserted(object sender, OutputInfoEventArgs e)
            {
                _outputNameMap.Add(e.OutputInfo.Name, e.OutputInfo);
            }

            private void OnOutputInfoRemoved(object sender, OutputInfoEventArgs e)
            {
                _outputNameMap.Remove(e.OutputInfo.Name);
            }

            private void OnOutputInfoClearing(object sender, EventArgs e)
            {
                OutputItem itemImpl = sender as OutputItem;

                foreach (IOutputInfo outputInfo in itemImpl.Outputs)
                {
                    _outputNameMap.Remove(outputInfo.Name);
                }
            }

            #endregion

            #region ** メソッド

            /// <summary>
            /// ユーザキーコレクションを取得します。
            /// </summary>
            /// <param name="userKeyMapID">ユーザキーマップID。</param>
            public Dictionary<string, IOutputItem>.KeyCollection GetUserKeys(string userKeyMapID)
            {
                return _userKeyMap[userKeyMapID].Keys;
            }

            /// <summary>
            /// 指定したユーザキーマップの出力が含まれているかどうかを確認します。
            /// </summary>
            /// <param name="userKeyMapID">ユーザキーマップID。</param>
            /// <returns>指定したキーを持つ出力を含む場合は true。それ以外の場合は false。</returns>
            public bool ContainsUserKeyMap(string userKeyMapID)
            {
                return _userKeyMap.ContainsKey(userKeyMapID);
            }

            /// <summary>
            /// 指定したユーザキーを持つ出力が含まれているかどうかを確認します。
            /// </summary>
            /// <param name="userKeyMapID">ユーザキーマップID。</param>
            /// <param name="userKey">検索されるユーザキー。</param>
            /// <returns>指定したユーザキーを持つ出力を含む場合は true。それ以外の場合は false。</returns>
            public bool ContainsUserKey(string userKeyMapID, string userKey)
            {
                if (!_userKeyMap.ContainsKey(userKeyMapID)) { return false; }
                if (!_userKeyMap[userKeyMapID].ContainsKey(userKey)) { return false; }

                return (GetValueFromUserKey(userKeyMapID, userKey) != null);
            }

            public bool ContainsOutputName(string outputName)
            {
                return _outputNameMap.ContainsKey(outputName);
            }

            /// <summary>
            /// 既存の出力にユーザキーを設定します。
            /// </summary>
            /// <param name="userKeyMapID">ユーザキーマップID。</param>
            /// <param name="userKey">設定するユーザキー。</param>
            /// <param name="output">既存の出力。</param>
            public void SetUserKey(string userKeyMapID, string userKey, IOutputItem output)
            {
                if (null == userKeyMapID) { throw new ArgumentNullException("userKeyMapID"); }
                if (0 == userKeyMapID.Length) { throw new ArgumentException("userKeyMapID"); }
                if (null == userKey) { throw new ArgumentNullException("userKey"); }
                if (0 == userKey.Length) { throw new ArgumentException("userKey"); }
                if (null == output) { throw new ArgumentNullException("output"); }

                if (!Contains(output.Key)) { throw new DependencyOutputNotFound(output.Key); }

                // 出力の参照カウントをインクリメントしてユーザキーを追加
                (output as OutputItem).AddReference();

                if (!_userKeyMap.ContainsKey(userKeyMapID))
                {
                    _userKeyMap.Add(userKeyMapID, new UserKeyDictionary());
                }

                if (!_userKeyMap[userKeyMapID].ContainsKey(userKey))
                {
                    _userKeyMap[userKeyMapID].Add(userKey, output);
                }
                else
                {
                    _userKeyMap[userKeyMapID][userKey] = output;
                }
            }

            /// <summary>
            /// 指定したユーザキーを削除します。
            /// </summary>
            /// <param name="userKeyMapID">ユーザキーマップID。</param>
            /// <param name="userKey">削除する出力のユーザキー。</param>
            public void RemoveByUserKey(string userKeyMapID, string userKey)
            {
                if (null == userKeyMapID) { throw new ArgumentNullException("userKeyMapID"); }
                if (0 == userKeyMapID.Length) { throw new ArgumentException("userKeyMapID"); }
                if (null == userKey) { throw new ArgumentNullException("userKey"); }
                if (0 == userKey.Length) { throw new ArgumentException("userKey"); }

                OutputItem output = GetValueFromUserKey(userKeyMapID, userKey) as OutputItem;

                // ユーザキーを削除して出力の参照カウントをデクリメント
                _userKeyMap[userKeyMapID].Remove(userKey);
                output.RemoveReference();

                // 誰からも参照されなくなったら出力を削除する
                if (0 >= output.ReferenceCount)
                {
                    base.Remove(output.Key);
                }
            }

            public void CollectGarbage()
            {
                Collection<UserKey> invalidKeys = new Collection<UserKey>();
                Collection<string> invalidKeyMaps = new Collection<string>();

                foreach (string userKeyMapID in _userKeyMap.Keys)
                {

                    foreach (string userKey in _userKeyMap[userKeyMapID].Keys)
                    {

                        if (GetValueFromUserKey(userKeyMapID, userKey) == null)
                        {
                            invalidKeys.Add(new UserKey(userKeyMapID, userKey));
                        }

                    }

                    if (0 == _userKeyMap[userKeyMapID].Count)
                    {
                        invalidKeyMaps.Add(userKeyMapID);
                    }

                }

                foreach (UserKey userKey in invalidKeys)
                {
                    _userKeyMap[userKey.MapID].Remove(userKey.Key);
                }

                foreach (string mapID in invalidKeyMaps)
                {
                    _userKeyMap.Remove(mapID);
                }
            }

            protected override string GetKeyForItem(IOutputItem item)
            {
                return item.Key;
            }

            protected override void InsertItem(int index, IOutputItem item)
            {
                base.InsertItem(index, item);

                foreach (IOutputInfo outputInfo in item.Outputs)
                {
                    _outputNameMap.Add(outputInfo.Name, outputInfo);
                }

                OutputItem itemImpl = item as OutputItem;
                itemImpl.ItemInserting += OnOutputInfoInserting;
                itemImpl.ItemInserted += OnOutputInfoInserted;
                itemImpl.ItemRemoved += OnOutputInfoRemoved;
                itemImpl.ItemClearing += OnOutputInfoClearing;
            }

            protected override void RemoveItem(int index)
            {
                OutputItem itemImpl = (Items[index] as OutputItem);

                itemImpl.ItemInserting -= OnOutputInfoInserting;
                itemImpl.ItemInserted -= OnOutputInfoInserted;
                itemImpl.ItemRemoved -= OnOutputInfoRemoved;
                itemImpl.ItemClearing -= OnOutputInfoClearing;

                foreach (IOutputInfo outputInfo in itemImpl.Outputs)
                {
                    _outputNameMap.Remove(outputInfo.Name);
                }

                itemImpl.ResetReference();
                base.RemoveItem(index);
            }

            protected override void ClearItems()
            {
                _userKeyMap.Clear();
                _outputNameMap.Clear();
                base.ClearItems();
            }

            /// <summary>
            /// 指定したユーザキーを持つ出力を取得します。
            /// </summary>
            /// <param name="userKeyMapID">ユーザキーマップID。</param>
            /// <param name="userKey">検索されるユーザキー。</param>
            /// <returns>指定したユーザキーを持つ出力。</returns>
            private IOutputItem GetValueFromUserKey(string userKeyMapID, string userKey)
            {
                IOutputItem output = _userKeyMap[userKeyMapID][userKey];
                if (!Contains(output)) { return null; }
                return output;
            }

            #endregion

            #region ** IOutputCollection の実装

            /// <summary>
            /// ユーザキーマップIDコレクションを取得します。
            /// </summary>
            public Dictionary<string, Dictionary<string, IOutputItem>>.KeyCollection UserKeyMapIDs
            {
                get { return _userKeyMap.Keys; }
            }

            /// <summary>
            /// 出力コレクションを取得します。
            /// </summary>
            public ReadOnlyCollection<IOutputItem> Values
            {
                get { return new ReadOnlyCollection<IOutputItem>(this); }
            }

            /// <summary>
            /// 指定したキーの出力が含まれているかどうかを確認します。
            /// </summary>
            /// <param name="key">取得する出力のキー。</param>
            /// <returns>指定したキーを持つ出力を含む場合は true。それ以外の場合は false。</returns>
            bool IOutputItemCollection.ContainsKey(string key)
            {
                return Contains(key);
            }

            #endregion

            #region ** 隠蔽するメソッド

            private new void Insert(int index, IOutputItem value)
            {
                // 使用しない
                Debug.Assert(false);
            }

            #endregion

            #region ** キーマップ

            public class UserKeyDictionary : Dictionary<string, IOutputItem> { }

            private struct UserKey
            {
                public string MapID;
                public string Key;

                public UserKey(string mapID, string key)
                {
                    MapID = mapID;
                    Key = key;
                }
            }

            #endregion
        }

        public class UserParameterDictionary : Dictionary<string, string> { }

        #endregion

        #region ** NoCaseStringCompare

        private class NoCaseStringCompare : IEqualityComparer<string>
        {
            public static NoCaseStringCompare Instance = new NoCaseStringCompare();

            private NoCaseStringCompare() { }

            /// <summary>
            /// 指定した string が等しいかどうかを判断します。
            /// </summary>
            /// <param name="x">string1。</param>
            /// <param name="y">string2。</param>
            /// <returns>指定した string が等しい場合は true。それ以外の場合は false。</returns>
            public bool Equals(string x, string y)
            {
                if (null == x) { throw new ArgumentNullException("x"); }
                if (null == y) { throw new ArgumentNullException("y"); }

                return string.Compare(x, y, true) == 0;
            }

            /// <summary>
            /// 指定した string のハッシュ コードを返します。
            /// </summary>
            /// <param name="text">string。</param>
            /// <returns>ハッシュコード</returns>
            public int GetHashCode(string text)
            {
                if (null == text) { throw new ArgumentNullException("text"); }
                return text.ToLower().GetHashCode();
            }
        }

        #endregion
    }

    #region ** インターフェイス

    /// <summary>
    /// 出力コレクション インターフェイス
    /// </summary>
    public interface IOutputItemCollection
    {
        /// <summary>
        /// キーコレクションを取得します。
        /// </summary>
        Dictionary<string, Dictionary<string, IOutputItem>>.KeyCollection UserKeyMapIDs { get; }

        /// <summary>
        /// 出力コレクションを取得します。
        /// </summary>
        ReadOnlyCollection<IOutputItem> Values { get; }

        #region ** インデクサ

        /// <summary>
        /// 指定したキーを持つ出力を取得します。
        /// </summary>
        /// <param name="key">取得する出力のキー。</param>
        /// <returns>指定したキーを持つ出力。</returns>
        IOutputItem this[string key] { get; }

        /// <summary>
        /// 指定したユーザキーを持つ出力を取得します。
        /// </summary>
        /// <param name="userKeyMapID">ユーザキーマップID。</param>
        /// <param name="userKey">取得する出力のユーザキー。</param>
        /// <returns>指定したキーを持つ出力。</returns>
        IOutputItem this[string userKeyMapID, string userKey] { get; }

        #endregion

        #region ** メソッド

        /// <summary>
        /// ユーザキーコレクションを取得します。
        /// </summary>
        /// <param name="userKeyMapID">ユーザキーマップID。</param>
        Dictionary<string, IOutputItem>.KeyCollection GetUserKeys(string userKeyMapID);

        /// <summary>
        /// 指定したキーを持つ出力が含まれているかどうかを確認します。
        /// </summary>
        /// <param name="key">取得する出力のキー。</param>
        /// <returns>指定した名前を持つ出力を含む場合は true。それ以外の場合は false。</returns>
        bool ContainsKey(string key);

        /// <summary>
        /// 指定したユーザキーマップの出力が含まれているかどうかを確認します。
        /// </summary>
        /// <param name="userKeyMapID">ユーザキーマップID。</param>
        /// <returns>指定したキーを持つ出力を含む場合は true。それ以外の場合は false。</returns>
        bool ContainsUserKeyMap(string userKeyMapID);

        /// <summary>
        /// 指定したユーザキーの出力が含まれているかどうかを確認します。
        /// </summary>
        /// <param name="userKeyMapID">ユーザキーマップID。</param>
        /// <param name="userKey">検索されるユーザキー。</param>
        /// <returns>指定したキーを持つ出力を含む場合は true。それ以外の場合は false。</returns>
        bool ContainsUserKey(string userKeyMapID, string userKey);

        #endregion
    }

    /// <summary>
    /// 出力情報コレクション インターフェイス
    /// </summary>
    public interface IOutputInfoCollection : ICollection<IOutputInfo>, IEnumerable<IOutputInfo>, IEnumerable
    {
        #region ** インデクサ

        /// <summary>
        /// 指定したインデックスを持つ出力情報を取得します。
        /// </summary>
        /// <param name="index">取得する出力情報のインデックス。</param>
        /// <returns>指定したインデックスを持つ出力情報。</returns>
        IOutputInfo this[int index] { get; }

        #endregion
    }

    /// <summary>
    /// 依存情報コレクション インターフェイス
    /// </summary>
    public interface IDependencyInfoCollection : ICollection<IDependencyFileInfo>, IEnumerable<IDependencyFileInfo>, IEnumerable
    {
        #region ** インデクサ

        /// <summary>
        /// 指定したインデックスを持つ出力情報を取得します。
        /// </summary>
        /// <param name="index">取得する出力情報のインデックス。</param>
        /// <returns>指定したインデックスを持つ出力情報。</returns>
        IDependencyFileInfo this[int index] { get; }

        #endregion
    }

    /// <summary>
    /// 出力インターフェイス
    /// </summary>
    public interface IOutputItem
    {
        #region ** プロパティ

        /// <summary>
        /// DependencyManagerを取得します。
        /// </summary>
        DependencyManager Owner { get; }

        /// <summary>
        /// 参照カウントを取得します。
        /// </summary>
        int ReferenceCount { get; }

        /// <summary>
        /// キーを取得します。
        /// </summary>
        string Key { get; }

        /// <summary>
        /// 出力情報コレクションを取得します。
        /// </summary>
        IOutputInfoCollection Outputs { get; }

        /// <summary>
        /// 依存情報コレクションを取得します。
        /// </summary>
        IDependencyInfoCollection Dependencies { get; }

        /// <summary>
        /// 出力ファイルの有効性を確認します。
        /// </summary>
        bool Dirty { get; }

        /// <summary>
        /// 強制的に無効にする設定を行います。
        /// </summary>
        bool ForceDirty { set; }

        /// <summary>
        /// 依存ファイルが存在しているかを確認します。
        /// </summary>
        bool DependencyFilesExists { get; }

        #endregion

        #region ** メソッド

        /// <summary>
        /// 出力ファイルを削除します。
        /// </summary>
        void Clean();

        /// <summary>
        /// 依存ファイルが存在しているかを確認します。
        /// </summary>
        void ValidateDependencies();

        #endregion
    }

    /// <summary>
    /// 出力情報インターフェイス
    /// </summary>
    public interface IOutputInfo
    {
        #region ** プロパティ

        /// <summary>
        /// DependencyManagerを取得します。
        /// </summary>
        DependencyManager Owner { get; }

        /// <summary>
        /// キーを取得します。
        /// </summary>
        string Key { get; }

        /// <summary>
        /// 出力情報名を取得します。
        /// </summary>
        string Name { get; }

        /// <summary>
        /// 出力ファイルのパスを取得します。
        /// </summary>
        string FilePath { get; }

        /// <summary>
        /// 出力ファイルの絶対パスを取得します。
        /// </summary>
        string AbsoluteFilePath { get; }

        /// <summary>
        /// 最終更新時間を取得します。
        /// </summary>
        DateTime LastUpdatedTime { get; }

        #endregion

        #region ** メソッド

        /// <summary>
        /// 出力ファイルを削除します。
        /// </summary>
        void DeleteFile();

        #endregion
    }

    /// <summary>
    /// 依存ファイル情報インターフェイス
    /// </summary>
    public interface IDependencyFileInfo
    {
        #region ** プロパティ

        /// <summary>
        /// DependencyManagerを取得します。
        /// </summary>
        DependencyManager Owner { get; }

        /// <summary>
        /// ファイルパスを取得します。
        /// </summary>
        string FilePath { get; }

        /// <summary>
        /// フルパスを取得します。
        /// </summary>
        string AbsoluteFilePath { get; }

        /// <summary>
        /// 現在の最終更新時間を取得します。
        /// </summary>
        DateTime LastUpdatedTime { get; }

        #endregion

        #region ** メソッド

        /// <summary>
        /// 依存ファイル情報をXML出力します。
        /// </summary>
        /// <param name="document">XMLドキュメント</param>
        /// <returns>XML要素</returns>
        XmlElement ToXml(XmlDocument document);

        #endregion
    }

    #endregion

    #region ** 例外クラス

    public class DependencyManagerInvalidFileFormatException : Exception
    {
        public override string Message
        {
            get { return "the dependency manager file is invalid format."; }
        }
    }

    public class DependencyManagerInvalidFileVersionException : Exception
    {
        public override string Message
        {
            get { return "the dependency manager file version is not supported."; }
        }
    }

    public class DependencyFileNotFoundException : FileNotFoundException
    {
        public DependencyFileNotFoundException(string fileName) : base("dependency file not found.", fileName) { }
    }

    public class DependencyOutputNotFound : Exception
    {
        private string _outputKey = string.Empty;

        public DependencyOutputNotFound(string outputKey)
        {
            _outputKey = (null == outputKey) ? string.Empty : outputKey;
        }

        public override string Message
        {
            get { return string.Format("output \"{0}\" not found.", _outputKey); }
        }
    }

    public class DependencyManagerInvalidOutputInfo : Exception
    {
        private string _name = string.Empty;

        public DependencyManagerInvalidOutputInfo(string name)
        {
            _name = (null == name) ? string.Empty : name;
        }

        public override string Message
        {
            get { return string.Format("invalid output info \"{0}\".", _name); }
        }
    }

    public class DependencyManagerInvalidDependencyInfo : Exception
    {
        private string _filePath = string.Empty;

        public DependencyManagerInvalidDependencyInfo(string filePath)
        {
            _filePath = (null == filePath) ? string.Empty : filePath;
        }

        public override string Message
        {
            get { return string.Format("invalid dependency info \"{0}\".", _filePath); }
        }
    }

    #endregion
}
