﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate;
    using NintendoWare.ToolDevelopmentKit;
    using MsgRsrc = NintendoWare.SoundFoundation.Resources;

    /// <summary>
    /// Cafe サウンドプロジェクトをインポートします。
    /// </summary>
    public class SoundDocumentImporterCafe : IDocumentImporter
    {
        private readonly DocumentService documentService = null;
        private readonly SoundIntermediateOutputTraits intermediateOutputTraits = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="documentService">ドキュメントサービスを指定します。</param>
        /// <param name="intermediateOutputTraits">中間出力特性を指定します。</param>
        public SoundDocumentImporterCafe(
            DocumentService documentService,
            SoundIntermediateOutputTraits intermediateOutputTraits)
        {
            Ensure.Argument.NotNull(documentService);
            Ensure.Argument.NotNull(intermediateOutputTraits);

            this.documentService = documentService;
            this.intermediateOutputTraits = intermediateOutputTraits;
        }

        /// <summary>
        /// インポート元のドキュメントタイプ名を取得します。
        /// </summary>
        public string SourceDocumentTypeName
        {
            get { return Platforms.Cafe.SoundProjectDocument; }
        }

        /// <summary>
        /// インポート後のドキュメントタイプ名を取得します。
        /// </summary>
        public string DestinationDocumentTypeName
        {
            get { return this.intermediateOutputTraits.SoundProjectDocumentTypeName; }
        }

        /// <summary>
        /// ドキュメントをインポートします。
        /// </summary>
        /// <param name="document">インポートするドキュメントを指定します。</param>
        /// <returns>インポートしたドキュメント参照の配列を返します。</returns>
        public DocumentReference[] Import(Document document)
        {
            Assertion.Argument.NotNull(document);
            Ensure.Argument.True(document is SoundProjectDocument);
            Ensure.Argument.True(document.Resource is FileResource);
#if false
            System.Reflection.AssemblyProductAttribute asmprd =
                (System.Reflection.AssemblyProductAttribute)
                Attribute.GetCustomAttribute(
                                             System.Reflection.Assembly.GetEntryAssembly(),
                                             typeof(System.Reflection.AssemblyProductAttribute)
                                             );

            DialogResult result =
                MessageBox.Show(MsgRsrc.MessageResourceCafe.Message_ImportNotesCafe,
                                asmprd.Product,
                                MessageBoxButtons.OKCancel,
                                MessageBoxIcon.Warning
                                );
            if (result == DialogResult.Cancel)
            {
                return null;
            }
#endif
            string typeName = this.intermediateOutputTraits.SoundProjectDocumentTypeName;
            List<DocumentReference> results = new List<DocumentReference>();
            try
            {
                ImportLog.WriteLine(
                                    MsgRsrc.MessageResource.Message_ImportFile,
                                    document.Resource.Key
                                    );
                DocumentReference soundProjectDocumentRef =
                    this.documentService.ConvertDocument(document, typeName);
                results.Add(soundProjectDocumentRef);

                DocumentReference[] importedDocumentRefs =
                    this.ImportSoundSetDocuments(document as SoundProjectDocument);
                results.AddRange(importedDocumentRefs);
            }
            catch
            {
                foreach (DocumentReference soundSetDocumentRef in results)
                {
                    soundSetDocumentRef.Close();
                }
                throw;
            }

            return results.ToArray();
        }

        /// <summary>
        /// サウンドセットドキュメントをインポートします。
        /// </summary>
        /// <param name="legacySoundProjectDocument">
        /// レガシーサウンドプロジェクトドキュメントを指定します。
        /// </param>
        /// <returns>インポートしたドキュメントへの参照配列を返します。</returns>
        private DocumentReference[] ImportSoundSetDocuments(
            SoundProjectDocument soundProjectDocument)
        {
            Assertion.Argument.NotNull(soundProjectDocument);

            List<DocumentReference> results = new List<DocumentReference>();

            string projectDirectoryPath = soundProjectDocument.Resource.Key;

            try
            {
                SoundSetDocumentImporterCafe importer =
                    new SoundSetDocumentImporterCafe(this.documentService,
                                                    this.intermediateOutputTraits,
                                                    projectDirectoryPath);
                foreach (ProjectSoundSetComponent child in soundProjectDocument.Project.Children)
                {
                    string soundSetFilePath = Path.Combine(projectDirectoryPath, child.TargetFilePath).GetFullPath();

                    DocumentReference[] importedDocumentRefs = importer.Import(soundSetFilePath);
                    results.AddRange(importedDocumentRefs);
                }
            }
            catch
            {
                foreach (DocumentReference importedDocumentRef in results)
                {
                    importedDocumentRef.Close();
                }

                throw;
            }

            return results.ToArray();
        }
    }
}
