﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.Foundation.Contracts;
using Nintendo.Foundation.IO;
using NintendoWare.SoundFoundation.Core.IO;
using NintendoWare.SoundFoundation.Core.Resources;
using NintendoWare.SoundFoundation.Documents;
using NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundMaker.Framework;
using NintendoWare.SoundMaker.Framework.Utilities;
using NintendoWare.SoundProjectUtility.Resources;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;

namespace NintendoWare.SoundProjectUtility
{
    /// <summary>
    /// サウンドプロジェクトファイルのバージョンを更新します。
    /// </summary>
    public class UpdateProjectStrategy
    {
        public class Settings : LocalizableSettings
        {
            [CommandLineOption(
                "backslash",
                Description = nameof(MessageResources.CommandLine_UsageMessage_UseBackslash),
                DescriptionConverterName = nameof(LocalizeDescription))]
            public bool UseBackslash { get; set; }

            [CommandLineOption(
                "backup",
                Description = nameof(MessageResources.CommandLine_UsageMessage_BackupOldFile),
                DescriptionConverterName = nameof(LocalizeDescription))]
            public bool BackupOldFile { get; set; }
            [CommandLineOption(
                "dry-run",
                Description = nameof(MessageResources.CommandLine_UsageMessage_DryRun),
                DescriptionConverterName = nameof(LocalizeDescription))]
            public bool DryRun { get; set; }

            [CommandLineOption(
                "force",
                IsHidden = true,
                Description = nameof(MessageResources.CommandLine_UsageMessage_ForceUpdate),
                DescriptionConverterName = nameof(LocalizeDescription))]
            public bool ForceUpdate { get; set; }

            [CommandLineOption(
                "relative",
                ValueName = "basepath",
                Description = nameof(MessageResources.CommandLine_UsageMessage_BasePath),
                DescriptionConverterName = nameof(LocalizeDescription))]
            public string BasePath { get; set; }

            [CommandLineOption(
                's',
                "silent",
                Description = nameof(MessageResources.CommandLine_UsageMessage_Silent),
                DescriptionConverterName = nameof(LocalizeDescription))]
            public bool Silent { get; set; }

            [CommandLineValues(
                ValueName = "input-filepath",
                CountMin = 1,
                Description = nameof(MessageResources.CommandLine_UsageMessage_InputFilePaths),
                DescriptionConverterName = nameof(LocalizeDescription))]
            public string[] InputFilePaths { get; set; }
        }

        private class RequiredServices : RequiredServicesBase
        {
            internal RequiredServices()
            {
                var intermediateOutputTraits = ServiceUtility.CreateIntermediateOutputTraits();
                this.DocumentService = new DocumentService(ServiceUtility.CreateDocumentServiceTraits());
                this.ProjectService = new SoundProjectServiceCommon(this.DocumentService, intermediateOutputTraits);
                this.BankServices = new BankServiceManager(this.DocumentService, intermediateOutputTraits);
            }
        }

        private const string BackupFileSuffix = ".bak";

        private readonly Settings _settings;
        private readonly DocumentService _documentService;
        private readonly SoundProjectService _projectService;
        private readonly BankServiceManager _bankServices;
        private readonly IList<string> _savedFiles = new List<string>();

        internal UpdateProjectStrategy(Settings settings)
        {
            Ensure.Argument.NotNull(settings, nameof(settings));

            _settings = settings;

            ComponentConfigurationCommon.Instance.Initialize();
            CoreServicesProvider.SetInstance(new RequiredServices());
            _documentService = CoreServicesProvider.Instance.DocumentService;
            _projectService = CoreServicesProvider.Instance.ProjectService;
            _bankServices = CoreServicesProvider.Instance.BankServices;

            // これらの設定をしないとドキュメントの情報が失われます。
            {
                SoundProjectParameterTranslateSndEdit.SndEditParamNames = new string[]
                {
                ProjectParameterNames.SndEditSetting.SyncPort,
                ProjectParameterNames.SndEditSetting.FuncPort,
                ProjectParameterNames.SndEditSetting.CtrlPort,
                };

                SoundProjectTranslatorSettings.EnabledSoundArchiveOutputType = true;
            }
        }

        internal void Execute(IResultWriter resultWriter)
        {
            Ensure.Argument.NotNull(resultWriter, nameof(resultWriter));

            foreach (var filePath in _settings.InputFilePaths)
            {
                this.UpdateProjectVersion(resultWriter, filePath);
            }
        }

        private void UpdateProjectVersion(IResultWriter resultWriter, string projFilePath)
        {
            IList<DocumentReference> documentReferences = null;

            try
            {
                var unsupportedVersionFiles = new List<string>();
                var oldVersionFiles = new List<string>();

                documentReferences = FileVersionUtility.GetDifferentVersionFilesWithProjectFile(
                    projFilePath,
                    ref unsupportedVersionFiles,
                    ref oldVersionFiles);

                if (unsupportedVersionFiles.Count > 0)
                {
                    this.AbortUnsupportedVersionFiles(unsupportedVersionFiles);
                }

                // プロジェクトを開きます。
                _projectService.Open(projFilePath);

                // バンクを開きます。
                _projectService.Project
                    .EnumerateComponentTree()
                    .OfType<SoundSetBankBase>()
                    .Select(component => component.GetFilePath())
                    .Where(path => !string.IsNullOrEmpty(path))
                    .ForEach(path => _bankServices.OpenItem(path));

                var savedFiles = new List<string>();

                // プロジェクトを保存します。
                foreach (var document in _projectService.Documents.Values)
                {
                    if (_settings.ForceUpdate || oldVersionFiles.Contains(document.Resource.Key))
                    {
                        this.SaveDocument(resultWriter, document, oldVersionFiles);
                    }
                }

                // バンクを保存します。
                foreach (var document in _bankServices.Values.Select(it => it.BankDocument))
                {
                    if (_settings.ForceUpdate || oldVersionFiles.Contains(document.Resource.Key))
                    {
                        this.SaveDocument(resultWriter, document, oldVersionFiles);
                    }
                }
            }
            finally
            {
                _projectService.Close();
                _bankServices.Clear();
                documentReferences.ForEach(it => it.Dispose());
            }
        }

        private void SaveDocument(
            IResultWriter resultWriter,
            Document document,
            IEnumerable<string> oldVersionFiles)
        {
            var filePath = document.Resource.Key;

            Ensure.Operation.True(File.Exists(filePath), nameof(filePath));

            if (_savedFiles.Contains(filePath))
            {
                return;
            }

            _savedFiles.Add(filePath);

            if (_settings.BackupOldFile && !_settings.DryRun)
            {
                this.MakeBackupFile(filePath);
            }

            if (!_settings.Silent)
            {
                this.PrintFilePath(resultWriter, filePath);
            }

            if (!_settings.DryRun)
            {
                _documentService.SaveDocument(document);
            }
        }

        private void MakeBackupFile(string filePath)
        {
            File.Copy(filePath, filePath + BackupFileSuffix);
        }

        private void PrintFilePath(IResultWriter resultWriter, string filePath)
        {
            resultWriter.Out.WriteLine(this.FormatFilePath(filePath));
        }

        private string FormatFilePath(string filePath)
        {
            return PathUtility.FormatFilePath(filePath, _settings.BasePath, _settings.UseBackslash);
        }

        private void AbortUnsupportedVersionFiles(List<string> unsupportedVersionFiles)
        {
            var stringBuilder = new StringBuilder();
            foreach (string newVersionFile in unsupportedVersionFiles)
            {
                stringBuilder.AppendLine(this.FormatFilePath(newVersionFile));
            }

            throw new ApplicationException(
                string.Format(
                    MessageResources.ErrorMessage_UnsupportedFileVersion,
                    stringBuilder.ToString()));
        }
    }
}
