﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Windows.Forms;

    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Resources;

    /// <summary>
    ///
    /// </summary>
    public class Finder
    {
        private FindArgs.TargetDocumentKinds targetDocumentKind;
        private SoundProjectService projectService = null;

        /// <summary>
        ///
        /// </summary>
        public static bool IsMatchCondition(FindArgs e, Component component)
        {
            List<string> list = new List<string>();

            if (e.EnableLabel != false)
            {
                list.Add(component.Name);
            }

            if (e.EnableFilePath != false)
            {
                list.AddRange(GetFilePaths(component));
            }

            if (e.EnableComment != false)
            {
                list.AddRange(GetComments(component));
            }

            if (e.EnableItemReference != false)
            {
                list.AddRange(GetItemReferences(component));
            }

            if (e.EnableStartPosition == true &&
                component is SequenceSoundBase)
            {
                SequenceSoundBase sequenceSound = component as SequenceSoundBase;
                list.Add(sequenceSound.StartPosition);
            }

            if (list.Count > 0)
            {
                return e.IsMatch(list);
            }

            return false;
        }

        /// <summary>
        ///
        /// </summary>
        public static string ReplaceMetaStrings(string text)
        {
            string[] metaStrings = {
                "\\", //これは必ず最初にすること
                ".",
                "$",
                "^",
                "{", "}",
                "[", "]",
                "(", ")",
                "|",
                //"*",
                "+",
                "?",
            };

            foreach (string metaString in metaStrings)
            {
                text = text.Replace(metaString, "\\" + metaString);
            }

            return text;
        }

        /// <summary>
        /// コマンドラインからのコマンド専用です。
        /// </summary>
        public Component FindInstrumentFromProject(SoundProjectService projectService, string name)
        {
            if (projectService == null)
            {
                return null;
            }
            this.projectService = projectService;

            FindArgs args = new FindArgs();
            args.Range = FindArgs.Ranges.Project;
            args.EnableLabel = true;
            args.EnableFilePath = false;
            args.EnableComment = false;
            args.EnableItemReference = false;
            args.EnableStartPosition = false;
            args.CaseSensitive = false;
            args.Repeat = false;
            args.OnlyOneThing = true;
            args.Text = "^" + name + "$";

            this.targetDocumentKind = FindArgs.TargetDocumentKinds.Bank;
            Component[] components = FindForProject(args);
            return components.Length > 0 ? components[0] : null;
        }

        /// <summary>
        /// 検索を行います。
        /// </summary>
        public Component[] Find(SoundProjectService projectService, FindArgs.TargetDocumentKinds targetDocumentKind, FindArgs args)
        {
            if (projectService == null)
            {
                return null;
            }

            this.projectService = projectService;
            this.targetDocumentKind = targetDocumentKind;

            //CommonTabPage page = null;
            //CommonPanel panel = null;
            Component[] components = null;

            switch (args.Range)
            {
                case FindArgs.Ranges.Project:
                    components = FindForProject(args);
                    break;

                case FindArgs.Ranges.SoundSet:
                    components = FindForSoundSet(args);
                    break;

                case FindArgs.Ranges.CurrentSoundSet:
                case FindArgs.Ranges.CurrentBank:
                case FindArgs.Ranges.CurrentList:
                    Debug.Assert(false, "Not implimented.");
                    break;
            }

            return components;
        }

        /// <summary>
        ///
        /// </summary>
        internal static string[] GetFilePaths(Component component)
        {
            List<Component> list = new List<Component>();

            if (component is Instrument)
            {
                foreach (KeyRegion keyRegion in component.Children)
                {
                    foreach (VelocityRegion velRegion in keyRegion.Children)
                    {
                        list.Add(velRegion);
                    }
                }
            }
            else
            {
                list.Add(component);
            }

            //
            string[] filePaths = list
                .Select(c => GetFilePath(c))
                .Where(s => s != null)
                .ToArray();

            return filePaths;
        }

        /// <summary>
        ///
        /// </summary>
        internal static string[] GetComments(Component component)
        {
            return new String[] {
                component.Comment,
                component.Comment1,
                component.Comment2,
                component.Comment3,
                component.Comment4,
                component.Comment5,
                component.Comment6,
                component.Comment7,
                component.Comment8,
                component.Comment9
            };
        }

        /// <summary>
        ///
        /// </summary>
        internal static string[] GetItemReferences(Component component)
        {
            List<string> list = new List<string>();

            if (component is Sound)
            {
                Sound sound = component as Sound;
                if (sound.TargetPlayer != null)
                {
                    list.Add(sound.TargetPlayer.Name);
                }
            }

            if (component is SequenceSoundBase)
            {
                SequenceSoundBase sequenceSound = component as SequenceSoundBase;
                foreach (ComponentReference reference in sequenceSound.SoundSetBankReferences)
                {
                    list.Add(reference.TargetName);
                }
            }

            if (component is GroupBase)
            {
                GroupBase group = component as GroupBase;
                foreach (GroupItemBase groupItem in group.Children)
                {
                    if (groupItem.Target != null)
                    {
                        list.Add(groupItem.Target.Name);
                    }
                }
            }

            return list.ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        private static string GetFilePath(Component component)
        {
            if (component.Parameters.ContainsKey(ProjectParameterNames.FilePath) == false)
            {
                return null;
            }

            return component.Parameters[ProjectParameterNames.FilePath].Value as String;
        }

        /// <summary>
        ///
        /// </summary>
        private FormsApplication Application
        {
            get { return FormsApplication.Instance; }
        }

        /// <summary>
        /// プロジェクト全体から検索結果を取得します。
        /// </summary>
        private Component[] FindForProject(FindArgs e)
        {
            List<Component> list = new List<Component>();
            ImaginaryDocument[] documents = null;

            documents = GetFindTargetDocuments(e);
            foreach (ImaginaryDocument document in documents)
            {
                list.AddRange(document.GetMatchConditionComponents(e));

                if (e.OnlyOneThing == true && list.Count > 0)
                {
                    break;
                }
            }

            return list.ToArray();
        }

        /// <summary>
        /// 指定サウンドセットから検索結果を取得します。
        /// </summary>
        private Component[] FindForSoundSet(FindArgs e)
        {
            List<Component> list = new List<Component>();

            HashSet<string> targetNames = new HashSet<string>();
            if (e.Range == FindArgs.Ranges.SoundSet &&
                e.TargetSoundSetNames != null)
            {
                foreach (string name in e.TargetSoundSetNames)
                {
                    targetNames.Add(name);
                }
            }

            //
            ImaginaryDocument[] documents = GetFindTargetDocuments(e);
            foreach (ImaginaryDocument document in documents)
            {
                if (document.Document is SoundSetDocument)
                {
                    SoundSetDocument soundSetDocument = document.Document as SoundSetDocument;
                    if (targetNames.Contains(soundSetDocument.SoundSet.Name) != false)
                    {
                        list.AddRange(document.GetMatchConditionComponents(e));
                    }
                }
            }
            return list.ToArray();
        }

#if false
        /// <summary>
        /// 現在のパネルから検索結果を取得
        /// </summary>
        private Component[] FindForCurrentPanel( FindArgs e, CommonPanel panel)
        {
            ImaginaryDocument   imaginaryDocument = null;
            string              filePath = null;

            if( panel is SoundSetPanel )
            {
                filePath = panel.Document.Resource.Key;
                imaginaryDocument = new ImaginarySoundSetDocument
                    ( panel.Document as SoundSetDocument, filePath);
            }

            if( panel is BankPanel )
            {
                filePath = panel.Document.Resource.Key;
                imaginaryDocument = new ImaginaryBankDocument
                    ( panel.Document as BankDocument, filePath);
            }

            return imaginaryDocument.GetMatchConditionComponents( e);
        }

        /// <summary>
        /// 現在のリストから検索結果を取得
        /// </summary>
        private Component[] FindForCurrentList( FindArgs e, CommonPanel panel)
        {
            List<Component>   list = new List<Component>();
            CommonListCtrl      listCtrl = null;

            listCtrl = panel.CurrentListCtrl;

            foreach( ComponentListItem item in listCtrl.ItemsSource.Items ) {
                if( IsMatchCondition( e, item) != false ) {

                    if( item.Target is Instrument ) {
                        BankDocument  bankDocument = null;
                        ImaginaryInstrument imaginaryInstrument = null;
                        bankDocument = panel.Document as BankDocument;

                        imaginaryInstrument = new ImaginaryInstrument
                            ( item.Target, bankDocument.Resource.Key);

                        list.Add( imaginaryInstrument);

                    } else {

                        list.Add( item.Target);
                    }
                }
            }

            return list.ToArray();
        }
#endif

        /// <summary>
        /// 検索対象となるドキュメントを取得します。
        /// </summary>
        private ImaginaryDocument[] GetFindTargetDocuments(FindArgs ea)
        {
            ImaginaryDocumentCollection collection = null;
            ImaginaryDocument imaginaryDocument = null;
            BankService bankService = null;
            BankDocument bankDocument = null;
            string filePath = null;

            collection = new ImaginaryDocumentCollection();

            switch (this.targetDocumentKind)
            {
                case FindArgs.TargetDocumentKinds.SoundSet:
                    foreach (SoundSetDocument soundSetDocument in this.projectService.SoundSetDocuments)
                    {
                        filePath = soundSetDocument.Resource.Key;
                        if (collection.Contains(filePath) == false)
                        {
                            imaginaryDocument = new ImaginarySoundSetDocument
                                (soundSetDocument, filePath);
                            collection.Add(imaginaryDocument);
                        }
                    }
                    break;

                case FindArgs.TargetDocumentKinds.Bank:
                    foreach (SoundSetBankBase bank in this.projectService.SoundSetBanks)
                    {
                        if (Application.BankServices.Contains(bank.FilePath) != false)
                        {
                            bankService = Application.BankServices[bank.FilePath];
                            bankDocument = bankService.BankDocument;
                        }
                        else
                        {
                            bankDocument = null;
                        }

                        filePath = bank.FilePath;
                        if (collection.Contains(filePath) == false)
                        {
                            imaginaryDocument = new ImaginaryBankDocument(bankDocument, filePath);
                            collection.Add(imaginaryDocument);
                        }
                    }
                    break;
            }

            return collection.ToArray();
        }

    }

    /// <summary>
    ///
    /// </summary>
    internal class ImaginaryDocumentCollection : KeyedCollection<string, ImaginaryDocument>
    {
        protected override string GetKeyForItem(ImaginaryDocument imaginaryDocument)
        {
            return imaginaryDocument.FilePath;
        }
    }
}
