﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using NintendoWare.SoundFoundation.Core.Parameters;
using NintendoWare.SoundFoundation.Core.Resources;
using NintendoWare.SoundFoundation.Documents;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundMaker.Framework.FileFormats;
using NintendoWare.SoundMaker.Framework.Resources;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    public class SaveHelper
    {
        public static event SaveHelperEventHandler Saving;
        public static event SaveHelperEventHandler Saved;

        // 編集があったものすべてを保存します。
        public static bool SaveAll(Form owner = null)
        {
            bool result = true;

            if (IsModifyProject == true)
            {
                string[] fileList = GetModifyFileList();
                result = Save(() =>
                {
                    ProgressDialog dialog = new ProgressDialog();
                    dialog.Owner = owner;
                    dialog.DoWork += DoWork;
                    dialog.ProgressChanged += ProgressChanged;
                    dialog.RunWorkerCompleted +=
                    (object sender, RunWorkerCompletedEventArgs e) =>
                    {
                        dialog.Close();
                    };
                    dialog.Title = MessageResource.Message_SaveProject;
                    dialog.Message = MessageResource.Message_SaveProjectFile;
                    dialog.ShowDialog();
                }, fileList);
            }

            return result;
        }

        // 変更が無くても指定されたファイルを強制的にセーブします。
        public static bool SaveForce(string[] fileList)
        {
            if (fileList == null)
            {
                return false;
            }

            bool result = Save(() =>
            {
                ProgressDialog dialog = new ProgressDialog();
                dialog.DoWork += ((object sender, DoWorkEventArgs e) =>
                        {
                            DoWorkSaveForce(sender, e, fileList);
                        });
                dialog.ProgressChanged += ProgressChanged;
                dialog.RunWorkerCompleted +=
                (object sender, RunWorkerCompletedEventArgs e) =>
                {
                    dialog.Close();
                };
                dialog.Title = MessageResource.Message_SaveFiles;
                dialog.Message = MessageResource.Message_SavingFiles;
                dialog.ShowDialog();
            }, fileList);

            return result;
        }

        // 保存の前後処理を実行します。
        public static bool Save(Action saveDelegate, string[] fileList)
        {
            bool result = false;
            string fileListPath = CreateSaveFileList(fileList);

            try
            {
                OnSaving(fileList);
                Application.RealtimeEditService.DisableParameterValueChanged();
                if (FileSavePreProcess(fileListPath) == true)
                {
                    saveDelegate();
                    result = FileSavePostProcess(fileListPath);
                }
            }
            finally
            {
                OnSaved(fileList, result);
                Application.RealtimeEditService.EnableParameterValueChanged();
                DeleteSaveFileList(fileListPath);
            }

            return result;
        }

        private static void SaveProject()
        {
            if (Application.ProjectService.IsOpened == true)
            {
                Application.ProjectService.Save();
            }

            SaveBanks();
        }

        private static void SaveBanks()
        {
            foreach (BankService bankService in Application.BankServices.Values)
            {
                if (bankService.IsOpened == false) { continue; }
                bankService.Save();
            }
        }

        private static ApplicationBase Application
        {
            get { return ApplicationBase.Instance; }
        }

        private static void DoWork(object sender, DoWorkEventArgs e)
        {
            SaveProject();
        }

        private static void DoWorkSaveForce(object sender, DoWorkEventArgs e, string[] fileList)
        {
            foreach (string filePath in fileList)
            {
                FileResource resource = new FileResource(filePath);
                using (DocumentReference docRef = Application.DocumentService.OpenDocument(resource))
                {
                    if (docRef != null)
                    {
                        Application.DocumentService.SaveDocument(docRef.Document);
                    }
                }
            }
        }

        private static void ProgressChanged(object sender, ProgressChangedEventArgs e)
        {
        }

        private static bool IsModifyProject
        {
            get
            {
                foreach (SoundDocument document in Application.ProjectService.Documents.Values)
                {
                    if (document.IsDirty == true)
                    {
                        return true;
                    }
                }

                foreach (BankService bankService in Application.BankServices.Values)
                {
                    if (null != bankService.BankDocument &&
                        null != bankService.BankDocument.Resource &&
                        bankService.BankDocument.IsDirty == true)
                    {
                        return true;
                    }
                }

                return false;
            }
        }

        private static Document[] GetModifyDocuments()
        {
            List<Document> docList = new List<Document>();

            foreach (SoundDocument document in Application.ProjectService.Documents.Values)
            {
                if (document.IsDirty == true)
                {
                    docList.Add(document);
                }
            }

            foreach (BankService bankService in Application.BankServices.Values)
            {
                if (null != bankService.BankDocument &&
                    null != bankService.BankDocument.Resource &&
                    bankService.BankDocument.IsDirty == true)
                {
                    docList.Add(bankService.BankDocument);
                }
            }

            return docList.ToArray();
        }

        private static string[] GetModifyFileList()
        {
            return GetModifyDocuments().Select(s => s.Resource.Key).ToArray();
        }

        private static string CreateSaveFileList(string[] fileList)
        {
            string fileListPath = Path.GetTempFileName();
            using (StreamWriter writer = new StreamWriter(fileListPath, false, Encoding.UTF8))
            {
                foreach (string filePath in fileList)
                {
                    writer.WriteLine(filePath);
                }
            }

            return fileListPath;
        }

        private static void DeleteSaveFileList(string fileListPath)
        {
            if (string.IsNullOrEmpty(fileListPath) == false &&
                File.Exists(fileListPath) == true)
            {
                File.Delete(fileListPath);
            }
        }

        private static bool FileSavePreProcess(string fileListPath)
        {
            FileSavePreProcessCommand();       // 内部処理用
            return FileSavePreCommand(fileListPath);  // ユーザーコマンド
        }

        private static bool FileSavePostProcess(string fileListPath)
        {
            FileSavePostProcessCommand();      // 内部処理用
            return FileSavePostCommand(fileListPath); // ユーザーコマンド
        }

        private static void FileSavePreProcessCommand()
        {
        }

        private static void FileSavePostProcessCommand()
        {
        }

        private static bool FileSavePreCommand(string fileListPath)
        {
            SoundProject project = Application.ProjectService.Project;
            if (project.IsFileSavePreCommandEnabled == true)
            {
                string path = Environment.ExpandEnvironmentVariables(project.FileSavePreCommandPath);
                if (File.Exists(path) == false)
                {
                    string msg = string.Format(Resources.MessageResource.Message_ErrorNoExistFileEventPreCommand, path);
                    MessageBox.Show(msg,
                                    Resources.MessageResource.Message_Error,
                                    MessageBoxButtons.OK,
                                    MessageBoxIcon.Error);
                    return false;
                }

                string errorMessage = Resources.MessageResource.Message_ErrorFileEventPreCommand;
                string exceptionMessage = Resources.MessageResource.Message_ErrorExecuteFileEventPreCommand;
                return ExecuteCommand(path, fileListPath, errorMessage, exceptionMessage);
            }

            return true;
        }

        private static bool FileSavePostCommand(string fileListPath)
        {
            SoundProject project = Application.ProjectService.Project;
            if (project.IsFileSavePostCommandEnabled == true)
            {
                string path = Environment.ExpandEnvironmentVariables(project.FileSavePostCommandPath);
                if (File.Exists(path) == false)
                {
                    string msg = string.Format(Resources.MessageResource.Message_ErrorNoExistFileEventPostCommand, path);
                    MessageBox.Show(msg,
                                    Resources.MessageResource.Message_Error,
                                    MessageBoxButtons.OK,
                                    MessageBoxIcon.Error);
                    return false;
                }

                string errorMessage = Resources.MessageResource.Message_ErrorFileEventPostCommand;
                string exceptionMessage = Resources.MessageResource.Message_ErrorExecuteFileEventPostCommand;
                return ExecuteCommand(path, fileListPath, errorMessage, exceptionMessage);
            }

            return true;
        }

        private static bool ExecuteCommand(string commandPath, string fileListPath, string errorMessage, string exceptionMessage)
        {
            using (Process commandProcess = new Process())
            {
                string outputMessages = string.Empty;

                commandProcess.StartInfo.FileName = commandPath;
                commandProcess.StartInfo.Arguments = "\"" + fileListPath + "\"";
                commandProcess.StartInfo.CreateNoWindow = true;
                commandProcess.StartInfo.UseShellExecute = false;
                commandProcess.StartInfo.RedirectStandardOutput = true;
                commandProcess.StartInfo.RedirectStandardError = true;
                commandProcess.OutputDataReceived +=
                    (object send, DataReceivedEventArgs outputLine) =>
                    {
                        if (outputLine.Data == null || outputLine.Data.Length == 0) { return; }
                        outputMessages += outputLine.Data + "\n";
                    };
                commandProcess.ErrorDataReceived +=
                    (object send, DataReceivedEventArgs outputLine) =>
                    {
                        if (outputLine.Data == null || outputLine.Data.Length == 0) { return; }
                        outputMessages += outputLine.Data + "\n";
                    };

                try
                {
                    commandProcess.Start();
                    commandProcess.BeginOutputReadLine();
                    commandProcess.BeginErrorReadLine();
                    commandProcess.WaitForExit();

                    if (commandProcess.ExitCode != 0)
                    {
                        TextDisplayMessageBox messageBox =
                            new TextDisplayMessageBox(errorMessage,
                                                      outputMessages,
                                                      TextDisplayMessageBoxStyle.OKButton);
                        messageBox.ShowDialog();
                        return false;
                    }
                }
                catch (Exception e)
                {
                    MessageBox.Show(exceptionMessage + "\n" + e.Message,
                                    Resources.MessageResource.Message_Error,
                                    MessageBoxButtons.OK,
                                    MessageBoxIcon.Error);
                    return false;
                }
            }

            return true;
        }

        private static void OnSaving(string[] filePaths)
        {
            if (Saving != null)
            {
                Saving(null, new SaveHelperEventArgs(filePaths));
            }
        }

        private static void OnSaved(string[] filePaths, bool isSuccess)
        {
            if (Saved != null)
            {
                Saved(null, new SaveHelperEventArgs(filePaths, isSuccess));
            }
        }
    }
}
