﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Drawing;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.Conversion;
    using NintendoWare.SoundFoundation.Core.Drawing;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Resources;

    /// <summary>
    /// 出力パネル
    /// </summary>
    public partial class ConvertLogPanel : DockingPage, IToolWindowPage
    {
        #region ** 固定値

        public const string PageName = "ConvertLog";

        #endregion

        private string _targetFilePath;
        private Control _invokeControl;

        private List<BankServiceReference> bankServiceRefs = new List<BankServiceReference>();

        public ConvertLogPanel()
            : base(PageName)
        {
            InitializeComponent();

            // テキストとアイコンをロードする
            Text = MessageResource.ToolWindowName_ConvertLog;
            Image = ImageResource.BitmapIconOutputWindow.
                                        MakeNewTransparent(UIServiceBase.TransparentColor);
            _toolMove2Item.Image = ImageResource.BitmapIconMove2Error.
                                        MakeNewTransparent(UIServiceBase.TransparentColor);

            DockVisible = false;

            // 初期配置を設定する
            FloatingBounds = new Rectangle(-1, -1,
                                            Width + (SystemInformation.FrameBorderSize.Width * 2),
                                            Height + SystemInformation.CaptionHeight + SystemInformation.FrameBorderSize.Height);

            // イベントハンドラを登録する
            ApplicationBase.Instance.AppActivated += OnAppActivated;
            ApplicationBase.Instance.AppDeactivated += OnAppDeactivated;

            ApplicationBase.Instance.ConvertService.BeginConvert += OnBeginConvert;
            ApplicationBase.Instance.ConvertService.OutputLine += OnOutputLineOnConvertThread;

            UpdateButtons();

            // コンテキストメニューを初期化する
            if (null != _log.ContextMenuStrip)
            {
                _log.ContextMenuStrip.Items.Add(_menuMove2Item);
            }

            _invokeControl = new Control();
            _invokeControl.CreateControl();
        }

        #region ** プロパティ

        private SoundProjectService ProjectService
        {
            get { return ApplicationBase.Instance.ProjectService; }
        }

        private BankServiceManager BankServices
        {
            get { return ApplicationBase.Instance.BankServices; }
        }

        private MainWindow MainWindow
        {
            get { return FormsApplication.Instance.UIService.MainWindow; }
        }

        private OutputLine SelectedLine
        {
            get
            {
                if (1 != _log.SelectedItems.Length) { return null; }
                return _log.SelectedItems[0].UserData as OutputLine;
            }
        }

        private Component TargetItem
        {
            get
            {
                if (null == SelectedLine || 0 == SelectedLine.Components.Length)
                {
                    return null;
                }

                _targetFilePath = null;

                Instrument targetInstrument = null;

                foreach (Component item in SelectedLine.Components)
                {
                    if (item is StreamSoundBase ||
                        item is WaveSoundSetBase ||
                        item is WaveSoundBase ||
                        item is SequenceSoundSetBase ||
                        item is SequenceSoundBase ||
                        item is PlayerBase ||
                        item is GroupBase)
                    {
                        if (!ProjectService.ComponentDictionary.Contains(item.Name))
                        {
                            return null;
                        }

                        return ProjectService.ComponentDictionary[item.Name][0];
                    }

                    if (item is SoundSetBankBase)
                    {
                        BankServiceReference bankServiceRef = null;

                        try
                        {
                            SoundSetBankBase soundSetBank = item as SoundSetBankBase;

                            bankServiceRef = BankServices.OpenItem(soundSetBank.FilePath);
                            if (bankServiceRef == null)
                            {
                                return null;
                            }

                            _targetFilePath = soundSetBank.FilePath;

                            this.bankServiceRefs.Add(bankServiceRef);

                            if (targetInstrument != null)
                            {
                                return targetInstrument;
                            }
                            else
                            {
                                return soundSetBank;
                            }
                        }
                        catch
                        {
                            if (bankServiceRef != null)
                            {
                                this.bankServiceRefs.Remove(bankServiceRef);
                                bankServiceRef.Close();
                            }

                            return null;
                        }
                    }

                    if (item is Instrument)
                    {
                        targetInstrument = item as Instrument;
                        continue;
                    }
                    else if (item is KeyRegion)
                    {
                        targetInstrument = item.Parent as Instrument;
                        continue;
                    }
                    else if (item is VelocityRegion)
                    {
                        targetInstrument = item.Parent.Parent as Instrument;
                        continue;
                    }
                }

                return null;
            }
        }

        #endregion

        #region ** イベントハンドラ

        private void OnBeginConvert(object sender, SoundProjectConvertEventArgs e)
        {
            if (!e.IsPartsConvert)
            {
                _log.Clear();
            }
        }

        private void OnOutputLine(object sender, OutputLineEventArgs e)
        {
            foreach (OutputLine line in e.Lines)
            {
                _log.Append(line.Text, line);
            }

            _log.ScrollToBottom();
        }

        private void OnOutputLineOnConvertThread(object sender, OutputLineEventArgs e)
        {
            // ConvertLogPanel が非表示のまま、ウィンドウハンドルが生成される前に呼び出されることもあるので、
            // Invoke 用のコントロールを使ってスレッド間同期をとる
            _invokeControl.BeginInvoke(new EventHandler<OutputLineEventArgs>(OnOutputLine), new object[] { sender, e });
        }

        private void OnClearLogClick(object sender, EventArgs e)
        {
            _log.Clear();
        }

        private void OnMove2ItemClick(object sender, EventArgs e)
        {
            Move2TargetItem();
        }

        private void OnLogSelectedItemsChanged(object sender, EventArgs e)
        {
            UpdateButtons();
        }

        private void OnLogMouseDoubleClick(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                Move2TargetItem();
            }
        }

        private void OnLogKeyDown(object sender, KeyEventArgs e)
        {
            switch (e.KeyCode)
            {
                case Keys.Enter:
                    Move2TargetItem();
                    break;
            }
        }

        private void OnAppActivated(object sender, EventArgs e)
        {
            _log.Invalidate();
        }

        private void OnAppDeactivated(object sender, EventArgs e)
        {
            _log.Invalidate();
        }

        #endregion

        #region ** メソッド

        private void UpdateButtons()
        {
            if (null == SelectedLine)
            {
                _toolMove2Item.Enabled = false;
                _menuMove2Item.Enabled = _toolMove2Item.Enabled;
                return;
            }

            switch (SelectedLine.Level)
            {
                case OutputLevel.Error:
                case OutputLevel.Warning:
                    _toolMove2Item.Enabled = TargetItem != null;
                    _menuMove2Item.Enabled = _toolMove2Item.Enabled;
                    break;

                default:
                    _toolMove2Item.Enabled = false;
                    _menuMove2Item.Enabled = _toolMove2Item.Enabled;
                    break;
            }
        }

        private void Move2TargetItem()
        {
            if (!_toolMove2Item.Enabled) { return; }

            Component item = TargetItem;

            try
            {
                if (null != _targetFilePath)
                {
                    MainWindow.OpenFile(_targetFilePath);
                }

                MainWindow.ShowPageByComponent(item);

                return;
            }
            catch
            {
            }
        }

        #endregion
    }
}
