﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core.Drawing;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Resources;

    /// <summary>
    ///
    /// </summary>
    public partial class IntParameterEditPanel : UserControl, IParameterEditPanel
    {
        private Component[] targetComponents = null;
        private string name = null;

        public event ValueChangingEventHandler ValueChanging;
        public event ValueChangedEventHandler ValueChanged;

        private bool suspendEvent = false;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public IntParameterEditPanel()
        {
            InitializeComponent();

            this.textBox.ValueChanged += OnFinished;
            this.textBox.Leave += OnLeave;
        }

        /// <summary>
        ///
        /// </summary>
        public Component[] Components
        {
            get
            {
                return this.targetComponents;
            }
            set
            {
                this.targetComponents = value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string ParameterName
        {
            set
            {
                this.name = value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string Label
        {
            set
            {
                this.label.Text = value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public float Minimum
        {
            get
            {
                return Convert.ToSingle(this.trackBar.Minimum);
            }
            set
            {
                Decimal decimalValue = Convert.ToDecimal(value);
                this.trackBar.Minimum = decimalValue;
                this.textBox.Minimum = decimalValue;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public float Maximum
        {
            get
            {
                return Convert.ToSingle(this.trackBar.Maximum);
            }
            set
            {
                Decimal decimalValue = Convert.ToDecimal(value);
                this.trackBar.Maximum = decimalValue;
                this.textBox.Maximum = decimalValue;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public int Value
        {
            get
            {
                return Convert.ToInt32(this.trackBar.Value);
            }
            set
            {
                SetValueToUI(value);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdateUI()
        {
            if (this.targetComponents == null)
            {
                return;
            }

            if (this.targetComponents.Length == 1)
            {
                Component component = this.targetComponents[0];
                IParameterValue value = component.Parameters[this.name];
                SetValueToUI((int)value.Value);
            }
            else
            {
                // 未実装
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void SetValueToUI(int value)
        {
            try
            {
                this.suspendEvent = true;

                this.trackBar.Value = (Decimal)value;
                this.textBox.Text = value.ToString();
            }
            finally
            {
                this.suspendEvent = false;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private bool IsValueChanged(int value)
        {
            if (this.targetComponents.Length == 1)
            {
                Component component = this.targetComponents[0];
                int origValue = (int)component.Parameters[this.name].Value;
                return origValue != value ? true : false;
            }
            else
            {
                // 未実装
                return false;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void SetValue(int value)
        {
            // 同じ値ならば再設定を行いません。
            if (IsValueChanged(value) == false)
            {
                return;
            }

            //
            SetValueToUI(value);

            if (ValueChanged != null)
            {
                ValueChanged(this, new IntValueChangedEventArgs
                              (this.targetComponents, this.name, value));
            }
        }

#if false
        /// <summary>
        ///
        /// </summary>
        private void SetValueByText( string text)
        {
            if( String.IsNullOrEmpty( text) != false )
            {
                return;
            }

            try
            {
                int value = int.Parse( text);
                if( value < Minimum ||
                    value > Maximum )
                {
                    return;
                }

                SetValue( value);
            }
            catch
            {
            }
        }
#endif

        /// <summary>
        ///
        /// </summary>
        private void OnFinished(object sender, ValueChangedEventArgs e)
        {
            if (this.suspendEvent != false)
            {
                return;
            }

            NNumericUpDown control = sender as NNumericUpDown;
            Debug.Assert(control != null, "Sender is not NNumericUpDown control");
            SetValue((int)control.Value);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnLeave(object sender, EventArgs e)
        {
            if (this.suspendEvent != false)
            {
                return;
            }

            NNumericUpDown control = sender as NNumericUpDown;
            Debug.Assert(control != null, "Sender is not NNumericUpDown control");
            SetValue((int)control.Value);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnValueChanged(object sender, ValueChangedEventArgs e)
        {
            if (this.suspendEvent != false)
            {
                return;
            }

            NTrackBar trackBar = sender as NTrackBar;
            int value = Convert.ToInt32(trackBar.Value);

            if (trackBar.IsDragging != false)
            {
                SetValueToUI(value);

                if (ValueChanging != null)
                {
                    ValueChanging(this, new IntValueChangingEventArgs
                                   (this.targetComponents, this.name, value));
                }
            }
            else
            {
                SetValue(value);
            }
        }

#if false
        /// <summary>
        ///
        /// </summary>
        private void OnValueChanging(object sender, TrackBarEventArgs e)
        {
            int value = trackBar.Value;
            SetValueToUI( value);

            if( ValueChanging != null )
            {
                ValueChanging( this, new IntValueChangingEventArgs
                               ( this.targetComponents, this.name, value));
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnValueChanged(object sender, TrackBarEventArgs e)
        {
            SetValue( trackBar.Value);
        }
#endif
    }

    /// <summary>
    ///
    /// </summary>
    public class ValueChangeEventArgs
    {
        private Component[] components = null;
        private string name = null;
        private object value = 0;

        /// <summary>
        ///
        /// </summary>
        public ValueChangeEventArgs(Component[] components, string name, object value)
        {
            this.components = components;
            this.name = name;
            this.value = value;
        }

        /// <summary>
        ///
        /// </summary>
        public Component[] Components
        {
            get
            {
                return this.components;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get
            {
                return this.name;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public object Value
        {
            get
            {
                return this.value;
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class IntValueChangeEventArgs : ValueChangeEventArgs
    {
        public IntValueChangeEventArgs(Component[] components, string name, object value) :
            base(components, name, value)
        {
        }
    }

    /// <summary>
    ///
    /// </summary>
    public delegate void ValueChangingEventHandler(object sender, ValueChangeEventArgs e);
    public delegate void ValueChangedEventHandler(object sender, ValueChangeEventArgs e);

    /// <summary>
    ///
    /// </summary>
    public class IntValueChangingEventArgs : IntValueChangeEventArgs
    {
        /// <summary>
        ///
        /// </summary>
        public IntValueChangingEventArgs(Component[] components, string name, int value) :
            base(components, name, value)
        {
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class IntValueChangedEventArgs : IntValueChangeEventArgs
    {
        /// <summary>
        ///
        /// </summary>
        public IntValueChangedEventArgs(Component[] components, string name, int value) :
            base(components, name, value)
        {
        }
    }


    ///
    public interface IParameterEditPanel
    {
        Component[] Components { get; set; }
        void UpdateUI();
    }
}
