﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Drawing;
    using System.IO;
    using System.Linq;
    using System.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.Preset;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers;

    /// <summary>
    ///
    /// </summary>
    public partial class OptionDialog : Form
    {
        private static IndexAndFunctionName[] IndexAndFunctionNames = new IndexAndFunctionName[]
            {
                new IndexAndFunctionName( 0, KeyAssignment.Functions.Preview_Play),
                new IndexAndFunctionName( 1, KeyAssignment.Functions.Preview_TogglePlayStop),
            };

        private Dictionary<string, Dictionary<string, XmlList>> presetXmlList =
            new Dictionary<string, Dictionary<string, XmlList>>();

        /// <summary>
        ///
        /// </summary>
        public OptionDialog()
        {
            InitializeComponent();
            InitializePresetListColumns();

            //
            this.cmb_KeySpace.SelectedIndex = 0;
            this.cmb_KeySpaceWithShift.SelectedIndex = 0;

            //
            ImageList imageList = new ImageList();
            imageList.TransparentColor = UIServiceBase.TransparentColor;
            imageList.Images.Add(ImageResource.BitmapIconRightArrow);
            this.ArgsButton.ImageList = imageList;
            this.ArgsButton.ImageIndex = 0;
            this.button_midiEditorArgs.ImageList = imageList;
            this.button_midiEditorArgs.ImageIndex = 0;
            this.button_wavEditorArgs.ImageList = imageList;
            this.button_wavEditorArgs.ImageIndex = 0;
            this.NotepadRadioButton.Checked = true;
            this.EditorRadioButton.Checked = false;
            this.comboBox_PresetListColumns.SelectedIndex = 0;
            OnCheckedChangedEditorRadioButton(this, EventArgs.Empty);
        }

        //-----------------------------------------------------------------
        // 全般タブパラメータへのアクセサ
        //-----------------------------------------------------------------

        public bool IsProjectAutoLoad
        {
            get { return this.isProjectAutoLoadCheckBox.Checked; }
            set { this.isProjectAutoLoadCheckBox.Checked = value; }
        }

        public bool IsShowProjectFilePath
        {
            get { return this.isShowProjectFilePath.Checked; }
            set { this.isShowProjectFilePath.Checked = value; }
        }

        public bool IsShowFullPath
        {
            get { return this.isShowFullPathCheckBox.Checked; }
            set { this.isShowFullPathCheckBox.Checked = value; }
        }

        public bool IsLineNumberVisible
        {
            get { return this.isLineNumberVisibleCheckBox.Checked; }
            set { this.isLineNumberVisibleCheckBox.Checked = value; }
        }

        public bool IsShowParameterBar
        {
            get { return this.isShowParameterBarCheckBox.Checked; }
            set { this.isShowParameterBarCheckBox.Checked = value; }
        }

        public bool IsShowGroupItemTargetComment
        {
            get { return this.chk_ShowGroupItemTargetComment.Checked; }
            set { this.chk_ShowGroupItemTargetComment.Checked = value; }
        }

        public bool PasteToItemLower
        {
            get { return this.chk_PasteToItemLower.Checked; }
            set { this.chk_PasteToItemLower.Checked = value; }
        }

        public bool MoveNextLineCaretByEnter
        {
            get { return this.chk_MoveNextLineCaretByEnter.Checked; }
            set { this.chk_MoveNextLineCaretByEnter.Checked = value; }
        }

        public bool BeginEditByEnter
        {
            get { return this.chk_BeginEditByEnter.Checked; }
            set { this.chk_BeginEditByEnter.Checked = value; }
        }

        public Color ParameterBarColor { get; set; }

        public bool IsNotCloseConvertDialog
        {
            get { return this.isNotCloseConvertDialog.Checked; }
            set { this.isNotCloseConvertDialog.Checked = value; }
        }

        public uint ParalellConversionCountMax
        {
            get { return (uint)this.parallelConversionCountMax.Value; }
            set { this.parallelConversionCountMax.Value = value; }
        }

        public bool EditorUseNotepad
        {
            get { return this.NotepadRadioButton.Checked; }
            set
            {
                this.NotepadRadioButton.Checked = value;
                this.EditorRadioButton.Checked = !value;
            }
        }

        public bool EditorUseEditor
        {
            get { return this.EditorRadioButton.Checked; }
            set
            {
                this.EditorRadioButton.Checked = value;
                this.NotepadRadioButton.Checked = !value;
            }
        }

        public string EditorCommand
        {
            get { return this.CommandTextBox.Text; }
            set { this.CommandTextBox.Text = value; }
        }

        public string EditorArguments
        {
            get { return this.ArgsTextBox.Text; }
            set { this.ArgsTextBox.Text = value; }
        }

        public string MidiEditorCommand
        {
            get { return this.textBox_midiEditorCommand.Text; }
            set { this.textBox_midiEditorCommand.Text = value; }
        }

        public string MidiEditorArguments
        {
            get { return this.textBox_midiEditorArgs.Text; }
            set { this.textBox_midiEditorArgs.Text = value; }
        }

        public string WavEditorCommand
        {
            get { return this.textBox_wavEditorCommand.Text; }
            set { this.textBox_wavEditorCommand.Text = value; }
        }

        public string WavEditorArguments
        {
            get { return this.textBox_wavEditorArgs.Text; }
            set { this.textBox_wavEditorArgs.Text = value; }
        }

        public int SelectedTabIndex
        {
            get { return this.optionTabCtrl.SelectedIndex; }
            set { this.optionTabCtrl.SelectedIndex = value; }
        }

        public bool UsePreviewWaveChooseDialog
        {
            get { return this.chk_UsePreviewWaveChooseDialog.Checked; }
            set { this.chk_UsePreviewWaveChooseDialog.Checked = value; }
        }

        public bool MaintainTabOpen
        {
            get { return this.chk_MaintainTabOpen.Checked; }
            set { this.chk_MaintainTabOpen.Checked = value; }
        }

        /// sndedit
        public bool AutoEnableMonitoringTarget
        {
            get
            {
                return this.chk_AutoEnableMonitoringTarget.Checked;
            }
            set
            {
                this.chk_AutoEnableMonitoringTarget.Checked = value;
            }
        }

        public bool AutoEnableBankBySequenceSound
        {
            get
            {
                return this.checkBox_InGameAutoSoundSetBankEdit.Checked;
            }
            set
            {
                this.checkBox_InGameAutoSoundSetBankEdit.Checked = value;
            }
        }

        /// 統計タブ
        public bool EnabledAutoMeasureLoudness
        {
            get
            {
                return this.checkBox_EnabledAutoMeasureLoudness.Checked;
            }
            set
            {
                this.checkBox_EnabledAutoMeasureLoudness.Checked = value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string FunctionOfKeySpace
        {
            get
            {
                return IndexToFunctionName(this.cmb_KeySpace.SelectedIndex);
            }
            set
            {
                int index = FunctionNameToIndex(value);
                this.cmb_KeySpace.SelectedIndex = index;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string FunctionOfKeySpaceWithShift
        {
            get
            {
                return IndexToFunctionName(this.cmb_KeySpaceWithShift.SelectedIndex);
            }
            set
            {
                int index = FunctionNameToIndex(value);
                this.cmb_KeySpaceWithShift.SelectedIndex = index;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void HideInGameTabPage()
        {
            this.tabPage_InGameEdit.Visible = false;
            this.tabPage_InGameEdit.Enabled = false;
            this.optionTabCtrl.Controls.Remove(this.tabPage_InGameEdit);
        }

        /// <summary>
        ///
        /// </summary>
        private class IndexAndFunctionName
        {
            private int index = -1;
            private string name = null;

            public IndexAndFunctionName(int index, string name)
            {
                this.index = index;
                this.name = name;
            }

            public int Index
            {
                get
                {
                    return this.index;
                }
            }

            public string Name
            {
                get
                {
                    return this.name;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private string IndexToFunctionName(int index)
        {
            IndexAndFunctionName[] datas = IndexAndFunctionNames
                .Where(d => d.Index == index)
                .ToArray();
            Debug.Assert(datas.Length == 1, "Not found index data.");
            return datas[0].Name;
        }

        /// <summary>
        ///
        /// </summary>
        private int FunctionNameToIndex(string name)
        {
            IndexAndFunctionName[] datas = IndexAndFunctionNames
                .Where(d => d.Name == name)
                .ToArray();
            Debug.Assert(datas.Length == 1, "Not found name data.");
            return datas[0].Index;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnOK(object sender, EventArgs e)
        {
            this.DialogResult = DialogResult.OK;
            UpdatePresetListColumns();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnCancel(object sender, EventArgs e)
        {
            this.DialogResult = DialogResult.Cancel;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnHelp(object sender, EventArgs e)
        {
            ShowHelpHandler.Execute(FormsApplication.Instance.OptionHelpURL);
        }

        /// <summary>
        ///
        /// </summary>
        private void ShowOpenFileDialog(TextBox targetTextBox)
        {
            var editorPath = targetTextBox.Text;

            FileDialog dialog = new OpenFileDialog();
            dialog.Filter = "Program(*.exe;*.bat)|*.exe;*.bat|All files (*.*)|*.*";

            if (string.IsNullOrEmpty(editorPath))
            {
                dialog.InitialDirectory = Environment.GetEnvironmentVariable("ProgramFiles");
            }
            else
            {
                dialog.InitialDirectory = Path.GetDirectoryName(editorPath);
            }
            if (dialog.ShowDialog() == DialogResult.OK)
            {
                targetTextBox.Text = dialog.FileName;
            }
        }

        private void OnClickRefButton(object sender, EventArgs e)
        {
            this.ShowOpenFileDialog(this.CommandTextBox);
        }

        private void OnClickMidiEditorRefButton(object sender, EventArgs e)
        {
            this.ShowOpenFileDialog(this.textBox_midiEditorCommand);
        }

        private void OnClickWavEditorRefButton(object sender, EventArgs e)
        {
            this.ShowOpenFileDialog(this.textBox_wavEditorCommand);
        }

        /// <summary>
        ///
        /// </summary>
        private void ShowArgsContextMenuStrip(Button anchorButton, TextBox targetTextBox)
        {
            this.ArgsContextMenuStrip.Tag = targetTextBox;
            this.ArgsContextMenuStrip.Show(anchorButton,
                                           anchorButton.Size.Width / 2,
                                           anchorButton.Size.Height / 2);
        }

        private void OnClickArgsButton(object sender, EventArgs e)
        {
            this.LineToolStripMenuItem.Visible = true;
            this.ShowArgsContextMenuStrip(this.ArgsButton, this.ArgsTextBox);
        }

        private void OnClickMidiEditorArgsButton(object sender, EventArgs e)
        {
            this.LineToolStripMenuItem.Visible = false;
            this.ShowArgsContextMenuStrip(this.button_midiEditorArgs, this.textBox_midiEditorArgs);
        }

        private void OnClickWavEditorArgsButton(object sender, EventArgs e)
        {
            this.LineToolStripMenuItem.Visible = false;
            this.ShowArgsContextMenuStrip(this.button_wavEditorArgs, this.textBox_wavEditorArgs);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnClickFileToolStripMenuItem(object sender, EventArgs e)
        {
            SetEditorArguments(sender, "$(file)");
        }

        /// <summary>
        ///
        /// </summary>
        private void OnClickLineToolStripMenuItem(object sender, EventArgs e)
        {
            SetEditorArguments(sender, "$(line)");
        }

        /// <summary>
        ///
        /// </summary>
        private void SetEditorArguments(object sender, string pattern)
        {
            var textBox = (sender as ToolStripMenuItem)?.Owner.Tag as TextBox;
            if (textBox == null)
            {
                return;
            }

            int position = textBox.SelectionStart;
            int length = textBox.SelectionLength;
            var text = textBox.Text;

            if (0 < length)
            {
                text = text.Remove(position, length);
            }

            text = text.Insert(position, pattern);
            textBox.Text = text;

            // 挿入した文字列の末尾にカーソルを移動します。
            textBox.SelectionStart = position + pattern.Length;
            textBox.Focus();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnCheckedChangedEditorRadioButton(object sender, EventArgs e)
        {
            bool value = this.EditorRadioButton.Checked;
            this.ArgsButton.Enabled = value;
            this.ArgsLabel.Enabled = value;
            this.ArgsTextBox.Enabled = value;
            this.CommandLabel.Enabled = value;
            this.CommandTextBox.Enabled = value;
            this.RefButton.Enabled = value;
        }

        /// <summary>
        /// リスト項目設定関連
        /// </summary>
        private string[] listNames =
            {
                ListTraits.ListName_StreamSound,
                ListTraits.ListName_StreamSoundTrack,
                ListTraits.ListName_WaveSoundSet,
                ListTraits.ListName_WaveSoundSetItem,
                ListTraits.ListName_SequenceSoundSet,
                ListTraits.ListName_SequenceSoundSetItem,
                ListTraits.ListName_SequenceSound,
                ListTraits.ListName_Bank,
                ListTraits.ListName_WaveArchive,
                ListTraits.ListName_Player,
                ListTraits.ListName_Group,
                ListTraits.ListName_GroupItem,
                ListTraits.ListName_Instrument,
                ListTraits.ListName_Percussion,
                ListTraits.ListName_FindResult,
            };

        private void InitializePresetListColumns()
        {
            if (ApplicationBase.Instance == null ||
                ApplicationBase.Instance.PresetListColumnsService == null)
            {
                return;
            }

            PresetListColumnsService preset = ApplicationBase.Instance.PresetListColumnsService;

            foreach (string listName in this.listNames)
            {
                Dictionary<string, XmlList> dictionary = new Dictionary<string, XmlList>();
                string[] names = preset.GetPresetNames(listName);
                foreach (string name in names)
                {
                    XmlList xmlList = preset.GetListColumn(listName, name);
                    dictionary.Add(name, xmlList);
                }
                this.presetXmlList.Add(listName, dictionary);
            }
        }

        private void UpdatePresetListColumns()
        {
            PresetListColumnsService preset = ApplicationBase.Instance.PresetListColumnsService;
            preset.Clear();
            foreach (string listName in this.presetXmlList.Keys)
            {
                foreach (string presetName in this.presetXmlList[listName].Keys)
                {
                    preset.Add(listName, presetName, this.presetXmlList[listName][presetName]);
                }
            }
        }

        private void OnSelectedIndexChanged_comboBox_PresetListColumns(object sender, EventArgs e)
        {
            int index = this.comboBox_PresetListColumns.SelectedIndex;
            UpdateListBoxPresetListColumns(this.listNames[index]);
        }

        private void UpdateListBoxPresetListColumns(string listName)
        {
            if (this.presetXmlList.ContainsKey(listName) == false)
            {
                return;
            }

            this.listBox_PresetListColumns.Items.Clear();

            foreach (string name in this.presetXmlList[listName].Keys)
            {
                this.listBox_PresetListColumns.Items.Add(name);
            }

            UpddateButtonsPresetListColumns();
        }

        private void UpddateButtonsPresetListColumns()
        {
            bool enabled = this.listBox_PresetListColumns.SelectedIndex >= 0;

            this.button_PresetListColumnsChangeName.Enabled = enabled;
            this.button_PresetListColumnsDelete.Enabled = enabled;
        }

        private void OnSelectedIndexChanged_listBox_PresetListColumns(object sender, EventArgs e)
        {
            UpddateButtonsPresetListColumns();
        }

        private void OnClick_button_PresetListColumnsDelete(object sender, EventArgs e)
        {
            int listIndex = this.comboBox_PresetListColumns.SelectedIndex;
            string presetName = this.listBox_PresetListColumns.SelectedItem as string;

            Dictionary<string, XmlList> dic = this.presetXmlList[this.listNames[listIndex]];
            dic.Remove(presetName);

            UpdateListBoxPresetListColumns(this.listNames[listIndex]);
        }

        private void OnClick_button_PresetListColumnsChangeName(object sender, EventArgs e)
        {
            int listIndex = this.comboBox_PresetListColumns.SelectedIndex;
            string presetName = this.listBox_PresetListColumns.SelectedItem as string;

            PresetListColumnsNameSettingDialog dialog = new PresetListColumnsNameSettingDialog();
            dialog.Text = MessageResource.Label_ChangeNamePresetListColumns;
            dialog.PresetName = presetName;
            show:
            if (dialog.ShowDialog() == DialogResult.OK)
            {
                if (dialog.PresetName == presetName) return;

                if (this.presetXmlList[this.listNames[listIndex]].ContainsKey(dialog.PresetName) == true)
                {
                    MessageBox.Show(MessageResource.Label_WarningSameNamePresetListColumns);
                    goto show;
                }
                Dictionary<string, XmlList> dic = this.presetXmlList[this.listNames[listIndex]];
                XmlList xmlList = dic[presetName];
                dic.Remove(presetName);
                dic.Add(dialog.PresetName, xmlList);

                UpdateListBoxPresetListColumns(this.listNames[listIndex]);
            }
        }
    }
}
