﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Data;
    using System.Diagnostics;
    using System.Drawing;
    using System.Linq;
    using System.Text;
    using System.Text.RegularExpressions;
    using System.Windows.Forms;

    using NintendoWare.SoundMaker.Framework.Resources;

    public partial class UserDataPanel : UserControl, IPartialValueManipulator
    {
        private string oldText = string.Empty;

        public event EventHandler ValueChanged;

        /// <summary>
        ///
        /// </summary>
        public class Candidate
        {
            public string Label { get; set; }
            public string Value { get; set; }

            public override string ToString()
            {
                return Value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public UserDataPanel()
        {
            InitializeComponent();

            this.textBox.Show();
            this.textBox.TextChanged += OnTextChanged;

            this.comboBox.Hide();
            this.comboBox.DrawMode = DrawMode.OwnerDrawFixed;
            this.comboBox.SelectedValueChanged += delegate (object sender, EventArgs e)
                {
                    UpdateControls();
                    this.OnValueChanged();
                };
            this.comboBox.SelectedIndexChanged += delegate (object sender, EventArgs e)
                {
                    if (SuspendComboBoxSelectedIndexChangedEvent == true)
                    {
                        return;
                    }

                    try
                    {
                        // SelectedIndexChanged が comboBox 内で呼ばれてしまい、
                        // comboBox.Text が上書きされ変更できない。
                        // BeginInvoke を用いる方法で対処。
                        Candidate candidate = candidates[this.comboBox.SelectedIndex];
                        string text = this.GetDisplayText(candidate);
                        Action action = delegate ()
                        {
                            try
                            {
                                this.SuspendComboBoxSelectedIndexChangedEvent = true;
                                this.SuspendComboBoxTextChangedEvent = true;
                                this.TextBoxText = text;
                            }
                            finally
                            {
                                this.SuspendComboBoxSelectedIndexChangedEvent = false;
                                this.SuspendComboBoxTextChangedEvent = false;
                            }
                            UpdateControls();
                            this.OnValueChanged();
                        };
                        BeginInvoke(action);
                    }
                    catch { }
                };

            this.comboBox.DrawItem += delegate (object sender, DrawItemEventArgs e)
                {
                    Candidate candidate = candidates[e.Index];
                    string text = this.GetDisplayText(candidate);

                    e.DrawBackground();
                    e.Graphics.DrawString(text, Font, Brushes.Black, e.Bounds);
                };

            UpdateControls();
        }

        /// <summary>
        ///
        /// </summary>
        public int BitLocation
        {
            get;
            set;
        }

        /// <summary>
        ///
        /// </summary>
        public virtual int BitSize
        {
            get;
            set;
        }

        /// <summary>
        ///
        /// </summary>
        public virtual object Value
        {
            get;
            set;
        }

        /// <summary>
        ///
        /// </summary>
        public string Label
        {
            get { return this.label.Text; }
            set { this.label.Text = value; }
        }

        public bool ValidateValue
        {
            get; private set;
        }

        /// <summary>
        ///
        /// </summary>
        private Candidate[] candidates = null;
        public Candidate[] Candidates
        {
            get
            {
                return this.candidates;
            }
            set
            {
                this.candidates = value;

                if (EnabledCandidates == false)
                {
                    this.textBox.Show();

                    this.comboBox.DataSource = null;
                    this.comboBox.Hide();
                }
                else
                {
                    this.textBox.Hide();
                    try
                    {
                        this.SuspendComboBoxSelectedIndexChangedEvent = true;
                        this.SuspendComboBoxTextChangedEvent = true;
                        this.comboBox.DataSource = this.candidates;
                    }
                    finally
                    {
                        this.SuspendComboBoxSelectedIndexChangedEvent = false;
                        this.SuspendComboBoxTextChangedEvent = false;
                    }
                    this.comboBox.Show();
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        ulong IPartialValueManipulator.GetPartialValue()
        {
            return GetPartialValueCore();
        }

        /// <summary>
        ///
        /// </summary>
        protected ulong GetMask(int size)
        {
            return (1UL << size) - 1;
        }

        /// <summary>
        ///
        /// </summary>
        protected string TextBoxText
        {
            get
            {
                if (EnabledCandidates == true)
                {
                    return this.comboBox.Text;
                }
                else
                {
                    return this.textBox.Text;
                }
            }
            set
            {
                if (EnabledCandidates == true)
                {
                    this.comboBox.Text = value;
                }
                else
                {
                    this.textBox.Text = value;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected Control TextBoxControl
        {
            get
            {
                if (EnabledCandidates != false)
                {
                    return this.comboBox;
                }
                else
                {
                    return this.textBox;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected bool EnabledCandidates
        {
            get
            {
                if (this.candidates != null &&
                    this.candidates.Length > 0)
                {
                    return true;
                }
                else
                {
                    return false;
                }
            }
        }

        protected bool SuspendComboBoxSelectedIndexChangedEvent { get; set; }
        protected bool SuspendComboBoxTextChangedEvent { get; set; }

        protected string InputPattern { get; set; }

        /// <summary>
        ///
        /// </summary>
        protected virtual ulong GetPartialValueCore()
        {
            return 0;
        }

        ///
        protected class ValidateResult
        {
            public enum Types
            {
                Done,
                Unknown,
                BadFormat,
                OutsideValueRange,
                InvalidValue,
            }

            private Types type = Types.Done;
            private string message = null;

            public ValidateResult()
            {
            }

            public ValidateResult(Types type, string message)
            {
                this.type = type;
                this.message = message;
            }

            public Types Type
            {
                get
                {
                    return this.type;
                }
            }

            public string Message
            {
                get
                {
                    return this.message;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual ValidateResult Validate(string text)
        {
            return new ValidateResult();
        }

        /// <summary>
        ///
        /// </summary>
        protected void UpdateControls()
        {
            this.ValidateValue = true;
            string text = this.GetValueText(TextBoxText);
            ValidateResult result = Validate(text);
            switch (result.Type)
            {
                case ValidateResult.Types.Done:
                    this.ValidateValue = true;
                    SetErrorMessage(null);
                    break;

                case ValidateResult.Types.BadFormat:
                case ValidateResult.Types.OutsideValueRange:
                case ValidateResult.Types.InvalidValue:
                    this.ValidateValue = false;
                    SetErrorMessage(result.Message);
                    break;
            }
        }

        protected virtual string GetValueText(string text)
        {
            string valueText = text;

            Match match = Regex.Match(text, "\\S+");
            if (match.Success == true)
            {
                valueText = match.Value;
            }

            return valueText;
        }

        protected virtual string GetLabelText(string value)
        {
            if (EnabledCandidates == true)
            {
                Candidate candidate = Array.Find<Candidate>(this.candidates, (Candidate c) => this.GetValueText(c.Value) == this.GetValueText(value));
                if (candidate != null)
                {
                    return this.GetDisplayText(candidate);
                }
            }

            return this.GetValueText(value);
        }

        protected string GetDisplayText(Candidate candidate)
        {
            return CommonListUtil.GetValueWithLabel(candidate.Value, candidate.Label);
        }

        /// <summary>
        ///
        /// </summary>
        private void SetErrorMessage(string message)
        {
            if (message != null)
            {
                errorProvider.SetError(TextBoxControl, message);
            }
            else
            {
                errorProvider.Clear();
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnTextChanged(object sender, EventArgs e)
        {
            if (this.SuspendComboBoxTextChangedEvent == true)
            {
                return;
            }
            if (Regex.IsMatch(this.TextBoxText, this.InputPattern) == true)
            {
                try
                {
                    this.SuspendComboBoxSelectedIndexChangedEvent = true;
                    this.SuspendComboBoxTextChangedEvent = true;
                    this.TextBoxText = this.oldText;
                }
                finally
                {
                    this.SuspendComboBoxSelectedIndexChangedEvent = false;
                    this.SuspendComboBoxTextChangedEvent = false;
                }
            }
            else
            {
                this.oldText = this.TextBoxText;
            }
            UpdateControls();
            this.OnValueChanged();
        }

        private void OnLeave(object sender, EventArgs e)
        {
            try
            {
                this.SuspendComboBoxSelectedIndexChangedEvent = true;
                this.SuspendComboBoxTextChangedEvent = true;
                this.TextBoxText = this.GetLabelText(this.TextBoxText);
            }
            finally
            {
                this.SuspendComboBoxSelectedIndexChangedEvent = false;
                this.SuspendComboBoxTextChangedEvent = false;
            }
            UpdateControls();
        }

        private void OnEnter(object sender, EventArgs e)
        {
            string text = this.GetValueText(this.TextBoxText);
            this.oldText = text;
            try
            {
                this.SuspendComboBoxSelectedIndexChangedEvent = true;
                this.SuspendComboBoxTextChangedEvent = true;
                this.TextBoxText = text;
            }
            finally
            {
                this.SuspendComboBoxSelectedIndexChangedEvent = false;
                this.SuspendComboBoxTextChangedEvent = false;
            }
        }

        private void OnValueChanged()
        {
            if (this.ValueChanged != null)
            {
                this.ValueChanged(this, EventArgs.Empty);
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class UserDataIntegerPanel : UserDataPanel
    {
        public UserDataIntegerPanel()
        {
            this.InputPattern = "[^0-9\\-\\+]+";
        }

        /// <summary>
        ///
        /// </summary>
        public override int BitSize
        {
            get
            {
                return base.BitSize;
            }
            set
            {
                if (0 < value)
                {
                    base.BitSize = value;
                    value = 1 << (value - 1);
                    this.min = -value;
                    this.max = value - 1;
                }
                else
                {
                    base.BitSize = 0;
                    this.min = 0;
                    this.max = 0;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public override object Value
        {
            set
            {
                try
                {
                    this.SuspendComboBoxSelectedIndexChangedEvent = true;
                    this.SuspendComboBoxTextChangedEvent = true;
                    TextBoxText = this.GetLabelText(value.ToString());
                    UpdateControls();
                }
                finally
                {
                    this.SuspendComboBoxSelectedIndexChangedEvent = false;
                    this.SuspendComboBoxTextChangedEvent = false;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected int min
        {
            get;
            set;
        }

        /// <summary>
        ///
        /// </summary>
        protected int max
        {
            get;
            set;
        }

        /// <summary>
        ///
        /// </summary>
        protected override ulong GetPartialValueCore()
        {
            int parsedValue = 0;
            string text = this.GetValueText(TextBoxText);

            if (int.TryParse(text, out parsedValue) == false)
            {
                return 0;
            }

            ulong value = (ulong)parsedValue;
            value &= GetMask(BitSize);
            return value << BitLocation;
        }

        /// <summary>
        ///
        /// </summary>
        protected override string GetValueText(string text)
        {
            int value;

            text = base.GetValueText(text);

            if (int.TryParse(text, out value) == true)
            {
                text = value.ToString();
            }

            return text;
        }

        /// <summary>
        ///
        /// </summary>
        protected override ValidateResult Validate(string text)
        {
            int value = 0;

            if (int.TryParse(text, out value) == true)
            {
                if (value >= min && value <= max)
                {
                    return new ValidateResult();
                }
                else
                {
                    return new ValidateResult
                        (ValidateResult.Types.OutsideValueRange,
                          String.Format(MessageResource.Message_ErrorProvider_OutsideValueRange,
                                         min, max));
                }
            }
            else
            {
                return new ValidateResult
                    (ValidateResult.Types.BadFormat,
                      MessageResource.Message_ErrorProvider_BadFormat);
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class UserDataUIntegerPanel : UserDataIntegerPanel
    {
        public UserDataUIntegerPanel()
        {
            this.InputPattern = "[^0-9\\+]+";
        }

        /// <summary>
        ///
        /// </summary>
        public override int BitSize
        {
            get
            {
                return base.BitSize;
            }
            set
            {
                if (0 < value)
                {
                    base.BitSize = value;
                    value = 1 << value;
                    this.min = 0;
                    this.max = value - 1;
                }
                else
                {
                    base.BitSize = 0;
                    this.min = 0;
                    this.max = 0;
                }
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class UserDataDecimalPanel : UserDataPanel
    {
        public UserDataDecimalPanel()
        {
            this.InputPattern = "[^0-9Ee\\.\\-\\+]+";
        }

        /// <summary>
        ///
        /// </summary>
        public override int BitSize
        {
            get
            {
                return 32;
            }
            set
            {
            }
        }

        /// <summary>
        ///
        /// </summary>
        public override object Value
        {
            set
            {
                try
                {
                    this.SuspendComboBoxSelectedIndexChangedEvent = true;
                    this.SuspendComboBoxTextChangedEvent = true;
                    TextBoxText = this.GetLabelText(value.ToString());
                    UpdateControls();
                }
                finally
                {
                    this.SuspendComboBoxSelectedIndexChangedEvent = false;
                    this.SuspendComboBoxTextChangedEvent = false;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected float min
        {
            get { return float.MinValue; }
            set { }
        }

        /// <summary>
        ///
        /// </summary>
        protected float max
        {
            get { return float.MaxValue; }
            set { }
        }

        /// <summary>
        ///
        /// </summary>
        protected override ulong GetPartialValueCore()
        {
            float parsedValue = 0;
            string text = this.GetValueText(TextBoxText);

            if (float.TryParse(text, out parsedValue) == false)
            {
                return 0;
            }

            ulong value = 0;
            byte[] bytes = BitConverter.GetBytes(parsedValue);

            for (int index = bytes.Length - 1; index >= 0; index--)
            {
                value <<= 8;
                value |= (ulong)bytes[index];
            }
            return value;
        }

        /// <summary>
        ///
        /// </summary>
        protected override string GetValueText(string text)
        {
            float value;

            text = base.GetValueText(text);

            if (float.TryParse(text, out value) == true)
            {
                text = value.ToString();
            }

            return text;
        }

        /// <summary>
        ///
        /// </summary>
        protected override ValidateResult Validate(string text)
        {
            float value = 0;

            try
            {
                value = float.Parse(text);
                if (value >= min && value <= max)
                {
                    return new ValidateResult();
                }
                else
                {
                    return new ValidateResult
                        (ValidateResult.Types.OutsideValueRange,
                          String.Format(MessageResource.Message_ErrorProvider_OutsideValueRange,
                                         min, max));
                }
            }

            catch (FormatException)
            {
                return new ValidateResult
                    (ValidateResult.Types.BadFormat,
                      MessageResource.Message_ErrorProvider_BadFormat);
            }

            catch (OverflowException)
            {
                return new ValidateResult
                    (ValidateResult.Types.OutsideValueRange,
                      String.Format(MessageResource.Message_ErrorProvider_OutsideValueRange,
                                     min, max));
            }

            catch (Exception e)
            {
                return new ValidateResult
                    (ValidateResult.Types.Unknown,
                      e.Message);
            }
        }
    }
}
