﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.IO;
    using System.Drawing;
    using System.Collections;
    using System.ComponentModel;
    using System.Windows.Forms;
    using System.Resources;
    using System.Reflection;
    using System.Diagnostics;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Utilities;

    ///--------------------------------------------------------------------------
    /// <summary>
    /// CreateWaveSoundDialog の概要の説明です。
    /// </summary>
    public partial class CreateWaveSoundDialog : CreateSoundDialog
    {
        private readonly WaveSoundBase _sound = ApplicationBase.Instance.CreateComponentService.Create<WaveSoundBase>();

        public static bool VisibleWaveEncodingPCM8
        {
            get; set;
        }

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CreateWaveSoundDialog(SoundProjectService projectService, SoundSetDocument soundSetDocument, string defaultPlayerName)
            : base(projectService, soundSetDocument, defaultPlayerName)
        {
            InitializeComponent();

            //
            SetUniqueNameToTextBox(textBox_Label, projectService.Project,
                                    ProjectService.ComponentDictionary,
                                    ProjectService.Project.WaveSoundNamePrefix);

            //
            if (VisibleWaveEncodingPCM8 == true)
            {
                comboBox_Type.Items.Add("PCM8");
            }
            comboBox_Type.SelectedIndex = 0;
            this.FileType = _sound.Encoding;

            this.Volume = _sound.Volume;
            this.ChannelPriority = _sound.ChannelPriority;
            this.PlayerPriority = _sound.PlayerPriority;

            textBox_Volume.Text = _sound.Volume.ToString();
            textBox_ChannelPriority.Text = _sound.ChannelPriority.ToString();
            textBox_PlayerPriority.Text = _sound.PlayerPriority.ToString();

            //プレイヤーの設定
            comboBox_Player.Items.AddRange(PlayerControlItems);
            comboBox_Player.SelectedIndex = 0;
            this.PlayerReference = _sound.PlayerReference;

            //
            InspectFilePath();
            UpdateControls();

            //
            //ResumeInitialParameter();
        }

        ///--------------------------------
        /// <summary>
        /// ラベルの取得
        /// </summary>
        private string Label
        {
            get { return textBox_Label.Text; }
        }

        ///--------------------------------
        /// <summary>
        /// ファイルタイプの取得
        /// </summary>
        private WaveEncoding FileType
        {
            get
            {
                switch (comboBox_Type.SelectedIndex)
                {
                    case 0:
                        return WaveEncoding.Adpcm;
                    case 1:
                        return WaveEncoding.Pcm16;
                    case 2:
                        return WaveEncoding.Pcm8;
                }

                throw new Exception("unexpected error");
            }
            set
            {
                foreach (var item in comboBox_Type.Items)
                {
                    if (item.ToString().ToLower() == value.ToString().ToLower())
                    {
                        comboBox_Type.SelectedItem = item;
                        break;
                    }
                }
            }
        }

        ///--------------------------------
        /// <summary>
        /// ファイルパスの取得
        /// </summary>
        private string FilePath
        {
            get
            {
                return textBox_FilePath.Text;
            }
            set
            {
                textBox_FilePath.Text = value;
            }
        }

        ///--------------------------------
        /// <summary>
        /// ボリュームの取得/設定
        /// </summary>
        private int Volume
        {
            get
            {
                int value = 0;
                try { value = int.Parse(textBox_Volume.Text); }
                catch { Debug.Assert(false); }
                return value;
            }
            set
            {
                textBox_Volume.Text = value.ToString();
            }
        }

        ///--------------------------------
        /// <summary>
        /// チャンネルプライオリティの取得/設定
        /// </summary>
        private int ChannelPriority
        {
            get
            {
                int value = 0;
                try { value = int.Parse(textBox_ChannelPriority.Text); }
                catch { Debug.Assert(false); }
                return value;
            }
            set
            {
                textBox_ChannelPriority.Text = value.ToString();
            }
        }

        ///--------------------------------
        /// <summary>
        /// プレイヤープライオリティの取得/設定
        /// </summary>
        private int PlayerPriority
        {
            get
            {
                int value = 0;
                try { value = int.Parse(textBox_PlayerPriority.Text); }
                catch { Debug.Assert(false); }
                return value;
            }
            set
            {
                textBox_PlayerPriority.Text = value.ToString();
            }
        }

        ///--------------------------------
        /// <summary>
        /// プレイヤーを取得します。
        /// </summary>
        private string PlayerReference
        {
            get
            {
                return comboBox_Player.SelectedItem.ToString();
            }
            set
            {
                foreach (var item in comboBox_Player.Items)
                {
                    if (item.ToString() == value)
                    {
                        comboBox_Player.SelectedItem = item;
                        break;
                    }
                }
            }
        }

        /// <summary>
        /// 作成したサウンドを取得します。
        /// </summary>
        public WaveSoundBase NewSound
        {
            get
            {
                _sound.Name = Label;
                _sound.Encoding = FileType;
                _sound.FilePath = FilePath;
                _sound.Volume = Volume;
                _sound.PlayerReference = PlayerReference;
                _sound.ChannelPriority = ChannelPriority;
                _sound.PlayerPriority = PlayerPriority;

                return _sound;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnClickReference(object sender, System.EventArgs e)
        {
            string filePath = null;

            if ((filePath = WaveFileQuester.QueryLoad(this)) != null)
            {
                this.FilePath = filePath;
            }

            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool CanCreate
        {
            get
            {
                try
                {
                    bool label = InspectLabel();
                    bool filePath = InspectFilePath();
                    bool volume = InspectVolume();
                    bool channelPriority = InspectChannelPriority();
                    bool playerPriority = InspectPlayerPriority();

                    if (label != false &&
                         filePath != false &&
                         volume != false &&
                         channelPriority != false &&
                         playerPriority != false)
                    {
                        return true;
                    }
                }
                catch { }
                return false;
            }
        }

        ///--------------------------------
        /// <summary>
        /// ラベルの検査
        /// </summary>
        private bool InspectLabel()
        {
            return ValidateName(textBox_Label.Text, textBox_Label);
        }

        ///--------------------------------
        /// <summary>
        /// ファイルパスの検査
        /// </summary>
        private bool InspectFilePath()
        {
            if (File.Exists(textBox_FilePath.Text) == false)
            {
                button_CreateLabel.Enabled = false;
                ShowError(textBox_FilePath, MessageResource.Message_FileNotFound);
                return false;
            }

            button_CreateLabel.Enabled = true;
            HideError(textBox_FilePath);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// ボリュームの検査
        /// </summary>
        private bool InspectVolume()
        {
            return ValidateIntParameter(_sound, ProjectParameterNames.Volume,
                                         textBox_Volume.Text, textBox_Volume);
        }

        ///--------------------------------
        /// <summary>
        /// チャンネルプライオリティの検査
        /// </summary>
        private bool InspectChannelPriority()
        {
            return ValidateIntParameter(_sound, ProjectParameterNames.Sound.ChannelPriority,
                                         textBox_ChannelPriority.Text, textBox_ChannelPriority);
        }

        ///--------------------------------
        /// <summary>
        /// プレイヤープライオリティの検査
        /// </summary>
        private bool InspectPlayerPriority()
        {
            return ValidateIntParameter(_sound, ProjectParameterNames.Sound.PlayerPriority,
                                         textBox_PlayerPriority.Text, textBox_PlayerPriority);
        }

        ///--------------------------------
        /// <summary>
        /// 確定に使用するボタンの取得
        /// </summary>
        protected override Control GetAcceptButton()
        {
            return button_OK;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnTextChanged(object sender, System.EventArgs e)
        {
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedLabel(object sender, System.EventArgs e)
        {
            InspectLabel();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedFilePath(object sender, System.EventArgs e)
        {
            InspectFilePath();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedVolume(object sender, System.EventArgs e)
        {
            InspectVolume();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedChannelPriority(object sender, System.EventArgs e)
        {
            InspectChannelPriority();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedPlayerPriority(object sender, System.EventArgs e)
        {
            InspectPlayerPriority();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnCreateLabel(object sender, EventArgs e)
        {
            textBox_Label.Text = CreateSoundSetItemNameFromFilePath(ProjectService.Project.WaveSoundNamePrefix, this.FilePath);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnClosing(object sender, FormClosingEventArgs e)
        {
            if (DialogResult == DialogResult.OK)
            {
                if (ProjectFilePathUtility.ConfirmFilePath(FilePath) == false)
                {
                    e.Cancel = true;
                    return;
                }
            }
        }
    }
}
