﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Windows.Forms;
    using SoundFoundation.Projects;
    using Framework.FileFormats;
    using Framework.Resources;
    using Framework.Utilities;

    /// <summary>
    /// CreateSequenceSoundDialog の概要の説明です。
    /// </summary>
    public partial class CreateSequenceSoundDialog : CreateSoundDialog
    {
        private readonly SequenceSoundBase _sound = ApplicationBase.Instance.CreateComponentService.Create<SequenceSoundBase>();

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CreateSequenceSoundDialog(SoundProjectService projectService, SoundSetDocument soundSetDocument, string defaultPlayerName)
            : base(projectService, soundSetDocument, defaultPlayerName)
        {
            if (null == projectService) { throw new ArgumentNullException("projectService"); }
            if (null == defaultPlayerName) { throw new ArgumentNullException("defaultPlayerName"); }

            InitializeComponent();

            //
            SetUniqueNameToTextBox(textBox_Label, projectService.Project,
                                    ProjectService.ComponentDictionary,
                                    ProjectService.Project.SequenceSoundNamePrefix);

            // ファイルタイプを設定します。
            this.FileType = _sound.FileType;

            this.Volume = _sound.Volume;
            this.ChannelPriority = _sound.ChannelPriority;
            this.PlayerPriority = _sound.PlayerPriority;

            //バンクの設定
            comboBox_Bank.Items.AddRange(BankControlItems);
            comboBox_Bank.SelectedIndex = 0;
            if (_sound.SoundSetBankReferences.Count > 0)
            {
                this.BankName = _sound.SoundSetBankReferences[0].TargetName;
            }

            //プレイヤーの設定
            comboBox_Player.Items.AddRange(PlayerControlItems);
            comboBox_Player.SelectedIndex = 0;
            this.PlayerReference = _sound.PlayerReference;

            //
            InspectFilePath();
            UpdateControls();
            UpdateStartPositionLabels();
        }

        ///--------------------------------
        /// <summary>
        /// ラベルの取得
        /// </summary>
        private string Label
        {
            get { return textBox_Label.Text; }
        }

        ///--------------------------------
        /// <summary>
        /// ファイルタイプの取得
        /// </summary>
        private SequenceSoundFileType FileType
        {
            get
            {
                switch (comboBox_Type.SelectedIndex)
                {
                    case 0:
                        return SequenceSoundFileType.Text;
                    case 1:
                        return SequenceSoundFileType.Smf;
                }
                Debug.Assert(false);
                throw new Exception("invalid sequence sound file type.");
            }
            set
            {
                int index = 0;

                switch (value)
                {
                    case SequenceSoundFileType.Text:
                        index = 0;
                        break;
                    case SequenceSoundFileType.Smf:
                        index = 1;
                        break;
                }
                comboBox_Type.SelectedIndex = index;
            }
        }

        ///--------------------------------
        /// <summary>
        /// ファイルパスの取得
        /// </summary>
        private string FilePath
        {
            get { return textBox_FilePath.Text; }
            set { textBox_FilePath.Text = value; }
        }

        ///--------------------------------
        /// <summary>
        /// ボリュームの取得/設定
        /// </summary>
        private int Volume
        {
            get
            {
                int value = 0;
                try { value = int.Parse(textBox_Volume.Text); }
                catch { Debug.Assert(false); }
                return value;
            }
            set
            {
                textBox_Volume.Text = value.ToString();
            }
        }

        ///--------------------------------
        /// <summary>
        /// チャンネルプライオリティの取得/設定
        /// </summary>
        private int ChannelPriority
        {
            get
            {
                int value = 0;
                try { value = int.Parse(textBox_ChannelPriority.Text); }
                catch { Debug.Assert(false); }
                return value;
            }
            set
            {
                textBox_ChannelPriority.Text = value.ToString();
            }
        }

        ///--------------------------------
        /// <summary>
        /// プレイヤープライオリティの取得/設定
        /// </summary>
        private int PlayerPriority
        {
            get
            {
                int value = 0;
                try { value = int.Parse(textBox_PlayerPriority.Text); }
                catch { Debug.Assert(false); }
                return value;
            }
            set
            {
                textBox_PlayerPriority.Text = value.ToString();
            }
        }

        ///--------------------------------
        /// <summary>
        /// 再生開始位置の取得/設定
        /// </summary>
        private string StartPositionLabel
        {
            get { return comboBox_StartPosition.Text; }
        }

        ///--------------------------------
        /// <summary>
        /// バンク名を取得します。
        /// </summary>
        private string BankName
        {
            get
            {
                return comboBox_Bank.SelectedItem.ToString();
            }
            set
            {
                foreach (var item in comboBox_Bank.Items)
                {
                    if (item.ToString() == value)
                    {
                        comboBox_Bank.SelectedItem = item;
                        break;
                    }
                }
            }
        }

        ///--------------------------------
        /// <summary>
        /// プレイヤーを取得します。
        /// </summary>
        private string PlayerReference
        {
            get
            {
                return comboBox_Player.SelectedItem.ToString();
            }
            set
            {
                foreach (var item in comboBox_Player.Items)
                {
                    if (item.ToString() == value)
                    {
                        comboBox_Player.SelectedItem = item;
                        break;
                    }
                }
            }
        }

        /// <summary>
        /// 作成したサウンドを取得します。
        /// </summary>
        public SequenceSoundBase NewSound
        {
            get
            {
                _sound.Name = Label;
                _sound.FileType = FileType;
                _sound.FilePath = FilePath;
                _sound.Volume = Volume;
                _sound.ChannelPriority = ChannelPriority;
                _sound.PlayerReference = PlayerReference;
                _sound.PlayerPriority = PlayerPriority;
                _sound.StartPosition = StartPositionLabel;
                if (_sound.SoundSetBankReferences.Count <= 0)
                {
                    _sound.InitializeBankReferences(BankName);
                }
                else
                {
                    _sound.SoundSetBankReferences[0].TargetName = BankName;
                }

                return _sound;
            }
        }

        private string[] StartPositionLabels
        {
            get
            {
                if (File.Exists(FilePath) == false)
                {
                    return new string[] { string.Empty };
                }

                try
                {
                    SequenceSoundFileParser parser;

                    switch (FileType)
                    {
                        case SequenceSoundFileType.Smf:
                            parser = ApplicationBase.Instance.FileParserStore.Parse<SmfParser>(FilePath);
                            break;

                        case SequenceSoundFileType.Text:
                            parser = ApplicationBase.Instance.FileParserStore.Parse<SequenceSoundTextParser>(FilePath);
                            break;

                        default:
                            return new string[] { string.Empty };
                    }

                    return (from SequenceSoundFileParser.Label label in parser.Labels
                            select label.Name).ToArray();
                }
                catch
                {
                    return new string[] { string.Empty };
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateStartPositionLabels()
        {
            comboBox_StartPosition.Items.Clear();
            comboBox_StartPosition.Items.AddRange(StartPositionLabels);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnClickReference(object sender, System.EventArgs e)
        {
            string filePath = null;

            switch (FileType)
            {
                case SequenceSoundFileType.Smf:
                    filePath = SeqFileQuester.QueryLoadSMF();
                    break;

                case SequenceSoundFileType.Text:
                    filePath = SeqFileQuester.QueryLoadText();
                    break;

                default:
                    Debug.Assert(false);
                    break;
            }

            if (filePath != null)
            {
                textBox_FilePath.Text = filePath;
            }

            UpdateControls();
            UpdateStartPositionLabels();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool CanCreate
        {
            get
            {
                try
                {
                    bool label = InspectLabel();
                    bool filePath = InspectFilePath();
                    bool volume = InspectVolume();
                    bool channelPriority = InspectChannelPriority();
                    bool playerPriority = InspectPlayerPriority();

                    if (label != false &&
                         filePath != false &&
                         volume != false &&
                         channelPriority != false &&
                         playerPriority != false)
                    {
                        return true;
                    }
                }
                catch { }
                return false;
            }
        }

        ///--------------------------------
        /// <summary>
        /// ラベルの検査
        /// </summary>
        private bool InspectLabel()
        {
            return ValidateName(textBox_Label.Text, textBox_Label);
        }

        ///--------------------------------
        /// <summary>
        /// ファイルパスの検査
        /// </summary>
        private bool InspectFilePath()
        {
            if (File.Exists(textBox_FilePath.Text) == false)
            {
                button_CreateLabel.Enabled = false;
                ShowError(textBox_FilePath, MessageResource.Message_FileNotFound);
                return false;
            }

            button_CreateLabel.Enabled = true;
            HideError(textBox_FilePath);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// ボリュームの検査
        /// </summary>
        private bool InspectVolume()
        {
            return ValidateIntParameter(_sound, ProjectParameterNames.Volume,
                                         textBox_Volume.Text, textBox_Volume);
        }

        ///--------------------------------
        /// <summary>
        /// チャンネルプライオリティの検査
        /// </summary>
        private bool InspectChannelPriority()
        {
            return ValidateIntParameter(_sound, ProjectParameterNames.Sound.ChannelPriority,
                                         textBox_ChannelPriority.Text, textBox_ChannelPriority);
        }

        ///--------------------------------
        /// <summary>
        /// プレイヤープライオリティの検査
        /// </summary>
        private bool InspectPlayerPriority()
        {
            return ValidateIntParameter(_sound, ProjectParameterNames.Sound.PlayerPriority,
                                         textBox_PlayerPriority.Text, textBox_PlayerPriority);
        }

        ///--------------------------------
        /// <summary>
        /// 確定に使用するボタンの取得
        /// </summary>
        protected override Control GetAcceptButton()
        {
            return button_OK;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnTextChanged(object sender, System.EventArgs e)
        {
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedLabel(object sender, System.EventArgs e)
        {
            InspectLabel();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedFilePath(object sender, System.EventArgs e)
        {
            InspectFilePath();
            UpdateControls();
            UpdateStartPositionLabels();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedVolume(object sender, System.EventArgs e)
        {
            InspectVolume();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedChannelPriority(object sender, System.EventArgs e)
        {
            InspectChannelPriority();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedPlayerPriority(object sender, System.EventArgs e)
        {
            InspectPlayerPriority();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnSelectIndexChangedFileType(object sender, EventArgs e)
        {
            UpdateStartPositionLabels();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnCreateLabel(object sender, EventArgs e)
        {
            textBox_Label.Text = CreateSoundSetItemNameFromFilePath(
                                    ProjectService.Project.SequenceSoundNamePrefix, textBox_FilePath.Text);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnClosing(object sender, FormClosingEventArgs e)
        {
            if (DialogResult == DialogResult.OK)
            {
                if (ProjectFilePathUtility.ConfirmFilePath(FilePath) == false)
                {
                    e.Cancel = true;
                    return;
                }
            }
        }
    }
}
