﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.IO;
    using System.Drawing;
    using System.Collections;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Windows.Forms;
    using System.Diagnostics;
    using System.Text.RegularExpressions;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.Wave;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Utilities;

    ///--------------------------------------------------------------------------
    /// <summary>
    /// CreateInstrumentDialog の概要の説明です。
    /// </summary>
    public partial class CreateInstrumentDialog : CreateItemDialog
    {
        private SoundProjectService _projectService;
        private BankService _bankService;
        private readonly Instrument _instrument = ApplicationBase.Instance.CreateComponentService.Create<Instrument>();

        public static bool VisibleWaveEncodingPCM8
        {
            get; set;
        }

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CreateInstrumentDialog(SoundProjectService projectService, BankService bankService)
        {
            if (null == projectService) { throw new ArgumentNullException("projectService"); }
            if (null == projectService.ProjectDocument)
            {
                throw new ArgumentException("projectService.ProjectDocument must not be null.");
            }
            if (null == bankService) { throw new ArgumentNullException("bankService"); }

            _projectService = projectService;
            _bankService = bankService;

            KeyRegion keyRegion = new KeyRegion()
            {
                KeyMin = 0,
                KeyMax = 127,
            };

            VelocityRegion velocityRegion = new VelocityRegion()
            {
                VelocityMin = 0,
                VelocityMax = 127,
            };

            keyRegion.Children.Add(velocityRegion);
            _instrument.Children.Add(keyRegion);


            InitializeComponent();

            //
            SetUniqueNameToTextBox(textBox_Label, projectService.Project,
                                    _bankService.ComponentDictionary,
                                    projectService.Project.InstrumentNamePrefix);

            //
            if (VisibleWaveEncodingPCM8 == true)
            {
                comboBox_Type.Items.Add("PCM8");
            }
            comboBox_Type.SelectedIndex = 0;

            //
            InspectLabel();
            InspectFilePath();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        /// ラベルの取得
        /// </summary>
        private string Label
        {
            get { return textBox_Label.Text; }
        }

        ///--------------------------------
        /// <summary>
        /// ファイルタイプの取得
        /// </summary>
        private WaveEncoding FileType
        {
            get
            {
                switch (comboBox_Type.SelectedIndex)
                {
                    case 0:
                        return WaveEncoding.Adpcm;
                    case 1:
                        return WaveEncoding.Pcm16;
                    case 2:
                        return WaveEncoding.Pcm8;
                }

                throw new Exception("unexpected error");
            }
        }

        ///--------------------------------
        /// <summary>
        /// ファイルパスの取得
        /// </summary>
        private string FilePath
        {
            get { return textBox_FilePath.Text; }
        }

        ///--------------------------------
        /// <summary>
        /// 作成したインストルメントを取得します。
        /// </summary>
        public Instrument Instrument
        {
            get
            {
                _instrument.Name = Label;
                _instrument.ProgramNo = ProgramNoCreator.Create(_bankService);

                VelocityRegion velocityRegion = _instrument.Children[0].Children[0] as VelocityRegion;
                velocityRegion.Encoding = FileType;
                velocityRegion.FilePath = FilePath;
                velocityRegion.OriginalKey = OriginalKeyCreator.FromFile(FilePath);

                return _instrument;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool CanCreate
        {
            get
            {
                bool label = InspectLabel();
                bool filePath = InspectFilePath();

                if (label != false &&
                        filePath != false)
                {
                    return true;
                }
                return false;
            }
        }

        ///--------------------------------
        /// <summary>
        /// ラベルの検査
        /// </summary>
        private bool InspectLabel()
        {
            return ValidateName(_bankService, textBox_Label.Text, textBox_Label);
        }

        ///--------------------------------
        /// <summary>
        /// ファイルパスの検査
        /// </summary>
        private bool InspectFilePath()
        {
            if (File.Exists(textBox_FilePath.Text) == false)
            {
                button_CreateLabel.Enabled = false;
                ShowError(textBox_FilePath, MessageResource.Message_FileNotFound);
                return false;
            }

            button_CreateLabel.Enabled = true;
            HideError(textBox_FilePath);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 確定に使用するボタンの取得
        /// </summary>
        protected override Control GetAcceptButton()
        {
            return OKButton;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnTextChanged(object sender, System.EventArgs e)
        {
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnReferenceSpecifyWaveFile(object sender, System.EventArgs e)
        {
            string filePath = null;

            if ((filePath = WaveFileQuester.QueryLoad(this)) != null)
            {
                textBox_FilePath.Text = filePath;
            }

            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedLabel(object sender, System.EventArgs e)
        {
            InspectLabel();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedFilePath(object sender, System.EventArgs e)
        {
            InspectFilePath();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnCreateLabel(object sender, EventArgs e)
        {
            textBox_Label.Text = ItemNamer.CreateUniqueNameFromFileName(
                                                _bankService.ComponentDictionary,
                                                new ItemNamingSettings(_projectService.Project),
                                                _projectService.Project.InstrumentNamePrefix,
                                                string.Empty, textBox_FilePath.Text);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnClosing(object sender, FormClosingEventArgs e)
        {
            if (DialogResult == DialogResult.OK)
            {
                if (ProjectFilePathUtility.ConfirmFilePath(FilePath) == false)
                {
                    e.Cancel = true;
                    return;
                }
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ProgramNoCreator
    {
        private HashSet<int> _Dictionary = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ProgramNoCreator(BankService bankService)
        {
            _Dictionary = CreateEntries(bankService);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int Create()
        {
            int programNo = Create(_Dictionary);
            _Dictionary.Add(programNo);
            return programNo;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public static int Create(BankService bankService)
        {
            return Create(CreateEntries(bankService));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static HashSet<int> CreateEntries(BankService bankService)
        {
            HashSet<int> dictionary = new HashSet<int>();

            foreach (Instrument instrument in bankService.Instruments)
            {
                dictionary.Add(instrument.ProgramNo);
            }
            return dictionary;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static int Create(HashSet<int> dictionary)
        {
            int programNo = 0;

            while (dictionary.Contains(programNo) != false)
            {
                programNo++;
            }
            return programNo;
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class OriginalKeyCreator
    {
        public static bool HasOriginalKeyFromFileName(string filePath)
        {
            try
            {
                // ファイル名に埋め込まれたキー文字列を検索
                Regex regex = new Regex(@"\W[a-gA-G][nbfsNFS#]?[mM-]?\d+\W");
                Match match = regex.Match(Path.GetFileName(filePath));
                if (match.Success && match.Value.Length > 2)
                {
                    return true;
                }
            }
            catch
            {
            }

            return false;
        }

        public static int FromFile(string filePath)
        {
            try
            {
                // ファイル名に埋め込まれたキー文字列を検索
                Regex regex = new Regex(@"\W[a-gA-G][nbfsNFS#]?[mM-]?\d+\W");
                Match match = regex.Match(Path.GetFileName(filePath));
                if (match.Success && match.Value.Length > 2)
                {
                    string keyString = match.Value.Substring(1, match.Value.Length - 2);
                    return ConvertStringToKeyNo(keyString);
                }
            }

            catch
            {
            }

            if (File.Exists(filePath) != false)
            {
                try
                {
                    using (WaveFileReader reader = WaveFileReader.CreateInstance(filePath))
                    {
                        WaveFile waveFile = reader.Open(filePath);
                        return waveFile.OriginalKey;
                    }
                }
                catch
                {
                    return 60; //cn4
                }
            }

            return 60; //cn4
        }

        ///
        private static int ConvertStringToKeyNo(string str)
        {
            int pos = 0;
            if (pos >= str.Length)
            {
                return -1;
            }

            int key = 0;
            switch (str[pos])
            {
                case 'c': case 'C': key = 0; break;
                case 'd': case 'D': key = 2; break;
                case 'e': case 'E': key = 4; break;
                case 'f': case 'F': key = 5; break;
                case 'g': case 'G': key = 7; break;
                case 'a': case 'A': key = 9; break;
                case 'b': case 'B': key = 11; break;
                default:
                    return -1;
            }

            pos++;
            if (pos >= str.Length)
            {
                return -1;
            }

            switch (str[pos])
            {
                case 'n': case 'N': pos++; break;
                case 'f': case 'F': case 'b': pos++; key--; break;
                case '#': case 's': case 'S': pos++; key++; break;
                default: break;
            }

            if (pos >= str.Length)
            {
                return -1;
            }

            bool minusFlag = false;
            if (str[pos] == 'm' || str[pos] == 'M' || str[pos] == '-')
            {
                minusFlag = true;
                pos++;
                if (pos >= str.Length)
                {
                    return -1;
                }
            }

            int octave = 0;
            while (pos < str.Length)
            {
                octave *= 10;

                switch (str[pos])
                {
                    case '0': octave += 0; break;
                    case '1': octave += 1; break;
                    case '2': octave += 2; break;
                    case '3': octave += 3; break;
                    case '4': octave += 4; break;
                    case '5': octave += 5; break;
                    case '6': octave += 6; break;
                    case '7': octave += 7; break;
                    case '8': octave += 8; break;
                    case '9': octave += 9; break;
                    default:
                        return -1;
                }

                pos++;
            }

            if (minusFlag) octave = -octave;

            key += (octave + 1) * 12;

            return key;
        }
    }
}
