﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Windows.Forms;

    using RegisterTypeResrcs = NintendoWare.SoundFoundation.Resources.MessageResource;
    using NintendoWare.SoundFoundation.Projects;
    using RegularExpressions = System.Text.RegularExpressions;


    ///--------------------------------------------------------------------------
    /// <summary>
    /// CreateGroupItemDialog の概要の説明です。
    /// </summary>
    public partial class CreateGroupItemDialog : CreateSoundSetItemDialog
    {
        private GroupBase _group;
        private SoundSetDocument _SoundSetDocument = null;
        private bool _SuspendUpdateDropDownList = false;

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CreateGroupItemDialog(
                                     SoundSetDocument soundSetDocument,
                                     SoundProjectService projectService,
                                     GroupBase group
                                     )
            : base(projectService)
        {
            if (null == group) { throw new ArgumentNullException("group"); }
            _group = group;
            _SoundSetDocument = soundSetDocument;

            InitializeComponent();

            comboBox_Label.DrawMode = DrawMode.OwnerDrawFixed;
            comboBox_Label.DrawItem += OnDrawItem;

            ComponentUIControlItem[] items = NotAddedItems;

            comboBox_Label.ItemHeight = 16;
            comboBox_Label.Items.AddRange(items);

            if (0 < items.Length)
            {
                comboBox_Label.SelectedIndex = 0;
            }

            UpdateRegisterTypesDropDownList();
        }

        public bool IsRegistableItemExisting
        {
            get { return (0 < NotAddedItems.Length); }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public GroupItemBase GroupItem
        {
            get
            {
                GroupItemBase groupItem = this.SelectGroupItem;
                groupItem.RegisterType = this.SelectRegisterType;

                return groupItem;
            }
        }

        protected ComponentUIControlItem[] NotAddedItems
        {
            get
            {
                return (from SoundSetItem item in ProjectService.SoundSetItems
                        where ((item is StreamSoundBase && AppConfiguration.EnabledPrefetch == true) ||
                               item is WaveSoundSetBase ||
                               item is SequenceSoundBase || item is SequenceSoundSetBase ||
                               item is SoundSetBankBase ||
                               item is GroupBase || item is WaveArchiveBase) &&
                        item != _group && !AlreadyContains(item)
                        select new ComponentUIControlItem(item)).ToArray();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override Control GetAcceptButton()
        {
            return button_OK;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool CanCreate
        {
            get
            {
                bool returnValue = true;

                if (comboBox_Label.Text == string.Empty)
                {
                    returnValue = false;
                }
                else
                {
                    try
                    {
                        int i = 0;
                        ComponentUIControlItem[] items = NotAddedItems;
                        foreach (ComponentUIControlItem item in items)
                        {
                            if (item.ToString().ToLower() == comboBox_Label.Text.ToLower())
                            {
                                ++i;
                                break;
                            }
                        }
                        if (i == 0)
                        {
                            returnValue = false;
                        }
                    }
                    catch { }
                }

                return returnValue;
            }
        }

        private GroupItemBase SelectGroupItem
        {
            get
            {
                GroupItemBase groupItem = null;
                Component component = null;

                if (comboBox_Label.SelectedItem != null &&
                    comboBox_Label.SelectedItem is ComponentUIControlItem)
                {
                    component = (comboBox_Label.SelectedItem as ComponentUIControlItem).Component;

                    groupItem = ApplicationBase.Instance.CreateComponentService.Create<GroupItemBase>();
                    groupItem.TargetItemReference = component.Name;
                    groupItem.TargetSoundSet = _SoundSetDocument.SoundSet.Name;
                    groupItem.Target = component as SoundSetItem;
                }

                return groupItem;
            }
        }

        private GroupItemRegisterType SelectRegisterType
        {
            get
            {
                GroupItemRegisterType type = GroupItemRegisterType.All;
                string text = RegisterTypeResrcs.Label_GroupItemRegisterType_All;

                if (comboBox_RegisterType.SelectedItem != null)
                {
                    text = comboBox_RegisterType.SelectedItem.ToString();
                }

                if (RegisterTypeResrcs.Label_GroupItemRegisterType_All == text)
                {
                    type = GroupItemRegisterType.All;
                }
                if (RegisterTypeResrcs.Label_GroupItemRegisterType_SequenceAndBank == text)
                {
                    type = GroupItemRegisterType.SequenceAndBank;
                }
                if (RegisterTypeResrcs.Label_GroupItemRegisterType_SequenceAndWaveArchive == text)
                {
                    type = GroupItemRegisterType.SequenceAndWaveArchive;
                }
                if (RegisterTypeResrcs.Label_GroupItemRegisterType_BankAndWaveArchive == text)
                {
                    type = GroupItemRegisterType.BankAndWaveArchive;
                }
                if (RegisterTypeResrcs.Label_GroupItemRegisterType_OnlySequence == text)
                {
                    type = GroupItemRegisterType.OnlySequence;
                }
                if (RegisterTypeResrcs.Label_GroupItemRegisterType_OnlyWaveSoundSet == text)
                {
                    type = GroupItemRegisterType.OnlyWaveSoundSet;
                }
                if (RegisterTypeResrcs.Label_GroupItemRegisterType_OnlyBank == text)
                {
                    type = GroupItemRegisterType.OnlyBank;
                }
                if (RegisterTypeResrcs.Label_GroupItemRegisterType_OnlyWaveArchive == text)
                {
                    type = GroupItemRegisterType.OnlyWaveArchive;
                }

                return type;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool AlreadyContains(SoundSetItem item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }

            return (from GroupItemBase groupItem in _group.Children
                    where groupItem.TargetItemReference == item.Name
                    select groupItem).FirstOrDefault() != null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void OnDrawItem(object sender, DrawItemEventArgs e)
        {
            ComboBox comboBox = sender as ComboBox;
            ComponentUIControlItem item = null;

            item = comboBox.Items[e.Index] as ComponentUIControlItem;
            if (item != null)
            {
                ListPartCommonRowHeaderDrawer.
                    DrawItem(e, comboBox.Font, item.Component, item.ToString());
            }
        }

        private void OnTextUpdate(object sender, EventArgs e)
        {
            UpdateDropDownList();
            UpdateControls();
        }

        private void OnDropDownClosed(object sender, EventArgs e)
        {
            UpdateRegisterTypesDropDownList();
            UpdateControls();
        }

        private void UpdateDropDownList()
        {
            if (_SuspendUpdateDropDownList != false) { return; }
            _SuspendUpdateDropDownList = true;

            string text = comboBox_Label.Text;
            int selectionStart = comboBox_Label.SelectionStart;
            int selectionLength = comboBox_Label.SelectionLength;

            List<ComponentUIControlItem> list = new List<ComponentUIControlItem>();
            RegularExpressions.Regex regex = null;
            string conditionText = @"^" + text.ToLower();

            ComponentUIControlItem[] items = NotAddedItems;

            try
            {
                regex = new RegularExpressions.Regex(conditionText);

                foreach (ComponentUIControlItem item in items)
                {
                    if (regex.IsMatch(item.ToString().ToLower()) != false)
                    {
                        list.Add(item);
                    }
                }
            }
            catch { }

            comboBox_Label.Items.Clear();

            if (list.Count <= 0)
            {
                comboBox_Label.Items.Add(string.Empty);
            }
            else
            {
                comboBox_Label.Items.AddRange(list.ToArray());
            }

            comboBox_Label.DroppedDown = true;
            comboBox_Label.SelectedIndex = -1;
            comboBox_Label.SelectedItem = null;

            comboBox_Label.Text = text;
            comboBox_Label.SelectionStart = selectionStart;
            comboBox_Label.SelectionLength = selectionLength;

            _SuspendUpdateDropDownList = false;
        }

        private void UpdateRegisterTypesDropDownList()
        {
            List<string> types = GetGroupItemRegisterTypes(this.SelectGroupItem);
            comboBox_RegisterType.Items.Clear();
            comboBox_RegisterType.Items.AddRange(types.ToArray());
            comboBox_RegisterType.SelectedIndex = 0;
        }

        private List<string> GetGroupItemRegisterTypes(GroupItemBase groupItem)
        {
            List<string> types = new List<string>();

            types.Add(RegisterTypeResrcs.Label_GroupItemRegisterType_All);

            if (groupItem != null)
            {
                if (groupItem.Target is WaveSoundSetBase)
                {
                    types.Add(RegisterTypeResrcs.Label_GroupItemRegisterType_OnlyWaveSoundSet);
                    types.Add(RegisterTypeResrcs.Label_GroupItemRegisterType_OnlyWaveArchive);
                }
                else if (groupItem.Target is SoundSetBankBase)
                {
                    types.Add(RegisterTypeResrcs.Label_GroupItemRegisterType_OnlyBank);
                    types.Add(RegisterTypeResrcs.Label_GroupItemRegisterType_OnlyWaveArchive);
                }
                else if (groupItem.Target is SequenceSoundBase || groupItem.Target is SequenceSoundSetBase)
                {
                    string label
                        = RegisterTypeResrcs.Label_GroupItemRegisterType_SequenceAndWaveArchive;
                    types.Add(label);
                    types.Add(RegisterTypeResrcs.Label_GroupItemRegisterType_SequenceAndBank);
                    types.Add(RegisterTypeResrcs.Label_GroupItemRegisterType_BankAndWaveArchive);
                    types.Add(RegisterTypeResrcs.Label_GroupItemRegisterType_OnlySequence);
                    types.Add(RegisterTypeResrcs.Label_GroupItemRegisterType_OnlyBank);
                    types.Add(RegisterTypeResrcs.Label_GroupItemRegisterType_OnlyWaveArchive);
                }
            }

            return types;
        }
    }
}
