﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareReport;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.Resources;
    using ToolDevelopmentKit;

    public partial class HtmlInstrumentListSettingsDialog : Form
    {
        private bool _lockMoveButtonState = false;

        private ListOutput listOutput = new ListOutput()
        {
            IsOutput = true,
        };

        public HtmlInstrumentListSettingsDialog()
        {
            InitializeComponent();
        }

        public ListOutput ListOutput
        {
            get { return this.listOutput; }
        }

        private ApplicationBase Application
        {
            get { return ApplicationBase.Instance; }
        }

        private SoundProjectConfiguration ProjectConfiguration
        {
            get { return Application.ProjectConfiguration; }
        }

        public void LoadSettings(XmlHtmlInstrumentList xmlData)
        {
            Ensure.Argument.NotNull(xmlData);

            foreach (XmlList xmlList in xmlData.Lists.List)
            {
                if (xmlList.Name != "HtmlInstrumentList")
                {
                    continue;
                }

                foreach (XmlListColumn xmlColumn in xmlList.ListColumns.ListColumn)
                {
                    this.listOutput.ColumnOutputs.Add(
                        new ListColumnOutput()
                        {
                            Name = xmlColumn.Name,
                            IsOutput = xmlColumn.Visible,
                        }
                        );
                }
            }

            _columnList.Items.Clear();

            foreach (ListColumnOutput columnOutput in listOutput.ColumnOutputs)
            {
                _columnList.Items.Add(new ColumnItem(columnOutput), columnOutput.IsOutput);
            }
        }

        public void SaveSettings(XmlHtmlInstrumentList xmlData)
        {
            Ensure.Argument.NotNull(xmlData);

            if (null == xmlData.Lists)
            {
                xmlData.Lists = new XmlLists();
            }

            if (null == xmlData.Lists.List)
            {
                xmlData.Lists.List = new XmlList[] {
                        new XmlList() {
                            Name = "HtmlInstrumentList",
                        }
                };
            }

            List<XmlListColumn> xmlColumns = new List<XmlListColumn>();

            foreach (ListColumnOutput columnOutput in this.listOutput.ColumnOutputs)
            {
                xmlColumns.Add(
                    new XmlListColumn()
                    {
                        Name = columnOutput.Name,
                        Visible = columnOutput.IsOutput,
                    });
            }

            xmlData.Lists.List[0].ListColumns.ListColumn = xmlColumns.ToArray();
        }

        protected override void OnFormClosed(FormClosedEventArgs e)
        {
            base.OnFormClosed(e);
            if (DialogResult != DialogResult.OK) { return; }

            this.listOutput.ColumnOutputs.Clear();

            int index = 0;

            foreach (ColumnItem columnItem in this._columnList.Items)
            {
                columnItem.ListColumnOutput.IsOutput = _columnList.GetItemChecked(index);
                this.listOutput.ColumnOutputs.Add(columnItem.ListColumnOutput);

                index++;
            }
        }

        private void OnMoveUpButtonClick(object sender, EventArgs e)
        {
            if (-1 == _columnList.SelectedIndex) { return; }

            SwapListItem(_columnList.SelectedIndex, _columnList.SelectedIndex - 1);
            _columnList.SelectedIndex--;
        }

        private void OnMoveDownButtonClick(object sender, EventArgs e)
        {
            if (-1 == _columnList.SelectedIndex) { return; }

            SwapListItem(_columnList.SelectedIndex, _columnList.SelectedIndex + 1);
            _columnList.SelectedIndex++;
        }

        private void OnCheckAllButtonClick(object sender, EventArgs e)
        {
            for (int i = 0; i < _columnList.Items.Count; i++)
            {
                _columnList.SetItemChecked(i, true);
            }
        }

        private void OnUncheckAllButtonClick(object sender, EventArgs e)
        {
            for (int i = 0; i < _columnList.Items.Count; i++)
            {
                _columnList.SetItemChecked(i, false);
            }
        }

        private void OnColumnListSelectedIndexChanged(object sender, EventArgs e)
        {
            if (_lockMoveButtonState) { return; }

            if (2 > _columnList.Items.Count || -1 == _columnList.SelectedIndex)
            {
                _moveUpButton.Enabled = false;
                _moveDownButton.Enabled = false;
                return;
            }

            _moveUpButton.Enabled = (1 <= _columnList.SelectedIndex);
            _moveDownButton.Enabled = (_columnList.SelectedIndex < _columnList.Items.Count - 1);
        }

        private void OnColumnListItemCheck(object sender, ItemCheckEventArgs e)
        {
            _okButton.Enabled = (CheckState.Unchecked != e.NewValue || 1 != _columnList.CheckedItems.Count);
        }

        private void SwapListItem(int index1, int index2)
        {
            Debug.Assert(0 <= index1, "unexpected error");
            Debug.Assert(0 <= index2, "unexpected error");

            _lockMoveButtonState = true;

            object itemWork = _columnList.Items[index1];
            bool checkWork = _columnList.GetItemChecked(index1);

            _columnList.Items[index1] = _columnList.Items[index2];
            _columnList.SetItemChecked(index1, _columnList.GetItemChecked(index2));

            _columnList.Items[index2] = itemWork;
            _columnList.SetItemChecked(index2, checkWork);

            _lockMoveButtonState = false;
        }

        private class ColumnItem
        {
            private static Dictionary<string, string> localizedTexts = new Dictionary<string, string>();

            static ColumnItem()
            {
                AddLocalizedTextItem(BankReportTraits.ColumnName.ItemIndex, MessageResource.HeaderText_ItemIndex);
                AddLocalizedTextItem(BankReportTraits.ColumnName.ProgramNo, MessageResource.HeaderText_ProgramNo);
                AddLocalizedTextItem(BankReportTraits.ColumnName.Name, MessageResource.HeaderText_Name);
                AddLocalizedTextItem(BankReportTraits.ColumnName.WaveEncoding, MessageResource.HeaderText_WaveEncoding);
                AddLocalizedTextItem(BankReportTraits.ColumnName.FileName, MessageResource.HeaderText_FilePath);

                AddLocalizedTextItem(BankReportTraits.ColumnName.WaveTime, MessageResource.HeaderText_WaveTime);
                AddLocalizedTextItem(BankReportTraits.ColumnName.WaveTick, MessageResource.HeaderText_WaveTick);
                AddLocalizedTextItem(BankReportTraits.ColumnName.SampleRate, MessageResource.HeaderText_SampleRate);
                AddLocalizedTextItem(BankReportTraits.ColumnName.WaveBitRate, MessageResource.HeaderText_WaveBitRate);
                AddLocalizedTextItem(BankReportTraits.ColumnName.WaveSampleBit, MessageResource.HeaderText_WaveSampleBit);
                AddLocalizedTextItem(BankReportTraits.ColumnName.OriginalKey, MessageResource.HeaderText_OriginalKey);
                AddLocalizedTextItem(BankReportTraits.ColumnName.EnvelopeAttack, MessageResource.HeaderText_EnvelopeAttack);
                AddLocalizedTextItem(BankReportTraits.ColumnName.EnvelopeHold, MessageResource.HeaderText_EnvelopeHold);
                AddLocalizedTextItem(BankReportTraits.ColumnName.EnvelopeDecay, MessageResource.HeaderText_EnvelopeDecay);
                AddLocalizedTextItem(BankReportTraits.ColumnName.EnvelopeSustain, MessageResource.HeaderText_EnvelopeSustain);
                AddLocalizedTextItem(BankReportTraits.ColumnName.EnvelopeRelease, MessageResource.HeaderText_EnvelopeRelease);
                AddLocalizedTextItem(BankReportTraits.ColumnName.Volume, MessageResource.HeaderText_Volume);
                AddLocalizedTextItem(BankReportTraits.ColumnName.PitchSemitones, MessageResource.HeaderText_PitchSemitones);
                AddLocalizedTextItem(BankReportTraits.ColumnName.PitchCents, MessageResource.HeaderText_PitchCents);

                AddLocalizedTextItem(BankReportTraits.ColumnName.InterpolationType, MessageResource.HeaderText_InterpolationType);

                AddLocalizedTextItem(BankReportTraits.ColumnName.Comment, MessageResource.HeaderText_Comment);
            }

            public ColumnItem(ListColumnOutput listColumnOutput)
            {
                Assertion.Argument.NotNull(listColumnOutput);
                this.ListColumnOutput = listColumnOutput;
            }

            public ListColumnOutput ListColumnOutput { get; private set; }

            public override string ToString()
            {
                return localizedTexts[this.ListColumnOutput.Name];
            }

            private static void AddLocalizedTextItem(string name, string localizedText)
            {
                Assertion.Argument.NotNull(name);
                Assertion.Argument.NotNull(localizedText);

                localizedTexts.Add(name, localizedText.Replace("\r\n", string.Empty));
            }
        }
    }
}
