﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Drawing;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.Conversion;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.ToolDevelopmentKit;
    using ComponentModel = System.ComponentModel;
    using Conversion = NintendoWare.SoundFoundation.Conversion;
    using Win32 = NintendoWare.SoundFoundation.Core.Win32;

    /// <summary>
    /// ConvertDialog の概要の説明です。
    /// </summary>
    public partial class ConvertDialog : Form
    {
        private const int ConvertingIconBegin = 1;
        private const int ConvertingIconCount = 8;
        private const int SucceededIconIndex = 9;
        private const int ErrorIconIndex = 0;

        private ISoundProjectConvertService _convertService;
        private DialogResult _result = DialogResult.None;

        private Icon[] _icons = null;
        private int _convertingIconIndex = 0;

        private string title = string.Empty;

        private string installDir;
        private List<string> logList = new List<string>();
        private System.Windows.Forms.Timer _iconTimer = new System.Windows.Forms.Timer();

        private string _targetFilePath;

        private HashSet<BankServiceReference> bankServiceRefs = new HashSet<BankServiceReference>();

        public ConvertDialog(ISoundProjectConvertService convertService)
        {
            if (null == convertService) { throw new ArgumentNullException("converter"); }
            _convertService = convertService;

            // Windows フォーム デザイナ サポートに必要です。
            InitializeComponent();

            _icons = new Icon[] {   ImageResource.IconConvert,
                                    ImageResource.IconConverting00,
                                    ImageResource.IconConverting01,
                                    ImageResource.IconConverting02,
                                    ImageResource.IconConverting03,
                                    ImageResource.IconConverting04,
                                    ImageResource.IconConverting05,
                                    ImageResource.IconConverting06,
                                    ImageResource.IconConverting07,
                                    ImageResource.IconConvertSucceeded  };
            SetIcon(ErrorIconIndex);

            this.title = this.Text;

            // コンバートイベント登録
            convertService.EndConvert += OnEndConvert;
            convertService.OutputLine += OnOutputLine;

            this.checkBox_NotClose.Checked = this.IsNotCloseConvertDialog;
        }

        #region ** プロパティ

        ///
        public string InstallDir
        {
            set { installDir = value; }
            get { return installDir; }
        }

        private SoundProjectService ProjectService
        {
            get { return ApplicationBase.Instance.ProjectService; }
        }

        private BankServiceManager BankServices
        {
            get { return ApplicationBase.Instance.BankServices; }
        }

        private MainWindow MainWindow
        {
            get { return FormsApplication.Instance.UIService.MainWindow; }
        }

        private OutputLine SelectedLine
        {
            get
            {
                if (1 != _logText.SelectedItems.Length) { return null; }
                return _logText.SelectedItems[0].UserData as OutputLine;
            }
        }

        private Component TargetItem
        {
            get
            {
                if (null == SelectedLine || 0 == SelectedLine.Components.Length)
                {
                    return null;
                }

                _targetFilePath = null;

                Instrument targetInstrument = null;

                foreach (Component item in SelectedLine.Components)
                {
                    if (item is StreamSoundBase ||
                        item is WaveSoundSetBase ||
                        item is WaveSoundBase ||
                        item is SequenceSoundSetBase ||
                        item is SequenceSoundBase ||
                        item is PlayerBase ||
                        item is GroupBase)
                    {
                        if (!ProjectService.ComponentDictionary.Contains(item.Name))
                        {
                            return null;
                        }

                        return ProjectService.ComponentDictionary[item.Name][0];
                    }

                    if (item is SoundSetBankBase)
                    {
                        BankServiceReference bankServiceRef = null;

                        try
                        {
                            SoundSetBankBase soundSetBank = item as SoundSetBankBase;

                            bankServiceRef = BankServices.OpenItem(soundSetBank.FilePath);
                            if (bankServiceRef == null)
                            {
                                return null;
                            }

                            _targetFilePath = soundSetBank.FilePath;

                            this.bankServiceRefs.Add(bankServiceRef);

                            if (targetInstrument != null)
                            {
                                return targetInstrument;
                            }
                            else
                            {
                                return soundSetBank;
                            }
                        }
                        catch
                        {
                            if (bankServiceRef != null)
                            {
                                this.bankServiceRefs.Remove(bankServiceRef);
                                bankServiceRef.Close();
                            }

                            return null;
                        }
                    }

                    if (item is Instrument)
                    {
                        targetInstrument = item as Instrument;
                        continue;
                    }
                    else if (item is KeyRegion)
                    {
                        targetInstrument = item.Parent as Instrument;
                        continue;
                    }
                    else if (item is VelocityRegion)
                    {
                        targetInstrument = item.Parent.Parent as Instrument;
                        continue;
                    }
                }

                if (targetInstrument != null)
                {
                    return targetInstrument;
                }

                return null;
            }
        }

        private bool IsConverting
        {
            get { return !buttonClose.Visible; }
        }

        private bool IsNotCloseConvertDialog
        {
            get
            {
                return ApplicationBase.Instance.AppConfiguration.Options.Application.General.NotCloseConvertDialog;
            }
            set
            {
                ApplicationBase.Instance.AppConfiguration.Options.Application.General.NotCloseConvertDialog = value;
            }
        }

        #endregion

        #region ** イベントハンドラ

        ///
        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            if (!_convertService.IsConverting && _result == DialogResult.OK)
            {
                DialogResult = DialogResult.OK;
                return;
            }

            _iconTimer.Interval = 200;
            _iconTimer.Tick += OnIconTimer;
            _iconTimer.Start();

            XmlWindow xmlWindow = ApplicationBase.Instance.AppConfiguration.Windows["ConvertWindow"];
            if (null != xmlWindow)
            {
                WindowConfigurationApplier.Apply(this, xmlWindow);
            }

            UpdateButtons();
        }

        protected override void OnFormClosing(FormClosingEventArgs e)
        {
            if (_convertService.IsConverting)
            {
                e.Cancel = true;
                return;
            }

            XmlWindow xmlWindow = ApplicationBase.Instance.AppConfiguration.Windows["ConvertWindow"];
            if (null != xmlWindow)
            {
                WindowConfigurationApplier.Extract(this, xmlWindow);
            }

            base.OnFormClosing(e);
        }

        protected override void OnFormClosed(FormClosedEventArgs e)
        {
            _iconTimer.Dispose();
            _iconTimer = null;

            // コンバートイベント登録解除
            _convertService.OutputLine -= OnOutputLine;
            _convertService.EndConvert -= OnEndConvert;

            foreach (BankServiceReference bankServiceRef in this.bankServiceRefs)
            {
                bankServiceRef.Close();
            }
            this.bankServiceRefs.Clear();

            base.OnFormClosed(e);
        }

        protected override void OnLayout(LayoutEventArgs levent)
        {
            base.OnLayout(levent);

            // 最大化時等にシステムメニューがリセットされることがあるので、ここで更新する
            UpdateSystemMenu();
        }

        private void OnIconTimer(object sender, EventArgs e)
        {
            if (!_convertService.IsConverting) { return; }

            _convertingIconIndex++;

            if (_convertingIconIndex >= ConvertingIconBegin + ConvertingIconCount)
            {
                _convertingIconIndex = ConvertingIconBegin;
            }

            BeginInvoke((SetIconDelegate)SetIcon, new object[] { _convertingIconIndex });
        }

        private void OnCancelClick(object sender, System.EventArgs e)
        {
            _convertService.Cancel();
        }

        private void OnMove2ItemClick(object sender, EventArgs e)
        {
            Move2TargetItem();
        }

        private void OnCloseClick(object sender, EventArgs e)
        {
            DialogResult = DialogResult.OK;
        }

        private void OnLogSelectedItemsChanged(object sender, EventArgs e)
        {
            UpdateButtons();
        }

        private void OnOutputLine(object sender, OutputLineEventArgs e)
        {
            AppendLine(e.Lines);

            if (this._convertService.ProgressMax > 0)
            {
                this.Text = string.Format(
                    "{0} ({1}%)",
                    this.title,
                    Convert.ToInt32(this._convertService.ProgressCurrent * 100 / this._convertService.ProgressMax)
                    );
            }

            if (null == SelectedLine &&
                e.Lines[e.Lines.Length - 1].Level == OutputLevel.Error)
            {
                _logText.SelectLastItem();
            }
        }

        private void OnEndConvert(object sender, Conversion.SoundProjectConvertEventArgs e)
        {
            if (e.Succeeded || e.Canceled)
            {
                ConvertFinish();
            }
            else
            {
                ConvertError();
            }
        }

        #endregion

        #region ** メソッド

        private void SetMessage(string msg)
        {
            Text = msg;
        }

        private void AppendLine(string msg)
        {
            _logText.Append(msg);
            _logText.ScrollToBottom();
        }

        private void AppendLine(IEnumerable<OutputLine> lines)
        {
            foreach (OutputLine line in lines)
            {
                _logText.Append(line.Text, line);
            }

            _logText.ScrollToBottom();
        }

        private void ConvertFinish()
        {
            _iconTimer.Stop();
            SetIcon(SucceededIconIndex);

            _result = DialogResult.OK;
            DialogResult = DialogResult.OK;
        }

        private void ConvertError()
        {
            _result = DialogResult.Cancel;

            _iconTimer.Stop();
            SetIcon(ErrorIconIndex);

            if (this.IsNotCloseConvertDialog == true)
            {
                buttonClose.DialogResult = DialogResult.Cancel;
                buttonClose.Enabled = true;
                buttonClose.Visible = true;

                buttonCancel.Visible = false;
                buttonCancel.Enabled = false;

                UpdateButtons();
            }
            else
            {
                DialogResult = DialogResult.OK;
            }
        }

        private void UpdateSystemMenu()
        {
            IntPtr hSystemMenu = Win32.User32.GetSystemMenu(Handle, false);
            Win32.User32.EnableMenuItem(hSystemMenu, Win32.SC.SC_CLOSE,
                                                buttonClose.Visible ? Win32.MF.MF_ENABLED : Win32.MF.MF_GRAYED);
        }

        private void UpdateButtons()
        {
            if (null == SelectedLine || IsConverting)
            {
                _buttonMove2Item.Enabled = false;
                return;
            }

            switch (SelectedLine.Level)
            {
                case OutputLevel.Error:
                case OutputLevel.Warning:
                    if (TargetItem is Instrument)
                    {
                        _buttonMove2Item.Enabled = MainWindow.FindPage(TargetItem.Parent) != null;
                    }
                    else
                    {
                        _buttonMove2Item.Enabled = TargetItem != null;
                    }
                    break;

                default:
                    _buttonMove2Item.Enabled = false;
                    break;
            }
        }

        private void Move2TargetItem()
        {
            if (IsConverting) { return; }

            Component item = TargetItem;

            try
            {
                if (null != _targetFilePath)
                {
                    MainWindow.OpenFile(_targetFilePath);
                }

                MainWindow.ShowPageByComponent(item);

                DialogResult = DialogResult.OK;
                return;
            }
            catch
            {
            }
        }

        private void SetIcon(int iconIndex)
        {
            if (_icons.Length <= iconIndex) { return; }
            Icon = _icons[iconIndex];
        }

        #endregion

        #region ** デリゲート

        private delegate void SetIconDelegate(int iconIndex);

        #endregion

        private void OnCheckedChanged_NotClose(object sender, EventArgs e)
        {
            this.IsNotCloseConvertDialog = this.checkBox_NotClose.Checked;
        }
    }
}
