﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Drawing;
    using System.Drawing.Imaging;
    using System.Globalization;
    using System.IO;
    using System.Linq;
    using System.Security.Cryptography;
    using System.Text;
    using System.Windows.Forms;
    using System.Windows.Forms.VisualStyles;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate;
    using NintendoWare.SoundFoundation.FileFormats.Wave;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.FileFormats;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.ToolDevelopmentKit.Collections;

    public static class BatchEditUtility
    {
        public delegate string ConvertTextHandler(string name, string text);
        public delegate IParameterValue ConvertValueHandler(string name, string text, IParameterValue value);
        public delegate bool ValidateHandler(Component component, string name, string text, object value);

        /// <summary>
        ///
        /// </summary>
        public static Operation[] SetEffect(ComponentListItem targetItem, string name, EffectKind effectKind, string text, ConvertTextHandler convertTextHandler, ConvertValueHandler convertValueHandler, ValidateHandler validator)
        {
            List<Operation> list = new List<Operation>();
            CommonListItem item = targetItem as CommonListItem;

            Debug.Assert(item != null, "Item is not CommonListItem");
            var component = item.GetTargetByName(name);

            if (component == null ||
                (component.Parameters.ContainsKey(name) == false &&
                ListTraits.IsSoundSetBankReference(name) == false &&
                ListTraits.IsUserParameters(name) == false))
            {
                return null;
            }

            if (component is GroupItemBase)
            {
                return null;
            }

            try
            {
                string answerText = null;
                var value = item.GetValue(name, true);

                text = convertTextHandler(name, text);
                value = convertValueHandler(name, text, value);

                switch (effectKind)
                {
                    case EffectKind.Set:
                        answerText = text;
                        break;

                    case EffectKind.Addition:
                        {
                            var sourceValue = decimal.Parse(value.Value.ToString(), NumberStyles.Float);
                            var destValue = decimal.Parse(text, NumberStyles.Float);
                            answerText = (sourceValue + destValue).ToString();
                        }
                        break;

                    case EffectKind.Multiplication:
                        {
                            var sourceValue = decimal.Parse(value.Value.ToString(), NumberStyles.Float);
                            var destValue = decimal.Parse(text, NumberStyles.Float);
                            answerText = (sourceValue * destValue).ToString();
                        }
                        break;
                }

                object parsedValue = null;
                if (value.Value is int ||
                    value.Value is uint ||
                    value.Value is long ||
                    value.Value is ulong)
                {
                    try
                    {
                        parsedValue = value.ParseValue(answerText);
                    }
                    catch (FormatException)
                    {
                        parsedValue = Convert.ChangeType(Math.Truncate(decimal.Parse(answerText)), value.Value.GetType());
                    }
                }
                else
                {
                    parsedValue = value.ParseValue(answerText);
                }

                if (validator(component, name, answerText, parsedValue) == false)
                {
                    return null;
                }

                var result = value.ValidateValue(parsedValue);
                if (result.IsValid == false)
                {
                    return null;
                }

                switch (name)
                {
                    case ProjectParameterNames.SequenceSound.SoundSetBankReference0:
                    case ProjectParameterNames.SequenceSound.SoundSetBankReference1:
                    case ProjectParameterNames.SequenceSound.SoundSetBankReference2:
                    case ProjectParameterNames.SequenceSound.SoundSetBankReference3:
                        {
                            var operations = BankReferenceSetter.Set(component as SequenceSoundBase, name, answerText);
                            if (operations != null)
                            {
                                list.AddRange(operations);
                            }
                        }
                        break;

                    case ProjectParameterNames.SndEdit:
                        component.Parameters[name].Value = parsedValue;
                        break;

                    case ProjectParameterNames.SampleRate:
                        {
                            if (component is StreamSoundBase == false &&
                                component is WaveSoundBase == false &&
                                component is Instrument == false)
                            {
                                break;
                            }

                            var operation = new SetParameterOperation(component.Parameters, name, parsedValue);
                            operation.Execute();
                            list.Add(operation);
                            if ((bool)component.Parameters[ProjectParameterNames.IsResampleEnabled].Value == false)
                            {
                                operation = new SetParameterOperation(component.Parameters, ProjectParameterNames.IsResampleEnabled, true);
                                operation.Execute();
                                list.Add(operation);
                            }
                        }
                        break;

                    default:
                        {
                            if (ListTraits.IsUserParameters(name) == true)
                            {
                                UserParameterStructure structure = UserParameterSettingWatcher.GetStructure(name);
                                string userParameterName = UserParameterSettingWatcher.GetUserParameterName(name);
                                ulong oldValue = (ulong)component.Parameters.GetValue(userParameterName).Value;
                                parsedValue = UserParameterUtility.SetValue(structure, oldValue, parsedValue);
                                name = userParameterName;
                            }
                            var operation = new SetParameterOperation(component.Parameters, name, parsedValue);
                            if (operation.Execute() == true)
                            {
                                list.Add(operation);
                            }
                        }
                        break;
                }
            }
            catch
            {
            }

            return list.Count > 0 ? list.ToArray() : null;
        }
    }

    /// <summary>
    ///
    /// </summary>
    public static class ValueTextUtility
    {
        private static string MultiBankReferenceSeparator = "|";

        /// <summary>
        ///
        /// </summary>
        public static string CreateBankReferenceText(IList<ComponentReference> list)
        {
            if (list.Count <= 0)
            {
                return null;
            }

            StringBuilder sb = new StringBuilder();
            sb.Append(list[0].TargetName);

            for (int index = 1; index < list.Count; index++)
            {
                sb.Append(MultiBankReferenceSeparator);
                sb.Append(list[index].TargetName);
            }
            return sb.ToString();
        }

        /// <summary>
        ///
        /// </summary>
        public static string[] DecompositionBankReferenceText(string text)
        {
            string[] delimiters = new string[] { " ", MultiBankReferenceSeparator };
            return text.Split(delimiters, StringSplitOptions.RemoveEmptyEntries);
        }
    }

    /// <summary>
    /// 参照バンクを設定するクラスです。
    /// </summary>
    public static class BankReferenceSetter
    {
        /// <summary>
        ///
        /// </summary>
        public static Operation[] Set(SequenceSoundBase sequenceSound, string name, string value)
        {
            int index = -1;
            switch (name)
            {
                case ProjectParameterNames.SequenceSound.SoundSetBankReference0: index = 0; break;
                case ProjectParameterNames.SequenceSound.SoundSetBankReference1: index = 1; break;
                case ProjectParameterNames.SequenceSound.SoundSetBankReference2: index = 2; break;
                case ProjectParameterNames.SequenceSound.SoundSetBankReference3: index = 3; break;
                default: return new Operation[0];
            }

            string parameterName = ProjectParameterNames.SequenceSound.SoundSetBankReferences;
            ObservableList<ComponentReference> reference =
                (ObservableList<ComponentReference>)(sequenceSound.Parameters[parameterName].Value);
            SetParameterOperation operation = new SetParameterOperation
                (reference[index].Parameters,
                  ProjectParameterNames.TargetName,
                  value);
            operation.Execute();
            return new Operation[] { operation };
        }
    }

    /// <summary>
    /// リストに関するユーティリティークラスです。
    /// </summary>
    public static class CommonListUtil
    {
        private const string ValueWithLabelFormat = "{0} ({1})";

        public static string GetValueWithLabel(int value, string label)
        {
            return String.Format(ValueWithLabelFormat, value, label);
        }

        public static string GetValueWithLabel(ulong value, string label)
        {
            return String.Format(ValueWithLabelFormat, value, label);
        }

        public static string GetValueWithLabel(float value, string label)
        {
            return String.Format(ValueWithLabelFormat, value, label);
        }

        public static string GetValueWithLabel(string value, string label)
        {
            return String.Format(ValueWithLabelFormat, value, label);
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class DuplicationNameChecker
    {
        /// <summary>
        ///
        /// </summary>
        public static bool Inquire(Component component)
        {
            if (component == null)
            {
                return false;
            }

            BankService bankService = null;
            Instrument instrument = null;
            string name = component.Name;
            Component[] components = null;

            if (component is StreamSoundTrackBase ||
                component is KeyRegion || component is VelocityRegion)
            {
                return false;
            }

            if (component is GroupItemBase)
            {
                name = (component as GroupItemBase).TargetItemReference;
                if (ProjectService.ComponentDictionary.Contains(name) == true)
                {
                    if (ProjectService.ComponentDictionary[name].Length > 1)
                    {
                        return true;
                    }
                }
            }

            if (component is Instrument)
            {
                instrument = component as Instrument;
                if ((bankService = GetBankService(instrument.Bank)) != null)
                {
                    if (bankService.ComponentDictionary.Contains(name) == true)
                    {
                        components = bankService.ComponentDictionary[name];
                        if (components.Length > 1)
                        {
                            return true;
                        }
                    }
                }
            }
            else
            {
                if (ProjectService.ComponentDictionary.Contains(name) == true)
                {
                    components = ProjectService.ComponentDictionary[name]
                        .Where(c => !(c is SoundSet))
                        .Where(c => !(c is FolderComponent))
                        .Where(c => !(c is SoundProject))
                        .ToArray();

                    if (components.Length > 1)
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        /// <summary>
        ///
        /// </summary>
        private static SoundProjectService ProjectService
        {
            get { return FormsApplication.Instance.ProjectService; }
        }

        /// <summary>
        ///
        /// </summary>
        private static BankService GetBankService(Bank bank)
        {
            foreach (BankService service in FormsApplication.Instance.BankServices.Values)
            {
                if (service.Bank == bank)
                {
                    return service;
                }
            }
            return null;
        }
    }

    /// <summary>
    /// ユーザーパラメータの設定を取得するクラスです。
    /// </summary>
    internal class UserParameterSettingWatcher
    {
        /// <summary>
        /// ユーザーパラメータの設定を取得します。
        /// </summary>
        public static UserParameterStructureSetting Get(string name)
        {
            int index = ToIndex(name);

            UserParameterStructureSettings settings =
                ApplicationBase.Instance.ProjectService.Project.UserDataStructureSettings;
            Debug.Assert(index >= 0 && index < settings.Settings.Count,
                          "Out of range user parameter index");
            return settings.Settings[index];
        }

        /// <summary>
        ///
        /// </summary>
        public static int ToIndex(string name)
        {
            switch (name)
            {
                case ProjectParameterNames.Sound.UserParameter: return 0;
                case ProjectParameterNames.Sound.UserParameter1: return 1;
                case ProjectParameterNames.Sound.UserParameter2: return 2;
                case ProjectParameterNames.Sound.UserParameter3: return 3;
            }
            Debug.Assert(false, "Bad user parameter name");
            return -1;
        }

        /// <summary>
        /// ユーザーパラメータの設定を取得します。
        /// </summary>
        public static UserParameterStructure GetStructure(string name)
        {
            UserParameterStructureSettings settings =
                ApplicationBase.Instance.ProjectService.Project.UserDataStructureSettings;

            int i = 0;
            foreach (UserParameterStructureSetting setting in settings.Settings)
            {
                int j = 0;
                foreach (UserParameterStructure structure in setting.Structures)
                {
                    if (name == ListTraits.ColumnName_UserParameters[i, j])
                    {
                        return structure;
                    }
                    j++;
                }
                i++;
            }

            return null;
        }

        /// <summary>
        /// カラム名からユーザーパラメータを取得します。
        /// </summary>
        public static string GetUserParameterName(string name)
        {
            UserParameterStructureSettings settings =
                ApplicationBase.Instance.ProjectService.Project.UserDataStructureSettings;

            int i = 0;
            foreach (UserParameterStructureSetting setting in settings.Settings)
            {
                int j = 0;
                foreach (UserParameterStructure structure in setting.Structures)
                {
                    if (name == ListTraits.ColumnName_UserParameters[i, j])
                    {
                        switch (i)
                        {
                            case 0: return ProjectParameterNames.Sound.UserParameter;
                            case 1: return ProjectParameterNames.Sound.UserParameter1;
                            case 2: return ProjectParameterNames.Sound.UserParameter2;
                            case 3: return ProjectParameterNames.Sound.UserParameter3;
                        }
                    }
                    j++;
                }
                i++;
            }

            return string.Empty;
        }
    }
}
