﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.Audio;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate;
    using NintendoWare.SoundFoundation.FileFormats.Wave;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Utilities;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.FileFormats;
    using NintendoWare.SoundMaker.Framework.Projects;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Utilities;
    using NintendoWare.ToolDevelopmentKit.Collections;

    // 処理を実行した場合にはtrueを返す必要があります。
    public delegate IConstParameterValue GetConstValueHandler(Component component, string name, ref bool cancel);
    public delegate IParameterValue GetValueHandler(Component component, string name, ref bool cancel);

    /// <summary>
    /// 共用のアイテム
    /// </summary>
    public class CommonListItem : ComponentListItem, IListItemToolTipProvider
    {
        ///
        private const string NotAvailableText = "-";

        ///
        private Dictionary<string, GetConstValueHandler> getterConstValueDictionary = new Dictionary<string, GetConstValueHandler>();
        private Dictionary<string, GetValueHandler> getterValueDictionary = new Dictionary<string, GetValueHandler>();
        private HashSet<string> readOnlyNames = new HashSet<string>();

        /// <summary>
        ///
        /// </summary>
        public Dictionary<string, GetConstValueHandler> ConstValueGetters
        {
            get
            {
                return this.getterConstValueDictionary;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public Dictionary<string, GetValueHandler> ValueGetters
        {
            get
            {
                return this.getterValueDictionary;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public HashSet<string> ReadOnlyNames
        {
            get
            {
                return this.readOnlyNames;
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CommonListItem(Component component)
            : base(component)
        {
            Length = 20;

            //
            ConstValueGetters.Add("RowHeader", RowHeaderConstValueGetter);
            ConstValueGetters.Add(ProjectParameterNames.WaveTime, WaveTimeConstValueGetter);
            ConstValueGetters.Add(ProjectParameterNames.SampleRate, SampleRateConstValueGetter);
            ConstValueGetters.Add(ProjectParameterNames.WaveBitRate, WaveBitRateConstValueGetter);
            ConstValueGetters.Add(ProjectParameterNames.WaveSampleBit, WaveSampleBitConstValueGetter);
            ConstValueGetters.Add(ProjectParameterNames.WaveChannel, WaveChannelConstValueGetter);
            ConstValueGetters.Add(ProjectParameterNames.StreamSoundTrack.TrackNo, TrackNoConstValueGetter);
            ConstValueGetters.Add(ListTraits.ColumnName_ChannelNo, ChannelNoConstValueGetter);
            ConstValueGetters.Add(ProjectParameterNames.SoundSetItem.PreviewPlay, PreviewPlayConstValueGetter);
            //ConstValueGetters.Add( ProjectParameterNames.LoopType, LoopTypeConstValueGetter);
            ConstValueGetters.Add(ProjectParameterNames.Sound.SinglePlayEffectiveDuration, SinglePlayEffectiveDurationConstValueGetter);

            ConstValueGetters.Add(ProjectParameterNames.FrontBypass, StreamConstValueGetter);

            ConstValueGetters.Add(ProjectParameterNames.SequenceSound.SoundSetBankReference0,
                                   BankReference0ConstValueGetter);
            ConstValueGetters.Add(ProjectParameterNames.SequenceSound.SoundSetBankReference1,
                                   BankReference1ConstValueGetter);
            ConstValueGetters.Add(ProjectParameterNames.SequenceSound.SoundSetBankReference2,
                                   BankReference2ConstValueGetter);
            ConstValueGetters.Add(ProjectParameterNames.SequenceSound.SoundSetBankReference3,
                                   BankReference3ConstValueGetter);

            ConstValueGetters.Add(ListTraits.ColumnName_MaximumVoiceCount, MaximumVoiceCountConstValueGetter);

            ConstValueGetters.Add(ProjectParameterNames.IntegratedLoudness, this.IntegratedLoudnessConstValueGetter);

            //
            foreach (string name in ListTraits.GetAllCommentColumnNames())
            {
                ConstValueGetters.Add(name, CommentConstValueGetter);
            }

            //
            foreach (string userParameter in ListTraits.GetAllUserParametersColumnNames())
            {
                ConstValueGetters.Add(userParameter, UserParametersConstValueGetter);
            }

            //
            ValueGetters.Add(ProjectParameterNames.WaveTime, ReadOnlyValueGetter);
            ValueGetters.Add(ProjectParameterNames.SampleRate, SampleRateValueGetter);
            ValueGetters.Add(ProjectParameterNames.IsResampleEnabled, IsResampleEnabledValueGetter);
            ValueGetters.Add(ProjectParameterNames.IsDownMixEnabled, IsDownMixEnabledValueGetter);
            ValueGetters.Add(ProjectParameterNames.WaveBitRate, ReadOnlyValueGetter);
            ValueGetters.Add(ProjectParameterNames.WaveChannel, ReadOnlyValueGetter);
            ValueGetters.Add(ProjectParameterNames.WaveSampleBit, ReadOnlyValueGetter);
            ValueGetters.Add(ProjectParameterNames.StreamSoundTrack.TrackNo, ReadOnlyValueGetter);
            ValueGetters.Add(ListTraits.ColumnName_ChannelNo, ReadOnlyValueGetter);
            ValueGetters.Add(ProjectParameterNames.SoundSetItem.PreviewPlay, PreviewPlayValueGetter);
            ValueGetters.Add(ProjectParameterNames.Sound.SinglePlayEffectiveDuration, SinglePlayEffectiveDurationValueGetter);
            //ValueGetters.Add( ProjectParameterNames.LoopType, LoopTypeValueGetter);

            ValueGetters.Add(ProjectParameterNames.FilePath, StreamValueGetter);
            ValueGetters.Add(ProjectParameterNames.FrontBypass, StreamValueGetter);

            ValueGetters.Add(ProjectParameterNames.SequenceSound.SoundSetBankReference0,
                              BankReference0ValueGetter);
            ValueGetters.Add(ProjectParameterNames.SequenceSound.SoundSetBankReference1,
                              BankReference1ValueGetter);
            ValueGetters.Add(ProjectParameterNames.SequenceSound.SoundSetBankReference2,
                              BankReference2ValueGetter);
            ValueGetters.Add(ProjectParameterNames.SequenceSound.SoundSetBankReference3,
                              BankReference3ValueGetter);

            ValueGetters.Add(ListTraits.ColumnName_MaximumVoiceCount, ReadOnlyValueGetter);

            foreach (string name in ListTraits.GetAllCommentColumnNames())
            {
                ValueGetters.Add(name, CommentValueGetter);
            }

            foreach (string userParameter in ListTraits.GetAllUserParametersColumnNames())
            {
                ValueGetters.Add(userParameter, UserParametersValueGetter);
            }

            // 読み取り用のパラメータ名の追加
            ReadOnlyNames.Add(ProjectParameterNames.GroupItem.GroupItemTargetSoundSet);
            ReadOnlyNames.Add(ProjectParameterNames.DataSize);
            ReadOnlyNames.Add(ProjectParameterNames.IntegratedLoudness);
        }

        /// <summary>
        ///
        /// </summary>
        public virtual Component GetTargetByName(string name)
        {
            StreamSoundTrackBase track = null;

            if (name == ProjectParameterNames.FilePath ||
                name == ProjectParameterNames.FrontBypass)
            {
                if (Target is StreamSoundBase)
                {
                    if ((track = GetSingleStreamSoundTrack(Target)) != null)
                    {
                        return track;
                    }
                }
            }

            return Target;
        }

        /// <summary>
        ///
        /// </summary>
        public virtual string FilePath
        {
            get
            {
                StreamSoundTrackBase track = null;

                if (Target is StreamSoundBase)
                {
                    if ((track = GetSingleStreamSoundTrack(Target)) == null)
                    {
                        return String.Empty;
                    }
                    return track.FilePath;
                }

                if (Target == null ||
                    Target.Parameters.ContainsKey(ProjectParameterNames.FilePath) == false)
                {
                    return String.Empty;
                }


                return (string)Target.Parameters[ProjectParameterNames.FilePath].Value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public override IParameterValue GetValue(string name)
        {
            return GetValue(name, false);
        }

        /// <summary>
        ///
        /// </summary>
        public IParameterValue GetValue(string name, bool force)
        {
            if (Target == null)
            {
                return null;
            }

            // パラメータには入っていないが、Getterによって取得できる場合があるので
            // パラメータに入っていないだけでは nullを返しません。
            if (Target.Parameters.ContainsKey(name) == false &&
                this.getterValueDictionary.ContainsKey(name) == false)
            {
                return null;
            }

            // 読み取り用のパラメータ名なのか？
            if (this.readOnlyNames.Contains(name) != false)
            {
                return null;
            }

            // Getterに含まれている場合には、Getterに処理を委譲します。
            if (this.getterValueDictionary.ContainsKey(name) == true)
            {
                bool cancel = false;
                IParameterValue value = this.getterValueDictionary[name](Target, name, ref cancel);

                if (cancel == false)
                {
                    return value;
                }
            }

            //
            switch (name)
            {
                case ProjectParameterNames.Name:
                    {
                        // ストリームサウンドトラックの名前の編集は禁止です。
                        if (Target is StreamSoundTrackBase)
                        {
                            return null;
                        }

                        return CreateParameterEditor
                            (name, new NameParameterValue((string)Target.Parameters[name].Value));
                    }

                case ProjectParameterNames.WaveArchive.LoadType:
                    WaveArchiveBase waveArchive = Target as WaveArchiveBase;
                    bool loadType = waveArchive.LoadType == WaveArchiveLoadType.Individual;
                    return CreateParameterEditor(name, new BoolParameterValue(loadType));

                case ProjectParameterNames.SequenceSound.SoundSetBankReferences:
                    {
                        IList<ComponentReference> list =
                            Target.Parameters[name].Value as IList<ComponentReference>;
                        if (list == null)
                        {
                            return null;
                        }

                        string text = null;
                        if (list.Count > 1)
                        {
                            if (force == false)
                            {
                                return null;
                            }

                            text = ValueTextUtility.CreateBankReferenceText(list);
                        }
                        else
                        {
                            text = list[0].TargetName;
                        }

                        return CreateParameterEditor
                            (name, new TextParameterValue(text));
                    }

                case ProjectParameterNames.SoundSetItem.WaveArchiveReference:
                    string warc = Target.Parameters[name].Value as string;
                    if (warc == null)
                    {
                        return null;
                    }
                    else
                    {
                        IParameterValue paraText = null;
                        switch (warc)
                        {
                            case WaveArchiveConsts.AutoShared:
                                paraText =
                                    new TextParameterValue(MessageResource.
                                                           Label_WaveArchiveReference_Shared);
                                break;

                            case WaveArchiveConsts.AutoIndividual:
                                paraText =
                                    new TextParameterValue(MessageResource.
                                                           Label_WaveArchiveReference_Individual);
                                break;

                            default:
                                paraText = new TextParameterValue(warc);
                                break;
                        }
                        return CreateParameterEditor(name, paraText);
                    }

                case ProjectParameterNames.UserParameter:
                case ProjectParameterNames.UserParameter1:
                case ProjectParameterNames.UserParameter2:
                case ProjectParameterNames.UserParameter3:
                    if (UserParameterSettingWatcher.Get(name).Enabled == false)
                    {
                        return null;
                    }
                    break;
            }

            //
            if (Target.Parameters.ContainsKey(name) == false)
            {
                return null;
            }

            return CreateParameterEditor(name, Target.Parameters[name]);
        }

        /// <summary>
        ///
        /// </summary>
        public override IConstParameterValue GetConstValue(string name)
        {
            return GetConstValueInternal(Target, name);
        }

        /// <summary>
        ///
        /// </summary>
        protected override IConstParameterValue GetConstValueInternal(Component target, string name)
        {
            if (target == null)
            {
                return null;
            }

            // パラメータには入っていないが、Getterによって取得できる場合があるので
            // パラメータに入っていないだけでは nullを返しません。
            if (target.Parameters.ContainsKey(name) == false &&
                this.getterConstValueDictionary.ContainsKey(name) == false &&
                name != "SndEditState" && name != ProjectParameterNames.FilePath)
            {
                return null;
            }

            //
            if (this.getterConstValueDictionary.ContainsKey(name) == true)
            {
                bool cancel = false;
                IConstParameterValue value =
                    this.getterConstValueDictionary[name](target, name, ref cancel);

                if (cancel == false)
                {
                    return value;
                }
            }

            //
            switch (name)
            {
                case ProjectParameterNames.Name:
                    if (target is StreamSoundTrackBase)
                    {
                        StreamSoundBase streamSound = target.Parent as StreamSoundBase;
                        Debug.Assert(streamSound != null, "Target parent is not StreamSoundBase.");
                        StreamSoundTrackBase streamSoundTrack = target as StreamSoundTrackBase;
                        Debug.Assert(streamSoundTrack != null, "Target is not StreamSoundTrackBase.");
                        int index = GetTrackNo(streamSoundTrack);
                        string indexText = index < 0 ? NotAvailable.Text : index.ToString();
                        string text = String.Format("{0} / {1}", streamSound.Name, indexText);
                        return new TextParameterValue(text);
                    }
                    break;

                case ProjectParameterNames.FilePath:
                    if (target is StreamSoundBase == true)
                    {
                        StreamSoundTrackBase track = GetSingleStreamSoundTrack(target);
                        if (track != null)
                        {
                            return new FilePathParameterValue(track.FilePath);
                        }
                        else if ((target as StreamSoundBase).IsMultiChannelAAC() == true)
                        {
                            StreamSoundTrackBase firstTrack = target.Children[0] as StreamSoundTrackBase;
                            return new FilePathParameterValue(firstTrack.FilePath);
                        }
                        else
                        {
                            return new TextParameterValue("*");
                        }
                    }
                    else if (target is StreamSoundTrackBase == true)
                    {
                        StreamSoundTrackBase track = target as StreamSoundTrackBase;

                        if (track.IsMultiChannelAACTrack() == true)
                        {
                            return new TextParameterValue(track.FilePath);
                        }
                        else
                        {
                            return new FilePathParameterValue(track.FilePath);
                        }
                    }
                    break;

                case ProjectParameterNames.GroupItem.GroupItemTargetSoundSet:
                    GroupItemBase groupItem = target as GroupItemBase;
                    Debug.Assert(groupItem != null, "Target is not GroupItemBase.");
                    return new TextParameterValue(groupItem.TargetSoundSet);

                case ProjectParameterNames.WaveArchive.LoadType:
                    WaveArchiveBase waveArchive = target as WaveArchiveBase;
                    Debug.Assert(waveArchive != null, "Target is not WaveArchiveBase.");
                    bool loadType = waveArchive.LoadType == WaveArchiveLoadType.Individual;
                    return new BoolParameterValue(loadType);

                case ProjectParameterNames.SequenceSound.SoundSetBankReference0:
                    {
                        IList<ComponentReference> list =
                            target.Parameters[name].Value as IList<ComponentReference>;
                        return new TextParameterValue(list[0].TargetName);
                    }
                case ProjectParameterNames.SequenceSound.SoundSetBankReference1:
                    {
                        IList<ComponentReference> list =
                            target.Parameters[name].Value as IList<ComponentReference>;
                        return new TextParameterValue(list[1].TargetName);
                    }
                case ProjectParameterNames.SequenceSound.SoundSetBankReference2:
                    {
                        IList<ComponentReference> list =
                            target.Parameters[name].Value as IList<ComponentReference>;
                        return new TextParameterValue(list[2].TargetName);
                    }
                case ProjectParameterNames.SequenceSound.SoundSetBankReference3:
                    {
                        IList<ComponentReference> list =
                            target.Parameters[name].Value as IList<ComponentReference>;
                        return new TextParameterValue(list[3].TargetName);
                    }

                case ProjectParameterNames.SoundSetItem.WaveArchiveReference:
                    string warc = target.Parameters[name].Value as string;
                    if (warc == null)
                    {
                        return null;
                    }
                    else if (warc == WaveArchiveConsts.AutoShared)
                    {
                        return new TextParameterValue(MessageResource.
                                                      Label_WaveArchiveReference_Shared);
                    }
                    else if (warc == WaveArchiveConsts.AutoIndividual)
                    {
                        return new TextParameterValue(MessageResource.
                                                      Label_WaveArchiveReference_Individual);
                    }
                    else
                    {
                        return new TextParameterValue(warc);
                    }
            }

            return base.GetConstValueInternal(target, name);
        }

        /// <summary>
        ///
        /// </summary>
        protected IParameterValue ReadOnlyValueGetter(Component component, string name, ref bool cancel)
        {
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        protected IParameterValue PreviewPlayValueGetter(Component component, string name, ref bool cancel)
        {
            return CreateParameterEditor(name, new BoolParameterValue(false));
        }

        /// <summary>
        /// サンプルレートカラムのセル編集
        /// </summary>
        protected IParameterValue SampleRateValueGetter(Component component, string name, ref bool cancel)
        {
            if (component is StreamSoundBase == false &&
                component is WaveSoundBase == false &&
                component is Instrument == false)
            {
                cancel = true;
                return null;
            }

            var param = (SampleRateParameterValue)component.Parameters[ProjectParameterNames.SampleRate];

            // リサンプルが無効の場合、サンプルレートは編集不可にします。
            if ((bool)component.Parameters[ProjectParameterNames.IsResampleEnabled].Value == false)
            {
                return null;
            }

            // Opus ファイルの場合、サンプルレートは編集不可にします。
            if (component is StreamSoundBase == true &&
                component.Children.Cast<StreamSoundTrackBase>().Any(t => FileUtil.IsOpusFile(t.FilePath)) == true)
            {
                return null;
            }

            // 編集初期値をサンプルレートにする
            return CreateSampleRateParameterEditor(name, new SampleRateParameterValue(param.Value));
        }

        private IParameterValue CreateSampleRateParameterEditor(string name, IParameterValue value)
        {
            ParameterEditor parameterEditor = new SampleRateParameterEditor(name, value);
            parameterEditor.ValueChanged += OnValueChanged;

            return parameterEditor;
        }

        /// <summary>
        /// ダウンサンプルカラムのセル編集
        /// </summary>
        protected IParameterValue IsResampleEnabledValueGetter(Component component, string name, ref bool cancel)
        {
            if (component is StreamSoundBase == true &&
                component.Children.Cast<StreamSoundTrackBase>().Any(t => FileUtil.IsOpusFile(t.FilePath)) == true)
            {
                // Opus ファイルが含まれる場合、ダウンサンプルは編集不可にします。
                return null;
            }

            // 条件外なのでここはキャンセルし、呼び出し元の GetValue() にまかせます。
            cancel = true;
            return null;
        }

        /// <summary>
        /// ダウンミックスのセル編集
        /// </summary>
        protected IParameterValue IsDownMixEnabledValueGetter(Component component, string name, ref bool cancel)
        {
            if (component is StreamSoundBase == true &&
                component.Children.Cast<StreamSoundTrackBase>().Any(t => FileUtil.IsOpusFile(t.FilePath)) == true)
            {
                // Opus ファイルが含まれる場合、ダウンミックスは編集不可にします。
                return null;
            }

            // 条件外なのでここはキャンセルし、呼び出し元の GetValue() にまかせます。
            cancel = true;
            return null;
        }

        /// <summary>
        /// 単一再生の有効時間の有効、無効を切り替える
        /// </summary>
        protected IParameterValue SinglePlayEffectiveDurationValueGetter(Component component, string name, ref bool cancel)
        {
            if (name == ProjectParameterNames.Sound.SinglePlayEffectiveDuration &&
                component is Sound == true)
            {
                Sound sound = component as Sound;
                if (sound.SinglePlayType == SinglePlayType.PrioritizeOldestEffectiveDuration ||
                    sound.SinglePlayType == SinglePlayType.PrioritizeNewestEffectiveDuration)
                {
                    // 時間内後着優先、時間内先着優先であれば
                    // ここはキャンセルし、呼び出し元の GetValue() にまかせます。
                    cancel = true;
                    return null;
                }
            }

            // 条件外の場合、単一再生の有効時間は常に編集不可にします。
            return null;
        }

#if false
        /// <summary>
        ///
        /// </summary>
        protected IParameterValue LoopTypeValueGetter(Component component, string name, ref bool cancel)
        {
            if (component is Instrument)
            {
                if (component.Children.Count == 1 &&
                    component.Children[0].Children.Count == 1 &&
                    component.Children[0].Children[0] is VelocityRegion)
                {
                    VelocityRegion velocityRegion =
                        component.Children[0].Children[0] as VelocityRegion;
                    return CreateParameterEditor(name, velocityRegion.Parameters[name]);
                }
                else
                {
                    return null;
                }
            }

            cancel = true;
            return null;
        }
#endif

        /// <summary>
        ///
        /// </summary>
        protected IParameterValue StreamValueGetter(Component component, string name, ref bool cancel)
        {
            if (component is StreamSoundBase)
            {
                StreamSoundTrackBase track = GetSingleStreamSoundTrack(component);
                if (track == null)
                {
                    if (AppConfiguration.EnabledMultiChannelAAC == false)
                    {
                        return null;
                    }
                    else
                    {
                        track = component.Children[0] as StreamSoundTrackBase;
                        if (AACUtil.IsAACFile(track.FilePath) == false)
                        {
                            return null;
                        }
                    }
                }

                return CreateParameterEditor(name, track.Parameters[name], OnValueChanged);
            }
            else if (component is StreamSoundTrackBase == true &&
                     name == ProjectParameterNames.FilePath &&
                     (component as StreamSoundTrackBase).IsMultiChannelAACTrack() == true)
            {
                return null;
            }

            cancel = true;
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        protected IParameterValue UserParametersValueGetter(Component component, string name, ref bool cancel)
        {
            string userParameterName = UserParameterSettingWatcher.GetUserParameterName(name);
            if (UserParameterSettingWatcher.Get(userParameterName).Enabled == false ||
                component.Parameters.ContainsKey(userParameterName) == false)
            {
                return null;
            }

            ulong uValue = (ULongParameterValue)component.Parameters.GetValue(userParameterName);
            UserParameterStructure structure = UserParameterSettingWatcher.GetStructure(name);
            cancel = false;

            // GetValue() の値は編集時にドロップダウンで表示されます。
            // ユーザーが選択肢を把握しやすいように、ユーザーが記述した値をそのまま表示します。
            // 例：
            // ユーザーの記述 A = +10.0
            // 実際の値　　　 10
            // ドロップダウン +10.0 (A)　←　ユーザーの記述の値をそのまま表示。

            switch (structure.StructureType)
            {
                case StructureTypes.Integer:
                    {
                        int value = UserParameterUtility.GetPartialIntegerValue(structure, uValue);

                        Dictionary<string, string> valueCandidates = UserParameterUtility.GetValueCandidate(structure);
                        int min = (int)UserParameterUtility.GetMinimum(structure);
                        int max = (int)UserParameterUtility.GetMaximum(structure);
                        IntLabelParameterValue pv = new IntLabelParameterValue(value, min, max);
                        foreach (var pair in valueCandidates)
                        {
                            pv.LabelDictionary.Add(pv.ParseValue(pair.Key), CommonListUtil.GetValueWithLabel(pair.Key, pair.Value));
                        }

                        return CreateParameterEditor(name, pv, OnValueChanged);
                    }

                case StructureTypes.UInteger:
                    {
                        ulong value = UserParameterUtility.GetPartialUIntegerValue(structure, uValue);
                        Dictionary<string, string> valueCandidates = UserParameterUtility.GetValueCandidate(structure);
                        ulong max = (ulong)UserParameterUtility.GetMaximum(structure);
                        ULongLabelParameterValue pv = new ULongLabelParameterValue(value, 0UL, max);
                        foreach (var pair in valueCandidates)
                        {
                            pv.LabelDictionary.Add(pv.ParseValue(pair.Key), CommonListUtil.GetValueWithLabel(pair.Key, pair.Value));
                        }

                        return CreateParameterEditor(name, pv, OnValueChanged);
                    }

                case StructureTypes.Decimal:
                    {
                        float value = UserParameterUtility.GetPartialDecimalValue(structure, uValue);

                        Dictionary<string, string> valueCandidates = UserParameterUtility.GetValueCandidate(structure);
                        FloatLabelParameterValue pv = new FloatLabelParameterValue(value);
                        foreach (var pair in valueCandidates)
                        {
                            pv.LabelDictionary.Add(pv.ParseValue(pair.Key), CommonListUtil.GetValueWithLabel(pair.Key, pair.Value));
                        }

                        return CreateParameterEditor(name, pv, OnValueChanged);
                    }

                case StructureTypes.Boolean:
                    {
                        bool value = UserParameterUtility.GetPartialBooleanValue(structure, uValue);
                        return CreateParameterEditor(name, new BoolParameterValue(value), OnValueChanged);
                    }
            }

            return null;
        }

        /// <summary>
        /// 行ヘッダ
        /// </summary>
        protected IConstParameterValue RowHeaderConstValueGetter(Component component, string name, ref bool cancel)
        {
            return new TextParameterValue("*");
        }

        /// <summary>
        /// 再生時間を取得します。
        /// </summary>
        protected IConstParameterValue WaveTimeConstValueGetter(Component component, string name, ref bool cancel)
        {
            if (this.HasMultiWaveFiles(component) == true && component is StreamSoundBase == false)
            {
                return new TextParameterValue("*");
            }

            double waveTime; // 再生時間
            bool isLoop; // ループ情報

            // ストリームサウンドの再生時間はトラックの中で最長の再生時間を使用する
            if (component is StreamSoundBase == true)
            {
                // トラックを抽出します。
                var tracks = component.Children.Cast<StreamSoundTrackBase>();

                // 波形ファイルの情報がなければ更新します。
                tracks
                    .Where(t => t.WaveFile == null)
                    .ToList()
                    .ForEach(t => t.UpdateWaveFile());

                // opus 以外の波形情報を抽出します。
                var waveFiles = tracks
                    .Where(t => FileUtil.IsOpusFile(t.FilePath) == false)
                    .Select(t => t.WaveFile)
                    .Where(w => w != null);

                if (waveFiles.Count() == 0)
                {
                    return new TextWithNAParameterValue(NotAvailable.Text, false);
                }

                // 一番長い再生時間を抽出します。
                waveTime = waveFiles.Max(w => w.WaveTime);

                // ストリームサウンドのループ情報は先頭のトラックのループ情報を使用する
                isLoop = tracks.First().WaveFile?.IsLoop ?? false;
            }
            else // ストリームサウンド以外
            {
                WaveFile waveFile = GetComponentReferenceWaveFile(component);
                if (waveFile == null)
                {
                    return new TextWithNAParameterValue(NotAvailable.Text, false);
                }

                waveTime = waveFile.WaveTime;
                isLoop = waveFile.IsLoop;
            }

            string text = WaveFileUtility.GetWaveTimeStringWithLoop(waveTime, isLoop);

            // GetWaveTimeStringWithLoopが NotAvailable.Textを返すことを前提としています。
            bool available = text == NotAvailable.Text ? false : true;
            return new TextWithNAParameterValue(text, available);
        }

        /// <summary>
        /// サンプルレートを取得します。
        /// </summary>
        protected IConstParameterValue SampleRateConstValueGetter(Component component, string name, ref bool cancel)
        {
            WaveFile waveFile = GetComponentReferenceWaveFile(component);
            if (waveFile == null)
            {
                return new TextWithNAParameterValue(NotAvailableText, false);
            }

            // リサンプルが有効ならサンプルレートを返す。
            if (component.Parameters.ContainsKey(ProjectParameterNames.IsResampleEnabled) == true &&
                (bool)component.Parameters[ProjectParameterNames.IsResampleEnabled].Value == true)
            {
                SampleRateParameterValue param = component.Parameters[ProjectParameterNames.SampleRate] as SampleRateParameterValue;
                // サンプルレートが設定されていれば（未設定でなければ）
                if (param.IsUnsettingValue == false)
                {
                    // サンプルレートを返す。
                    return new TextParameterValue(WaveFileUtility.GetSampleRateString(param.Value));
                }
            }
            else if (this.HasMultiWaveFiles(component) == true)
            {
                return new TextParameterValue("*");
            }

            // 波形ファイルのサンプルレートを返す。
            string text = WaveFileUtility.GetSampleRateString(waveFile);

            // GetSampleRateStringが NotAvailable.Textを返すことを前提としています。
            bool available = text == NotAvailable.Text ? false : true;
            return new TextWithNAParameterValue(text, available);
        }

        /// <summary>
        /// 平均ラウドネス値を取得します。
        /// </summary>
        protected IConstParameterValue IntegratedLoudnessConstValueGetter(Component component, string name, ref bool cancel)
        {
            if (component.Parameters.ContainsKey(ProjectParameterNames.IntegratedLoudness) == false)
            {
                return new TextWithNAParameterValue(NotAvailableText, true);
            }

            var status = component.Parameters[ProjectParameterNames.IntegratedLoudnessStatus];

            switch ((IntegratedLoudnessStatus)status.Value)
            {
                case IntegratedLoudnessStatus.Unmeasured:     // 未計測なら「未計測」を返す。
                case IntegratedLoudnessStatus.WaitingMeasure: // 計測待ちなら「計測待ち」を返す。
                case IntegratedLoudnessStatus.Measuring:      // 計測中なら「計測中」を返す。
                case IntegratedLoudnessStatus.Error:          // 計測失敗なら「計測失敗」を返す。
                    return new TextParameterValue(status.ToString());

                case IntegratedLoudnessStatus.Measured:       // 計測済みなら平均ラウドネス値を返す。
                    float value = (float)component.Parameters[ProjectParameterNames.IntegratedLoudness].Value;
                    value = (float)(Math.Ceiling(value * 10) / 10); // 有効桁数小数点以下一桁（二桁目は切り上げ）
                    return new TextParameterValue(string.Format("{0:F1} LKFS", value));
            }

            return new TextWithNAParameterValue(NotAvailable.Text, true);
        }

        /// <summary>
        /// ビットレートを取得します。
        /// </summary>
        protected IConstParameterValue WaveBitRateConstValueGetter(Component component, string name, ref bool cancel)
        {
            if (this.HasMultiWaveFiles(component) == true)
            {
                return new TextParameterValue("*");
            }

            WaveFile waveFile = GetComponentReferenceWaveFile(component);
            if (waveFile == null)
            {
                return new TextWithNAParameterValue(NotAvailableText, false);
            }

            string text = NotAvailable.Text;

            if (waveFile is WaveFileAiff == true || waveFile is WaveFileWav == true)
            {
                // リサンプルが有効なら設定されたサンプルレートからビットレートを求める。
                if (component.Parameters.ContainsKey(ProjectParameterNames.IsResampleEnabled) == true &&
                    (bool)component.Parameters[ProjectParameterNames.IsResampleEnabled].Value == true)
                {
                    SampleRateParameterValue param = component.Parameters[ProjectParameterNames.SampleRate] as SampleRateParameterValue;
                    // サンプルレートが設定されていれば（未設定でなければ）
                    if (param.IsUnsettingValue == false)
                    {
                        text = WaveFileUtility.GetWaveBitRateString(waveFile, param.Value);

                        return new TextWithNAParameterValue(text, text == NotAvailable.Text ? false : true);
                    }
                }

                // 波形ファイルのサンプルレートからビットレートを求める。
                text = WaveFileUtility.GetWaveBitRateString(waveFile);
            }

            // GetWaveBitRateStringが NotAvailable.Textを返すことを前提としています。
            bool available = text == NotAvailable.Text ? false : true;

            return new TextWithNAParameterValue(text, available);
        }

        /// <summary>
        /// 量子化ビット数を取得します。
        /// </summary>
        protected IConstParameterValue WaveSampleBitConstValueGetter(Component component, string name, ref bool cancel)
        {
            if (this.HasMultiWaveFiles(component) == true)
            {
                return new TextParameterValue("*");
            }

            WaveFile waveFile = GetComponentReferenceWaveFile(component);
            if (waveFile == null)
            {
                return new TextWithNAParameterValue(NotAvailableText, false);
            }

            if ((waveFile is WaveFileAiff == false) && (waveFile is WaveFileWav == false))
            {
                waveFile = null; // Wav, Aiff ファイルではないので null にする。
            }
            string text = WaveFileUtility.GetWaveSampleBitString(waveFile);

            // GetWaveSampleBitStringが NotAvailable.Textを返すことを前提としています。
            bool available = text == NotAvailable.Text ? false : true;

            return new TextWithNAParameterValue(text, available);
        }

        /// <summary>
        /// チャンネル数を取得します。
        /// </summary>
        protected IConstParameterValue WaveChannelConstValueGetter(Component component, string name, ref bool cancel)
        {
            // ダウンミックスが有効ならモノラルを表示する。
            if (this.IsDownMixEnabled(component) == true)
            {
                var str = WaveFileUtility.GetChannelString(1); // モノラル

                return new TextWithNAParameterValue(str, str != NotAvailable.Text);
            }

            if (this.HasMultiWaveFiles(component) == true)
            {
                return new TextParameterValue("*");
            }

            WaveFile waveFile = GetComponentReferenceWaveFile(component);
            if (waveFile == null)
            {
                return new TextWithNAParameterValue(NotAvailableText, false);
            }

            string text = WaveFileUtility.GetChannelString(waveFile);

            // GetChannelStringが NotAvailable.Textを返すことを前提としています。
            bool available = text == NotAvailable.Text ? false : true;

            return new TextWithNAParameterValue(text, available);
        }

        private bool IsResampleEnabled(Component component)
        {
            if (component.Parameters.ContainsKey(ProjectParameterNames.IsResampleEnabled) == true &&
                (bool)component.Parameters[ProjectParameterNames.IsResampleEnabled].Value == true)
            {
                return true;
            }
            return false;
        }

        private bool IsDownMixEnabled(Component component)
        {
            if (component.Parameters.ContainsKey(ProjectParameterNames.IsDownMixEnabled) == true &&
                (bool)component.Parameters[ProjectParameterNames.IsDownMixEnabled].Value == true)
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// トラック番号を取得します。
        /// </summary>
        protected IConstParameterValue TrackNoConstValueGetter(Component component, string name, ref bool cancel)
        {
            StreamSoundTrackBase streamSoundTrack = component as StreamSoundTrackBase;
            int index = GetTrackNo(streamSoundTrack);
            if (index < 0)
            {
                return new TextWithNAParameterValue(NotAvailableText, false);
            }
            else
            {
                return new IntParameterValue(index);
            }
        }

        /// <summary>
        /// チャンネル番号を取得します。
        /// </summary>
        protected IConstParameterValue ChannelNoConstValueGetter(Component component, string name, ref bool cancel)
        {
            StreamSoundTrackBase streamSoundTrack = component as StreamSoundTrackBase;
            string channelNoString = GetChannelNoString(streamSoundTrack);
            return new TextParameterValue(channelNoString);
        }

        /// <summary>
        /// プレビュー再生
        /// </summary>
        protected IConstParameterValue PreviewPlayConstValueGetter(Component component, string name, ref bool cancel)
        {
            if (component is StreamSoundBase == true ||
                component is SequenceSoundBase == true ||
                component is WaveSoundBase == true)
            {
                return new BoolParameterValue(false);
            }

            return new TextWithNAParameterValue(NotAvailableText, false);
        }

        /// <summary>
        /// 単一再生の有効時間
        /// </summary>
        protected IConstParameterValue SinglePlayEffectiveDurationConstValueGetter(Component component, string name, ref bool cancel)
        {
            if (name == ProjectParameterNames.Sound.SinglePlayEffectiveDuration &&
                component is Sound == true)
            {
                Sound sound = component as Sound;

                if (sound.SinglePlayType == SinglePlayType.PrioritizeOldestEffectiveDuration ||
                    sound.SinglePlayType == SinglePlayType.PrioritizeNewestEffectiveDuration)
                {
                    // 時間内後着優先、時間内先着優先であれば値を返す。（値が表示される）
                    return new TextParameterValue(string.Format("{0} msec", sound.SinglePlayEffectiveDuration));
                }
            }

            // それ以外。（NotAvailableText "-" が表示される）
            return new TextWithNAParameterValue(NotAvailableText, false);
        }

#if false
        /// <summary>
        ///
        /// </summary>
        protected IConstParameterValue LoopTypeConstValueGetter(Component component, string name, ref bool cancel)
        {
            if (component is Instrument)
            {
                if (component.Children.Count == 1 &&
                    component.Children[0].Children.Count == 1 &&
                    component.Children[0].Children[0] is VelocityRegion)
                {
                    VelocityRegion velocityRegion =
                        component.Children[0].Children[0] as VelocityRegion;
                    return velocityRegion.Parameters[name];
                }
                else
                {
                    return new TextParameterValue("*");
                }
            }

            cancel = true;
            return null;
        }
#endif

        /// <summary>
        ///
        /// </summary>
        protected IConstParameterValue StreamConstValueGetter(Component component, string name, ref bool cancel)
        {
            if (component is StreamSoundBase == true)
            {
                StreamSoundTrackBase track = GetSingleStreamSoundTrack(component);
                switch (name)
                {
                    case ProjectParameterNames.FrontBypass:
                        if (track != null)
                        {
                            return new BoolParameterValue(track.FrontBypass);
                        }
                        else
                        {
                            return new TextParameterValue("*");
                        }
                }
            }

            cancel = true;
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        protected IConstParameterValue BankReference0ConstValueGetter(Component component, string name, ref bool cancel)
        {
            return GetBankReference(0);
        }

        /// <summary>
        ///
        /// </summary>
        protected IConstParameterValue BankReference1ConstValueGetter(Component component, string name, ref bool cancel)
        {
            return GetBankReference(1);
        }

        /// <summary>
        ///
        /// </summary>
        protected IConstParameterValue BankReference2ConstValueGetter(Component component, string name, ref bool cancel)
        {
            return GetBankReference(2);
        }

        /// <summary>
        ///
        /// </summary>
        protected IConstParameterValue BankReference3ConstValueGetter(Component component, string name, ref bool cancel)
        {
            return GetBankReference(3);
        }

        /// <summary>
        /// シーケンスの最大発音数を取得します。
        /// </summary>
        protected IConstParameterValue MaximumVoiceCountConstValueGetter(Component component, string name, ref bool cancel)
        {
            SequenceSoundBase sequenceSound = component as SequenceSoundBase;

            if (sequenceSound == null || sequenceSound.MaximumVoiceCount == null)
            {
                return new TextParameterValue(NotAvailable.Text);
            }

            return new IntParameterValue((int)sequenceSound.MaximumVoiceCount);
        }

        /// <summary>
        ///
        /// </summary>
        protected IConstParameterValue CommentConstValueGetter(Component component, string name, ref bool cancel)
        {
            if (component is GroupItemBase &&
                GeneralOptions.ShowGroupItemTargetComment != false)
            {
                GroupItemBase groupItem = component as GroupItemBase;
                if (groupItem.Target != null)
                {
                    return GetConstValueInternal(groupItem.Target, name);
                }
            }

            cancel = true;
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        protected IParameterValue BankReference0ValueGetter(Component component, string name, ref bool cancel)
        {
            return GetBankReferenceValue(name, 0);
        }

        /// <summary>
        ///
        /// </summary>
        protected IParameterValue BankReference1ValueGetter(Component component, string name, ref bool cancel)
        {
            return GetBankReferenceValue(name, 1);
        }

        /// <summary>
        ///
        /// </summary>
        protected IParameterValue BankReference2ValueGetter(Component component, string name, ref bool cancel)
        {
            return GetBankReferenceValue(name, 2);
        }

        /// <summary>
        ///
        /// </summary>
        protected IParameterValue BankReference3ValueGetter(Component component, string name, ref bool cancel)
        {
            return GetBankReferenceValue(name, 3);
        }

        /// <summary>
        ///
        /// </summary>
        protected IParameterValue GetBankReferenceValue(string name, int index)
        {
            string parameterName = ProjectParameterNames.SequenceSound.SoundSetBankReferences;
            IList<ComponentReference> list =
                Target.Parameters[parameterName].Value as IList<ComponentReference>;
            if (list == null)
            {
                return null;
            }

            string text = list[index].TargetName;
            return CreateParameterEditor(name, new TextParameterValue(text), OnValueChanged);
        }

        /// <summary>
        ///
        /// </summary>
        protected IConstParameterValue GetBankReference(int index)
        {
            string parameterName = ProjectParameterNames.SequenceSound.SoundSetBankReferences;
            IList<ComponentReference> list =
                Target.Parameters[parameterName].Value as IList<ComponentReference>;
            if (list == null)
            {
                return null;
            }
            return new TextParameterValue(list[index].TargetName);
        }

        /// <summary>
        ///
        /// </summary>
        protected IParameterValue CommentValueGetter(Component component, string name, ref bool cancel)
        {
            if (component is GroupItemBase &&
                GeneralOptions.ShowGroupItemTargetComment != false)
            {
                return null;
            }

            cancel = true;
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        protected IConstParameterValue UserParametersConstValueGetter(Component component, string name, ref bool cancel)
        {
            string userParameterName = UserParameterSettingWatcher.GetUserParameterName(name);
            if (component.Parameters.ContainsKey(userParameterName) == false)
            {
                return null;
            }
            ulong uValue = (ULongParameterValue)component.Parameters.GetValue(userParameterName);
            UserParameterStructure structure = UserParameterSettingWatcher.GetStructure(name);

            // GetConstValue() の値はリスト中に表示されます。
            // 設定された実際の値が確認できるように、ユーザーの記述をパースした結果（実際の値）を表示します。
            // 例：
            // ユーザーの記述 A = +10.0
            // 実際の値　　　 10
            // リスト表示　　 10 (A)　←　ユーザーの記述をパースした結果（実際の値）を表示。

            switch (structure.StructureType)
            {
                case StructureTypes.Integer:
                    {
                        int value = UserParameterUtility.GetPartialIntegerValue(structure, uValue);
                        Dictionary<string, string> valueCandidates = UserParameterUtility.GetValueCandidate(structure);
                        IntLabelParameterValue pv = new IntLabelParameterValue(value);
                        foreach (var pair in valueCandidates)
                        {
                            var iValue = pv.ParseValue(pair.Key);
                            pv.LabelDictionary.Add(iValue, CommonListUtil.GetValueWithLabel(iValue, pair.Value));
                        }
                        return pv;
                    }

                case StructureTypes.UInteger:
                    {
                        ulong value = UserParameterUtility.GetPartialUIntegerValue(structure, uValue);
                        Dictionary<string, string> valueCandidates = UserParameterUtility.GetValueCandidate(structure);
                        ULongLabelParameterValue pv = new ULongLabelParameterValue(value);
                        foreach (var pair in valueCandidates)
                        {
                            var uiValue = pv.ParseValue(pair.Key);
                            pv.LabelDictionary.Add(uiValue, CommonListUtil.GetValueWithLabel(uiValue, pair.Value));
                        }
                        return pv;
                    }

                case StructureTypes.Decimal:
                    {
                        float value = UserParameterUtility.GetPartialDecimalValue(structure, uValue);
                        Dictionary<string, string> valueCandidates = UserParameterUtility.GetValueCandidate(structure);
                        FloatLabelParameterValue pv = new FloatLabelParameterValue(value);
                        foreach (var pair in valueCandidates)
                        {
                            var fValue = pv.ParseValue(pair.Key);
                            pv.LabelDictionary.Add(fValue, CommonListUtil.GetValueWithLabel(fValue, pair.Value));
                        }
                        return pv;
                    }

                case StructureTypes.Boolean:
                    {
                        bool value = UserParameterUtility.GetPartialBooleanValue(structure, uValue);
                        return new BoolParameterValue(value);
                    }
            }

            return new TextParameterValue(string.Empty);
        }

        /// <summary>
        ///
        /// </summary>
        protected override Component GetTarget(string name)
        {
            StreamSoundTrackBase track = null;

            if (name == ProjectParameterNames.FilePath ||
                name == ProjectParameterNames.FrontBypass)
            {
                if ((track = GetSingleStreamSoundTrack(Target)) != null)
                {
                    return track;
                }
                else if (AppConfiguration.EnabledMultiChannelAAC == true &&
                         name == ProjectParameterNames.FilePath &&
                         Target is StreamSoundBase == true)
                {
                    return Target.Children[0];
                }
            }

            return base.GetTarget(name);
        }

        /// <summary>
        ///
        /// </summary>
        public void ResetWaveFile()
        {
            Component component = Target;

            if (component is StreamSoundBase)
            {
                foreach (StreamSoundTrackBase track in component.Children)
                {
                    track.WaveFile = null;
                }
            }
            else if (component is Instrument)
            {
                foreach (KeyRegion keyRegion in component.Children)
                {
                    foreach (VelocityRegion velocityRegion in keyRegion.Children)
                    {
                        velocityRegion.WaveFile = null;
                    }
                }
            }
            else if (component is StreamSoundTrackBase)
            {
                (component as StreamSoundTrackBase).WaveFile = null;
            }
            else if (component is WaveSoundBase)
            {
                (component as WaveSoundBase).WaveFile = null;
            }
            else if (component is VelocityRegion)
            {
                (component as VelocityRegion).WaveFile = null;
            }
        }

        /// <summary>
        /// Componentが参照している波形ファイルを取得します。
        /// 参照している波形ファイルのインスタンスが作成されていない場合には作成も行います。
        /// </summary>
        protected virtual WaveFile GetComponentReferenceWaveFile(Component component)
        {
            if (component is StreamSoundBase == true)
            {
                // デフォルトは先頭のトラック
                component = component.Children[0];
            }

            if (component is StreamSoundTrackBase == true)
            {
                StreamSoundTrackBase track = component as StreamSoundTrackBase;
                if (track.WaveFile == null)
                {
                    track.UpdateWaveFile();
                }

                return track.WaveFile;
            }
            else if (component is WaveSoundBase == true)
            {
                WaveSoundBase waveSound = component as WaveSoundBase;
                if (waveSound.WaveFile == null)
                {
                    waveSound.UpdateWaveFile();
                }

                return waveSound.WaveFile;
            }

            return null;
        }

        /// <summary>
        ///
        /// </summary>
        public XmlOptionsApplicationGeneral GeneralOptions
        {
            get
            {
                return ApplicationBase.Instance.AppConfiguration.Options.Application.General;
            }
        }

        /// <summary>
        /// ストリームサウンドトラックのインデックスを取得します。
        /// 親ストリームサウンドの子供としてのインデックスです。
        /// </summary>
        private int GetTrackNo(StreamSoundTrackBase targetStreamSoundTrack)
        {
            if (targetStreamSoundTrack == null)
            {
                return -1;
            }

            StreamSoundBase streamSound = targetStreamSoundTrack.Parent as StreamSoundBase;
            Debug.Assert(streamSound != null, "StreamSoundTrackBase parent is not StreamSoundBase.");

            int index = 0;
            foreach (StreamSoundTrackBase streamSoundTrack in streamSound.Children)
            {
                if (streamSoundTrack == targetStreamSoundTrack)
                {
                    return streamSoundTrack.IsEnabled == false ? -1 : index;
                }

                if (streamSoundTrack.IsEnabled != false)
                {
                    index++;
                }
            }

            Debug.Assert(false, "No contains StreamSoundTrackBase in StreamSoundBase children.");
            return -1;
        }

        /// <summary>
        /// ストリームサウンドトラックに格納されるチャンネル番号を取得します。
        /// </summary>
        private string GetChannelNoString(StreamSoundTrackBase targetStreamSoundTrack)
        {
            if (targetStreamSoundTrack == null ||
                targetStreamSoundTrack.IsEnabled == false ||
                targetStreamSoundTrack.ChannelCount <= 0)
            {
                return NotAvailableText;
            }

            StreamSoundBase streamSound = targetStreamSoundTrack.Parent as StreamSoundBase;
            Debug.Assert(streamSound != null, "StreamSoundTrackBase parent is not StreamSoundBase.");

            if (AACUtil.IsAACFile((streamSound.Children[0] as StreamSoundTrackBase).FilePath) == false)
            {
                return NotAvailableText;
            }

            int channel = 1;
            foreach (StreamSoundTrackBase streamSoundTrack in streamSound.Children)
            {
                if (streamSoundTrack == targetStreamSoundTrack)
                {
                    string channelNumber = string.Empty;
                    for (int i = channel; i < channel + targetStreamSoundTrack.ChannelCount; i++)
                    {
                        if (channel < i)
                        {
                            channelNumber += ",";
                        }
                        channelNumber += i.ToString();
                    }

                    return channelNumber;
                }

                if (streamSoundTrack.IsEnabled == true)
                {
                    channel += streamSoundTrack.ChannelCount;
                }
            }

            Debug.Assert(false, "No contains StreamSoundTrackBase in StreamSoundBase children.");
            return NotAvailableText;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnValueChanged(object sender, EventArgs e)
        {
            ParameterEditor parameterEditor = sender as ParameterEditor;
            string name = parameterEditor.Name;

            Adapter.SetValue(GetTarget(name), name, parameterEditor.Value);
        }

        /// <summary>
        ///
        /// </summary>
        private StreamSoundTrackBase GetSingleStreamSoundTrack(Component component)
        {
            StreamSoundBase streamSound = null;

            if ((streamSound = component as StreamSoundBase) == null ||
               streamSound.Children.Count > 1)
            {
                return null;
            }

            return streamSound.Children[0] as StreamSoundTrackBase;
        }

        private bool HasMultiWaveFiles(Component component)
        {
            if (component is StreamSoundBase == true)
            {
                return component.Children.Count > 1;
            }

            return false;
        }

        ListItemToolTipInfo IListItemToolTipProvider.GetToolTip(string name, IListItem item)
        {
            var evaluator = (Adapter as CommonListAdapter)?.DecorationEvaluator;
            return evaluator?.GetToolTip(name, item);
        }
    }

    public class SampleRateParameterEditor : ParameterEditor
    {
        public SampleRateParameterEditor(string name, IParameterValue targetParameter) : base(name, targetParameter)
        {
        }

        public override ValidationResult ValidateValue(object value)
        {
            int ivalue;

            if (value is string == true &&
                int.TryParse(value as string, out ivalue) == true)
            {
                return base.ValidateValue(ivalue);
            }

            return base.ValidateValue(value);
        }

        public override object Value
        {
            get
            {
                return base.Value;
            }

            set
            {
                if (value is string == true)
                {
                    base.Value = this.ParseValue(value as string);
                }
                else
                {
                    base.Value = value;
                }
            }
        }
    }
}
