﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundFoundation.Windows.Forms;
using NintendoWare.SoundMaker.Framework.FileManagement;
using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Linq;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    /// <summary>
    /// <see cref="CommonListCtrl"/> に表示される状態アイコンの表示に影響するファイルの監視を行います。
    /// </summary>
    /// <seealso cref="CommonListDecorationEvaluator"/>
    public class CommonListFileWatcher : IDisposable
    {
        private HashSet<string> _watchingFilePaths;
        private bool _requestedUpdateFileWatch;
        private bool _requestedNotifyFileChanged;

        /// <summary>
        /// イベント処理に利用されるコントロールです。
        /// </summary>
        public Control Control { get; }

        public CommonListCtrl[] ListControls { get; }

        private FileWatcherService FileWatcherService => FormsApplication.Instance.FileWatcherService;

        /// <summary>
        /// リストに表示されている <see cref="Component"/> の状態に影響する
        /// ファイルが変更された場合に発行されます。
        /// </summary>
        public event EventHandler FileChanged;

        /// <summary>
        /// コンストラクタです。
        /// <para>
        /// <paramref name="listControls"/> で指定したリストコントロールに表示される <see cref="Component"/> のファイル監視を行います。
        /// <see cref="Start()"/> で監視を開始し、<see cref="Stop()"/> で終了します。
        /// </para>
        /// <para>
        /// ファイルの変更が検出されると <see cref="FileChanged"/> イベントが発行されます。
        /// イベントの発行には <paramref name="control"/> に対して <see cref="Control.BeginInvoke"/> が使われます。
        /// </para>
        /// </summary>
        /// <param name="control"></param>
        /// <param name="listControls"></param>
        public CommonListFileWatcher(Control control, CommonListCtrl[] listControls)
        {
            if (control == null)
            {
                throw new ArgumentNullException(nameof(control));
            }

            if (listControls == null)
            {
                throw new ArgumentNullException(nameof(listControls));
            }

            this.Control = control;
            this.ListControls = listControls.ToArray();

            foreach (var listControl in this.ListControls)
            {
                listControl.VerticalScrollValueChanged += VerticalScrollValueChanged;

                if (listControl.ItemsSource?.Items != null)
                {
                    listControl.ItemsSource.Items.CollectionChanged += CollectionChanged;
                }

                if (listControl.ItemsSource is CommonListAdapter)
                {
                    var listAdapter = (CommonListAdapter)listControl.ItemsSource;
                    listAdapter.OperationExecuted += OperationExecuted;
                }
            }
        }

        private void VerticalScrollValueChanged(object sender, EventArgs e)
        {
            this.RequestUpdateFileWatch();
        }

        private void CollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            this.RequestUpdateFileWatch();
        }

        private void OperationExecuted(object sender, OperationExecutedEventArgs e)
        {
            this.RequestUpdateFileWatch();
        }

        /// <summary>
        /// ファイル監視を終了します。
        /// </summary>
        public void Dispose()
        {
            Stop();
        }

        /// <summary>
        /// ファイルの監視を開始します。
        /// <para>
        /// イベントの送信等で <see cref="Control.BeginInvoke"/> を使用するため、
        /// <see cref="Control"/> が表示されてから呼び出してください。
        /// </para>
        /// </summary>
        public void Start()
        {
            if (_watchingFilePaths == null)
            {
                _watchingFilePaths = new HashSet<string>();

                if (_requestedUpdateFileWatch == true)
                {
                    this.UpdateFileWatch();
                }
            }
        }

        /// <summary>
        /// ファイル監視を終了します。
        /// </summary>
        public void Stop()
        {
            if (_watchingFilePaths != null)
            {
                foreach (var filePath in _watchingFilePaths)
                {
                    this.FileWatcherService.Remove(filePath, OnWatchingFileChanged, watchChangeOnly: false);
                }

                _watchingFilePaths.Clear();

                _watchingFilePaths = null;
            }
        }

        /// <summary>
        /// ファイル監視の設定を更新します（非同期）。
        /// </summary>
        private void RequestUpdateFileWatch()
        {
            if (_requestedUpdateFileWatch == false)
            {
                _requestedUpdateFileWatch = true;

                if (_watchingFilePaths != null)
                {
                    this.Control.BeginInvoke((Action)UpdateFileWatch);
                }
            }
        }

        /// <summary>
        /// ファイル監視の設定を更新します。
        /// </summary>
        private void UpdateFileWatch()
        {
            if (_watchingFilePaths == null)
            {
                return;
            }

            _requestedUpdateFileWatch = false;

            // 現在表示されているアイテムの状態アイコンに影響するファイルパスを収集します。
            var shownItems = new HashSet<string>(
                this.ListControls
                    .SelectMany(it => it.GetDisplayedItems())
                    .Cast<ComponentListItem>()
                    .SelectMany(it => EnumerateFilePathsToWatch(it.Target))
                    .Where(it => string.IsNullOrWhiteSpace(it) == false));

            // 表示されるようになったアイテムのファイル監視を登録します。
            shownItems.ForEach(it =>
            {
                if (_watchingFilePaths.Contains(it) == false)
                {
                    this.FileWatcherService.Add(it, OnWatchingFileChanged, watchChangeOnly: false);
                }
            });

            // 表示されなくなったアイテムのファイル監視を終了します。
            _watchingFilePaths.ForEach(it =>
            {
                if (shownItems.Contains(it) == false)
                {
                    this.FileWatcherService.Remove(it, OnWatchingFileChanged, watchChangeOnly: false);
                }
            });

            _watchingFilePaths = shownItems;
        }

        private void OnWatchingFileChanged(string filePath)
        {
            this.RequestNotifyFileChanged();
        }

        /// <summary>
        /// <see cref="FileChanged"/> イベントを送信します（非同期）。
        /// </summary>
        private void RequestNotifyFileChanged()
        {
            if (_requestedNotifyFileChanged == false)
            {
                _requestedNotifyFileChanged = true;
                this.Control.BeginInvoke((Action)NotifyFileChanged);
            }
        }

        /// <summary>
        /// <see cref="FileChanged"/> イベントを送信します。
        /// </summary>
        private void NotifyFileChanged()
        {
            _requestedNotifyFileChanged = false;
            if (_watchingFilePaths != null)
            {
                this.FileChanged?.Invoke(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// 状態アイコンの表示に影響するファイルパスを列挙します。
        /// <para>
        /// 要素として null または空白文字列を返した場合は無視されます。
        /// </para>
        /// </summary>
        /// <param name="item"></param>
        /// <returns></returns>
        protected virtual IEnumerable<string> EnumerateFilePathsToWatch(Component component)
        {
            if (component is StreamSoundBase)
            {
                foreach (var track in component.Children.OfType<StreamSoundTrackBase>())
                {
                    yield return track.FilePath;
                }

                yield break;
            }

            if (component is Sound)
            {
                yield return component.GetFilePath();

                yield break;
            }

            if (component is SoundSetBankBase)
            {
                yield return component.GetFilePath();

                yield break;
            }

            if (component is Instrument)
            {
                foreach (var key in component.Children.OfType<KeyRegion>())
                {
                    foreach (VelocityRegion vel in key.Children)
                    {
                        yield return vel.FilePath;
                    }
                }

                yield break;
            }

            if (component is KeyRegion)
            {
                foreach (var vel in component.Children.OfType<VelocityRegion>())
                {
                    yield return vel.FilePath;
                }

                yield break;
            }

            if (component is VelocityRegion)
            {
                var vel = (VelocityRegion)component;

                yield return vel.FilePath;

                yield break;
            }
        }
    }
}
