﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundFoundation.Windows.Forms;
using NintendoWare.SoundMaker.Framework.FileFormats;
using NintendoWare.SoundMaker.Framework.Projects;
using NintendoWare.SoundMaker.Framework.Resources;
using NintendoWare.SoundMaker.Framework.Utilities;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    /// <summary>
    /// 無効アイコンを表示するかどうか、および無効化の理由を取得するデリゲートです。
    /// <para>
    /// string を返すと無効アイコンが表示されます。
    /// 空文字列でない場合はツールチップに無効化の理由が表示されます。
    /// </para>
    /// </summary>
    /// <param name="component"></param>
    /// <returns>無効化の理由を説明する文字列、空文字列、または null を返します。</returns>
    public delegate string DoShowDisabledIconDelegate(Component component);

    /// <summary>
    /// エラーアイコンを表示するかどうか、およびエラーの内容を取得するデリゲートです。
    /// <para>
    /// string を返すとエラーアイコンが表示されます。
    /// 空文字列でない場合はツールチップにエラーの内容が表示されます。
    /// </para>
    /// </summary>
    /// <param name="component"></param>
    /// <returns>エラーを説明する文字列、空文字列、または null を返します。</returns>
    public delegate string DoShowErrorIconDelegate(Component component);

    /// <summary>
    /// ワーニングアイコンを表示するかどうか、およびワーニングの内容を取得するデリゲートです。
    /// <para>
    /// string を返すとワーニングアイコンが表示されます。
    /// 空文字列でない場合はツールチップにワーニングの内容が表示されます。
    /// </para>
    /// </summary>
    /// <param name="component"></param>
    /// <returns>ワーニングを説明する文字列、空文字列、または null を返します。</returns>
    public delegate string DoShowWarningIconDelegate(Component component);

    /// <summary>
    /// ツールチップに表示したい内容を取得するデリゲートです。
    /// </summary>
    /// <param name="name"></param>
    /// <param name="listItem"></param>
    /// <returns>ツールチップに表示する文字列、または null を返します。</returns>
    public delegate string DoShowToolTipDelegate(string name, IListItem listItem);

    /// <summary>
    /// セルにアイコンを表示するかの情報を提供します。
    /// </summary>
    public class CommonListDecorationEvaluator : IListItemToolTipProvider
    {
        public const string RowHeaderName = "RowHeader";

        /// <summary>
        /// 情報の提供を有効にするかを指定および取得します。
        /// <para>
        /// デフォルト値は true です。false に指定した場合、全てのタイプのアイコンおよびツールチップが表示されなくなります。
        /// </para>
        /// </summary>
        public bool IsEnabled { get; set; } = true;

        private class DelegateDictionary<T> : Dictionary<string, List<T>>
        {
            public new List<T> this[string name]
            {
                get
                {
                    List<T> list;
                    if (!this.TryGetValue(name, out list))
                    {
                        list = new List<T>();
                        this.Add(name, list);
                    }
                    return list;
                }
            }
        }

        private readonly List<DoShowDisabledIconDelegate> _doShowHeaderDisabledIconDelegates =
            new List<DoShowDisabledIconDelegate>();

        private readonly List<DoShowErrorIconDelegate> _doShowHeaderErrorIconDelegates =
            new List<DoShowErrorIconDelegate>();

        private readonly List<DoShowWarningIconDelegate> _doShowHeaderWarningIconDelegates =
            new List<DoShowWarningIconDelegate>();

        private readonly DelegateDictionary<DoShowDisabledIconDelegate> _doShowDisabledIconDelegates =
            new DelegateDictionary<DoShowDisabledIconDelegate>();

        private readonly DelegateDictionary<DoShowErrorIconDelegate> _doShowErrorIconDelegates =
            new DelegateDictionary<DoShowErrorIconDelegate>();

        private readonly DelegateDictionary<DoShowWarningIconDelegate> _doShowWarningIconDelegates =
            new DelegateDictionary<DoShowWarningIconDelegate>();

        private readonly List<DoShowToolTipDelegate> _doShowToolTipDelegates =
            new List<DoShowToolTipDelegate>();

        public CommonListDecorationEvaluator()
        {
            AddDoShowDisabledIconDelegate(null, Delegates.ShowDisabledIconTargetDisabled, true);
            AddDoShowDisabledIconDelegate(ProjectParameterNames.TargetItemReference, Delegates.ShowDisabledIconGroupItemDisabled);
            AddDoShowDisabledIconDelegate(ProjectParameterNames.Sound.PlayerReference, Delegates.ShowDisabledIconPlayerDisabled);
            AddDoShowDisabledIconDelegate(ProjectParameterNames.SequenceSound.SoundSetBankReference0, c => Delegates.ShowDisabledIconBankDisabled(c, 0));
            AddDoShowDisabledIconDelegate(ProjectParameterNames.SequenceSound.SoundSetBankReference1, c => Delegates.ShowDisabledIconBankDisabled(c, 1));
            AddDoShowDisabledIconDelegate(ProjectParameterNames.SequenceSound.SoundSetBankReference2, c => Delegates.ShowDisabledIconBankDisabled(c, 2));
            AddDoShowDisabledIconDelegate(ProjectParameterNames.SequenceSound.SoundSetBankReference3, c => Delegates.ShowDisabledIconBankDisabled(c, 3));
            AddDoShowDisabledIconDelegate(ProjectParameterNames.SoundSetItem.WaveArchiveReference, Delegates.ShowDisabledIconWaveArchiveDisabled);

            AddDoShowErrorIconDelegate(null, Delegates.ShowErrorIconNotLinkedAllSoundSetBank);
            AddDoShowErrorIconDelegate(null, Delegates.ShowErrorIconStreamSoundTrackNumberExceedLimit);
            AddDoShowErrorIconDelegate(ProjectParameterNames.Name, Delegates.ShowErrorIconDuplicateName);
            AddDoShowErrorIconDelegate(ProjectParameterNames.TargetItemReference, Delegates.ShowErrorIconDuplicateName);
            AddDoShowErrorIconDelegate(ProjectParameterNames.Sound.PlayerReference, Delegates.ShowErrorIconInvalidPlayer);
            AddDoShowErrorIconDelegate(ProjectParameterNames.SequenceSound.SoundSetBankReference0, Delegates.ShowErrorIconMissingBank0);
            AddDoShowErrorIconDelegate(ProjectParameterNames.SequenceSound.SoundSetBankReference1, Delegates.ShowErrorIconMissingBank1);
            AddDoShowErrorIconDelegate(ProjectParameterNames.SequenceSound.SoundSetBankReference2, Delegates.ShowErrorIconMissingBank2);
            AddDoShowErrorIconDelegate(ProjectParameterNames.SequenceSound.SoundSetBankReference3, Delegates.ShowErrorIconMissingBank3);
            AddDoShowErrorIconDelegate(ProjectParameterNames.SoundSetItem.WaveArchiveReference, Delegates.ShowErrorIconInvalidWaveArchive);
            AddDoShowErrorIconDelegate(ProjectParameterNames.GroupItem.RegisterType, Delegates.ShowErrorIconInvalidGroupItemRegisterType);
            AddDoShowErrorIconDelegate(ProjectParameterNames.TargetItemReference, Delegates.ShowErrorIconInvalidGroupItem);
            AddDoShowErrorIconDelegate(ProjectParameterNames.FilePath, Delegates.ShowErrorIconInvalidFilePath);
            AddDoShowErrorIconDelegate(ProjectParameterNames.FilePath, Delegates.ShowErrorIconInvalidWaveFile);

            AddDoShowToolTipDelegate(Delegates.ShowToolTipConstValue);
        }

        /// <summary>
        /// 無効アイコンの表示情報を提供するデリゲートを登録します。
        /// <para>
        /// <paramref name="name"/> に null を指定した場合は、セルとの関連付けはされません。
        /// その場合でも <paramref name="showInHeader"/> に true が指定されているときはヘッダでは使用されます。
        /// </para>
        /// </summary>
        /// <param name="name"></param>
        /// <param name="f"></param>
        /// <param name="showInHeader"></param>
        public void AddDoShowDisabledIconDelegate(string name, DoShowDisabledIconDelegate f, bool showInHeader = false)
        {
            if (showInHeader && !_doShowHeaderDisabledIconDelegates.Contains(f))
            {
                _doShowHeaderDisabledIconDelegates.Add(f);
            }

            if (name != null)
            {
                _doShowDisabledIconDelegates[name].Add(f);
            }
        }

        /// <summary>
        /// エラーアイコンの表示情報を提供するデリゲートを登録します。
        /// <para>
        /// <paramref name="name"/> に null を指定した場合は、セルとの関連付けはされません。
        /// その場合でも <paramref name="showInHeader"/> に true が指定されているときはヘッダでは使用されます。
        /// </para>
        /// </summary>
        /// <param name="name"></param>
        /// <param name="f"></param>
        /// <param name="showInHeader"></param>
        public void AddDoShowErrorIconDelegate(string name, DoShowErrorIconDelegate f, bool showInHeader = true)
        {
            if (showInHeader && !_doShowHeaderErrorIconDelegates.Contains(f))
            {
                _doShowHeaderErrorIconDelegates.Add(f);
            }

            if (name != null)
            {
                _doShowErrorIconDelegates[name].Add(f);
            }
        }

        /// <summary>
        /// ワーニングアイコンの表示情報を提供するデリゲートを登録します。
        /// <para>
        /// <paramref name="name"/> に null を指定した場合は、セルとの関連付けはされません。
        /// その場合でも <paramref name="showInHeader"/> に true が指定されているときはヘッダでは使用されます。
        /// </para>
        /// </summary>
        /// <param name="name"></param>
        /// <param name="f"></param>
        /// <param name="showInHeader"></param>
        public void AddDoShowWarningIconDelegate(string name, DoShowWarningIconDelegate f, bool showInHeader = true)
        {
            if (showInHeader && !_doShowHeaderWarningIconDelegates.Contains(f))
            {
                _doShowHeaderWarningIconDelegates.Add(f);
            }

            if (name != null)
            {
                _doShowWarningIconDelegates[name].Add(f);
            }
        }

        /// <summary>
        /// ツールチップの表示情報を提供するデリゲートを登録します。
        /// </summary>
        public void AddDoShowToolTipDelegate(DoShowToolTipDelegate f)
        {
            _doShowToolTipDelegates.Add(f);
        }

        /// <summary>
        /// <see cref="CommonListDecorationEvaluator"/> を取得します。
        /// </summary>
        /// <param name="listItem"></param>
        /// <returns></returns>
        public static CommonListDecorationEvaluator GetEvaluator(IListItem listItem)
        {
            return ((listItem as CommonListItem)?.Adapter as CommonListAdapter)?.DecorationEvaluator;
        }

        /// <summary>
        /// 無効アイコンを表示するかを判定します。
        /// <para>
        /// <paramref name="name"/> に null を指定した場合はヘッダー用に評価します。
        /// </para>
        /// </summary>
        /// <param name="name"></param>
        /// <param name="listItem"></param>
        /// <returns></returns>
        public bool DoShowDisabledIcon(string name, IListItem listItem)
        {
            if (this.IsEnabled == false)
            {
                return false;
            }

            return EnumerateDisabledReasons(name, listItem).Any();
        }

        /// <summary>
        /// エラーアイコンを表示するかを判定します。
        /// <para>
        /// <paramref name="name"/> に null を指定した場合はヘッダー用に評価します。
        /// </para>
        /// </summary>
        /// <param name="name"></param>
        /// <param name="listItem"></param>
        /// <returns></returns>
        public bool DoShowErrorIcon(string name, IListItem listItem)
        {
            if (this.IsEnabled == false)
            {
                return false;
            }

            return EnumerateErrors(name, listItem).Any();
        }

        /// <summary>
        /// ワーニングアイコンを表示するかを判定します。
        /// <para>
        /// <paramref name="name"/> に null を指定した場合はヘッダー用に評価します。
        /// </para>
        /// </summary>
        /// <param name="name"></param>
        /// <param name="listItem"></param>
        /// <returns></returns>
        public bool DoShowWarningIcon(string name, IListItem listItem)
        {
            if (this.IsEnabled == false)
            {
                return false;
            }

            return EnumerateWarnings(name, listItem).Any();
        }

        /// <summary>
        /// ツールチップの表示内容を取得します。
        /// <para>
        /// <paramref name="name"/> に null を指定した場合はヘッダー用に評価します。
        /// </para>
        /// </summary>
        /// <param name="name"></param>
        /// <param name="item"></param>
        /// <returns></returns>
        public ListItemToolTipInfo GetToolTip(string name, IListItem item)
        {
            if (this.IsEnabled == false)
            {
                return null;
            }

            string message;

            message = EnumerateDisabledReasons(name, item).FirstOrDefault();
            if (message != null)
            {
                if (!string.IsNullOrEmpty(message))
                {
                    return new ListItemToolTipInfo(message, ListItemToolTipType.Disabled);
                }
                return null;
            }

            message = EnumerateErrors(name, item).FirstOrDefault();
            if (message != null)
            {
                if (!string.IsNullOrEmpty(message))
                {
                    return new ListItemToolTipInfo(message, ListItemToolTipType.Error);
                }
                return null;
            }

            message = EnumerateWarnings(name, item).FirstOrDefault();
            if (message != null)
            {
                if (!string.IsNullOrEmpty(message))
                {
                    return new ListItemToolTipInfo(message, ListItemToolTipType.Warning);
                }
            }

            if (name != null)
            {
                message = EnumerateToolTips(name, item).FirstOrDefault();
                if (message != null)
                {
                    if (!string.IsNullOrEmpty(message))
                    {
                        return new ListItemToolTipInfo(message, ListItemToolTipType.Normal);
                    }
                }
            }

            return null;
        }
        private IEnumerable<string> EnumerateDisabledReasons(string name, IListItem listItem)
        {
            var component = (listItem as CommonListItem)?.Target;
            if (component == null)
            {
                yield break;
            }

            var delegates = (name == null) ? _doShowHeaderDisabledIconDelegates : _doShowDisabledIconDelegates[name];
            foreach (var func in delegates)
            {
                var message = func(component);
                if (message != null)
                {
                    yield return message;
                }
            }
        }

        private IEnumerable<string> EnumerateErrors(string name, IListItem listItem)
        {
            var component = (listItem as CommonListItem)?.Target;
            if (component == null)
            {
                yield break;
            }

            var delegates = (name == null) ? _doShowHeaderErrorIconDelegates : _doShowErrorIconDelegates[name];
            foreach (var func in delegates)
            {
                var message = func(component);
                if (message != null)
                {
                    yield return message;
                }
            }
        }

        private IEnumerable<string> EnumerateWarnings(string name, IListItem listItem)
        {
            var component = (listItem as CommonListItem)?.Target;
            if (component == null)
            {
                yield break;
            }

            var delegates = (name == null) ? _doShowHeaderWarningIconDelegates : _doShowWarningIconDelegates[name];
            foreach (var func in delegates)
            {
                var message = func(component);
                if (message != null)
                {
                    yield return message;
                }
            }
        }

        private IEnumerable<string> EnumerateToolTips(string name, IListItem listItem)
        {
            if (listItem == null)
            {
                yield break;
            }

            foreach (var func in _doShowToolTipDelegates)
            {
                var message = func(name, listItem);
                if (!string.IsNullOrEmpty(message))
                {
                    yield return message;
                }
            }
        }

        public static class Delegates
        {
            public static string ShowDisabledIconTargetDisabled(Component component)
            {
                if (component?.IsEnabled == false)
                {
                    return MessageResource.Message_ItemDisabled;
                }
                return null;
            }

            public static string ShowDisabledIconGroupItemDisabled(Component component)
            {
                GroupItemBase groupItem = component as GroupItemBase;

                if (groupItem != null && groupItem.Target != null)
                {
                    if (groupItem.Target.IsEnabled == false)
                    {
                        return string.Empty;
                    }
                }

                return null;
            }

            public static string ShowDisabledIconPlayerDisabled(Component component)
            {
                Sound sound = component as Sound;

                // ストリームサウンドトラックの場合は、親のストリームサウンドの情報を表示します。
                if (component is StreamSoundTrackBase)
                {
                    sound = component.Parent as Sound;
                }

                if (sound != null && sound.TargetPlayer != null)
                {
                    if (sound.TargetPlayer.IsEnabled == false)
                    {
                        return string.Empty;
                    }
                }

                return null;
            }

            public static string ShowDisabledIconBankDisabled(Component component, int bankIndex)
            {
                SequenceSoundBase sequenceSound = component as SequenceSoundBase;

                if (sequenceSound != null)
                {
                    var soundSetBank = sequenceSound.SoundSetBanks.ElementAtOrDefault(bankIndex);
                    if (soundSetBank != null && soundSetBank.IsEnabled == false)
                    {
                        return string.Empty;
                    }
                }

                return null;
            }

            public static string ShowDisabledIconWaveArchiveDisabled(Component component)
            {
                WaveSoundSetBase waveSoundSet = component as WaveSoundSetBase;
                SoundSetBankBase soundSetBank = component as SoundSetBankBase;

                if (waveSoundSet != null)
                {
                    if (waveSoundSet.TargetWaveArchive != null)
                    {
                        if (waveSoundSet.TargetWaveArchive.IsEnabled == false)
                        {
                            return string.Empty;
                        }
                    }
                    else
                    {
                        if (waveSoundSet.LinkedWaveArchive == false)
                        {
                            return string.Empty;
                        }
                    }
                }
                else if (soundSetBank != null)
                {
                    if (soundSetBank.TargetWaveArchive != null)
                    {
                        if (soundSetBank.TargetWaveArchive.IsEnabled == false)
                        {
                            return string.Empty;
                        }
                    }
                    else
                    {
                        if (soundSetBank.LinkedWaveArchive == false)
                        {
                            return string.Empty;
                        }
                    }
                }

                return null;
            }

            /// <summary>
            /// プレイヤーへの参照を確認します。
            /// </summary>
            /// <param name="component"></param>
            /// <returns>不正な場合はエラーメッセージを返します。</returns>
            public static string ShowErrorIconInvalidPlayer(Component component)
            {
                var sound = component as Sound;
                if (sound != null)
                {
                    if (sound.LinkedPlayer == false)
                    {
                        // プレイヤーが不正です。
                        return MessageResource.ErrorMessage_PlayerNotLinked;
                    }

                    if (AddonSoundArchiveUtility.IsItemInAddonSoundArchive(sound.TargetPlayer) == true)
                    {
                        // 追加サウンドアーカイブ内のプレイヤーは参照できません。
                        return MessageResource.ErrorMessage_LinkToPlayerInAddonSoundArchive;
                    }
                }
                return null;
            }

            /// <summary>
            /// ストリームサウンドのトラック数が範囲内か確認します。
            /// </summary>
            /// <param name="component"></param>
            /// <returns>不正な場合はエラーメッセージを返します。</returns>
            public static string ShowErrorIconStreamSoundTrackNumberExceedLimit(Component component)
            {
                var streamSoundTrack = component as StreamSoundTrackBase;
                if (streamSoundTrack != null &&
                    streamSoundTrack.Parent != null &&
                    ComponentConfiguration.Instance.StreamSoundTrackNumberMaximum <=
                    streamSoundTrack.Parent.Children.IndexOf(streamSoundTrack))
                {
                    // トラック数が上限を超えています。
                    return MessageResource.ErrorMessage_StreamSoundTrackNumberExceedLimit;
                }
                return null;
            }

            /// <summary>
            /// シーケンスサウンドのすべてのサウンドセットバンクが正しくリンクされていることを確認します。
            /// </summary>
            /// <param name="component"></param>
            /// <returns>不正な場合はエラーメッセージを返します。</returns>
            public static string ShowErrorIconNotLinkedAllSoundSetBank(Component component)
            {
                var sequenceSound = component as SequenceSoundBase;
                if (sequenceSound != null)
                {
                    for (int index = 0; index < sequenceSound.SoundSetBanks.Count; index++)
                    {
                        var message = ShowErrorIconMissingBank(sequenceSound, index);
                        if (message != null)
                        {
                            return message;
                        }
                    }
                }
                return null;
            }

            /// <summary>
            /// サウンドセットバンクが正しくリンクされていることを確認します。
            /// </summary>
            /// <param name="component"></param>
            /// <param name="index"></param>
            /// <returns>不正な場合はエラーメッセージを返します。</returns>
            public static string ShowErrorIconMissingBank(Component component, int index)
            {
                var sequenceSound = component as SequenceSoundBase;
                if (sequenceSound == null)
                {
                    return null;
                }

                var soundSetBank = sequenceSound.SoundSetBanks[index];
                if (soundSetBank != null)
                {
                    switch (AddonSoundArchiveUtility.GetSoundArchiveOutputType(sequenceSound.SoundSet))
                    {
                        case SoundArchiveOutputTypes.None:
                            // 出力しないときには、サウンドセット間の参照の正当性は判断できないのでチェックを行いません。
                            break;

                        case SoundArchiveOutputTypes.SoundArchive:
                        case SoundArchiveOutputTypes.AddonSoundArchive:
                        default:
                            if (soundSetBank.SoundSet != sequenceSound.SoundSet &&
                                AddonSoundArchiveUtility.GetSoundArchiveOutputType(soundSetBank.SoundSet) != SoundArchiveOutputTypes.SoundArchive)
                            {
                                // 自身のサウンドアーカイブに含まれているバンクか、メインサウンドアーカイブのバンクしか参照できません。
                                return MessageResource.ErrorMessage_LinkToBankInInvalidSoundArchive;
                            }
                            break;
                    }
                }

                if (!sequenceSound.IsLinkedSoundSetBank(index))
                {
                    // バングが不正です。
                    return MessageResource.ErrorMessage_InvalidBankReference;
                }

                return null;
            }

            /// <summary>
            /// サウンドセットバンク 0 が正しくリンクされていることを確認します。
            /// </summary>
            /// <param name="component"></param>
            /// <param name="index"></param>
            /// <returns>不正な場合はエラーメッセージを返します。</returns>
            public static string ShowErrorIconMissingBank0(Component component)
            {
                return ShowErrorIconMissingBank(component, 0);
            }

            /// <summary>
            /// サウンドセットバンク 1 が正しくリンクされていることを確認します。
            /// </summary>
            /// <param name="component"></param>
            /// <param name="index"></param>
            /// <returns>不正な場合はエラーメッセージを返します。</returns>
            public static string ShowErrorIconMissingBank1(Component component)
            {
                return ShowErrorIconMissingBank(component, 1);
            }

            /// <summary>
            /// サウンドセットバンク 2 が正しくリンクされていることを確認します。
            /// </summary>
            /// <param name="component"></param>
            /// <param name="index"></param>
            /// <returns>不正な場合はエラーメッセージを返します。</returns>
            public static string ShowErrorIconMissingBank2(Component component)
            {
                return ShowErrorIconMissingBank(component, 2);
            }

            /// <summary>
            /// サウンドセットバンク 3 が正しくリンクされていることを確認します。
            /// </summary>
            /// <param name="component"></param>
            /// <param name="index"></param>
            /// <returns>不正な場合はエラーメッセージを返します。</returns>
            public static string ShowErrorIconMissingBank3(Component component)
            {
                return ShowErrorIconMissingBank(component, 3);
            }

            /// <summary>
            /// 波形アーカイブへのリンクが正常か確認します。
            /// </summary>
            /// <param name="component"></param>
            /// <returns>不正な場合はエラーメッセージを返します。</returns>
            public static string ShowErrorIconInvalidWaveArchive(Component component)
            {
                if (component is WaveSoundSetBase)
                {
                    var waveSoundSet = component as WaveSoundSetBase;

                    var waveArchive = waveSoundSet.TargetWaveArchive;

                    if (AddonSoundArchiveUtility.IsItemInAddonSoundArchive(waveSoundSet) == true)
                    {
                        if (waveArchive != null &&
                            waveSoundSet.SoundSet != waveArchive.SoundSet)
                        {
                            // 追加サウンドアーカイブ内から他のサウンドアーカイブの波形アーカイブは参照できません。
                            return MessageResource.ErrorMessage_LinkToWaveArchiveInOtherSoundArchive;
                        }
                    }
                    else
                    {
                        if (AddonSoundArchiveUtility.IsItemInAddonSoundArchive(waveArchive) == true)
                        {
                            // メインサウンドアーカイブ内から追加サウンドアーカイブの波形アーカイブは参照できません。
                            return MessageResource.ErrorMessage_LinkToWaveArchiveInAddonSoundArchive;
                        }
                    }

                    if (waveSoundSet.LinkedWaveArchive == false)
                    {
                        // 波形アーカイブのリンクが不正です。
                        return MessageResource.ErrorMessage_WaveArchiveNotLinked;
                    }
                }
                else if (component is SoundSetBankBase)
                {
                    var soundSetBank = component as SoundSetBankBase;

                    var waveArchive = soundSetBank.TargetWaveArchive;

                    if (AddonSoundArchiveUtility.IsItemInAddonSoundArchive(soundSetBank) == true)
                    {
                        if (waveArchive != null &&
                            soundSetBank.SoundSet != waveArchive.SoundSet)
                        {
                            // 追加サウンドアーカイブ内から他のサウンドアーカイブの波形アーカイブは参照できません。
                            return MessageResource.ErrorMessage_LinkToWaveArchiveInOtherSoundArchive;
                        }
                    }
                    else
                    {
                        if (AddonSoundArchiveUtility.IsItemInAddonSoundArchive(waveArchive) == true)
                        {
                            // メインサウンドアーカイブ内から追加サウンドアーカイブの波形アーカイブは参照できません。
                            return MessageResource.ErrorMessage_LinkToWaveArchiveInAddonSoundArchive;
                        }
                    }

                    if (soundSetBank.LinkedWaveArchive == false)
                    {
                        // 波形アーカイブのリンクが不正です。
                        return MessageResource.ErrorMessage_WaveArchiveNotLinked;
                    }
                }

                return null;
            }

            /// <summary>
            /// グループアイテムの参照が適正か確認します。
            /// </summary>
            /// <param name="component"></param>
            /// <returns>不正な場合はエラーメッセージを返します。</returns>
            public static string ShowErrorIconInvalidGroupItem(Component component)
            {
                var groupItem = component as GroupItemBase;
                if (groupItem != null)
                {
                    if (groupItem.Target == null)
                    {
                        // アイテムの参照が不正です。
                        return MessageResource.ErrorMessage_InvalidGroupItem;
                    }
                }
                return null;
            }

            /// <summary>
            /// グループの関連アイテムの登録が適正か確認します。
            /// </summary>
            /// <param name="component"></param>
            /// <returns>不正な場合はエラーメッセージを返します。</returns>
            public static string ShowErrorIconInvalidGroupItemRegisterType(Component component)
            {
                var groupItem = component as GroupItemBase;
                if (groupItem != null)
                {
                    if (groupItem.Target == null)
                    {
                        // アイテムの参照が不正です。
                        return MessageResource.ErrorMessage_InvalidGroupItem;
                    }
                    else if (GroupItemRegisterTypeValidator.Validate(groupItem).IsValid == false)
                    {
                        // グループの関連アイテムの登録が不正です。
                        return MessageResource.ErrorMessage_InvalidGroupItemRegisterType;
                    }
                }
                return null;
            }

            /// <summary>
            /// 不正な FilePath か確認します。
            /// </summary>
            /// <param name="component"></param>
            /// <returns>不正な場合はエラーメッセージを返します。</returns>
            public static string ShowErrorIconInvalidFilePath(Component component)
            {
                if (component is StreamSoundBase)
                {
                    var tracks = component.Children.OfType<StreamSoundTrackBase>();

                    foreach (var track in tracks)
                    {
                        var result = ShowErrorIconInvalidFilePath_FilePath(track.FilePath);
                        if (result != null)
                        {
                            return result;
                        }
                    }

                    bool isMultiTrack = tracks.Take(2).Count() > 1;
                    if (isMultiTrack && tracks.Any(t => FileUtil.IsOpusFile(t.FilePath)))
                    {
                        // 複数トラックで Opus ファイルはサポートされません。
                        return MessageResource.ErrorMessage_StreamSoundOpusInMultiTrackNotSupported;
                    }

                    return null;
                }

                if (component is StreamSoundTrackBase)
                {
                    var track = (StreamSoundTrackBase)component;

                    var result = ShowErrorIconInvalidFilePath_FilePath(track.FilePath);
                    if (result != null)
                    {
                        return result;
                    }

                    if (track.Parent is StreamSoundBase)
                    {
                        bool isMultiTrack = track.Parent.Children.OfType<StreamSoundTrackBase>().Take(2).Count() > 1;
                        if (isMultiTrack && FileUtil.IsOpusFile(track.FilePath))
                        {
                            // 複数トラックで Opus ファイルはサポートされません。
                            return MessageResource.ErrorMessage_StreamSoundOpusInMultiTrackNotSupported;
                        }
                    }

                    return null;
                }

                if (component is Sound)
                {
                    var sound = (Sound)component;

                    return ShowErrorIconInvalidFilePath_FilePath(sound.FilePath);
                }

                if (component is Instrument)
                {
                    var instrument = (Instrument)component;

                    foreach (var key in instrument.Children.OfType<KeyRegion>())
                    {
                        var result = ShowErrorIconInvalidFilePath_KeyRegion(key);
                        if (result != null)
                        {
                            return result;
                        }
                    }

                    if (instrument.LinkedRegion == false)
                    {
                        // リージョンのファイル設定が不正です。
                        return MessageResource.ErrorMessage_RegionNotLinked;
                    }

                    return null;
                }

                if (component is KeyRegion)
                {
                    return ShowErrorIconInvalidFilePath_KeyRegion((KeyRegion)component);
                }

                if (component is VelocityRegion)
                {
                    return ShowErrorIconInvalidFilePath_VelocityRegion((VelocityRegion)component);
                }

                if (component is SoundSetBankBase)
                {
                    var bank = (SoundSetBankBase)component;

                    return ShowErrorIconInvalidFilePath_FilePath(bank.FilePath);
                }

                return null;
            }

            private static string ShowErrorIconInvalidFilePath_FilePath(string filePath)
            {
                if (string.IsNullOrWhiteSpace(filePath) == true)
                {
                    // パスが不正です
                    return MessageResource.ErrorMessage_InvalidFilePath;
                }

                if (File.Exists(filePath) == false)
                {
                    // ファイルが存在しません。
                    return MessageResource.Message_FileNotFound;
                }

                if (AppConfiguration.EnabledAAC == false && AACUtil.IsAACFilePrimitive(filePath) == true)
                {
                    // AAC 形式の波形ファイルはサポートしていません。
                    return MessageResource.Message_UnsupportedAACFile;
                }

                return null;
            }

            private static string ShowErrorIconInvalidFilePath_KeyRegion(KeyRegion component)
            {
                foreach (var vel in component.Children.OfType<VelocityRegion>())
                {
                    var result = ShowErrorIconInvalidFilePath_VelocityRegion(vel);
                    if (result != null)
                    {
                        return result;
                    }
                }

                return null;
            }

            private static string ShowErrorIconInvalidFilePath_VelocityRegion(VelocityRegion component)
            {
                return ShowErrorIconInvalidFilePath_FilePath(component.FilePath);
            }

            /// <summary>
            /// 不正な WaveFile か確認します。
            /// </summary>
            /// <param name="component"></param>
            /// <returns>不正な場合はエラーメッセージを返します。</returns>
            public static string ShowErrorIconInvalidWaveFile(Component component)
            {
                component.UpdateWaveFile(force: false);
                var message = component.GetUpdateWaveFileErrorMessage();
                if (string.IsNullOrEmpty(message) == false)
                {
                    return message;
                }
                return null;
            }

            /// <summary>
            /// 名前が重複しているか確認します。
            /// </summary>
            /// <param name="component"></param>
            /// <returns>不正な場合はエラーメッセージを返します。</returns>
            public static string ShowErrorIconDuplicateName(Component component)
            {
                if (DuplicationNameChecker.Inquire(component))
                {
                    // 名前が重複しています。
                    return MessageResource.ErrorMessage_DuplicateName;
                }
                return null;
            }

            /// <summary>
            /// ツールチップとして <see cref="IListItem.GetConstValue"/> を表示します。
            /// <para>
            /// 対象名:
            /// <list type="bullet">
            /// <item><see cref="ProjectParameterNames.Name"/></item>
            /// <item><see cref="ProjectParameterNames.FilePath"/></item>
            /// </list>
            /// </para>
            /// </summary>
            /// <param name="name"></param>
            /// <param name="listItem"></param>
            /// <returns>不正な場合はエラーメッセージを返します。</returns>
            public static string ShowToolTipConstValue(string name, IListItem listItem)
            {
                switch (name)
                {
                    case ProjectParameterNames.Name:
                    case ProjectParameterNames.FilePath:
                        return listItem.GetConstValue(name)?.ToString();
                }

                return null;
            }
        }
    }
}
