﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Drawing;
    using System.Drawing.Imaging;
    using System.IO;
    using System.Linq;
    using System.Security.Cryptography;
    using System.Windows.Forms;
    using System.Windows.Forms.VisualStyles;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate;
    using NintendoWare.SoundFoundation.FileFormats.Wave;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.FileFormats;
    using NintendoWare.SoundMaker.Framework.Projects;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Utilities;
    using NintendoWare.ToolDevelopmentKit.Collections;

    /// <summary>
    ///
    /// </summary>
    public interface IAdapterManipulator
    {
        void BeginTransaction();
        void EndTransaction();
        void OperationExecuted(Operation operation);
        void Set(Component targetComponent, string name, object value);
    }

    // 処理を実行した場合にはtrueを返す必要があります。
    public delegate bool SetValueHandler(IAdapterManipulator adapter, Component targetComponent, string name, object value);

    /// <summary>
    /// 共用リストアダプタ
    /// </summary>
    public class CommonListAdapter : ComponentListAdapter, IAdapterManipulator
    {
        private Dictionary<string, SetValueHandler> setterDictionary = new Dictionary<string, SetValueHandler>();

        /// <summary>
        ///
        /// </summary>
        public delegate ComponentListItem ListItemCreateHandler(Component component);
        public ListItemCreateHandler ListItemCreator
        {
            get;
            set;
        }

        /// <summary>
        /// セルの装飾に関する情報を提供します。
        /// </summary>
        public CommonListDecorationEvaluator DecorationEvaluator { get; set; }

        /// <summary>
        ///
        /// </summary>
        public event EventHandler PreviewPlay;
        public event EventHandler PreviewMute;
        public event EventHandler PreviewSoloPlay;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CommonListAdapter(CommonListDecorationEvaluator decorationEvaluator)
        {
            DecorationEvaluator = decorationEvaluator;

            // 参照バンクのSetterを設定します。
            Setters.Add(ProjectParameterNames.SequenceSound.SoundSetBankReference0,
                         BankReferenceValueSetter);
            Setters.Add(ProjectParameterNames.SequenceSound.SoundSetBankReference1,
                         BankReferenceValueSetter);
            Setters.Add(ProjectParameterNames.SequenceSound.SoundSetBankReference2,
                         BankReferenceValueSetter);
            Setters.Add(ProjectParameterNames.SequenceSound.SoundSetBankReference3,
                         BankReferenceValueSetter);

            //
            Setters.Add(ProjectParameterNames.WaveArchive.LoadType,
                         delegate (IAdapterManipulator adapter, Component targetComponent, string name, object value)
                             {
                                 WaveArchiveLoadType type = WaveArchiveLoadType.Individual;
                                 if ((bool)value == false)
                                 {
                                     type = WaveArchiveLoadType.Whole;
                                 }
                                 adapter.Set(targetComponent, name, type);
                                 return true;
                             });

            //
            Setters.Add(ProjectParameterNames.SoundSetItem.WaveArchiveReference,
                         delegate (IAdapterManipulator adapter, Component targetComponent, string name, object value)
                             {
                                 string label = (string)value;
                                 if ((string)value == MessageResource.Label_WaveArchiveReference_Shared)
                                 {
                                     label = WaveArchiveConsts.AutoShared;
                                 }
                                 else if ((string)value == MessageResource.Label_WaveArchiveReference_Individual)
                                 {
                                     label = WaveArchiveConsts.AutoIndividual;
                                 }
                                 adapter.Set(targetComponent, name, label);
                                 return true;
                             });

            // リサンプルカラムにチェックが入った時
            Setters.Add(ProjectParameterNames.IsResampleEnabled,
                        delegate (IAdapterManipulator adapter, Component targetComponent, string name, object value)
                        {
                            BeginTransaction();

                            var operation = new SetParameterOperation(targetComponent.Parameters, name, value);
                            operation.Execute();
                            OnOperationExecuted(operation);

                            // サンプルレートが設定されていない場合、サンプルレートを設定します。
                            if (((SampleRateParameterValue)targetComponent.Parameters[ProjectParameterNames.SampleRate]).IsUnsettingValue == true)
                            {
                                this.UpdateWaveFiles(targetComponent);

                                operation = new SetParameterOperation(targetComponent.Parameters, ProjectParameterNames.SampleRate, this.GetSampleRate(targetComponent));
                                operation.Execute();
                                OnOperationExecuted(operation);
                            }

                            EndTransaction();

                            return true;
                        });

            // サンプルレートカラムのセルが編集された時
            Setters.Add(ProjectParameterNames.SampleRate,
                        delegate (IAdapterManipulator adapter, Component targetComponent, string name, object value)
                        {
                            BeginTransaction();

                            var operation = new SetParameterOperation(targetComponent.Parameters, name, value);
                            operation.Execute();
                            OnOperationExecuted(operation);

                            // リサンプルを有効にします。
                            if ((bool)targetComponent.Parameters[ProjectParameterNames.IsResampleEnabled].Value == false)
                            {
                                operation = new SetParameterOperation(targetComponent.Parameters, ProjectParameterNames.IsResampleEnabled, true);
                                operation.Execute();
                                OnOperationExecuted(operation);
                            }

                            EndTransaction();

                            return true;
                        });

            //
            Setters.Add(ProjectParameterNames.SoundSetItem.PreviewPlay,
                         delegate (IAdapterManipulator adapter, Component targetComponent, string name, object value)
                             {
                                 CommonListAdapter commonAdapter = adapter as CommonListAdapter;
                                 commonAdapter.ExecutePreviewPlay(targetComponent);
                                 return true;
                             });

            //
            Setters.Add(ProjectParameterNames.StreamSoundTrack.PreviewMute,
                         delegate (IAdapterManipulator adapter, Component targetComponent, string name, object value)
                             {
                                 CommonListAdapter commonAdapter = adapter as CommonListAdapter;
                                 commonAdapter.ExecutePreviewMute(targetComponent);
                                 return true;
                             });

            //
            Setters.Add(ProjectParameterNames.StreamSoundTrack.PreviewSoloPlay,
                         delegate (IAdapterManipulator adapter, Component targetComponent, string name, object value)
                             {
                                 CommonListAdapter commonAdapter = adapter as CommonListAdapter;
                                 commonAdapter.ExecutePreviewSoloPlay(targetComponent);
                                 return true;
                             });

            //
            Setters.Add(ProjectParameterNames.GroupItem.RegisterType,
                         delegate (IAdapterManipulator adapter, Component targetComponent, string name, object value)
                             {
                                 GroupItemBase groupItem = targetComponent as GroupItemBase;
                                 GroupItemRegisterType registerType = (GroupItemRegisterType)value;

                                 if (groupItem == null || groupItem.Target == null)
                                 {
                                     return false;
                                 }
                                 else
                                 {
                                     ValidationResult result =
                                         GroupItemRegisterTypeValidator.Validate(groupItem.Target, registerType);

                                     if (result.IsValid)
                                     {
                                         return false;
                                     }
                                     else
                                     {
                                         FormsApplication.Instance.UIService.
                                             ShowMessageBox(result.ToString(),
                                                            AppMessageBoxButton.OK,
                                                            AppMessageBoxImage.Warning);
                                     }
                                 }
                                 return true;
                             });

            //
            Setters.Add(ProjectParameterNames.FilePath,
                         delegate (IAdapterManipulator adapter, Component targetComponent, string name, object value)
            {
                if (ProjectFilePathUtility.Inquire(adapter, targetComponent, name, value) == true)
                {
                    return true;
                }

                if (targetComponent is StreamSoundTrackBase)
                {
                    string filepath = value as string;
                    StreamSoundTrackBase track = targetComponent as StreamSoundTrackBase;
                    StreamSoundBase streamSound = track.Parent as StreamSoundBase;
                    if (AppConfiguration.EnabledMultiChannelAAC == false)
                    {
                        if (streamSound.Children.Count == 1)
                        {
                            BeginTransaction();
                            SetParameterOperation operation;
                            operation = new SetParameterOperation(track.Parameters, ProjectParameterNames.FilePath, filepath);
                            operation.Execute();
                            OnOperationExecuted(operation);
                            if (AACUtil.IsAACFile(filepath) == true)
                            {
                                operation = new SetParameterOperation(streamSound.Parameters, ProjectParameterNames.WaveEncoding, WaveEncoding.NoConvert);
                                operation.Execute();
                                OnOperationExecuted(operation);
                            }
                            else
                            {
                                operation = new SetParameterOperation(streamSound.Parameters, ProjectParameterNames.WaveEncoding, WaveEncoding.Adpcm);
                                operation.Execute();
                                OnOperationExecuted(operation);

                                if (FileUtil.IsOpusFile(filepath) == true)
                                {
                                    // 圧縮方式を「変更しない」に設定します。
                                    operation = new SetParameterOperation(streamSound.Parameters, ProjectParameterNames.WaveEncoding, WaveEncoding.NoConvert);
                                    operation.Execute();
                                    OnOperationExecuted(operation);

                                    // リサンプルを無効に設定します。
                                    if ((bool)streamSound.Parameters[ProjectParameterNames.IsResampleEnabled].Value == true)
                                    {
                                        operation = new SetParameterOperation(streamSound.Parameters, ProjectParameterNames.IsResampleEnabled, false);
                                        operation.Execute();
                                        OnOperationExecuted(operation);
                                    }
                                }
                            }
                            EndTransaction();

                            return true;
                        }
                    }
                    else // AppConfiguration.EnabledMultiChannelAAC == true
                    {
                        StreamSoundTrackBase first = streamSound.Children[0] as StreamSoundTrackBase;

                        if (AACUtil.IsAACFile(filepath) == false &&
                            AACUtil.IsAACFile(first.FilePath) == false)
                        {
                            try
                            {
                                BeginTransaction();
                                SetParameterOperation operation;

                                operation = new SetParameterOperation(track.Parameters, ProjectParameterNames.FilePath, filepath);
                                operation.Execute();
                                OnOperationExecuted(operation);

                                if (FileUtil.IsOpusFile(filepath) == true)
                                {
                                    // 圧縮方式を「変更しない」に設定します。
                                    operation = new SetParameterOperation(streamSound.Parameters, ProjectParameterNames.WaveEncoding, WaveEncoding.NoConvert);
                                    operation.Execute();
                                    OnOperationExecuted(operation);

                                    // リサンプルを無効に設定します。
                                    if ((bool)streamSound.Parameters[ProjectParameterNames.IsResampleEnabled].Value == true)
                                    {
                                        operation = new SetParameterOperation(streamSound.Parameters, ProjectParameterNames.IsResampleEnabled, false);
                                        operation.Execute();
                                        OnOperationExecuted(operation);
                                    }
                                }

                                EndTransaction();
                            }
                            catch
                            {
                                CancelTransaction();
                            }

                            return true;
                        }

                        if (AACUtil.IsAACFile(filepath) != AACUtil.IsAACFile(first.FilePath))
                        {
                            if (FormsApplication.Instance.UIService.ShowMessageBox
                                (MessageResource.Message_ConfirmStreamSoundTrackAAC,
                                 MessageResource.DialogTitle_ConfirmStreamSoundTrack,
                                 AppMessageBoxButton.YesNo,
                                 AppMessageBoxImage.Warning) != AppMessageBoxResult.Yes)
                            {
                                // Cancel なので以後の SetValue を実行しないように true を返す。
                                return true;
                            }
                        }

                        try
                        {
                            BeginTransaction();
                            SetParameterOperation operation;

                            foreach (StreamSoundTrackBase t in streamSound.Children)
                            {
                                operation = new SetParameterOperation(t.Parameters, ProjectParameterNames.FilePath, filepath);
                                operation.Execute();
                                OnOperationExecuted(operation);
                            }

                            if (AACUtil.IsAACFile(filepath) == true)
                            {
                                operation = new SetParameterOperation(streamSound.Parameters, ProjectParameterNames.WaveEncoding, WaveEncoding.NoConvert);
                            }
                            else
                            {
                                operation = new SetParameterOperation(streamSound.Parameters, ProjectParameterNames.WaveEncoding, WaveEncoding.Adpcm);
                            }
                            operation.Execute();
                            OnOperationExecuted(operation);

                            EndTransaction();
                        }
                        catch
                        {
                            CancelTransaction();
                        }

                        return true;
                    }
                }

                return false;
            });

            foreach (string userParameter in ListTraits.GetAllUserParametersColumnNames())
            {
                Setters.Add(userParameter, UserParametersValueSetter);
            }

            ListItemCreator = delegate (Component component)
                {
                    return new CommonListItem(component);
                };

            PreviewPlay = null;
        }

        /// <summary>
        ///
        /// </summary>
        void IAdapterManipulator.OperationExecuted(Operation operation)
        {
            OnOperationExecuted(operation);
        }

        /// <summary>
        ///
        /// </summary>
        void IAdapterManipulator.Set(Component targetComponent, string name, object value)
        {
            base.SetValue(targetComponent, name, value);
        }

        /// <summary>
        ///
        /// </summary>
        public Dictionary<string, SetValueHandler> Setters
        {
            get
            {
                return this.setterDictionary;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public override void SetValue(Component targetComponent, string name, object value)
        {
            if (targetComponent == null ||
                name != ProjectParameterNames.SoundSetItem.PreviewPlay &&
                targetComponent.Parameters.ContainsKey(name) == false &&
                ListTraits.IsUserParameters(name) == false &&
                ListTraits.IsSoundSetBankReference(name) == false)
            {
                return;
            }

            if (this.setterDictionary.ContainsKey(name) == true)
            {
                if (this.setterDictionary[name](this, targetComponent, name, value) == true)
                {
                    return;
                }
            }

            base.SetValue(targetComponent, name, value);
        }

        /// <summary>
        ///
        /// </summary>
        public void ExecutePreviewPlay(Component targetComponent)
        {
            if (FormsApplication.Instance.IsErrorAudioDevice == true)
            {
                FormsApplication.Instance.ShowWarningMessageCanNotInitializeAudioDevice();
            }
            else
            {
                OnPreviewPlay(targetComponent, new EventArgs());
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void ExecutePreviewMute(Component targetComponent)
        {
            if (PreviewMute != null)
            {
                PreviewMute(targetComponent, new EventArgs());
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void ExecutePreviewSoloPlay(Component targetComponent)
        {
            if (PreviewSoloPlay != null)
            {
                PreviewSoloPlay(targetComponent, new EventArgs());
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected override ComponentListItem CreateListItem(Component component)
        {
            return ListItemCreator(component);
        }

        /// <summary>
        ///
        /// </summary>
        private bool BankReferenceValueSetter(IAdapterManipulator adapter, Component targetComponent, string name, object value)
        {
            int index = -1;
            switch (name)
            {
                case ProjectParameterNames.SequenceSound.SoundSetBankReference0: index = 0; break;
                case ProjectParameterNames.SequenceSound.SoundSetBankReference1: index = 1; break;
                case ProjectParameterNames.SequenceSound.SoundSetBankReference2: index = 2; break;
                case ProjectParameterNames.SequenceSound.SoundSetBankReference3: index = 3; break;
                default: return false;
            }

            string parameterName = ProjectParameterNames.SequenceSound.SoundSetBankReferences;
            ObservableList<ComponentReference> reference = (ObservableList<ComponentReference>)
                (targetComponent.Parameters[parameterName].Value);

            if (value is string &&
                value as string == String.Empty)
            {
                int emptyCount = 1;
                for (int i = 0; i < reference.Count; i++)
                {
                    if (i != index)
                    {
                        IParameterValue parameterValue =
                            reference[i].Parameters[ProjectParameterNames.TargetName];
                        if ((string)parameterValue.Value == String.Empty)
                        {
                            if (++emptyCount >= reference.Count)
                            {
                                return false;
                            }
                        }
                    }
                }
            }

            //
            adapter.BeginTransaction();
            SetParameterOperation operation = new SetParameterOperation
                (reference[index].Parameters,
                  ProjectParameterNames.TargetName,
                  value);
            operation.Execute();
            adapter.OperationExecuted(operation);
            adapter.EndTransaction();
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        private bool UserParametersValueSetter(IAdapterManipulator adapter, Component targetComponent, string name, object value)
        {
            string userParameterName = UserParameterSettingWatcher.GetUserParameterName(name);
            ulong uValue = (ULongParameterValue)targetComponent.Parameters.GetValue(userParameterName);
            UserParameterStructure structure = UserParameterSettingWatcher.GetStructure(name);

            uValue = UserParameterUtility.SetValue(structure, uValue, value);
            adapter.Set(targetComponent, userParameterName, uValue);

            return true;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnPreviewPlay(object sender, EventArgs e)
        {
            if (PreviewPlay != null)
            {
                PreviewPlay(sender, e);
            }
        }

        /// <summary>
        /// サンプルレートを取得します。
        /// </summary>
        private int GetSampleRate(Component component)
        {
            if (component is StreamSoundBase == true)
            {
                return component.Children
                    .Cast<StreamSoundTrackBase>()
                    .Where(t => t.WaveFile != null)
                    .Min(t => t.WaveFile.SampleRate);
            }
            else if (component is WaveSoundBase == true)
            {
                WaveSoundBase wave = component as WaveSoundBase;

                if (wave.WaveFile != null)
                {
                    return wave.WaveFile.SampleRate;
                }
            }
            else if (component is Instrument == true)
            {
                return component.Children
                    .Cast<KeyRegion>()
                    .SelectMany(k => k.Children.Cast<VelocityRegion>())
                    .Where(v => v.WaveFile != null)
                    .Min(v => v.WaveFile.SampleRate);
            }

            return SampleRateParameterValue.SampleRateMinValue;
        }

        /// <summary>
        /// 波形ファイルの情報を更新します。
        /// </summary>
        private void UpdateWaveFiles(Component component)
        {
            if (component is StreamSoundBase == true)
            {
                component.Children
                    .Cast<StreamSoundTrackBase>()
                    .Where(t => t.WaveFile == null)
                    .ToList()
                    .ForEach(t => t.UpdateWaveFile());
            }
            else if (component is WaveSoundBase == true)
            {
                WaveSoundBase wave = component as WaveSoundBase;
                if (wave.WaveFile == null)
                {
                    wave.UpdateWaveFile();
                }
            }
            else if (component is Instrument == true)
            {
                component.Children
                    .Cast<KeyRegion>()
                    .SelectMany(k => k.Children.Cast<VelocityRegion>())
                    .Where(k => k.WaveFile == null)
                    .ToList()
                    .ForEach(v => v.UpdateWaveFile());
            }
        }
    }
}
