﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;

using NintendoWare.SoundFoundation.Windows.Forms;
using NintendoWare.SoundFoundation.Windows.Forms.Windowless;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundMaker.Framework.Resources;
using NintendoWare.SoundMaker.Framework.Utilities;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    public class TabCtrl : NTabControl
    {
        #region ** 固定値

        /// <summary>
        /// タブの配置
        /// </summary>
        public enum ControlAlignment
        {
            Top,    // 上
            Bottom,   // 下
            Left,   // 左
            Right,    // 右
        }

        /// <summary>
        /// コントロールの状態
        /// </summary>
        public enum ControlState
        {
            Normal = 0,     // 通常
            Minimized = 1,    // 最小化
        }

        /// <summary>
        /// アイコンの種類
        /// </summary>
        public enum IconType
        {
            None = -1,
            StreamFolder = 0,
            WaveSoundSetFolder,
            WaveSoundSet,
            SeqSoundSetFolder,
            SeqSoundSet,
            SeqSoundFolder,
            BankFolder,
            Bank,
            WaveArchiveFolder,
            PlayerFolder,
            GroupFolder,
            Loupe,
            AddonPlayerFolder,
            AddonGroupFolder,
        }

        #endregion

        private ControlAlignment _alignment = ControlAlignment.Top;
        private ControlState _state = ControlState.Normal;
        private ImageList _imageList = new ImageList();

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TabCtrl()
        {
            CoreControlHost.RootInitialized += OnCoreControlInitialized;
            CoreControlHost.RootUninitialize += OnCoreControlUninitialize;

            ItemMaxWidth = 200;
            AllowDrop = true;

            Color color = Color.FromArgb(255, 255, 0, 255);
            _imageList.ColorDepth = ColorDepth.Depth32Bit;

            _imageList.Images.Add(ImageResource.BitmapIconStreamSoundFolder, color);
            _imageList.Images.Add(ImageResource.BitmapIconWaveSoundSetFolder, color);
            _imageList.Images.Add(ImageResource.BitmapIconWaveSoundSet, color);
            _imageList.Images.Add(ImageResource.BitmapIconSequenceSoundSetFolder, color);
            _imageList.Images.Add(ImageResource.BitmapIconSequenceSoundSet, color);
            _imageList.Images.Add(ImageResource.BitmapIconSequenceSoundFolder, color);
            _imageList.Images.Add(ImageResource.BitmapIconBankFolder, color);
            _imageList.Images.Add(ImageResource.BitmapIconBank, color);
            _imageList.Images.Add(ImageResource.BitmapIconWaveArchiveFolder, color);
            _imageList.Images.Add(ImageResource.BitmapIconPlayerFolder, color);
            _imageList.Images.Add(ImageResource.BitmapIconGroupFolder, color);
            _imageList.Images.Add(ImageResource.BitmapIconLoupe, color);
            _imageList.Images.Add(ImageResource.BitmapIconAddonPlayerFolder, color);
            _imageList.Images.Add(ImageResource.BitmapIconAddonGroupFolder, color);

            ImageList = _imageList;
        }

        #region ** イベント

        public event EventHandler StateChanged;
        public event RotateEventHandler RotateSelecting;

        public event NTabControlEventHandler TabDoubleClick;

        #endregion

        #region ** プロパティ

        public ControlAlignment Alignment
        {
            get { return _alignment; }
            set
            {
                if (value == _alignment) { return; }

                _alignment = value;
                UpdateControlState();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private MainWindow MainWindow
        {
            get { return FormsApplication.Instance.UIService.MainWindow; }
        }

        /// <summary>
        /// コントロールの状態を取得します。
        /// </summary>
        public ControlState State
        {
            get { return _state; }
            set
            {
                if (value == _state) { return; }

                _state = value;
                UpdateControlState();

                if (null != StateChanged)
                {
                    StateChanged(this, new EventArgs());
                }
            }
        }

        protected override NTabControlStyle DefaultTabControlStyle
        {
            get { return NTabControlStyle.NormalMinimizableTopTab; }
        }

        #endregion

        #region ** イベントハンドラ
        #region ** コアコントロールイベント

        private void OnCoreControlInitialized(object sender, EventArgs e)
        {
            // NWTab.MouseDoubleClick
            foreach (NWTabControl.NWTab tab in CoreControl.TabPanel.Controls)
            {
                tab.MouseDoubleClick += OnTabMouseDoubleClicked;
            }

            // NWTabPanel.ControlInserted
            CoreControl.TabPanel.ControlInserted += OnTabInserted;
            CoreControl.TabPanel.ControlRemoved += OnTabRemoved;

            UpdateIcons();
        }

        private void OnCoreControlUninitialize(object sender, EventArgs e)
        {
            // NWTab.MouseDoubleClick
            foreach (NWTabControl.NWTab tab in CoreControl.TabPanel.Controls)
            {
                tab.MouseDoubleClick -= OnTabMouseDoubleClicked;
            }

            // NWTabPanel.ControlInserted
            CoreControl.TabPanel.ControlInserted -= OnTabInserted;
            CoreControl.TabPanel.ControlRemoved -= OnTabRemoved;
        }

        private void OnTabInserted(object sender, NWControlEventArgs e)
        {
            Debug.Assert(e.Control is NWTabControl.NWTab);
            e.Control.MouseDoubleClick += OnTabMouseDoubleClicked;
        }

        private void OnTabRemoved(object sender, NWControlEventArgs e)
        {
            Debug.Assert(e.Control is NWTabControl.NWTab);
            e.Control.MouseDoubleClick -= OnTabMouseDoubleClicked;
        }

        private void OnTabMouseDoubleClicked(object sender, MouseEventArgs e)
        {
            if (ControlState.Minimized != _state) { return; }
            if (MouseButtons.Left != e.Button) { return; }

            State = ControlState.Normal;

            NWTabControl.NWTab tab = sender as NWTabControl.NWTab;
            Debug.Assert(null != tab);

            OnTabDoubleClick(new NTabControlEventArgs(tab.Model.Page as NTabPage,
                                                        CoreControl.Model.TabPages.IndexOf(tab.Model),
                                                        NTabControlAction.Selected));
        }

        #region ** 最小化イベント

        protected override void OnMinimizeButtonClick(object sender, EventArgs e)
        {
            State = ControlState.Minimized;
            OnStateChanged(new EventArgs());
        }

        protected override void OnRestoreButtonClick(object sender, EventArgs e)
        {
            State = ControlState.Normal;
            OnStateChanged(new EventArgs());


            if (null == SelectedTab) { return; }

            CommonPanel panel = (SelectedTab as CommonTabPage).Panel;
            if (null == panel) { return; }

            panel.ActivatePanel(false, true);
        }

        #endregion
        #endregion

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnResize(System.EventArgs e)
        {
            base.OnResize(e);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnDragEnter(DragEventArgs drgevent)
        {
            if (MainWindow.CanDropFiles(drgevent))
            {
                drgevent.Effect = DragDropEffects.Copy;
                return;
            }

            base.OnDragEnter(drgevent);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnDragOver(DragEventArgs e)
        {
            if (0 < TabPages.Count)
            {
                NWHitResult result = base.CoreControl.HitTest<NWTabControl.NWTabPanel>(
                                                            PointToClient(new Point(e.X, e.Y)));
                if (null == result.Control)
                {
                    e.Effect = DragDropEffects.None;
                    return;
                }
            }

            if (MainWindow.CanDropFiles(e))
            {
                e.Effect = DragDropEffects.Copy;
                return;
            }

            base.OnDragOver(e);

            if (e.Data.GetDataPresent(DataFormats.FileDrop) != false)
            {
                e.Effect = DragDropEffects.Copy;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnDragDrop(DragEventArgs e)
        {
            if (0 < TabPages.Count)
            {
                NWHitResult result = base.CoreControl.HitTest<NWTabControl.NWTabPanel>(
                                                            PointToClient(new Point(e.X, e.Y)));
                if (null == result.Control)
                {
                    e.Effect = DragDropEffects.None;
                    return;
                }
            }

            base.OnDragDrop(e);

            if (FileDroping != null)
            {
                FileDroping(this, new EventArgs());
            }

            //
            if (e.Data.GetDataPresent(DataFormats.FileDrop) != false)
            {
                string[] filePaths = (string[])e.Data.GetData(DataFormats.FileDrop);
                Array.Sort(filePaths);
                MainWindow.DropFiles(filePaths);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnDragged(NTabControlDragEventArgs e)
        {
            base.OnDragged(e);

            if (null != e.TabPage) { return; }

            // タブページの並び替えを許可する
            // 新しくタブページを作成した場合は、ここに追加する

            e.TabPage = e.Data.GetData(typeof(CommonTabPage)) as NTabPage;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnClosing(NTabControlCancelEventArgs e)
        {
            if (SuspendClosedEvent != false)
            {
                return;
            }

            base.OnClosing(e);
            if (e.Cancel) { return; }

            CommonTabPage page = null;
            CommonPanel panel = null;

            if ((page = e.TabPage as CommonTabPage) == null)
            {
                e.Cancel = true;
                return;
            }

            panel = page.Panel;

            if (null == panel)
            {
                e.Cancel = true;
                return;
            }

            if (panel.OnClosing() == false)
            {
                e.Cancel = true;
                return;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnClosed(NTabControlEventArgs e)
        {
            if (SuspendClosedEvent != false)
            {
                return;
            }

            base.OnClosed(e);

            CommonPanel panel = (e.TabPage as CommonTabPage).Panel;
            Debug.Assert(null != panel);

            panel.OnClosed();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual void OnStateChanged(EventArgs e)
        {
            if (null != StateChanged)
            {
                StateChanged(this, e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual void OnRotateSelecting(RotateSelectionEventArgs e)
        {
            if (null != RotateSelecting)
            {
                RotateSelecting(this, e);
            }
        }

        protected virtual void OnTabDoubleClick(NTabControlEventArgs e)
        {
            if (null != TabDoubleClick)
            {
                TabDoubleClick(this, e);
            }
        }

        #endregion

        #region ** メソッド

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void UpdatedClipboard()
        {
            foreach (CommonTabPage page in TabPages)
            {
                if (page.Panel != null)
                {
                    page.Panel.UpdatedClipboard();
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public static IconType ToImageIndex(SoundSetItem item)
        {
            if (item is StreamSoundPack) { return IconType.StreamFolder; }
            if (item is WaveSoundSetPack) { return IconType.WaveSoundSetFolder; }
            if (item is WaveSoundSetBase) { return IconType.WaveSoundSet; }
            if (item is SequenceSoundSetPack) { return IconType.SeqSoundSetFolder; }
            if (item is SequenceSoundSetBase) { return IconType.SeqSoundSet; }
            if (item is SequenceSoundPack) { return IconType.SeqSoundFolder; }
            if (item is SoundSetBankPack) { return IconType.BankFolder; }
            if (item is SoundSetBankBase) { return IconType.Bank; }
            if (item is WaveArchivePack) { return IconType.WaveArchiveFolder; }

            // 追加サウンドアーカイブに所属していたらアイコンを変更します。
            if (AddonSoundArchiveUtility.IsItemInAddonSoundArchive(item) == true)
            {
                if (item is PlayerPack) { return IconType.AddonPlayerFolder; }
                if (item is GroupPack) { return IconType.AddonGroupFolder; }
            }
            else
            {
                if (item is PlayerPack) { return IconType.PlayerFolder; }
                if (item is GroupPack) { return IconType.GroupFolder; }
            }

            return IconType.None;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateControlState()
        {
            switch (_state)
            {
                case ControlState.Normal:
                    TabControlStyle = DefaultTabControlStyle;
                    break;

                case ControlState.Minimized:

                    switch (_alignment)
                    {
                        case ControlAlignment.Top:
                            TabControlStyle = NTabControlStyle.MinimizedTopTab;
                            break;

                        case ControlAlignment.Bottom:
                            TabControlStyle = NTabControlStyle.MinimizedBottomTab;
                            break;

                        case ControlAlignment.Left:
                            TabControlStyle = NTabControlStyle.MinimizedLeftTab;
                            break;

                        case ControlAlignment.Right:
                            TabControlStyle = NTabControlStyle.MinimizedRightTab;
                            break;
                    }

                    break;
            }

            UpdateIcons();
        }

        private void UpdateIcons()
        {
            // アイコンを更新する
            NTabControlCore core = CoreControl as NTabControlCore;
            if (null == core) { return; }

            switch (_alignment)
            {
                case ControlAlignment.Top:
                    core.SetResource(NTabControlCore.MinimizeButton.ControlName, Icons.Images[CollapseUpImageIndex]);
                    core.SetResource(NTabControlCore.RestoreButton.ControlName, Icons.Images[CollapseDownImageIndex]);
                    break;

                case ControlAlignment.Bottom:
                    core.SetResource(NTabControlCore.MinimizeButton.ControlName, Icons.Images[CollapseDownImageIndex]);
                    core.SetResource(NTabControlCore.RestoreButton.ControlName, Icons.Images[CollapseUpImageIndex]);
                    break;

                case ControlAlignment.Left:
                    core.SetResource(NTabControlCore.MinimizeButton.ControlName, Icons.Images[CollapseLeftImageIndex]);
                    core.SetResource(NTabControlCore.RestoreButton.ControlName, Icons.Images[CollapseRightImageIndex]);
                    break;

                case ControlAlignment.Right:
                    core.SetResource(NTabControlCore.MinimizeButton.ControlName, Icons.Images[CollapseRightImageIndex]);
                    core.SetResource(NTabControlCore.RestoreButton.ControlName, Icons.Images[CollapseLeftImageIndex]);
                    break;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private CommonTabPage FindPage(CommonPanel panel)
        {
            foreach (CommonTabPage page in TabPages)
            {
                if (page.Panel == panel)
                {
                    return page;
                }
            }
            return null;
        }

        ///--------------------------------
        /// <summary>
        /// ページをアクティブにする
        /// </summary>
        public bool ActivatePage(CommonPanel panel)
        {
            CommonTabPage page = null;

            if ((page = FindPage(panel)) == null)
            {
                return false;
            }

            SelectedTab = page;
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool ClosePage(CommonTabPage page)
        {
            TabPages.Remove(page);
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void CloseSelectedPage()
        {
            CommonTabPage page = null;
            CommonPanel panel = null;

            if ((page = SelectedTab as CommonTabPage) == null)
            {
                return;
            }

            panel = page.Panel;
            Debug.Assert(panel != null);

            if (panel.OnClosing() == false)
            {
                return;
            }

            ClosePage(page);
            panel.OnClosed();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool ClosePage(CommonPanel panel)
        {
            CommonTabPage page = null;

            if ((page = FindPage(panel)) == null)
            {
                return false;
            }
            return ClosePage(page);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void ClosePageBySelfJudgement()
        {
            List<CommonTabPage> list = new List<CommonTabPage>();

            foreach (NTabPage page in TabPages)
            {
                list.Add(page as CommonTabPage);
            }

            foreach (CommonTabPage page in list)
            {
                //page.Panel.CloseBySelfJudgement();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool SuspendClosedEvent { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void CloseAllTabPagesWithoutClosedEvent()
        {
            SuspendClosedEvent = true;
            TabPages.Clear();
            SuspendClosedEvent = false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void CloseAllTabPages()
        {
            List<CommonTabPage> list = new List<CommonTabPage>();

            foreach (CommonTabPage page in TabPages)
            {
                list.Add(page);
            }

            foreach (CommonTabPage page in list)
            {
                ClosePage(page.Panel);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool SelectNextTab()
        {
            if (TabPages.Count <= 0) { return true; }

            if (SelectedIndex + 1 >= TabPages.Count)
            {

                RotateSelectionEventArgs e = new RotateSelectionEventArgs(true);
                OnRotateSelecting(e);

                if (e.Cancel) { return true; }

            }

            return base.SelectNextTab();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool SelectPrevTab()
        {
            if (TabPages.Count <= 0) { return true; }

            if (SelectedIndex - 1 < 0)
            {

                RotateSelectionEventArgs e = new RotateSelectionEventArgs(false);
                OnRotateSelecting(e);

                if (e.Cancel) { return true; }

            }

            return base.SelectPrevTab();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public event EventHandler FileDroping;

        #endregion

        #region ** イベントクラス

        public delegate void RotateEventHandler(object sender, RotateSelectionEventArgs e);

        public class RotateSelectionEventArgs : CancelEventArgs
        {
            bool _forward = true;

            public RotateSelectionEventArgs(bool forward) : base(false)
            {
                _forward = forward;
            }

            public bool Forward
            {
                get { return _forward; }
            }
        }

        #endregion
    }
}
