﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Drawing;
    using System.Drawing.Drawing2D;
    using System.Drawing.Imaging;
    using System.IO;
    using System.Linq;
    using System.Security.Cryptography;
    using System.Windows.Forms;
    using System.Windows.Forms.VisualStyles;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate;
    using NintendoWare.SoundFoundation.FileFormats.Wave;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.FileFormats;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Utilities;
    using NintendoWare.ToolDevelopmentKit.Collections;

    public delegate IInplaceEditorArgument ReplaceArgumentDelegate(ListColumnStyle style, IInplaceEditorArgument origArgs, ComponentListItem item);

    /// <summary>
    /// 共用リストコントロール
    /// </summary>
    public class CommonListCtrl : ListCtrl
    {
        ///
        private static Brush _DisableTextBrush = SystemBrushes.GrayText;

        ///
        private Dictionary<string, ReplaceArgumentDelegate> replaceArgumentDelegateDictionary = new Dictionary<string, ReplaceArgumentDelegate>();

        ///
        private IParameterValue[] _EncodingValues = new[] {
            new WaveEncodingParameterValue( WaveEncoding.Adpcm),
            new WaveEncodingParameterValue( WaveEncoding.Pcm16),
            new WaveEncodingParameterValue( WaveEncoding.Pcm8),
        };

        /// <summary>
        /// 単一再生のプルダウンメニュー
        /// </summary>
        private IParameterValue[] _SinglePlayTypeValues = new[] {
            new SinglePlayTypeParameterValue(SinglePlayType.None),
            new SinglePlayTypeParameterValue(SinglePlayType.PrioritizeNewest),
            new SinglePlayTypeParameterValue(SinglePlayType.PrioritizeNewestEffectiveDuration),
            new SinglePlayTypeParameterValue(SinglePlayType.PrioritizeOldest),
            new SinglePlayTypeParameterValue(SinglePlayType.PrioritizeOldestEffectiveDuration),
        };

        /// <summary>
        ///
        /// </summary>
        private IParameterValue[] _ActorPlayerValues = new[] {
            new IntParameterValue( 0),
            new IntParameterValue( 1),
            new IntParameterValue( 2),
            new IntParameterValue( 3),
        };

        ///
        private IParameterValue[] _3DDecayCurveValues = new[] {
            new DecayCurve3DParameterValue( DecayCurve3D.Log),
            new DecayCurve3DParameterValue( DecayCurve3D.Linear),
        };

        ///
        private IParameterValue[] _PanModeValues = new[] {
            new PanModeParameterValue( PanMode.Dual),
            new PanModeParameterValue( PanMode.Balance),
        };

        ///
        private IParameterValue[] _PanCurveValues = new[] {
            new PanCurveParameterValue( PanCurve.Sqrt),
            new PanCurveParameterValue( PanCurve.Sqrt_0Db),
            new PanCurveParameterValue( PanCurve.Sqrt_0Db_Clamp),
            new PanCurveParameterValue( PanCurve.SinCos),
            new PanCurveParameterValue( PanCurve.SinCos_0Db),
            new PanCurveParameterValue( PanCurve.SinCos_0Db_Clamp),
            new PanCurveParameterValue( PanCurve.Linear),
            new PanCurveParameterValue( PanCurve.Linear_0Db),
            new PanCurveParameterValue( PanCurve.Linear_0Db_Clamp),
        };

        ///
        private IParameterValue[] _FileTypeValues = new[] {
            new SequenceSoundFileTypeParameterValue( SequenceSoundFileType.Text),
            new SequenceSoundFileTypeParameterValue( SequenceSoundFileType.Smf),
        };

        ///
        private IParameterValue[] _GroupOutputTypeValues = new[] {
            new GroupOutputTypeParameterValue( GroupOutputType.Embedding ),
            new GroupOutputTypeParameterValue( GroupOutputType.Link ),
            new GroupOutputTypeParameterValue( GroupOutputType.UserManagement ),
            new GroupOutputTypeParameterValue( GroupOutputType.None ),
        };

        ///
        private IParameterValue[] _GroupItemRegisterTypeValues = new[] {
            new GroupItemRegisterTypeParameterValue( GroupItemRegisterType.All ),
            new GroupItemRegisterTypeParameterValue( GroupItemRegisterType.SequenceAndBank ),
            new GroupItemRegisterTypeParameterValue( GroupItemRegisterType.SequenceAndWaveArchive ),
            new GroupItemRegisterTypeParameterValue( GroupItemRegisterType.BankAndWaveArchive ),
            new GroupItemRegisterTypeParameterValue( GroupItemRegisterType.OnlySequence ),
            new GroupItemRegisterTypeParameterValue( GroupItemRegisterType.OnlyWaveSoundSet ),
            new GroupItemRegisterTypeParameterValue( GroupItemRegisterType.OnlyBank ),
            new GroupItemRegisterTypeParameterValue( GroupItemRegisterType.OnlyWaveArchive ),
        };

        ///
        private IParameterValue[] _BiquadTypeValues = new[] {
            new BiquadTypeParameterValue(BiquadType.None),
            new BiquadTypeParameterValue(BiquadType.LPF),
            new BiquadTypeParameterValue(BiquadType.HPF),
            new BiquadTypeParameterValue(BiquadType.BPF512),
            new BiquadTypeParameterValue(BiquadType.BPF1024),
            new BiquadTypeParameterValue(BiquadType.BPF2048),
            new BiquadTypeParameterValue(BiquadType.USR0),
            new BiquadTypeParameterValue(BiquadType.USR1),
            new BiquadTypeParameterValue(BiquadType.USR2),
            new BiquadTypeParameterValue(BiquadType.USR3),
            new BiquadTypeParameterValue(BiquadType.USR4),
            new BiquadTypeParameterValue(BiquadType.USR5),
            new BiquadTypeParameterValue(BiquadType.USR6),
            new BiquadTypeParameterValue(BiquadType.USR7),
            new BiquadTypeParameterValue(BiquadType.USR8),
            new BiquadTypeParameterValue(BiquadType.USR9),
            new BiquadTypeParameterValue(BiquadType.USR10),
            new BiquadTypeParameterValue(BiquadType.USR11),
            new BiquadTypeParameterValue(BiquadType.USR12),
            new BiquadTypeParameterValue(BiquadType.USR13),
            new BiquadTypeParameterValue(BiquadType.USR14),
            new BiquadTypeParameterValue(BiquadType.USR15),
            new BiquadTypeParameterValue(BiquadType.USR16),
            new BiquadTypeParameterValue(BiquadType.USR17),
            new BiquadTypeParameterValue(BiquadType.USR18),
            new BiquadTypeParameterValue(BiquadType.USR19),
            new BiquadTypeParameterValue(BiquadType.USR20),
            new BiquadTypeParameterValue(BiquadType.USR21),
            new BiquadTypeParameterValue(BiquadType.USR22),
            new BiquadTypeParameterValue(BiquadType.USR23),
            new BiquadTypeParameterValue(BiquadType.USR24),
            new BiquadTypeParameterValue(BiquadType.USR25),
            new BiquadTypeParameterValue(BiquadType.USR26),
            new BiquadTypeParameterValue(BiquadType.USR27),
            new BiquadTypeParameterValue(BiquadType.USR28),
            new BiquadTypeParameterValue(BiquadType.USR29),
            new BiquadTypeParameterValue(BiquadType.USR30),
            new BiquadTypeParameterValue(BiquadType.USR31),
            new BiquadTypeParameterValue(BiquadType.USR32),
            new BiquadTypeParameterValue(BiquadType.USR33),
            new BiquadTypeParameterValue(BiquadType.USR34),
            new BiquadTypeParameterValue(BiquadType.USR35),
            new BiquadTypeParameterValue(BiquadType.USR36),
            new BiquadTypeParameterValue(BiquadType.USR37),
            new BiquadTypeParameterValue(BiquadType.USR38),
            new BiquadTypeParameterValue(BiquadType.USR39),
            new BiquadTypeParameterValue(BiquadType.USR40),
            new BiquadTypeParameterValue(BiquadType.USR41),
            new BiquadTypeParameterValue(BiquadType.USR42),
            new BiquadTypeParameterValue(BiquadType.USR43),
            new BiquadTypeParameterValue(BiquadType.USR44),
            new BiquadTypeParameterValue(BiquadType.USR45),
            new BiquadTypeParameterValue(BiquadType.USR46),
            new BiquadTypeParameterValue(BiquadType.USR47),
            new BiquadTypeParameterValue(BiquadType.USR48),
            new BiquadTypeParameterValue(BiquadType.USR49),
            new BiquadTypeParameterValue(BiquadType.USR50),
            new BiquadTypeParameterValue(BiquadType.USR51),
            new BiquadTypeParameterValue(BiquadType.USR52),
            new BiquadTypeParameterValue(BiquadType.USR53),
            new BiquadTypeParameterValue(BiquadType.USR54),
            new BiquadTypeParameterValue(BiquadType.USR55),
            new BiquadTypeParameterValue(BiquadType.USR56),
            new BiquadTypeParameterValue(BiquadType.USR57),
            new BiquadTypeParameterValue(BiquadType.USR58),
            new BiquadTypeParameterValue(BiquadType.USR59),
            new BiquadTypeParameterValue(BiquadType.USR60),
            new BiquadTypeParameterValue(BiquadType.USR61),
            new BiquadTypeParameterValue(BiquadType.USR62),
            new BiquadTypeParameterValue(BiquadType.USR63),
        };

        ///
        /// データ名、サブアイテム名、型、サブアイテムに渡す引数
        ///
        private ListColumnStyle[] _ColumnStyles = new[] {
            new ListColumnStyle( ListTraits.ColumnName_Name,                     "Text", typeof(string)),
            new ListColumnStyle( ListTraits.ColumnName_Comment,                  "Text", typeof(string)),
            new ListColumnStyle( ListTraits.ColumnName_Comment1,                 "Text", typeof(string)),
            new ListColumnStyle( ListTraits.ColumnName_Comment2,                 "Text", typeof(string)),
            new ListColumnStyle( ListTraits.ColumnName_Comment3,                 "Text", typeof(string)),
            new ListColumnStyle( ListTraits.ColumnName_Comment4,                 "Text", typeof(string)),
            new ListColumnStyle( ListTraits.ColumnName_Comment5,                 "Text", typeof(string)),
            new ListColumnStyle( ListTraits.ColumnName_Comment6,                 "Text", typeof(string)),
            new ListColumnStyle( ListTraits.ColumnName_Comment7,                 "Text", typeof(string)),
            new ListColumnStyle( ListTraits.ColumnName_Comment8,                 "Text", typeof(string)),
            new ListColumnStyle( ListTraits.ColumnName_Comment9,                 "Text", typeof(string)),
            new ListColumnStyle( ListTraits.ColumnName_GroupItemTargetSoundSet,  "Text", typeof(string)),
            new ListColumnStyle( ListTraits.ColumnName_ItemLocation,             "Text", typeof(string)),
            new ListColumnStyle( ListTraits.ColumnName_Volume,                   "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_ChannelPriority,          "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_PlayerPriority,           "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_Pan,                      "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_SurroundPan,              "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_AuxASend,                 "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_AuxBSend,                 "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_MainSend,                 "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_DopplerFactor3D,          "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_PlayerSoundLimit,         "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_PlayerHeapSize,           "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_ProgramNo,                "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_EnvelopeAttack,           "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_EnvelopeHold,             "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_EnvelopeDecay,            "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_EnvelopeSustain,          "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_EnvelopeRelease,          "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_PitchSemitones,           "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_PitchCents,               "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_KeyGroup,                 "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_WaveSoundEnvelopeRelease, "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_TrackVolume,              "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_TrackAuxASend,            "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_TrackAuxBSend,            "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_TrackMainSend,            "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_MaximumVoiceCount,        "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_UserParameter0,   "UserData", typeof(ulong)),
            new ListColumnStyle( ListTraits.ColumnName_UserParameter1,   "UserData", typeof(ulong)),
            new ListColumnStyle( ListTraits.ColumnName_UserParameter2,   "UserData", typeof(ulong)),
            new ListColumnStyle( ListTraits.ColumnName_UserParameter3,   "UserData", typeof(ulong)),

            new ListColumnStyle( ListTraits.ColumnName_Pitch,        "Float", typeof(float),
                                 new ListSubItemArgument( new ListDecimalDrawerArgument( 5),
                                                          null)),
            new ListColumnStyle( ListTraits.ColumnName_DecayRatio3D, "Float", typeof(float),
                                 new ListSubItemArgument( new ListDecimalDrawerArgument( 2),
                                                          null)),

            new ListColumnStyle( ListTraits.ColumnName_Enable3DVolume,       "CheckBox", typeof(bool)),
            new ListColumnStyle( ListTraits.ColumnName_Enable3DPan,          "CheckBox", typeof(bool)),
            new ListColumnStyle( ListTraits.ColumnName_Enable3DSurroundPan,  "CheckBox", typeof(bool)),
            new ListColumnStyle( ListTraits.ColumnName_Enable3DPriority,     "CheckBox", typeof(bool)),
            new ListColumnStyle( ListTraits.ColumnName_Enable3DFilter,       "CheckBox", typeof(bool)),
            new ListColumnStyle( ListTraits.ColumnName_ReleasePriorityFixed, "CheckBox", typeof(bool)),
            new ListColumnStyle( ListTraits.ColumnName_PercussionMode,       "CheckBox", typeof(bool)),

            new ListColumnStyle( ListTraits.ColumnName_WaveArchiveLoadType,  "CheckBox", typeof(bool)),

            new ListColumnStyle( ListTraits.ColumnName_WaveTime,             "Text", typeof(string),
                                 new ListSubItemArgument( new ListTextDrawerArgument()
                                                              {
                                                                  Alignment = StringAlignment.Far
                                                              },
                                                          null)),
            new ListColumnStyle( ListTraits.ColumnName_WaveTick,             "Integer", typeof(int)),
            new ListColumnStyle(ListTraits.ColumnName_IsPrefetchEnabled, "CheckBox", typeof(bool)),
            new ListColumnStyle(ListTraits.ColumnName_IsResampleEnabled, "CheckBox", typeof(bool)),
            new ListColumnStyle( ListTraits.ColumnName_WaveBitRate, "Text", typeof(string),
                                 new ListSubItemArgument( new ListTextDrawerArgument()
                                                              {
                                                                  Alignment = StringAlignment.Far
                                                              },
                                                          null)),

            new ListColumnStyle( ListTraits.ColumnName_WaveSampleBit, "Text", typeof(string),
                                 new ListSubItemArgument( new ListTextDrawerArgument()
                                                              {
                                                                  Alignment = StringAlignment.Far
                                                              },
                                                          null)),

            new ListColumnStyle(ListTraits.ColumnName_IsDownMixEnabled, "CheckBox", typeof(bool)),
            new ListColumnStyle( ListTraits.ColumnName_WaveChannel, "Text", typeof(string),
                                 new ListSubItemArgument( new ListTextDrawerArgument()
                                                              {
                                                                  Alignment = StringAlignment.Far
                                                              },
                                                          null)),

            new ListColumnStyle( ListTraits.ColumnName_TrackNo, "Text", typeof(string),
                                 new ListSubItemArgument( new ListTextDrawerArgument()
                                                              {
                                                                  Alignment = StringAlignment.Far
                                                              },
                                                          null)),

            new ListColumnStyle( ListTraits.ColumnName_ChannelNo, "Text", typeof(string),
                                 new ListSubItemArgument( new ListTextDrawerArgument()
                                                              {
                                                                  Alignment = StringAlignment.Far
                                                              },
                                                          null)),

            new ListColumnStyle( ListTraits.ColumnName_DataSize,             "Size", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_PreviewPlay,          "PreviewPlay", typeof(bool)),

            new ListColumnStyle( ListTraits.ColumnName_PreviewMute,          "PreviewMute", typeof(bool)),
            new ListColumnStyle( ListTraits.ColumnName_PreviewSoloPlay,      "PreviewSoloPlay", typeof(bool)),

            new ListColumnStyle( ListTraits.ColumnName_FrontBypass,          "CheckBox", typeof(bool)),
            new ListColumnStyle( ListTraits.ColumnName_LPF,                  "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_Biquad,               "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_DateTime,             "Text", typeof(string)),
            new ListColumnStyle( ListTraits.ColumnName_ReferenceCount,       "Integer", typeof(int)),

            new ListColumnStyle( ListTraits.ColumnName_SinglePlayEffectiveDuration, "Integer", typeof(int)),

            new ListColumnStyle( ListTraits.ColumnName_IntegratedLoudness,   "Text", typeof(string),
                                 new ListSubItemArgument( new ListTextDrawerArgument()
                                                              {
                                                                  Alignment = StringAlignment.Far
                                                              },
                                                          null)),
        };

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CommonListCtrl()
        {
            DropDownInplaceEditor.Argument dropDownArg = null;
            DropDownListInplaceEditor.Argument arg = null;

            foreach (ListColumnStyle style in _ColumnStyles)
            {
                ColumnStyles.Add(style);
            }

            //
            ColumnStyles.Add(new ListColumnStyle
                              ("RowHeader", "RowHeader", typeof(string), null, false, false));

            //
            OpenFilePathDialogInplaceEditor.Argument fileDialogArg = null;

            fileDialogArg = new OpenFilePathDialogInplaceEditor.Argument();
            ColumnStyles.Add(new ListColumnStyle
                              (ListTraits.ColumnName_FilePath, "FilePath", typeof(string),
                                new ListSubItemArgument(null, fileDialogArg)));

            //
            ListTextDrawerArgument drawerArg = null;

            //
            drawerArg = new ListTextDrawerArgument();
            drawerArg.LineAlignment = StringAlignment.Far;
            drawerArg.Alignment = StringAlignment.Far;

            ColumnStyles.Add(new ListColumnStyle
                              ("Key", "RowHeader", typeof(string),
                                new ListSubItemArgument(drawerArg, null), false, false));

            //
            arg = new DropDownListInplaceEditor.Argument(OriginalKeyValues);
            ColumnStyles.Add(new ListColumnStyle
                              (ListTraits.ColumnName_OriginalKey, "DropDownList", typeof(string),
                                new ListSubItemArgument(null, arg)));

            arg = new DropDownListInplaceEditor.Argument(_EncodingValues);
            ColumnStyles.Add(new ListColumnStyle
                              (ListTraits.ColumnName_WaveEncoding, "DropDownList", typeof(string),
                                new ListSubItemArgument(null, arg)));

            // サンプルレート
            dropDownArg = new DropDownInplaceEditor.Argument(null);
            ColumnStyles.Add(new ListColumnStyle
                             (ListTraits.ColumnName_SampleRate, "DropDown", typeof(int),
                              new ListSubItemArgument(new ListTextDrawerArgument() { Alignment = StringAlignment.Far }, dropDownArg)));

            // 単一再生の種類
            arg = new DropDownListInplaceEditor.Argument(_SinglePlayTypeValues);
            ColumnStyles.Add(new ListColumnStyle
                             (ListTraits.ColumnName_SinglePlayType, "DropDownList", typeof(string),
                              new ListSubItemArgument(null, arg)));

#if false
            arg = new DropDownListInplaceEditor.Argument(_LoopTypeValues);
            ColumnStyles.Add(new ListColumnStyle
                              ( ListTraits.ColumnName_LoopType, "DropDownList", typeof(string),
                                new ListSubItemArgument(null, arg)));
#endif

            arg = new DropDownListInplaceEditor.Argument(_BiquadTypeValues);
            ColumnStyles.Add(new ListColumnStyle
                              (ListTraits.ColumnName_BiquadType, "DropDownList", typeof(string),
                                new ListSubItemArgument(null, arg)));

            //
            dropDownArg = new DropDownInplaceEditor.Argument(null);
            ColumnStyles.Add(new ListColumnStyle
                              (ListTraits.ColumnName_PlayerReference, "AssistDropDown", typeof(string),
                                new ListSubItemArgument(null, dropDownArg)));

            dropDownArg = new DropDownInplaceEditor.Argument(null);
            ColumnStyles.Add(new ListColumnStyle
                             (ListTraits.ColumnName_SoundSetBankReferences, "BankAssistDropDown", typeof(string),
                               new ListSubItemArgument(null, dropDownArg)));

            //
            dropDownArg = new DropDownInplaceEditor.Argument(null);
            ColumnStyles.Add(new ListColumnStyle
                              (ListTraits.ColumnName_SoundSetBankReference0, "BankAssistDropDown", typeof(string),
                                new ListSubItemArgument(null, dropDownArg)));
            dropDownArg = new DropDownInplaceEditor.Argument(null);
            ColumnStyles.Add(new ListColumnStyle
                              (ListTraits.ColumnName_SoundSetBankReference1, "BankAssistDropDown", typeof(string),
                                new ListSubItemArgument(null, dropDownArg)));
            dropDownArg = new DropDownInplaceEditor.Argument(null);
            ColumnStyles.Add(new ListColumnStyle
                              (ListTraits.ColumnName_SoundSetBankReference2, "BankAssistDropDown", typeof(string),
                                new ListSubItemArgument(null, dropDownArg)));
            dropDownArg = new DropDownInplaceEditor.Argument(null);
            ColumnStyles.Add(new ListColumnStyle
                              (ListTraits.ColumnName_SoundSetBankReference3, "BankAssistDropDown", typeof(string),
                                new ListSubItemArgument(null, dropDownArg)));

            //
            dropDownArg = new DropDownInplaceEditor.Argument(null);
            ColumnStyles.Add(new ListColumnStyle
                              (ListTraits.ColumnName_WaveArchiveReference, "AssistDropDown", typeof(string),
                                new ListSubItemArgument(null, dropDownArg)));
            //
            arg = new DropDownListInplaceEditor.Argument(_GroupOutputTypeValues);
            ColumnStyles.Add(new ListColumnStyle
                              (ListTraits.ColumnName_GroupOutputType, "DropDownList", typeof(string),
                                new ListSubItemArgument(null, arg)));

            //
            arg = new DropDownListInplaceEditor.Argument(_GroupItemRegisterTypeValues);
            ColumnStyles.Add(new ListColumnStyle
                              (ListTraits.ColumnName_GroupItemRegisterType, "DropDownList", typeof(string),
                                new ListSubItemArgument(null, arg)));

            //
            dropDownArg = new DropDownInplaceEditor.Argument(null);
            ColumnStyles.Add(new ListColumnStyle
                              (ListTraits.ColumnName_GroupItemTargetName,
                                "GroupItemDropDownList", typeof(string),
                                new ListSubItemArgument(null, dropDownArg)));

            //
            drawerArg = new ListTextDrawerArgument();
            drawerArg.LineAlignment = StringAlignment.Far;
            drawerArg.Alignment = StringAlignment.Far;

            arg = new DropDownListInplaceEditor.Argument(_ActorPlayerValues);
            ColumnStyles.Add(new ListColumnStyle
                              (ListTraits.ColumnName_ActorPlayer, "DropDownList", typeof(string),
                                new ListSubItemArgument(drawerArg, arg)));

            //
            arg = new DropDownListInplaceEditor.Argument(_3DDecayCurveValues);
            ColumnStyles.Add(new ListColumnStyle
                              (ListTraits.ColumnName_DecayCurve3D, "DropDownList", typeof(string),
                                new ListSubItemArgument(null, arg)));

            arg = new DropDownListInplaceEditor.Argument(_PanModeValues);
            ColumnStyles.Add(new ListColumnStyle
                              (ListTraits.ColumnName_PanMode, "DropDownList", typeof(string),
                                new ListSubItemArgument(null, arg)));

            arg = new DropDownListInplaceEditor.Argument(_PanCurveValues);
            ColumnStyles.Add(new ListColumnStyle
                              (ListTraits.ColumnName_PanCurve, "DropDownList", typeof(string),
                                new ListSubItemArgument(null, arg)));

            arg = new DropDownListInplaceEditor.Argument(_FileTypeValues);
            ColumnStyles.Add(new ListColumnStyle
                              (ListTraits.ColumnName_FileType, "DropDownList", typeof(string),
                                new ListSubItemArgument(null, arg)));

            //
            dropDownArg = new DropDownInplaceEditor.Argument(null);
            ColumnStyles.Add(new ListColumnStyle
                              (ListTraits.ColumnName_StartPosition, "DropDown", typeof(string),
                                new ListSubItemArgument(null, dropDownArg)));

            //
            BackgroundDrawer = new CommonListBackgroundDrawer();
            ContentDrawer = new CommonListContentDrawer();

            //Drawerの登録
            PartDrawers.Add(new ListPartCommonRowHeaderDrawer());
            PartDrawers.Add(new ListPartUserParameterDrawer());
            PartDrawers.Add(new ListPartUserParameterIntegerDrawer());
            PartDrawers.Add(new ListPartUserParameterULongDrawer());
            PartDrawers.Add(new ListPartUserParameterFloatDrawer());
            PartDrawers.Add(new ListPartUserParameterBooleanDrawer());
            PartDrawers.Add(new ListPartUserParameterAssistDropDownDrawer());

            PartDrawers.Remove("Text");
            PartDrawers.Add(new ListPartTextDrawer());

            PartDrawers.Remove("Integer");
            PartDrawers.Add(new ListPartIntegerDrawer());

            PartDrawers.Add(new ListPartPreviewPlayDrawer());
            PartDrawers.Add(new ListPartPreviewMuteDrawer());
            PartDrawers.Add(new ListPartPreviewSoloPlayDrawer());

            //SubItemの登録
            SubItems.Remove("RowHeader");
            SubItems.Add(new ListSubItemCommonRowHeader());
            SubItems.Add(new ListSubItemUserData());
            SubItems.Add(new ListSubItemUserParameterInteger());
            SubItems.Add(new ListSubItemUserParameterULong());
            SubItems.Add(new ListSubItemUserParameterFloat());
            SubItems.Add(new ListSubItemUserParameterBoolean());
            SubItems.Add(new ListSubItemUserParameterAssistDropDown());
            SubItems.Add(new ListSubItemDropDownList
                          ("GroupItemDropDownList", "GroupItemDropDownList"));
            SubItems.Add(new ListSubItemPreviewPlay());
            SubItems.Add(new ListSubItemPreviewMute());
            SubItems.Add(new ListSubItemPreviewSoloPlay());
            SubItems.Add(new ListSubItemBankAssistDropDown());

            //Editorの登録
            InplaceEditor editor = new UserDataEditDialogInplaceEditor();
            editor.BeginEditing += delegate (object sender, EventArgs e)
                {
                    UserDataEditDialogInplaceEditor ie = sender as UserDataEditDialogInplaceEditor;
                    Debug.Assert(ie != null, "Editor is null");
                    ie.UserDataStructureSettings = ProjectService.Project.UserDataStructureSettings;
                };
            InplaceEditors.Add("UserDataEditDialog", editor);

            //
            MultipleBankEditDialogInplaceEditor bankEditor = new MultipleBankEditDialogInplaceEditor();
            InplaceEditors.Add("BankDialog", bankEditor);

            //
            InplaceEditors.Add("GroupItemDropDownList", new GroupItemDropDownList());
            InplaceEditors.Add("PreviewPlay", new PreviewPlayInplaceEditor());
            InplaceEditors.Add("PreviewMute", new PreviewMuteInplaceEditor());
            InplaceEditors.Add("PreviewSoloPlay", new PreviewSoloPlayInplaceEditor());

            //
            ColumnStyles.Add(new ListColumnStyle(ListTraits.ColumnName_SndEdit, "SndEdit", typeof(bool)));
            ColumnStyles.Add(new ListColumnStyle(ListTraits.ColumnName_SndEditEnabled, "CheckBox", typeof(bool)));
            SubItems.Add(new ListSubItemSndEditWithCheck());
            PartDrawers.Add(new ListPartSndEditDrawer());

            //
            this.ReplaceArgumentDelegates.Add(ProjectParameterNames.Sound.PlayerReference, this.ReplaceArgumentReferencePlayer);
            this.ReplaceArgumentDelegates.Add(ProjectParameterNames.SequenceSound.SoundSetBankReference0, this.ReplaceArgumentReferenceBank);
            this.ReplaceArgumentDelegates.Add(ProjectParameterNames.SequenceSound.SoundSetBankReference1, this.ReplaceArgumentReferenceBank);
            this.ReplaceArgumentDelegates.Add(ProjectParameterNames.SequenceSound.SoundSetBankReferences, this.ReplaceArgumentReferenceBank);
            this.ReplaceArgumentDelegates.Add(ProjectParameterNames.SequenceSound.SoundSetBankReference2, this.ReplaceArgumentReferenceBank);
            this.ReplaceArgumentDelegates.Add(ProjectParameterNames.SequenceSound.SoundSetBankReference3, this.ReplaceArgumentReferenceBank);
            this.ReplaceArgumentDelegates.Add(ProjectParameterNames.SoundSetItem.WaveArchiveReference, this.ReplaceArgumentReferenceWaveArchive);
            this.ReplaceArgumentDelegates.Add(ProjectParameterNames.TargetItemReference, this.ReplaceArgumentReferenceTargetItem);

            //
            this.AddUserParameterColumnStyles();

            //
            ApplyOptionConfigurations();
        }

        /// <summary>
        /// ReplaceArgument のデリゲートのコレクションを取得します。
        /// </summary>
        public IDictionary<string, ReplaceArgumentDelegate> ReplaceArgumentDelegates
        {
            get
            {
                return this.replaceArgumentDelegateDictionary;
            }
        }

        /// <summary>
        /// オプションのコンフィグレーションを適用します。
        /// </summary>
        public void ApplyOptionConfigurations()
        {
            if (ApplicationBase.Instance == null)
            {
                return;
            }

            XmlOptionsApplicationGeneral general = null;
            general = ApplicationBase.Instance.AppConfiguration.Options.Application.General;

            //
            MoveNextLineCaretAfterEndEditByEnter = general.MoveNextLineByEnter;

            //
            if (general.BeginEditByEnter != false)
            {
                if (ImmediateEditBeginKeys.Contains(Keys.Enter) == false)
                {
                    ImmediateEditBeginKeys.Add(Keys.Enter);
                }
            }
            else
            {
                if (ImmediateEditBeginKeys.Contains(Keys.Enter) != false)
                {
                    ImmediateEditBeginKeys.Remove(Keys.Enter);
                }
            }
        }

        /// <summary>
        /// エディタの引数の取得
        /// </summary>
        public override IInplaceEditorArgument GetEditorArgument(ListColumnStyle style, IListItem item)
        {
            IInplaceEditorArgument origArgs = base.GetEditorArgument(style, item);
            IInplaceEditorArgument arg = null;

            if ((arg = ReplaceArgument(style, origArgs, item as ComponentListItem)) != null)
            {
                return arg;
            }

            return origArgs;
        }

        /// <summary>
        ///
        /// </summary>
        public override DataObject TrackDragDrop()
        {
            Component[] components = null;
            DataObject dataObject = null;

            components = GetSelectedItems().Select(i => i.Target).ToArray();
            dataObject = ClipboardService.SerializeComponents(components);

            // 名前を外部ツールにドラッグ＆ドロップできるようにします。
            if (dataObject != null)
            {
                if (components.Length == 1)
                {
                    dataObject.SetText(components[0].Name);
                }
            }

            if (dataObject != null)
            {
                Clipboard.SetDataObject(dataObject, true);
            }

            return dataObject;
        }

        /// <summary>
        /// 指定Componentからアイテムを選択します。
        /// </summary>
        public void SelectItemsByComponent(Component[] components)
        {
            if (components == null ||
                components.Length <= 0)
            {
                return;
            }

            ListItemSelectedDictionary selectedItems = GetItemSelecteds(null);

            Dictionary<Component, IListItem> dictionary = new Dictionary<Component, IListItem>();
            foreach (IListItemInfo info in ItemInfos)
            {
                CommonListItem item = info.Item as CommonListItem;
                if (item != null)
                {
                    dictionary.Add(item.Target, item);
                }
            }

            //
            selectedItems.Clear();
            bool setCaret = false;
            foreach (Component component in components)
            {
                if (dictionary.ContainsKey(component) != false)
                {
                    selectedItems.Select(dictionary[component], true);

                    if (setCaret == false)
                    {
                        this.SetCaret(dictionary[component]);
                        setCaret = true;
                    }
                }
            }
        }

        /// <summary>
        /// 選択されているアイテムが参照している Componentを取得
        /// </summary>
        public virtual Component[] GetComponentsBySelectedItem()
        {
            return SelectedItems.Cast<ComponentListItem>().Select(i => i.Target).ToArray();
        }

        /// <summary>
        /// 選択されているアイテムの取得
        /// </summary>
        public CommonListItem[] GetSelectedItems()
        {
            return SelectedItems.Cast<CommonListItem>().ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        public override void SetupDrawDescriptor(IListItem listItem, string name, ListDrawDescriptor desc)
        {
            ComponentListItem item = null;

            item = listItem as CommonListItem;

            if (desc.Selected == false && desc.SubSelected == false &&
                item.Target.IsEnabled == false)
            {
                desc.TextBrush = _DisableTextBrush;
            }
        }

        /// <summary>
        /// 表示されている順番でソートを行います。
        /// </summary>
        public Component[] SortToDisplayedOrder(Component[] components)
        {
            HashSet<Component> hashSet = new HashSet<Component>();
            foreach (Component component in components)
            {
                hashSet.Add(component);
            }

            List<Component> list = new List<Component>();
            foreach (ListItemInfo itemInfo in ItemInfos)
            {
                CommonListItem item = itemInfo.Item as CommonListItem;
                Component component = item.Target;
                if (hashSet.Contains(component) != false)
                {
                    list.Add(component);
                }
            }

            return list.ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdatePartDrawerOptions()
        {
            ListPartCommonRowHeaderDrawer rowHeaderDrawer = null;
            ListPartFilePathDrawer drawer = null;
            bool showFullPath = true;
            bool showRowNumber = true;

            XmlOptionsApplicationGeneral general = null;
            general = ApplicationBase.Instance.AppConfiguration.Options.Application.General;

            showFullPath = general.ShowFullFilePath;
            showRowNumber = general.ShowRowNumber;

            drawer = PartDrawers["FilePath"] as ListPartFilePathDrawer;
            drawer.ShowFullPath = showFullPath;

            rowHeaderDrawer = PartDrawers["RowHeader"] as ListPartCommonRowHeaderDrawer;
            rowHeaderDrawer.ShowRowNumber = showRowNumber;

            // ユーザーパラメータドロワーの設定
            if (ProjectService.Project != null)
            {
                ListPartUserParameterDrawer userParameterDrawer = null;
                userParameterDrawer = PartDrawers["UserParameter"] as ListPartUserParameterDrawer;
                userParameterDrawer.Enableds = new bool[4] {
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter).Enabled,
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter1).Enabled,
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter2).Enabled,
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter3).Enabled,
                };

                ListPartUserParameterIntegerDrawer integerDrawer =
                    PartDrawers["UserParameterInteger"] as ListPartUserParameterIntegerDrawer;
                integerDrawer.Enableds = new bool[4] {
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter).Enabled,
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter1).Enabled,
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter2).Enabled,
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter3).Enabled,
                };

                ListPartUserParameterULongDrawer ulongDrawer =
                    PartDrawers["UserParameterULong"] as ListPartUserParameterULongDrawer;
                ulongDrawer.Enableds = new bool[4] {
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter).Enabled,
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter1).Enabled,
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter2).Enabled,
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter3).Enabled,
                };

                ListPartUserParameterFloatDrawer floatDrawer =
                    PartDrawers["UserParameterFloat"] as ListPartUserParameterFloatDrawer;
                floatDrawer.Enableds = new bool[4] {
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter).Enabled,
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter1).Enabled,
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter2).Enabled,
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter3).Enabled,
                };

                ListPartUserParameterBooleanDrawer boolDrawer =
                    PartDrawers["UserParameterBoolean"] as ListPartUserParameterBooleanDrawer;
                boolDrawer.Enableds = new bool[4] {
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter).Enabled,
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter1).Enabled,
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter2).Enabled,
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter3).Enabled,
                };

                ListPartUserParameterAssistDropDownDrawer assistDrawer =
                    PartDrawers["UserParameterAssistDropDown"] as ListPartUserParameterAssistDropDownDrawer;
                assistDrawer.Enableds = new bool[4] {
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter).Enabled,
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter1).Enabled,
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter2).Enabled,
                    UserParameterSettingWatcher.Get( ProjectParameterNames.UserParameter3).Enabled,
                };
            }

            //
            CommonListBackgroundDrawer backgroundDrawer = null;
            backgroundDrawer = BackgroundDrawer as CommonListBackgroundDrawer;
            if (backgroundDrawer != null)
            {
                backgroundDrawer.ShowParameterBar = general.ParameterBar.Enable;
                backgroundDrawer.ParameterBarSize = general.ParameterBar.Size;

                backgroundDrawer.ParameterBarColor = Color.FromArgb
                    (general.ParameterBar.Color.Alpha,
                      general.ParameterBar.Color.Red,
                      general.ParameterBar.Color.Green,
                      general.ParameterBar.Color.Blue);

                backgroundDrawer.SelectedParameterBarColor = Color.FromArgb
                    (general.ParameterBar.SelectedColor.Alpha,
                      general.ParameterBar.SelectedColor.Red,
                      general.ParameterBar.SelectedColor.Green,
                      general.ParameterBar.SelectedColor.Blue);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public IListItem GetItem(Component component)
        {
            foreach (ComponentListItem item in ItemsSource.Items)
            {
                if (item.Target == component)
                {
                    return item;
                }
            }
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        public IListItem GetItem(string name)
        {
            foreach (ComponentListItem item in ItemsSource.Items)
            {
                if (item.Target.Name == name)
                {
                    return item;
                }
            }
            return null;
        }

        /// <summary>
        /// ユーザーパラメータの定義からカラムを再登録
        /// </summary>
        public void UpdateUserParameterColumnStyles()
        {
            List<string> userParameters = new List<string>();

            foreach (string key in ColumnStyles.Keys)
            {
                if (ListTraits.IsUserParameters(key) == true)
                {
                    userParameters.Add(key);
                }
            }

            foreach (string userParameter in userParameters)
            {
                ColumnStyles.Remove(userParameter);
            }

            this.AddUserParameterColumnStyles();
        }

        /// <summary>
        ///
        /// </summary>
        protected override void OnEnter(EventArgs e)
        {
            base.OnEnter(e);
        }

        protected virtual IInplaceEditorArgument ReplaceArgument(ListColumnStyle style, IInplaceEditorArgument origArgs, ComponentListItem item)
        {
            if (this.replaceArgumentDelegateDictionary.ContainsKey(style.Name) == true)
            {
                IInplaceEditorArgument resultArgs = this.replaceArgumentDelegateDictionary[style.Name](style, origArgs, item);
                if (resultArgs != null)
                {
                    return resultArgs;
                }
            }

            DropDownListInplaceEditor.Argument args = null;
            SequenceSoundBase sequenceSound = null;

            switch (style.Name)
            {
                case ProjectParameterNames.VelocityRegion.OriginalKey:
                    InstrumentListItem instrumentListItem = null;
                    PercussionListItem percussionListItem = null;
                    args = origArgs as DropDownListInplaceEditor.Argument;

                    if (item is InstrumentListItem)
                    {
                        instrumentListItem = item as InstrumentListItem;
                        args.Index = instrumentListItem.OriginalKey;
                    }

                    if (item is PercussionListItem)
                    {
                        percussionListItem = item as PercussionListItem;
                        args.Index = percussionListItem.OriginalKey;
                    }
                    return args;

                case ProjectParameterNames.SequenceSound.StartPosition:
                    sequenceSound = item.Target as SequenceSoundBase;
                    if (sequenceSound != null)
                    {
                        return new DropDownInplaceEditor.Argument
                            (GetStartPositionLabels(sequenceSound.FilePath, sequenceSound.FileType));
                    }
                    break;

                case ProjectParameterNames.FilePath:
                    OpenFilePathDialogInplaceEditor.OpenDialogHandler openDialog = null;
                    OpenFilePathDialogInplaceEditor.Argument fileDialogArgs = null;
                    fileDialogArgs = origArgs as OpenFilePathDialogInplaceEditor.Argument;
                    string filter = null;

                    if (item.Target is WaveSoundBase ||
                        item.Target is Instrument ||
                        item.Target is KeyRegion ||
                        item.Target is VelocityRegion)
                    {
                        filter = WaveFileQuester.Filter;
                        openDialog = WaveOpenFileDialog;
                    }

                    if (item.Target is StreamSoundBase)
                    {
                        filter = FormsApplication.Instance.AppConfiguration.StreamWaveFileFilter;
                        openDialog = WaveOpenFileDialog;
                    }

                    if (item.Target is StreamSoundTrackBase)
                    {
                        StreamSoundBase streamSound = item.Target.Parent as StreamSoundBase;
                        filter = (streamSound.Children.Count <= 1 ?
                                  FormsApplication.Instance.AppConfiguration.StreamWaveFileFilter :
                                  WaveFileQuester.Filter);
                        openDialog = WaveOpenFileDialog;
                    }

                    if (item.Target is SequenceSoundBase)
                    {
                        sequenceSound = item.Target as SequenceSoundBase;
                        switch (sequenceSound.FileType)
                        {
                            case SequenceSoundFileType.Text:
                                filter = SeqFileQuester.FilterText;
                                break;

                            case SequenceSoundFileType.Smf:
                                filter = SeqFileQuester.FilterSmf;
                                break;
                        }
                    }

                    if (item.Target is SoundSetBankBase)
                    {
                        filter = BankFileQuester.Filter;
                    }

                    fileDialogArgs.Filter = filter;
                    fileDialogArgs.OpenDialog = openDialog;
                    return fileDialogArgs;

                case ProjectParameterNames.GroupItem.RegisterType:
                    var groupItem = item.Target as GroupItemBase;
                    IParameterValue[] para = GetGroupItemRegisterTypeLabel(groupItem);
                    return new DropDownListInplaceEditor.Argument(para);

                // サンプルレート
                case ProjectParameterNames.SampleRate:
                    return new DropDownInplaceEditor.Argument(GetSampleRateLabel());
            }

            return null;
        }

        /// <summary>
        /// 参照可能なプレイヤーのラベルを InplaceEditorの引数として取得します。
        /// </summary>
        private IInplaceEditorArgument ReplaceArgumentReferencePlayer(ListColumnStyle style, IInplaceEditorArgument origArgs, ComponentListItem item)
        {
            var soundSetItem = item.Target as SoundSetItem;
            if (soundSetItem == null)
            {
                return new DropDownInplaceEditor.Argument(new IParameterValue[0]);
            }

            PlayerBase[] players = SoundSetUtility.GetPlayers(AddonSoundArchiveUtility.GetMainSoundSets()).ToArray();

            return new DropDownInplaceEditor.Argument(GetComponentsLabel(players, soundSetItem.SoundSet));
        }

        private IInplaceEditorArgument ReplaceArgumentReferenceBank(ListColumnStyle style, IInplaceEditorArgument origArgs, ComponentListItem item)
        {
            var soundSetItem = item.Target as SoundSetItem;
            if (soundSetItem == null)
            {
                return new DropDownInplaceEditor.Argument(new IParameterValue[0]);
            }

            SoundSetBankBase[] soundSetBanks = null;

            if (AddonSoundArchiveUtility.IsItemInAddonSoundArchive(soundSetItem) == true)
            {
                soundSetBanks = SoundSetUtility.GetSoundSetBanks(AddonSoundArchiveUtility.GetMainSoundSets()
                    .Concat(new[] { soundSetItem.SoundSet }))
                    .ToArray();
            }
            else
            {
                soundSetBanks = SoundSetUtility.GetSoundSetBanks(AddonSoundArchiveUtility.GetMainSoundSets()).ToArray();
            }

            return new DropDownInplaceEditor.Argument(GetComponentsLabel(soundSetBanks, soundSetItem.SoundSet));
        }

        private IInplaceEditorArgument ReplaceArgumentReferenceWaveArchive(ListColumnStyle style, IInplaceEditorArgument origArgs, ComponentListItem item)
        {
            var soundSetItem = item.Target as SoundSetItem;
            if (soundSetItem == null)
            {
                return new DropDownInplaceEditor.Argument(new IParameterValue[0]);
            }

            WaveArchiveBase[] waveArchives = null;

            if (AddonSoundArchiveUtility.IsItemInAddonSoundArchive(soundSetItem.SoundSet) == true)
            {
                waveArchives = SoundSetUtility.GetWaveArchives(new[] { soundSetItem.SoundSet }).ToArray();
            }
            else
            {
                waveArchives = SoundSetUtility.GetWaveArchives(AddonSoundArchiveUtility.GetMainSoundSets()).ToArray();
            }

            return new DropDownInplaceEditor.Argument(GetWaveArchiveReferenceLabel(waveArchives, soundSetItem.SoundSet));
        }

        private IInplaceEditorArgument ReplaceArgumentReferenceTargetItem(ListColumnStyle style, IInplaceEditorArgument origArgs, ComponentListItem item)
        {
            var groupItem = item.Target as GroupItemBase;
            if (groupItem == null)
            {
                return new DropDownInplaceEditor.Argument(new IParameterValue[0]);
            }

            GroupBase group = groupItem.Parent as GroupBase;
            IEnumerable<SoundSetItem> soundSetItems = null;

            if (AddonSoundArchiveUtility.IsItemInAddonSoundArchive(group.SoundSet) == true)
            {
                soundSetItems = new SoundSetItem[0];
            }
            else
            {
                soundSetItems = SoundSetUtility.GetSoundSetItems(AddonSoundArchiveUtility.GetMainSoundSets());
            }

            return new DropDownInplaceEditor.Argument(GetComponentsLablNotRegisteredOnGroup(soundSetItems, group, groupItem));
        }

        /// <summary>
        /// ユーザーパラメータの定義からカラムを登録
        /// </summary>
        private void AddUserParameterColumnStyles()
        {
            if (ProjectService != null && ProjectService.Project != null)
            {
                List<UserParameterStructureSetting> settings =
                    ProjectService.Project.UserDataStructureSettings.Settings;

                int i = 0;
                foreach (UserParameterStructureSetting setting in settings)
                {
                    int j = 0;
                    foreach (UserParameterStructure structure in setting.Structures)
                    {
                        string columnName = ListTraits.ColumnName_UserParameters[i, j];
                        ListColumnStyle style = this.CreateUserParameterColumnStyle(columnName, structure);
                        ColumnStyles.Add(style);
                        j++;
                    }
                    i++;
                }
            }
        }

        private ListColumnStyle CreateUserParameterColumnStyle(string columnName, UserParameterStructure structure)
        {
            ListSubItemArgument subArg = null;
            ListTextDrawerArgument drawArg = new ListTextDrawerArgument() { Alignment = StringAlignment.Far };

            Type type;
            string subItemName = "UserParameterAssistDropDown";
            switch (structure.StructureType)
            {
                default:
                case StructureTypes.Integer:
                    type = typeof(int);
                    if (string.IsNullOrEmpty(structure.ValueCandidate) == true)
                    {
                        subItemName = "UserParameterInteger";
                    }
                    else
                    {
                        List<IParameterValue> labelValues = new List<IParameterValue>();
                        Dictionary<string, string> valueCandidates = UserParameterUtility.GetValueCandidate(structure);
                        foreach (var pair in valueCandidates)
                        {
                            IntLabelParameterValue value = new IntLabelParameterValue();
                            value.Parse(pair.Key);
                            // ドロップダウンには "定義数値文字 (ラベル)" を使用する。A=+10 → "+10 (A)"
                            value.LabelDictionary.Add(value.Value, CommonListUtil.GetValueWithLabel(pair.Key, pair.Value));
                            labelValues.Add(value);
                        }

                        if (0 < labelValues.Count)
                        {
                            DropDownInplaceEditor.Argument arg = new DropDownInplaceEditor.Argument(labelValues.ToArray());
                            subArg = new ListSubItemArgument(drawArg, arg);
                        }
                    }
                    break;

                case StructureTypes.UInteger:
                    type = typeof(ulong);
                    if (string.IsNullOrEmpty(structure.ValueCandidate) == true)
                    {
                        subItemName = "UserParameterULong";
                    }
                    else
                    {
                        List<IParameterValue> labelValues = new List<IParameterValue>();
                        Dictionary<string, string> valueCandidates = UserParameterUtility.GetValueCandidate(structure);
                        foreach (var pair in valueCandidates)
                        {
                            ULongLabelParameterValue value = new ULongLabelParameterValue();
                            value.Parse(pair.Key);
                            // ドロップダウンには "定義数値文字 (ラベル)" を使用する。A=+10 → "+10 (A)"
                            value.LabelDictionary.Add(value.Value, CommonListUtil.GetValueWithLabel(pair.Key, pair.Value));
                            labelValues.Add(value);
                        }

                        if (0 < labelValues.Count)
                        {
                            DropDownInplaceEditor.Argument arg = new DropDownInplaceEditor.Argument(labelValues.ToArray());
                            subArg = new ListSubItemArgument(drawArg, arg);
                        }
                    }
                    break;

                case StructureTypes.Decimal:
                    type = typeof(float);
                    if (string.IsNullOrEmpty(structure.ValueCandidate) == true)
                    {
                        subItemName = "UserParameterFloat";
                    }
                    else
                    {
                        List<IParameterValue> labelValues = new List<IParameterValue>();
                        Dictionary<string, string> valueCandidates = UserParameterUtility.GetValueCandidate(structure);
                        foreach (var pair in valueCandidates)
                        {
                            FloatLabelParameterValue value = new FloatLabelParameterValue();
                            value.Parse(pair.Key);
                            // ドロップダウンには "定義数値文字 (ラベル)" を使用する。A=10.0 → "10.0 (A)"
                            value.LabelDictionary.Add(value.Value, CommonListUtil.GetValueWithLabel(pair.Key, pair.Value));
                            labelValues.Add(value);
                        }

                        if (0 < labelValues.Count)
                        {
                            DropDownInplaceEditor.Argument arg = new DropDownInplaceEditor.Argument(labelValues.ToArray());
                            subArg = new ListSubItemArgument(drawArg, arg);
                        }
                    }
                    break;

                case StructureTypes.Boolean:
                    type = typeof(bool);
                    subItemName = "UserParameterBoolean";
                    break;
            }

            return new ListColumnStyle(columnName, subItemName, type, subArg);
        }

        /// <summary>
        ///
        /// </summary>
        private bool WaveOpenFileDialog(string title, IParameterValue initialValue, OpenFilePathDialogInplaceEditor.Argument argument, out string result)
        {
            string filter = null;

            if (argument != null)
            {
                filter = argument.Filter;
            }

            string filePath = initialValue.Value != null ? initialValue.Value.ToString() : null;
            if (WaveFileQuester.QueryLoad
                (this, title, filter, filePath, out result) != false)
            {
                if (result != null)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        ///
        /// </summary>
        private SoundProjectService ProjectService
        {
            get
            {
                if (FormsApplication.Instance == null)
                {
                    return null;
                }
                else
                {
                    return FormsApplication.Instance.ProjectService;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private TextParameterValue[] OriginalKeyValues
        {
            get
            {
                return KeyNoteConverter.Notes
                    .Select(s => new TextParameterValue(s)).ToArray();
            }
        }

        /// <summary>
        ///
        /// </summary>
        private IParameterValue[] GetStartPositionLabels(string filePath, SequenceSoundFileType origFileType)
        {
            SequenceSoundFileParser parser = null;

            switch (origFileType)
            {
                case SequenceSoundFileType.Smf:
                    parser = ApplicationBase.Instance.FileParserStore.Parse<SmfParser>(filePath);
                    break;

                case SequenceSoundFileType.Text:
                    try
                    {
                        parser = ApplicationBase.Instance.FileParserStore.Parse<SequenceSoundTextParser>(filePath);
                    }
                    catch (Exception e)
                    {
                        LexerException le = e as LexerException;
                        if (le != null)
                        {
                            string message =
                                MessageResource.Message_IllegalFileFormat + "\r\n\r\n" +
                                MessageResource.Message_FileName + " " +
                                Path.GetFileName(le.FilePath) + "\r\n" +
                                MessageResource.Message_LineNo + " " + le.LineNumber + "\r\n" +
                                MessageResource.Message_IllegalStrings + " " + le.LastLine;

                            MessageBox.Show(message);
                        }

                        return new TextParameterValue[] { new TextParameterValue(String.Empty) };
                    }
                    break;

                default:
                    return new TextParameterValue[] { new TextParameterValue(String.Empty) };
            }

            return (from SequenceSoundFileParser.Label label in parser.Labels
                    select new TextParameterValue(label.Name)).ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        private IParameterValue[] GetComponentsLabel(Component[] components, SoundSet soundSet)
        {
            return components
                .Cast<SoundSetItem>()
                .OrderBy(i => i.SoundSet != soundSet)
                .Select(i => new TextParameterValue(i.Name)).ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        private IParameterValue[] GetComponentsLablNotRegisteredOnGroup(IEnumerable<SoundSetItem> soundSetItems, GroupBase group, GroupItemBase groupItem)
        {
            return soundSetItems
                .Where(i => ((i is StreamSoundBase && AppConfiguration.EnabledPrefetch == true) ||
                             i is WaveSoundSetBase ||
                             i is SequenceSoundSetBase ||
                             i is SequenceSoundBase ||
                             i is SoundSetBankBase ||
                             i is WaveArchiveBase ||
                             i is GroupBase) &&
                       (i != group) &&
                       (i == groupItem.Target || !AlreadyRegisteredOnGroup(group, i)))
                .Select(i => new GroupItemDropDownListItem(i, i.Parameters["Name"])).ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        private bool AlreadyRegisteredOnGroup(GroupBase groupItem, SoundSetItem item)
        {
            return (groupItem.Children
                        .Cast<GroupItemBase>()
                        .Where(i => i.TargetItemReference == item.Name)
                        .FirstOrDefault() != null);
        }

        private IParameterValue[] GetWaveArchiveReferenceLabel(
                                                               Component[] components,
                                                               SoundSet soundSet
                                                               )
        {
            List<IParameterValue> paraList = new List<IParameterValue>();

            IParameterValue shared =
                new TextParameterValue(MessageResource.Label_WaveArchiveReference_Shared);
            IParameterValue individual =
                new TextParameterValue(MessageResource.Label_WaveArchiveReference_Individual);

            paraList.Add(shared);
            paraList.Add(individual);

            IParameterValue[] para = GetComponentsLabel(components, soundSet);

            paraList.AddRange(para);

            return paraList.ToArray();
        }

        private IParameterValue[] GetSampleRateLabel()
        {
            int[] presets = { 8000, 11025, 16000, 22050, 24000, 32000, 44100 };

            List<IParameterValue> paraList = new List<IParameterValue>();

            presets.ToList().ForEach(i => paraList.Add(new IntParameterValue(i)));

            return paraList.ToArray();
        }

        private IParameterValue[] GetGroupItemRegisterTypeLabel(GroupItemBase groupItem)
        {
            List<IParameterValue> paraList = new List<IParameterValue>();

            if (groupItem != null)
            {
                GroupItemRegisterTypeParameterValue flag;

                if (groupItem.Target is WaveSoundSetBase)
                {
                    flag = new GroupItemRegisterTypeParameterValue(GroupItemRegisterType.All);
                    paraList.Add(flag);
                    flag = new GroupItemRegisterTypeParameterValue(GroupItemRegisterType.OnlyWaveSoundSet);
                    paraList.Add(flag);
                    flag = new GroupItemRegisterTypeParameterValue(GroupItemRegisterType.OnlyWaveArchive);
                    paraList.Add(flag);
                }
                else if (groupItem.Target is SoundSetBankBase)
                {
                    flag = new GroupItemRegisterTypeParameterValue(GroupItemRegisterType.All);
                    paraList.Add(flag);
                    flag = new GroupItemRegisterTypeParameterValue(GroupItemRegisterType.OnlyBank);
                    paraList.Add(flag);
                    flag = new GroupItemRegisterTypeParameterValue(GroupItemRegisterType.OnlyWaveArchive);
                    paraList.Add(flag);
                }
                else if ((groupItem.Target is StreamSoundBase && AppConfiguration.EnabledPrefetch == true) ||
                         groupItem.Target is WaveArchiveBase ||
                         groupItem.Target is GroupBase)
                {
                    flag = new GroupItemRegisterTypeParameterValue(GroupItemRegisterType.All);
                    paraList.Add(flag);
                }
                else
                {
                    flag = new GroupItemRegisterTypeParameterValue(GroupItemRegisterType.All);
                    paraList.Add(flag);
                    flag = new GroupItemRegisterTypeParameterValue(GroupItemRegisterType.SequenceAndBank);
                    paraList.Add(flag);
                    flag =
                     new GroupItemRegisterTypeParameterValue(GroupItemRegisterType.SequenceAndWaveArchive);
                    paraList.Add(flag);
                    flag =
                        new GroupItemRegisterTypeParameterValue(GroupItemRegisterType.BankAndWaveArchive);
                    paraList.Add(flag);
                    flag = new GroupItemRegisterTypeParameterValue(GroupItemRegisterType.OnlySequence);
                    paraList.Add(flag);
                    flag = new GroupItemRegisterTypeParameterValue(GroupItemRegisterType.OnlyBank);
                    paraList.Add(flag);
                    flag = new GroupItemRegisterTypeParameterValue(GroupItemRegisterType.OnlyWaveArchive);
                    paraList.Add(flag);
                }
            }

            return paraList.ToArray();
        }
    }
}
