﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Linq;
using System.Text;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Preset;
    using NintendoWare.SoundMaker.Framework.Utilities;

    public class BookmarkAdapter : ComponentTreeAdapter
    {
        private BookmarkFolder bookmarkRoot;

        public void AttachBookmark(BookmarkFolder bookmarkRoot)
        {
            this.bookmarkRoot = bookmarkRoot;
            Items.Clear();

            if (bookmarkRoot != null)
            {
                BookmarkTreeItem bookmarkItem = this.Create(bookmarkRoot);
                bookmarkItem.Expanded = true;
                Items.Add(bookmarkItem);
            }
        }

        public void UpdateBookmark()
        {
            this.UpdateBookmark(this.Items);

            this.Update();
        }

        public ComponentTreeItem FindBookmarkItem(Component component)
        {
            return FindTreeItem(component);
        }


        protected override ComponentTreeItem CreateTreeItem(Component component)
        {
            if (component is BookmarkFolder == false &&
                component is Bookmark == false &&
                component is BookmarkFind == false)
            {
                return null;
            }

            BookmarkTreeItem item = null;

            item = new BookmarkTreeItem(component);
            if (component is BookmarkFolder == true)
            {
                item.Expanded = (component as BookmarkFolder).Expanded;
                item.Container = true;
                item.Expander = true;
                item.CanEdit = true;
            }
            else
            {
                item.Container = false;
                item.Expander = false;
                item.CanEdit = false;
            }
            string iconName = this.GetIconName(component);
            item.ExpandIconName = iconName;
            item.CollapseIconName = iconName;

            return item;
        }

        private void UpdateBookmark(ITreeItemCollection items)
        {
            foreach (BookmarkTreeItem item in items)
            {
                this.UpdateBookmark(item.Children);

                if (item.Target is BookmarkFolder == true)
                {
                    item.Expander = (item.Children.Count != 0);
                }
                else if (item.Target is Bookmark == true)
                {
                    Bookmark bookmark = item.Target as Bookmark;
                    string iconName;
                    if (bookmark.ValidateTarget() == true)
                    {
                        iconName = this.GetIconName(bookmark.Target);
                    }
                    else
                    {
                        iconName = this.GetIconName(null);
                    }

                    item.ExpandIconName = iconName;
                    item.CollapseIconName = iconName;
                }
                else if (item.Target is BookmarkFind == true)
                {
                    string iconName = this.GetIconName(item.Target);
                    item.ExpandIconName = iconName;
                    item.CollapseIconName = iconName;
                }
            }
        }

        private BookmarkTreeItem Create(Component bookmarkFolder)
        {
            BookmarkTreeItem bookmarkItem = null;

            bookmarkItem = CreateTreeItem(bookmarkFolder) as BookmarkTreeItem;

            foreach (Component component in bookmarkFolder.Children)
            {
                bookmarkItem.Children.Add(this.Create(component));
            }

            bookmarkItem.UpdateExpander();

            return bookmarkItem;
        }

        private string GetIconName(Component component)
        {
            if (component == null)
            {
                return "IconError";
            }

            if (component is BookmarkFolder)
            {
                return "IconBookmarkFolder";
            }

            if (component is BookmarkFind)
            {
                return "IconBookmarkFind";
            }

            Component target = component;
            if (component is Bookmark)
            {
                Bookmark bookmark = component as Bookmark;
                target = bookmark.Target;
            }

            if (target is StreamSoundPack)
            {
                return "IconStreamSoundPack";
            }
            if (target is WaveSoundSetPack)
            {
                return "IconWaveSoundSetPack";
            }
            if (target is WaveSoundSetBase)
            {
                return "IconWaveSoundSet";
            }
            if (target is SequenceSoundPack)
            {
                return "IconSequenceSoundPack";
            }
            if (target is SequenceSoundSetPack)
            {
                return "IconSequenceSoundSetPack";
            }
            if (target is SequenceSoundSetBase)
            {
                return "IconSequenceSoundSet";
            }
            if (target is SoundSetBankPack)
            {
                return "IconBankPack";
            }
            if (target is SoundSetBankBase)
            {
                return "IconBank";
            }
            if (target is WaveArchivePack)
            {
                return "IconWaveArchivePack";
            }
            if (target is GroupPack)
            {
                return "IconGroupPack";
            }
            if (target is PlayerPack)
            {
                return "IconPlayerPack";
            }

            return "IconError";
        }
    }


    //
    public class BookmarkTreeItem : ComponentTreeItem
    {
        private bool expanded = false;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public BookmarkTreeItem(Component component)
            : base(component)
        {
            this.Container = false;
            this.Expander = false;

            if (component is BookmarkFolder == true)
            {
                BookmarkFolder folder = component as BookmarkFolder;
                this.expanded = folder.Expanded;

                this.Children.CollectionChanged += OnCollectionChanged;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override Component Target
        {
            get { return base.Target; }
            set
            {
                if (Target != null)
                {
                    Target.NameChanged -= OnNameChanged;
                }

                base.Target = value;

                if (Target != null)
                {
                    Target.NameChanged += OnNameChanged;
                }
            }
        }

        public override bool Expanded
        {
            get
            {
                return this.expanded;
            }
            set
            {
                this.expanded = value;
                if (this.Target is BookmarkFolder == true)
                {
                    BookmarkFolder folder = this.Target as BookmarkFolder;
                    folder.Expanded = value;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdateExpander()
        {
            this.Expander = this.Children.Count > 0 ? true : false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private MainWindow MainWindow
        {
            get { return FormsApplication.Instance.UIService.MainWindow; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private BookmarkService BookmarkService
        {
            get { return FormsApplication.Instance.BookmarkService; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnNameChanged(object sender, NameChangedEventArgs e)
        {
            ComponentTreeAdapter adapter = null;

            if ((adapter = Adapter) == null)
            {
                return;
            }

            adapter.Update();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            UpdateExpander();
        }
    }
}
