﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;

    public partial class AdsrControl
    {
        public enum AdsrEnvelopeStatus
        {
            Attack,
            Hold,
            Decay,
            Sustain,
            Release
        }

        /// <summary>
        /// AdsrEnvelope の概要の説明です。
        /// </summary>
        public class AdsrEnvelope
        {
            private readonly int[] cAttackTable = new int[127 - 109 + 1]
            {
                0, 1, 5, 14, 26, 38, 51, 63, 73, 84,
                92, 100, 109, 116, 123, 127, 132, 137, 143
            };
            private readonly int[] cDecibelSquareTable = new int[128]
            {
                -723, -722, -721, -651, -601, -562, -530, -503,
                -480, -460, -442, -425, -410, -396, -383, -371,
                -360, -349, -339, -330, -321, -313, -305, -297,
                -289, -282, -276, -269, -263, -257, -251, -245,
                -239, -234, -229, -224, -219, -214, -210, -205,
                -201, -196, -192, -188, -184, -180, -176, -173,
                -169, -165, -162, -158, -155, -152, -149, -145,
                -142, -139, -136, -133, -130, -127, -125, -122,
                -119, -116, -114, -111, -109, -106, -103, -101,
                -99,  -96,  -94,  -91,  -89,  -87,  -85,  -82,
                -80,  -78,  -76,  -74,  -72,  -70,  -68,  -66,
                -64,  -62,  -60,  -58,  -56,  -54,  -52,  -50,
                -49,  -47,  -45,  -43,  -42,  -40,  -38,  -36,
                -35,  -33,  -31,  -30,  -28,  -27,  -25,  -23,
                -22,  -20,  -19,  -17,  -16,  -14,  -13,  -11,
                -10,   -8,   -7,   -6,   -4,   -3,   -1,    0
            };

            private AdsrEnvelopeStatus status;
            private double current;
            private double attack;
            private int hold;
            private double decay;
            private int sustain;
            private double release;
            private int holdCounter;

            public AdsrEnvelope()
            {
                Attack = 127;
                Hold = 0;
                Decay = 127;
                Sustain = 127;
                Release = 127;

                current = -904;
                holdCounter = 0;
                status = AdsrEnvelopeStatus.Attack;
            }

            public void NoteOff()
            {
                status = AdsrEnvelopeStatus.Release;
            }

            public AdsrEnvelopeStatus Status
            {
                get { return status; }
            }
            public bool IsFinishedRelease
            {
                get
                {
                    if (status != AdsrEnvelopeStatus.Release) return false;
                    return (current < -904);
                }
            }
            public double Current
            {
                get { return current; }
            }
            public int Attack
            {
                set
                {
                    if (value < 109)
                    {
                        attack = 255 - value;
                    }
                    else
                    {
                        attack = cAttackTable[127 - value];
                    }
                    attack /= 256.0f;
                    attack = Math.Pow(attack, 1.0 / 5);
                }
            }
            public int Hold
            {
                set
                {
                    hold = value;
                    hold++;
                    hold *= hold;
                    hold /= 4;
                }
            }
            public int Decay
            {
                set
                {
                    decay = CalcRelease(value);
                }
            }
            public int Sustain
            {
                set
                {
                    sustain = value;
                }
            }
            public int Release
            {
                set
                {
                    release = CalcRelease(value);
                }
            }
            public void Update(int msec)
            {
                switch (status)
                {
                    case AdsrEnvelopeStatus.Attack:
                        while (msec > 0)
                        {
                            current *= attack;
                            msec--;

                            if (current > -1.0f / 32.0f)
                            {
                                current = 0.0f;
                                status = AdsrEnvelopeStatus.Hold;
                                break;
                            }
                        }
                        break;

                    case AdsrEnvelopeStatus.Hold:
                        if (msec + holdCounter < hold)
                        {
                            holdCounter += msec;
                        }
                        else
                        {
                            msec -= hold - holdCounter;
                            holdCounter = hold;
                            status = AdsrEnvelopeStatus.Decay;
                            goto case AdsrEnvelopeStatus.Decay;
                        }
                        break;

                    case AdsrEnvelopeStatus.Decay:
                        double sustainDecay = CalcDecibelSquare(sustain);

                        current -= decay * msec;
                        if (current > sustainDecay)
                        {
                            break;
                        }
                        current = sustainDecay;
                        status = AdsrEnvelopeStatus.Sustain;
                        break;

                    case AdsrEnvelopeStatus.Sustain:
                        // do nothing
                        break;

                    case AdsrEnvelopeStatus.Release:
                        current -= release * msec;
                        break;
                }
            }

            private double CalcRelease(int release)
            {
                if (release == 127) return 65535;
                if (release == 126) return (120 / 5.0f);

                if (release < 50)
                {
                    return ((release << 1) + 1) / 128.0f / 5.0f;
                }
                else
                {
                    return (60.0f / (126 - release)) / 5.0f;
                }
            }

            private int CalcDecibelSquare(int value)
            {
                return cDecibelSquareTable[value];
            }
        }

    }
}
