﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using NintendoWare.SoundFoundation.Core;
using NintendoWare.SoundFoundation.Core.Resources;
using NintendoWare.SoundFoundation.Documents;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundMaker.Framework.Configurations.Schemas;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    public partial class MainWindow
    {
        /// <summary>
        /// タブ設定を適用または抽出します。
        /// </summary>
        public class TabConfigurationApplier
        {
            private TabCtrl _tabControl;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            /// <param name="tabControl">タブコントロール。</param>
            public TabConfigurationApplier(TabCtrl tabControl)
            {
                if (null == tabControl) { throw new ArgumentNullException("tabControl"); }
                _tabControl = tabControl;
            }

            private XmlTabState TabState
            {
                get
                {
                    return (TabCtrl.ControlState.Minimized == _tabControl.State) ?
                        XmlTabState.Minimized : XmlTabState.Normal;
                }
            }

            private FormsApplication Application
            {
                get { return FormsApplication.Instance; }
            }

            private SoundProjectService ProjectService
            {
                get { return ApplicationBase.Instance.ProjectService; }
            }

            /// <summary>
            /// タブ設定を適用します。
            /// </summary>
            /// <param name="tabControl">タブコントロール。</param>
            /// <param name="mainWindow">メインウィンドウ。</param>
            /// <param name="xmlTab">タブ設定。</param>
            public static void Apply(TabCtrl tabControl, MainWindow mainWindow, XmlTab xmlTab)
            {
                new TabConfigurationApplier(tabControl).Apply(mainWindow, xmlTab);
            }

            /// <summary>
            /// タブ設定を抽出します。
            /// </summary>
            /// <param name="tabControl">タブコントロール。</param>
            /// <param name="xmlTab">抽出したデータの格納先。</param>
            public static void Extract(TabCtrl tabControl, XmlTab xmlTab)
            {
                new TabConfigurationApplier(tabControl).Extract(xmlTab);
            }

            /// <summary>
            /// タブ設定を適用します。
            /// </summary>
            /// <param name="mainWindow">メインウィンドウ。</param>
            /// <param name="xmlTab">タブ設定。</param>
            public void Apply(MainWindow mainWindow, XmlTab xmlTab)
            {
                if (null == mainWindow) { throw new ArgumentNullException("mainWindow"); }
                if (null == xmlTab) { throw new ArgumentNullException("xmlTab"); }

                if (null == xmlTab.TabPages.Items) { return; }

                mainWindow.SetCurrentTabCtrl(_tabControl);

                foreach (XmlTabPage xmlTabPage in xmlTab.TabPages.Items)
                {
                    DocumentReference docReference = null;

                    try
                    {
                        if (xmlTabPage is XmlFindTabPage)
                        {
                            XmlFindTabPage xmlFindTabPage = xmlTabPage as XmlFindTabPage;
                            AddFindTabPage(mainWindow, xmlFindTabPage);
                            continue;
                        }

                        //
                        docReference = Application.DocumentService.OpenDocument
                            (new FileResource(xmlTabPage.FilePath));
                        if (null == docReference)
                        {
                            continue;
                        }

                        //
                        if (docReference.Document is SoundSetDocument)
                        {

                            if ((from document in ProjectService.SoundSetDocuments
                                 where document.Resource.Key == xmlTabPage.FilePath
                                 select document).FirstOrDefault() == null)
                            {
                                continue;
                            }

                            XmlSoundSetTabPage xmlSoundSetTabPage = xmlTabPage as XmlSoundSetTabPage;
                            if (null == xmlSoundSetTabPage) { continue; }

                            SoundSet soundSet = (docReference.Document as SoundSetDocument).SoundSet;
                            mainWindow.AddPage(soundSet, false);

                            OpenPage(mainWindow, soundSet,
                                      xmlSoundSetTabPage.TargetItemPath,
                                      xmlSoundSetTabPage.TargetViewType.IsMultipleView());

                        }
                        else if (docReference.Document is BankDocument)
                        {
                            XmlBankTabPage xmlBankTabPage = xmlTabPage as XmlBankTabPage;

                            SoundSetBankBase soundSetBank = null;

                            if (null != xmlBankTabPage.OpenedBy && 0 < xmlBankTabPage.OpenedBy.Length)
                            {
                                soundSetBank = new TreeTracer('/').
                                                Trace(ProjectService.Project, xmlBankTabPage.OpenedBy)
                                                as SoundSetBankBase;
                            }

                            mainWindow.AddPage((docReference.Document as BankDocument).Bank, soundSetBank, false);

                        }

                    }
                    catch
                    {
                        // ドキュメントを開けなかった場合も処理を続行します
                    }
                    finally
                    {
                        if (docReference != null)
                        {
                            docReference.Close();
                        }
                    }

                }

                try
                {
                    _tabControl.SelectedIndex = xmlTab.SelectedIndex;
                }
                catch
                {
                }

                if (XmlTabState.Minimized == xmlTab.State)
                {
                    _tabControl.State = TabCtrl.ControlState.Minimized;
                }
            }

            /// <summary>
            /// タブ設定を抽出します。
            /// </summary>
            /// <param name="xmlTab">抽出したデータの格納先。</param>
            public void Extract(XmlTab xmlTab)
            {
                if (null == xmlTab) { throw new ArgumentNullException("xmlTab"); }

                List<XmlTabPage> xmlTabPages = new List<XmlTabPage>();
                foreach (CommonTabPage page in _tabControl.TabPages)
                {
                    if (page.Panel is SoundSetPanel)
                    {
                        SoundSetPanel panel = page.Panel as SoundSetPanel;

                        xmlTabPages.Add(
                            new XmlSoundSetTabPage()
                            {
                                FilePath = panel.SoundSetDocument.Resource.Key,
                                TargetViewType = XmlSoundSetViewTypeEx.FromType(panel.TargetItemType),
                                TargetItemPath = panel.TargetItemPath,
                            });
                    }
                    else if (page.Panel is BankPanel)
                    {
                        BankPanel panel = page.Panel as BankPanel;

                        xmlTabPages.Add(
                            new XmlBankTabPage()
                            {
                                FilePath = panel.BankDocument.Resource.Key,
                                OpenedBy = (null == panel.SoundSetBank) ?
                                            string.Empty :
                                            new TreeTracer('/').GetPath(
                                                panel.SoundSetBank, ProjectService.Project),
                            });
                    }
                    else if (page.Panel is FindResultPanel2)
                    {
                        FindResultPanel2 panel = page.Panel as FindResultPanel2;
                        xmlTabPages.Add(GetXmlFindTabPage(panel));
                    }
                }

                xmlTab.TabPages.Items = xmlTabPages.ToArray();
                xmlTab.SelectedIndex = _tabControl.SelectedIndex;
                xmlTab.State = TabState;
            }

            /// <summary>
            ///
            /// </summary>
            private void OpenPage(MainWindow mainWindow, SoundSet soundSet,
                                   string targetItemPath, bool isMultipleView)
            {
                if (null == mainWindow) { throw new ArgumentNullException("mainWindow"); }
                if (null == soundSet) { throw new ArgumentNullException("soundSet"); }

                Component targetItem = null;

                if (null != targetItemPath && 0 < targetItemPath.Length)
                {
                    targetItem = new TreeTracer('/').Trace(soundSet, targetItemPath) as Component;
                }

                if (null == targetItem)
                {
                    targetItem = FindItemPack(soundSet, typeof(StreamSoundPack));
                }

                mainWindow.ShowSoundSetPageByComponent(targetItem,
                    !isMultipleView ? ShowComponentMode.SingleList : ShowComponentMode.MultipleList);
            }

            /// <summary>
            ///
            /// </summary>
            private SoundSetItemPack FindItemPack(SoundSet soundSet, Type type)
            {
                if (null == type) { throw new ArgumentNullException("type"); }

                return (from SoundSetItemPack pack in soundSet.Children
                        where pack.GetType() == type
                        select pack).FirstOrDefault();
            }

            /// <summary>
            ///
            /// </summary>
            private FindArgs.TargetDocumentKinds ToTargetDocumentKind(XmlTargetDocumentKind kind)
            {
                switch (kind)
                {
                    case XmlTargetDocumentKind.SoundSet:
                        return FindArgs.TargetDocumentKinds.SoundSet;
                    case XmlTargetDocumentKind.Bank:
                        return FindArgs.TargetDocumentKinds.Bank;
                    default:
                        Debug.Assert(false, "Illegal target document kind.");
                        return FindArgs.TargetDocumentKinds.SoundSet;
                }
            }

            /// <summary>
            ///
            /// </summary>
            private XmlTargetDocumentKind ToXmlTargetDocumentKind(FindArgs.TargetDocumentKinds kind)
            {
                switch (kind)
                {
                    case FindArgs.TargetDocumentKinds.SoundSet:
                        return XmlTargetDocumentKind.SoundSet;
                    case FindArgs.TargetDocumentKinds.Bank:
                        return XmlTargetDocumentKind.Bank;
                    default:
                        Debug.Assert(false, "Illegal target document kind.");
                        return XmlTargetDocumentKind.SoundSet;
                }
            }

            /// <summary>
            ///
            /// </summary>
            private XmlFindTabPage GetXmlFindTabPage(FindResultPanel2 panel)
            {
                XmlFindTabPage xmlPage = new XmlFindTabPage();

                xmlPage.TargetDocumentKind = ToXmlTargetDocumentKind
                    (panel.TargetDocumentKind);

                xmlPage.Name = panel.TitleName;
                xmlPage.Number = panel.TitleNumber.ToString();
                xmlPage.Text = panel.Keyword;
                xmlPage.TargetParameters = panel.GetTargetParameterNames();
                xmlPage.TargetSoundSets = panel.GetTargetSoundSetNames();
                xmlPage.TargetFilters = panel.GetTargetFilterNames();

                return xmlPage;
            }

            /// <summary>
            ///
            /// </summary>
            private void AddFindTabPage(MainWindow mainWindow, XmlFindTabPage xmlTabPage)
            {
                mainWindow.AddFindTabPage
                    (ToTargetDocumentKind(xmlTabPage.TargetDocumentKind),
                      delegate (FindResultPanel2 panel)
                          {
                              panel.TitleName = xmlTabPage.Name;
                              panel.TitleNumber = int.Parse(xmlTabPage.Number);
                              panel.Keyword = xmlTabPage.Text;
                              panel.SetTargetParameterNames(xmlTabPage.TargetParameters);
                              panel.SetTargetSoundSetNames(xmlTabPage.TargetSoundSets);
                              panel.SetTargetFilterNames(xmlTabPage.TargetFilters);
                              panel.Find();
                          });
            }
        }
    }
}
