﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.Core.Collections;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Resources;
    using Schemas = NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using SFresrc = NintendoWare.SoundFoundation.Resources;
    using ToolDevelopmentKit;

    public static class HorizontalAlignmentEx
    {
        public static HorizontalAlignment ToFormsAlignment(this ListColumnTrait.HorizontalAlignment alignment)
        {
            switch (alignment)
            {
                case ListColumnTrait.HorizontalAlignment.Left:
                    return HorizontalAlignment.Left;

                case ListColumnTrait.HorizontalAlignment.Right:
                    return HorizontalAlignment.Right;

                case ListColumnTrait.HorizontalAlignment.Center:
                    return HorizontalAlignment.Center;
            }

            throw new ArgumentException("invalid horizontal alignment");
        }
    }

    /// <summary>
    /// リスト設定を適用または抽出します。
    /// </summary>
    public class ListConfigurationApplier
    {
        private ListHeaderAdapterDictionary _headerAdapters;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="headerAdapters">ヘッダアダプタディクショナリ。</param>
        public ListConfigurationApplier(ListHeaderAdapterDictionary headerAdapters)
        {
            if (null == headerAdapters) { throw new ArgumentNullException("headerAdapters"); }
            _headerAdapters = headerAdapters;
        }

        protected bool IsEffectiveColumnTrait(string columnName)
        {
            // プリフェッチ無効の場合、カラムを無効化する
            if (!AppConfiguration.EnabledPrefetch && columnName == ListTraits.ColumnName_IsPrefetchEnabled)
            {
                return false;
            }

            return ApplicationBase.Instance.Traits.ListTraits.Columns.ContainsKey(columnName);
        }

        protected ListColumnTrait GetEffectiveColumnTrait(string columnName)
        {
            if (!IsEffectiveColumnTrait(columnName))
            {
                return null;
            }

            return ApplicationBase.Instance.Traits.ListTraits.Columns[columnName];
        }

        /// <summary>
        /// リスト設定を抽出します。
        /// </summary>
        /// <param name="xmlList">抽出したデータの格納先。</param>
        public void Extract(Schemas.XmlList xmlList)
        {
            if (null == xmlList) { throw new ArgumentNullException("xmlList"); }

            if (_headerAdapters.ContainsKey(xmlList.Name) == false ||
                _headerAdapters[xmlList.Name] == null) { return; }

            ListHeaderAdapterBase headerAdapter = _headerAdapters[xmlList.Name];
            List<Schemas.XmlListColumn> xmlColumns = new List<Schemas.XmlListColumn>();

            foreach (IHeaderItem item in headerAdapter.Headers)
            {

                if (!this.IsEffectiveColumnTrait(item.Name)) { continue; }

                xmlColumns.Add(
                    new Schemas.XmlListColumn()
                    {
                        Name = item.Name,
                        Length = item.Length,
                        Visible = item.Visible,
                        LengthSpecified = true,
                    });

            }

            int index = 0;
            foreach (Schemas.XmlListColumn column in xmlList.ListColumns.ListColumn)
            {
                bool exist =
                    xmlColumns.Exists(delegate (Schemas.XmlListColumn a)
                            {
                                return a.Name == column.Name;
                            });
                if (exist == false)
                {
                    xmlColumns.Insert(index, column);
                }
                ++index;
            }

            xmlList.ListColumns.ListColumn = xmlColumns.ToArray();

            xmlList.SortKey = (null == headerAdapter.SortItem) ?
                                    string.Empty : headerAdapter.SortItem.Name;

            switch (headerAdapter.SortOrder)
            {
                case SortOrder.Ascending:
                    xmlList.SortOrder = Schemas.XmlListSortOrder.Ascending;
                    break;

                case SortOrder.Descending:
                    xmlList.SortOrder = Schemas.XmlListSortOrder.Descending;
                    break;

                default:
                    xmlList.SortOrder = Schemas.XmlListSortOrder.None;
                    break;
            }
        }

        /// <summary>
        /// リスト設定を適用します。
        /// </summary>
        /// <param name="xmlList">リスト設定。</param>
        public void Apply(Schemas.XmlList xmlList)
        {
            this.Apply(xmlList, true);
        }

        /// <summary>
        /// リスト設定を適用しますが、カラムの追加はしません。
        /// </summary>
        /// <param name="xmlList">リスト設定。</param>
        public void ApplyWithout(Schemas.XmlList xmlList)
        {
            this.Apply(xmlList, false);
        }

        /// <summary>
        /// リスト設定を適用します。
        /// </summary>
        /// <param name="xmlList">リスト設定。</param>
        private void Apply(Schemas.XmlList xmlList, bool addColumn)
        {
            if (null == xmlList) { throw new ArgumentNullException("xmlList"); }

            if (xmlList.ListColumns == null || xmlList.ListColumns.ListColumn == null)
            {
                return;
            }

            if (!_headerAdapters.ContainsKey(xmlList.Name))
            {

                if (ListTraits.ListName_Percussion == xmlList.Name)
                {
                    _headerAdapters.Add(xmlList.Name, new PercussionListHeaderAdapter());
                }
                else
                {
                    _headerAdapters.Add(xmlList.Name, new CommonListHeaderAdapter());
                }

            }
            ListHeaderAdapterBase headerAdapter = _headerAdapters[xmlList.Name];
            Dictionary<string, HeaderColumnData> headerColumns = new Dictionary<string, HeaderColumnData>();
            if (headerAdapter.Columns != null)
            {
                foreach (HeaderColumnData data in headerAdapter.Columns)
                {
                    headerColumns.Add(data.Name, data);
                }
            }

            List<HeaderColumnData> columns = new List<HeaderColumnData>();
            Dictionary<string, HeaderColumnData> columnsDic = new Dictionary<string, HeaderColumnData>();

            foreach (Schemas.XmlListColumn xmlColumn in xmlList.ListColumns.ListColumn)
            {
                if (!this.IsEffectiveColumnTrait(xmlColumn.Name)) { continue; }

                ListColumnTrait columnTrait = this.GetEffectiveColumnTrait(xmlColumn.Name);
                Ensure.Operation.ObjectNotNull(columnTrait);

                if (!xmlColumn.LengthSpecified)
                {
                    xmlColumn.Length = columnTrait.Length;
                }

                HeaderColumnData headerColumnData =
                    new HeaderColumnData()
                    {
                        Name = xmlColumn.Name,
                        Text = columnTrait.Text,
                        Length = xmlColumn.Length,
                        Alignment = columnTrait.Alignment.ToFormsAlignment(),
                        Visible = xmlColumn.Visible,
                        MinimumLength = columnTrait.MinimumLength,
                        MaximumLength = columnTrait.MaximumLength
                    };

                if (addColumn == true || headerColumns.ContainsKey(xmlColumn.Name) == true)
                {
                    columns.Add(headerColumnData);
                    columnsDic.Add(headerColumnData.Name, headerColumnData);
                }
            }

            if (headerAdapter.Columns != null)
            {
                foreach (HeaderColumnData data in headerAdapter.Columns)
                {
                    if (columnsDic.ContainsKey(data.Name) == false)
                    {
                        HeaderColumnData headerColumnData =
                            new HeaderColumnData()
                            {
                                Name = data.Name,
                                Text = data.Text,
                                Length = data.Length,
                                Alignment = data.Alignment,
                                Visible = false,
                                MinimumLength = data.MinimumLength,
                                MaximumLength = data.MaximumLength
                            };
                        columns.Add(headerColumnData);
                        columnsDic.Add(headerColumnData.Name, headerColumnData);
                    }
                }
            }

            switch (xmlList.Name)
            {
                case ListTraits.ListName_StreamSound:
                case ListTraits.ListName_WaveSoundSetItem:
                case ListTraits.ListName_SequenceSoundSetItem:
                case ListTraits.ListName_SequenceSound:
                case ListTraits.ListName_FindResult:
                case ListTraits.ListName_FindResult2:
                    UpdateUserParameterColumn(headerAdapter, ref columns);
                    break;
            }

            headerAdapter.Columns = columns.ToArray();

            if (null == xmlList.SortKey || 0 == xmlList.SortKey.Length)
            {
                headerAdapter.SortItem = null;
            }
            else
            {

                foreach (IHeaderItem item in headerAdapter.Headers)
                {
                    if (item.Name != xmlList.SortKey) { continue; }
                    headerAdapter.SortItem = item;
                    break;
                }

            }

            switch (xmlList.SortOrder)
            {
                case Schemas.XmlListSortOrder.Ascending:
                    headerAdapter.SortOrder = SortOrder.Ascending;
                    break;

                case Schemas.XmlListSortOrder.Descending:
                    headerAdapter.SortOrder = SortOrder.Descending;
                    break;

                default:
                    headerAdapter.SortOrder = SortOrder.None;
                    break;
            }

            //
            CommentColumnTextApplier.Apply(headerAdapter);
        }
#if false
        public void UpdateXmlListsUserParameters(Schemas.XmlList[] xmlLists)
        {
            foreach(Schemas.XmlList xmlList in xmlLists)
            {
                switch (xmlList.Name)
                {
                case ListTraits.ListName_StreamSound:
                case ListTraits.ListName_WaveSoundSetItem:
                case ListTraits.ListName_SequenceSoundSetItem:
                case ListTraits.ListName_SequenceSound:
                    Schemas.XmlListColumn[] listColumn = xmlList.ListColumns.ListColumn;
                    xmlList.ListColumns.ListColumn = this.UpdateXmlListColumnUserParameters(listColumn);
                    break;
                }
            }
        }

        private Schemas.XmlListColumn[] UpdateXmlListColumnUserParameters(Schemas.XmlListColumn[] listColumns)
        {
            List<Schemas.XmlListColumn> listColumnList = new List<Schemas.XmlListColumn>();

            foreach (Schemas.XmlListColumn listColumn in listColumns)
            {
                if (ListTraits.IsUserParamenters(listColumn.Name) == false)
                {
                    listColumnList.Add(listColumn);
                }
            }

            if (FormsApplication.Instance.ProjectService.Project != null)
            {
                List<UserParameterStructureSetting> settings =
                    FormsApplication.Instance.ProjectService.Project.UserDataStructureSettings.Settings;

                int i = 0;
                foreach (UserParameterStructureSetting setting in settings)
                {
                    int j = 0;
                    foreach (UserParameterStructure structure in setting.Structures)
                    {
                        Schemas.XmlListColumn xmlListColumn = new Schemas.XmlListColumn();
                        xmlListColumn.Name = ListTraits.ColumnName_UserParameters[i,j];
                        xmlListColumn.Visible = true;
                        xmlListColumn.LengthSpecified = true;
                        if (structure.StructureType == StructureTypes.Boolean)
                        {
                            xmlListColumn.Length = 32;
                        }
                        else
                        {
                            xmlListColumn.Length = 48;
                        }
                        listColumnList.Add(xmlListColumn);
                        j++;
                    }
                    i++;
                }
            }

            return listColumnList.ToArray();
        }
#endif
        private void UpdateUserParameterColumn(IHeaderSource headerAdapter, ref List<HeaderColumnData> datas)
        {
            if (headerAdapter == null)
            {
                return;
            }

            SoundProject project = FormsApplication.Instance.ProjectService.Project;
            if (project == null)
            {
                return;
            }

            List<UserParameterStructureSetting> settings = project.UserDataStructureSettings.Settings;

            Dictionary<string, string[]> userParameterLabels = new Dictionary<string, string[]>();

            int i = 0;
            foreach (UserParameterStructureSetting setting in settings)
            {
                string orgLabel = string.Empty;
                switch (i)
                {
                    case 0: orgLabel = MessageResource.HeaderText_UserParameter; break;
                    case 1: orgLabel = MessageResource.HeaderText_UserParameter1; break;
                    case 2: orgLabel = MessageResource.HeaderText_UserParameter2; break;
                    case 3: orgLabel = MessageResource.HeaderText_UserParameter3; break;
                }
                int j = 0;
                foreach (UserParameterStructure structure in setting.Structures)
                {
                    string columnName = ListTraits.ColumnName_UserParameters[i, j];
                    string[] labels = new string[2] { structure.Label, orgLabel };
                    userParameterLabels.Add(columnName, labels);
                    j++;
                }
                i++;
            }

            List<HeaderColumnData> deleteColumnDatas = new List<HeaderColumnData>();
            foreach (HeaderColumnData headerItem in datas)
            {
                if (userParameterLabels.ContainsKey(headerItem.Name) == true)
                {
                    string[] labels = userParameterLabels[headerItem.Name];
                    headerItem.Text = String.Format("{0}\r\n({1})", labels[0], labels[1]);
                }
                else if (ListTraits.IsUserParameters(headerItem.Name) == true)
                {
                    deleteColumnDatas.Add(headerItem);
                }
            }

            if (0 < deleteColumnDatas.Count)
            {
                foreach (HeaderColumnData headerItem in deleteColumnDatas)
                {
                    datas.Remove(headerItem);
                }
            }

            //
            List<HeaderColumnData> addColumnDatas = new List<HeaderColumnData>();
            foreach (string columnName in userParameterLabels.Keys)
            {
                foreach (HeaderColumnData headerItem in datas)
                {
                    if (headerItem.Name == columnName)
                    {
                        goto next;
                    }
                }
                string[] labels = userParameterLabels[columnName];
                string text = String.Format("{0}\r\n({1})", labels[0], labels[1]);

                HeaderColumnData item =
                    new HeaderColumnData()
                    {
                        Name = columnName,
                        Text = text,
                        Length = 32,
                        Alignment = HorizontalAlignment.Left,
                        Visible = true,
                        MinimumLength = 32
                    };
                addColumnDatas.Add(item);

                next:
                continue;
            }

            foreach (HeaderColumnData item in addColumnDatas)
            {
                datas.Add(item);
            }
        }
    }
}
