﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Windows.Forms.CommandHandlers
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Conversion;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.Windows;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;

    /// <summary>
    /// smf ファイルをコンバートするコマンドを処理します。
    /// </summary>
    public class QueryAndConvertSmfFileHandler : CommandHandler
    {
        private string sequenceExtention = "." + FormsApplication.Instance.Traits.IntermediateOutputTraits.TextSequenceSoundFileExtension;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="queryParameter">パラメータの問い合わせ先。</param>
        public QueryAndConvertSmfFileHandler(IQueryCommandParameter queryParameter)
            : base(queryParameter)
        {
        }

        public ISmfConverter SmfConverter
        {
            get; set;
        }

        /// <summary>
        /// コマンドハンドラが実行可能なコマンドIDを取得します。
        /// </summary>
        public override string TargetCommandID
        {
            get { return ProjectCommands.ConvertSmfFile.ID; }
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command,
                                                             IQueryCommandParameter parameters)
        {
            if (null == GetTargetProjectService(parameters))
            {
                return CommandStatus.SupportedAndVisible;
            }

            Sound[] sounds = GetTargetSounds(parameters);
            if (sounds.Length == 0) { return CommandStatus.SupportedAndVisible; }

            foreach (Sound sound in sounds)
            {
                if (sound is SequenceSoundBase)
                {
                    SequenceSoundBase seqSound = sound as SequenceSoundBase;
                    if (seqSound.FileType == SequenceSoundFileType.Text)
                    {
                        return CommandStatus.SupportedAndVisible;
                    }
                }
                else
                {
                    return CommandStatus.SupportedAndVisible;
                }
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            Sound[] targets = GetTargetSounds(parameters);
            if (0 == targets.Length) { return false; }

            string initDestinationFolderPath = Path.GetDirectoryName(targets[0].FilePath);

            ConvertSmfFileDialog dialog = new ConvertSmfFileDialog(initDestinationFolderPath);
            dialog.IsOpenDestinationFolder = FormsApplication.Instance.AppConfiguration.Options.Application.General.ConvertSmfFileDialogOpenDestinationFolder;

            if (dialog.ShowDialog() != DialogResult.OK) { return false; }

            FormsApplication.Instance.AppConfiguration.Options.Application.General.ConvertSmfFileDialogOpenDestinationFolder = dialog.IsOpenDestinationFolder;

            List<string> paths = new List<string>();

            foreach (Sound sound in targets)
            {
                string path = Path.Combine(dialog.DestinationFolderPath,
                                           Path.GetFileNameWithoutExtension(sound.FilePath) +
                                           this.sequenceExtention);
                if (paths.Contains(path) == false)
                {
                    paths.Add(path);
                }
            }
            if (ConfirtmDestinationFilesOverwrite(paths.ToArray()) == false)
            {
                return false;
            }

            try
            {
                if (Directory.Exists(dialog.DestinationFolderPath) == false)
                {
                    Directory.CreateDirectory(dialog.DestinationFolderPath);
                }

                if (ConvertSmfFiles(targets, dialog.DestinationFolderPath) == false)
                {
                    return false;
                }
            }
            catch (DirectoryNotFoundException exception)
            {
                FormsApplication.Instance.UIService.ShowMessageBox(exception.Message);
                return false;
            }
            catch (Exception)
            {
                FormsApplication.Instance.UIService.ShowMessageBox
                    (FrameworkResources.MessageResource.Message_UnknownError);
                return false;
            }

            if (dialog.IsOpenDestinationFolder)
            {
                Process.Start("explorer.exe", dialog.DestinationFolderPath);
            }

            return true;
        }

        /// <summary>
        /// 対象プロジェクトサービスを取得します。
        /// </summary>
        private SoundProjectService GetTargetProjectService(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }
            return parameters.GetParameter(CommandParameterNames.TargetComponentService)
                as SoundProjectService;
        }

        /// <summary>
        /// 対象サウンドの一覧を取得します。
        /// </summary>
        private Sound[] GetTargetSounds(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            return (from Component component in
                    (parameters.GetParameter(CommandParameterNames.TargetComponents)
                     as IEnumerable<Component>)
                    where component is Sound
                    select component as Sound).ToArray();
        }

        private bool ConvertSmfFiles(Sound[] sounds, string filePath)
        {
            if (null == sounds) { throw new ArgumentNullException("sounds"); }
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            FormsApplication.Instance.ExecuteCommand(PreviewCommands.StopAll);
            FormsApplication.Instance.ExecuteCommand(ToolCommands.StopAllSounds);

            List<string> inputFiles = new List<string>();
            List<string> outputFiles = new List<string>();

            foreach (Sound sound in sounds)
            {
                if (inputFiles.Contains(sound.FilePath) == false)
                {
                    string outputFile =
                        Path.Combine(filePath,
                                     Path.GetFileNameWithoutExtension(sound.FilePath) +
                                     this.sequenceExtention);
                    outputFiles.Add(outputFile);
                    inputFiles.Add(sound.FilePath);
                }
            }

            return SaveHelper.Save(() =>
            {
                for (int i = 0; i < inputFiles.Count(); i++)
                {
                    ConvertSmfFile(inputFiles[i], outputFiles[i]);
                }
            }, outputFiles.ToArray());
        }

        private void ConvertSmfFile(string inputFile, string outputFile)
        {
            if (null == inputFile) { throw new ArgumentNullException("inputFile"); }
            if (null == outputFile) { throw new ArgumentNullException("outputFile"); }

            this.SmfConverter.InputPath = inputFile;
            this.SmfConverter.OutputPath = outputFile;
            this.SmfConverter.TimeBase =
                ApplicationBase.Instance.ProjectService.Project.SmfConvertTimeBase;

            this.SmfConverter.Run(new ConversionContext(ApplicationBase.Instance.Traits.ConversionTraits));
        }

        private bool ConfirtmDestinationFilesOverwrite(string[] filePaths)
        {
            if (null == filePaths) { throw new ArgumentNullException("filePaths"); }

            bool showMessage = false;
            string message = FrameworkResources.MessageResource.Message_ConfirmOverwriteFiles;

            foreach (string filePath in filePaths)
            {
                if (File.Exists(filePath) == false) { continue; }

                message += "\n" + filePath;
                showMessage = true;
            }

            if (showMessage == false) { return true; }

            if (FormsApplication.Instance.UIService.ShowMessageBox(
                               message, null, AppMessageBoxButton.OKCancel,
                               AppMessageBoxImage.Question, AppMessageBoxResult.OK)
                 != AppMessageBoxResult.OK)
            {
                return false;
            }

            return true;
        }
    }
}
