﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Windows.Forms;
using NintendoWare.SoundFoundation.Commands;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundMaker.Framework.Commands;
using NintendoWare.SoundMaker.Framework.Windows.Forms;

namespace NintendoWare.SoundMaker.Framework.CommandHandlers
{
    /// <summary>
    /// 新しいウェーブサウンドを作成します。
    /// </summary>
    public class SoundListQueryAndAddWaveSoundHandler : AddSoundHandler
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="insert">挿入する場合は false、追加する場合は false。</param>
        /// <param name="queryTarget">対象の問い合わせ先。</param>
        /// <param name="defaultPlayerName">デフォルトプレイヤー名。</param>
        public SoundListQueryAndAddWaveSoundHandler(bool insert, IQueryCommandParameter queryParameter,
                                            string defaultPlayerName)
            : base(insert ? SoundListCommands.InsertWaveSound.ID : SoundListCommands.AddWaveSound.ID,
                    insert, queryParameter, defaultPlayerName)
        { }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            CommandStatus status = base.QueryStatusInternal(command, parameters);
            if (!status.IsEnabled()) { return CommandStatus.SupportedAndVisible; }

            return (GetTargetWaveSoundSet(parameters) != null) ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            var soundSetDocument = GetTargetDocument(parameters);
            CreateWaveSoundDialog form = new CreateWaveSoundDialog(Application.ProjectService, soundSetDocument, DefaultPlayerName);
            if (form.ShowDialog() != DialogResult.OK) { return false; }

            if (!Execute(parameters, form.NewSound)) { return false; }

            (Application.UIService as FormsUIService).MainWindow.ShowPageByComponent(form.NewSound);
            return true;
        }

        /// <summary>
        /// アイテムの追加先を取得します。
        /// </summary>
        /// <param name="queryTarget">対象の問い合わせ先。</param>
        /// <returns>アイテムの追加先。</returns>
        protected override Component GetInsertParentComponent(IQueryCommandParameter queryParameter)
        {
            Component component = base.GetInsertParentComponent(queryParameter);
            if (null == component) { return null; }

            if (component is WaveSoundBase)
            {
                return component.Parent;
            }

            return component;
        }

        /// <summary>
        /// アイテムの追加先を取得します。
        /// </summary>
        /// <returns>アイテムの追加先。</returns>
        private WaveSoundSetBase GetTargetWaveSoundSet(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            Component component = GetInsertParentComponent(parameters);
            if (null == component) { return null; }

            if (component is WaveSoundSetBase)
            {
                return component as WaveSoundSetBase;
            }

            if (component is WaveSoundBase)
            {
                return component.Parent as WaveSoundSetBase;
            }

            return null;
        }
    }
}
