﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Linq;
using System.Windows.Forms;
using NintendoWare.SoundFoundation.Commands;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundMaker.Framework.Commands;
using NintendoWare.SoundMaker.Framework.Resources;
using NintendoWare.SoundMaker.Framework.Windows;
using NintendoWare.SoundMaker.Framework.Windows.Forms;
using System.Collections.Generic;

namespace NintendoWare.SoundMaker.Framework.CommandHandlers
{
    /// <summary>
    /// 新しいシーケンスサウンドを作成します。
    /// </summary>
    public class SoundListQueryAndAddSequenceSoundHandler : AddSoundHandler
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="insert">挿入する場合は false、追加する場合は false。</param>
        /// <param name="queryTarget">対象の問い合わせ先。</param>
        /// <param name="defaultPlayerName">デフォルトプレイヤー名。</param>
        public SoundListQueryAndAddSequenceSoundHandler(bool insert, IQueryCommandParameter queryParameter,
                                                string defaultPlayerName)
            : base(insert ? SoundListCommands.InsertSequenceSound.ID : SoundListCommands.AddSequenceSound.ID,
                    insert, queryParameter, defaultPlayerName)
        { }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            if (Application.ProjectService.SoundSetBanks.Length <= 0)
            {

                FormsApplication.Instance.UIService.ShowMessageBox(
                      MessageResource.Message_CanNotCreateSequenceSoundNothingBank,
                      AppMessageBoxButton.OK,
                      AppMessageBoxImage.Warning);

                return false;

            }

            var soundSetDocument = GetTargetDocument(parameters);
            CreateSequenceSoundDialog form = new CreateSequenceSoundDialog(Application.ProjectService, soundSetDocument, DefaultPlayerName);
            if (form.ShowDialog() != DialogResult.OK) { return false; }

            if (!Execute(parameters, form.NewSound)) { return false; }

            (Application.UIService as FormsUIService).MainWindow.ShowPageByComponent(form.NewSound);
            return true;
        }

        /// <summary>
        /// アイテムの追加先を取得します。
        /// </summary>
        /// <param name="queryTarget">対象の問い合わせ先。</param>
        /// <returns>アイテムの追加先。</returns>
        protected override Component GetInsertParentComponent(IQueryCommandParameter queryParameter)
        {
            SoundSetDocument targetDocument = GetTargetDocument(queryParameter);
            Component targetComponent = base.GetInsertParentComponent(queryParameter);

            if (null == targetDocument) { return null; }
            if (null == targetComponent) { return null; }

            if (targetComponent is SequenceSoundSetBase)
            {
                return targetComponent;
            }

            if (targetComponent is SequenceSoundBase)
            {
                return targetComponent.Parent;
            }

            return (from Component pack in targetDocument.SoundSet.Children
                    where pack is SequenceSoundPack
                    select pack as SequenceSoundPack).FirstOrDefault();
        }
    }
}
