﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.CommandHandlers
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.FileFormats;
    using NintendoWare.SoundMaker.Framework.Projects;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;

    public class QueryAndAddStreamSoundTrackHandler : AddComponentHandler
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="insert">挿入する場合は false、追加する場合は false。</param>
        /// <param name="queryTarget">対象の問い合わせ先。</param>
        public QueryAndAddStreamSoundTrackHandler(bool insert, IQueryCommandParameter queryParameter)
            : base(insert ? ProjectCommands.InsertStreamSoundTrack.ID : ProjectCommands.AddStreamSoundTrack.ID,
                   insert, queryParameter)
        { }

        /// <summary>
        /// ストリームサウンドにトラックを追加します。
        /// </summary>
        /// <param name="document">追加先のバンクドキュメント。</param>
        /// <param name="parentItem">追加先のコンポーネント。</param>
        /// <param name="insertTargetItem">挿入位置のコンポーネント。</param>
        /// <param name="newItem">追加するバンクアイテム。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        public static bool Execute(SoundDocument document, Component parentItem,
                                   Component insertTargetItem, StreamSoundTrackBase newTrack)
        {
            if (null == document) { throw new ArgumentNullException("document"); }
            if (null == parentItem) { throw new ArgumentNullException("parentItem"); }
            if (null == newTrack) { throw new ArgumentNullException("newTrack"); }

            if (parentItem is StreamSoundBase == false)
            {
                throw new ArgumentException("parentItem is invalid.");
            }

            return AddComponentHandler.Execute(document, parentItem, insertTargetItem, newTrack);
        }

        /// <summary>
        /// ストリームサウンドにトラックを追加します。
        /// </summary>
        /// <param name="newTrack">追加するトラック。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected virtual bool Execute(IQueryCommandParameter parameters, StreamSoundTrackBase newTrack)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }
            if (null == newTrack) { throw new ArgumentNullException("newTrack"); }
            if (!(newTrack is StreamSoundTrackBase)) { throw new ArgumentException("invalid item type."); }

            SoundDocument targetDocument = GetTargetDocument(parameters);
            Component targetComponent = GetInsertParentComponent(parameters);

            return Execute(targetDocument, targetComponent,
                           GetInsertTargetComponent(parameters), newTrack);
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            CommandStatus status = base.QueryStatusInternal(command, parameters);
            if (status.IsEnabled() == false) { return CommandStatus.SupportedAndVisible; }

            SoundDocument targetDocument = GetTargetDocument(parameters);
            if (null == targetDocument) { return CommandStatus.SupportedAndVisible; }
            if (targetDocument.IsReadOnly == true) { return CommandStatus.SupportedAndVisible; }

            StreamSoundBase streamSound = GetTargetStreamSound(parameters);
            if (streamSound == null || GetInsertParentComponent(parameters) == null)
            {
                return CommandStatus.SupportedAndVisible;
            }
            if (ComponentConfiguration.Instance.StreamSoundTrackNumberMaximum <= streamSound.Children.Count)
            {
                return CommandStatus.SupportedAndVisible;
            }
            if (AppConfiguration.EnabledMultiChannelAAC == false &&
                streamSound.Children.Count == 1 &&
                (streamSound.Children[0] as StreamSoundTrackBase).IsAACFile())
            {
                return CommandStatus.SupportedAndVisible;
            }
            if (streamSound.Children.Count == 1 &&
                FileUtil.IsOpusFile((streamSound.Children[0] as StreamSoundTrackBase).FilePath) == true)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            StreamSoundBase streamSound = GetInsertParentComponent(parameters) as StreamSoundBase;
            CreateStreamSoundTrackDialog form = new CreateStreamSoundTrackDialog(this.GetFirstTrackFilePath(streamSound));

            if (form.ShowDialog() != DialogResult.OK) { return false; }

            if (!Execute(parameters, form.NewTrack)) { return false; }

            (Application.UIService as FormsUIService).MainWindow.ShowPageByComponent(form.NewTrack);
            return true;
        }

        /// <summary>
        /// アイテムの追加先を取得します。
        /// </summary>
        /// <param name="queryTarget">対象の問い合わせ先。</param>
        /// <returns>アイテムの追加先。</returns>
        protected override Component GetInsertParentComponent(IQueryCommandParameter queryParameter)
        {
            Component component = base.GetInsertParentComponent(queryParameter);
            if (null == component) { return null; }

            if (component is StreamSoundTrackBase)
            {
                return component.Parent;
            }

            return component;
        }

        /// <summary>
        /// アイテムの追加先を取得します。
        /// </summary>
        /// <returns>アイテムの追加先。</returns>
        private StreamSoundBase GetTargetStreamSound(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            Component component = GetInsertParentComponent(parameters);
            if (null == component) { return null; }

            if (component is StreamSoundBase)
            {
                return component as StreamSoundBase;
            }

            if (component is StreamSoundTrackBase)
            {
                return component.Parent as StreamSoundBase;
            }

            return null;
        }

        private string GetFirstTrackFilePath(StreamSoundBase streamSound)
        {
            StreamSoundTrackBase track = streamSound.Children[0] as StreamSoundTrackBase;

            return track.FilePath;
        }
    }
}
