﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Windows.Forms;
using NintendoWare.SoundFoundation.Commands;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundMaker.Framework.Commands;
using NintendoWare.SoundMaker.Framework.Resources;
using NintendoWare.SoundMaker.Framework.Utilities;
using NintendoWare.SoundMaker.Framework.Windows.Forms;

namespace NintendoWare.SoundMaker.Framework.CommandHandlers
{
    /// <summary>
    /// 新しいグループを作成します。
    /// </summary>
    public class QueryAndAddGroupItemHandler : AddComponentHandler
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="insert">挿入する場合は false、追加する場合は false。</param>
        /// <param name="queryTarget">対象の問い合わせ先。</param>
        public QueryAndAddGroupItemHandler(bool insert, IQueryCommandParameter queryParameter)
            : base(insert ? ProjectCommands.InsertGroupItem.ID : ProjectCommands.AddGroupItem.ID,
                    insert, queryParameter)
        { }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            var soundSetDocument = this.GetTargetDocument(parameters);
            if (AddonSoundArchiveUtility.IsItemInAddonSoundArchive(soundSetDocument.SoundSet) == true)
            {
                return CommandStatus.SupportedAndVisible;
            }

            CommandStatus status = base.QueryStatusInternal(command, parameters);
            if (!status.IsEnabled()) { return CommandStatus.SupportedAndVisible; }

            return (GetTargetGroup(parameters) != null) ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            SoundSetDocument targetDocument = GetTargetDocument(parameters);
            GroupBase targetGroup = GetTargetGroup(parameters);

            CreateGroupItemDialog form = new CreateGroupItemDialog(targetDocument, Application.ProjectService, targetGroup);

            if (!form.IsRegistableItemExisting)
            {
                ApplicationBase.Instance.UIService.ShowMessageBox(MessageResource.Message_RegistableItemNotFound);
                return false;
            }

            if (form.ShowDialog() != DialogResult.OK)
            {
                return false;
            }

            if (GroupUtility.AddToGroup(targetDocument,
                                         targetGroup,
                                         GetInsertTargetComponent(parameters) as GroupItemBase,
                                         new GroupItemBase[] { form.GroupItem }) == false)
            {
                return false;
            }

            (Application.UIService as FormsUIService).MainWindow.ShowPageByComponent(form.GroupItem);
            return true;
        }

        /// <summary>
        /// 対象バンクドキュメントを取得します。
        /// </summary>
        private new SoundSetDocument GetTargetDocument(IQueryCommandParameter parameters)
        {
            return base.GetTargetDocument(parameters) as SoundSetDocument;
        }

        /// <summary>
        /// アイテムの追加先を取得します。
        /// </summary>
        /// <returns>アイテムの追加先。</returns>
        private GroupBase GetTargetGroup(IQueryCommandParameter parameters)
        {
            Component component = GetInsertParentComponent(parameters);
            if (null == component) { return null; }

            if (component is GroupBase)
            {
                return component as GroupBase;
            }

            if (component is GroupItemBase)
            {
                return component.Parent as GroupBase;
            }

            return null;
        }
    }
}
