﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using NintendoWare.SoundFoundation.Commands;
using NintendoWare.SoundFoundation.Operations;
using NintendoWare.SoundFoundation.Parameters;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundMaker.Framework.CommandHandlers;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers
{
    ///--------------------------------------------------------------------------
    /// <summary>
    /// Bank関連"削除"ハンドラ
    /// </summary>
    public class DeleteBankPanelHandler : DeleteHandler
    {
        private CaretItemSetter _CaretItemSetter = new CaretItemSetter();

        ///--------------------------------
        /// <summary>
        /// </summary>
        public DeleteBankPanelHandler(IQueryCommandParameter queryParameter)
            : base(queryParameter)
        {
        }

        public static void RemoveRegionComponents(SoundDocument document, Component[] components)
        {
            Operation operation = null;

            foreach (Component component in components)
            {
                if (component is KeyRegion)
                {
                    RemoveComponent(document, component);
                }
                else if (component is VelocityRegion)
                {
                    VelocityRegion velocityRegion = (VelocityRegion)component;
                    KeyRegion parent = (KeyRegion)(velocityRegion.Parent);
                    int index = parent.Children.IndexOf(velocityRegion);
                    if (velocityRegion.VelocityMin == 0)
                    {
                        if (velocityRegion.VelocityMax == 127)
                        {
                            RemoveComponent(document, parent);
                        }
                        else
                        {
                            VelocityRegion vel =
                                (VelocityRegion)(parent.Children[index + 1]);
                            RemoveComponent(document, velocityRegion);
                            operation = new SetParameterOperation(vel.Parameters, ProjectParameterNames.VelocityRegion.VelocityMin, 0);
                            operation.Execute();
                            document.OperationHistory.AddOperation(operation);
                        }
                    }
                    else
                    {
                        int max = velocityRegion.VelocityMax;
                        RemoveComponent(document, velocityRegion);
                        VelocityRegion vel =
                            (VelocityRegion)(parent.Children[index - 1]);
                        operation = new SetParameterOperation(vel.Parameters, ProjectParameterNames.VelocityRegion.VelocityMax, max);
                        operation.Execute();
                        document.OperationHistory.AddOperation(operation);
                    }
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            SoundDocument document = GetTargetDocument(parameters);
            Component[] components = GetTargetComponents(parameters);

            if (null == document) { return CommandStatus.SupportedAndVisible; }
            if (null == components) { return CommandStatus.SupportedAndVisible; }

            if (document.IsReadOnly == true ||
                 components.Length <= 0 ||
                 CheckOverlapComponents(components) != false)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            SoundDocument document = GetTargetDocument(parameters);
            Component[] components = GetTargetComponents(parameters);

            if (components == null)
            {
                return false;
            }

            _CaretItemSetter.PreserveNextItem(parameters);

            if (components[0] is KeyRegion || components[0] is VelocityRegion)
            {
                {
                    try
                    {
                        document.OperationHistory.BeginTransaction();
                        RemoveRegionComponents(document, components);
                        document.OperationHistory.EndTransaction();
                    }
                    catch
                    {
                        document.OperationHistory.CancelTransaction();
                    }
                }
            }
            else
            {
                base.ExecuteInternal(command, parameters);
            }

            _CaretItemSetter.SetCaretByPreservedItem(parameters);
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static void RemoveComponent(SoundDocument document, Component component)
        {
            Component[] comp = new Component[] { component };
            Operation operation = new RemoveComponentOperation(comp);
            operation.Execute();
            document.OperationHistory.AddOperation(operation);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool CheckOverlapComponents(Component[] components)
        {
            foreach (Component component in components)
            {
                if (ContainsComponentCount(components, component) > 1)
                {
                    return true;
                }
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private int ContainsComponentCount(Component[] components, Component targetComponent)
        {
            int count = 0;

            foreach (Component component in components)
            {
                if (component == targetComponent)
                {
                    count++;
                }
            }
            return count;
        }
    }
}
