﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using NintendoWare.SoundFoundation.Core;
using NintendoWare.SoundFoundation.Core.Parameters;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundMaker.Framework;
using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Security.Cryptography;
using System.Text;

namespace NintendoWare.SoundMaker.Preview.Service
{
    public class StatisticsService
    {
        public StatisticsService()
        {
        }

        public ComponentDictionary ComponentDictionary
        {
            get; set;
        }

        public Func<Sound, string> GetHashCodeFunc
        {
            get; set;
        }

        // <summary>
        // サウンドセット単位で統計ファイル *.statistics を読み込みます。
        // </summary>
        // <param name="filePath">統計ファイル *.statistics を指定します。</param>
        // <param name="soundSet">サウンドセットを指定します。</param>
        public void Load(string filePath, SoundSet soundSet, int sequenceMaxMeasureDuration)
        {
            var config = new StatisticsConfiguration();
            config.Load(filePath);

            if (config.Statistics == null || config.Statistics.SoundList.Sound == null)
            {
                return;
            }

            foreach (var soundData in config.Statistics.SoundList.Sound)
            {
                Sound sound = this.FindSound(soundData.Name);
                if (sound == null)
                {
                    continue;
                }

                // 平均ラウドネス値を設定します。
                sound.IntegratedLoudness = soundData.IntegratedLoudness;
                if (soundData.Hash == this.GetHashCodeFunc(sound))
                {
                    sound.IntegratedLoudnessStatus = IntegratedLoudnessStatus.Measured;
                }
                else
                {
                    sound.IntegratedLoudnessStatus = IntegratedLoudnessStatus.Unmeasured;
                }
            }
        }

        // <summary>
        // サウンドセット単位で統計ファイル *.statistics を出力します。
        // </summary>
        // <param name="filePath">統計ファイル *.statistics を指定します。</param>
        // <param name="soundSet">サウンドセットを指定します。</param>
        public void Save(string filePath, SoundSet soundSet, int sequenceMaxMeasureDuration)
        {
            var config = new StatisticsConfiguration();
            config.Statistics = this.CreateStatisticsConfiguration(soundSet, sequenceMaxMeasureDuration);

            // 計測済みがある、または、計測済みがなくても *.statistics が既に存在する場合は .statistics を出力（更新）する。
            if (config.Statistics.SoundList.Sound.Count() > 0 || File.Exists(filePath) == true)
            {
                config.Save(filePath);
            }
        }

        // <summary>
        // 統計ファイル名 .statistics を返します。
        // </summary>
        // <param name="filePath">サウンドセットのファイルパスを指定します。</param>
        public string GetStatisticsFilePath(string filePath)
        {
            return filePath + ".statistics";
        }

        private XmlStatisticsConfiguration CreateStatisticsConfiguration(SoundSet soundSet, int sequenceMaxMeasureDuration)
        {
            List<XmlSound> list = new List<XmlSound>();

            foreach (Sound sound in this.GetSounds(soundSet))
            {
                if (sound.IntegratedLoudnessStatus == IntegratedLoudnessStatus.Measured)
                {
                    var xmlSound = new XmlSound()
                    {
                        Name = sound.Name,                             // サウンド名
                        IntegratedLoudness = sound.IntegratedLoudness, // 平均ラウドネス値
                        Hash = this.GetHashCodeFunc(sound),            // サウンドのハッシュ値
                    };

                    list.Add(xmlSound);
                }
            }

            var xmlStatisticsConfiguration = new XmlStatisticsConfiguration();
            xmlStatisticsConfiguration.SoundList = new XmlSoundList();
            xmlStatisticsConfiguration.SoundList.Sound = list.ToArray();

            return xmlStatisticsConfiguration;
        }

        // <summary>
        // サウンドセットに含まれるサウンドを取得します。
        // </summary>
        // <param name="component">サウンドセットを指定します。（再帰的に使用するので Component になっています）</param>
        private IEnumerable<Sound> GetSounds(Component component)
        {
            if (component is Sound == true)
            {
                yield return component as Sound;
            }

            foreach (Component child in component.Children)
            {
                foreach (var sound in this.GetSounds(child))
                {
                    yield return sound;
                }
            }

            yield break;
        }

        // <summary>
        // サウンド名からサウンドを取得します。
        // </summary>
        private Sound FindSound(string name)
        {
            Debug.Assert(this.ComponentDictionary != null);

            if (this.ComponentDictionary.Contains(name) == true)
            {
                var components = this.ComponentDictionary[name];

                // 名前の重複がない場合のみサウンドを返します。
                if (components.Length == 1 && components[0] is Sound == true)
                {
                    return components[0] as Sound;
                }
            }

            return null;
        }
    }
}
