﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using System.Threading;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Preview.Service
{
    using NintendoWare.SoundFoundation.Conversion;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Tool;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Windows.Forms;
    using NW4R.ProtocolSound;

    /// <summary>
    ///
    /// </summary>
    public static class ComponentEx
    {
        public static bool HasSndEdit(this Component component)
        {
            return component.Parameters.ContainsKey(ProjectParameterNames.SndEdit);
        }

        public static bool IsEnabledSndEdit(this Component component)
        {
            IParameterValue value = component.Parameters[ProjectParameterNames.SndEdit];
            return (bool)value.Value;
        }

        public static bool IsEnabledSndEditEnabled(this Component component)
        {
            IParameterValue value = component.Parameters[ProjectParameterNames.SndEditEnabled];
            return (bool)value.Value;
        }
    }

    /// <summary>
    /// シーケンスサウンド拡張
    /// </summary>
    public static class SequenceSoundEx
    {
        private static RealtimeEditService Sre
        {
            get
            {
                ApplicationBase app = ApplicationBase.Instance;
                Debug.Assert(app != null, "Application is not FormsApplicationCafe");
                return app.RealtimeEditService;
            }
        }

        ///
        public static uint GetConversionStartOffset(this SequenceSoundBase sequenceSound)
        {
            string name = ParameterNames.SequenceSound.StartOffset;

            if (sequenceSound.Parameters.ContainsKey(name) == false)
            {
                return uint.MaxValue;
            }

            return (uint)sequenceSound.Parameters[name].Value;
        }

        ///
        public static uint AllocateTrackFlags(this SequenceSoundBase sequenceSound)
        {
            string name = ParameterNames.SequenceSound.AllocateTrackFlags;

            if (sequenceSound.Parameters.ContainsKey(name) == false)
            {
                return uint.MaxValue;
            }

            return (uint)sequenceSound.Parameters[name].Value;
        }

        ///
        public static string BfseqFilePath(this SequenceSoundBase sequenceSound)
        {
            string name = ParameterNames.SequenceSound.BinaryFilePathForPartsConvert;

            if (sequenceSound.Parameters.ContainsKey(name) == false)
            {
                return String.Empty;
            }

            string value = sequenceSound.Parameters[name].Value.ToString();
            if (value == String.Empty)
            {
                return String.Empty;
            }

            return Sre.GetFilePath(value);
        }

        ///
        public static string[] BankNames(this SequenceSoundBase sequenceSound)
        {
            return sequenceSound.SoundSetBankReferences
                .Select(s => s.TargetName)
                .ToArray();
        }


        ///
        public static string[] BfwarFilePaths(this SequenceSoundBase sequenceSound)
        {
            string name = ParameterNames.SequenceSound.WaveArchiveBinaryFilePathsForPartsConvert;

            if (sequenceSound.Parameters.ContainsKey(name) == false)
            {
                return null;
            }

            IList<string> list = sequenceSound.Parameters[name].Value as IList<string>;
            return list
                .Select(f => Sre.GetFilePath(f))
                .ToArray();
        }
    }

    /// <summary>
    /// ウェーブサウンド拡張
    /// </summary>
    public static class WaveSoundEx
    {
        private static RealtimeEditService Sre
        {
            get
            {
                ApplicationBase app = ApplicationBase.Instance;
                Debug.Assert(app != null, "Application is not FormsApplicationCafe");
                return app.RealtimeEditService;
            }
        }

        public static string BfwsdFilePath(this WaveSoundBase waveSound)
        {
            string name = ParameterNames.WaveSound.BinaryFilePathForPartsConvert;

            if (waveSound.Parameters.ContainsKey(name) == false)
            {
                return String.Empty;
            }

            string value = waveSound.Parameters[name].Value.ToString();
            if (value == String.Empty)
            {
                return String.Empty;
            }

            return Sre.GetFilePath(value);
        }

        public static string BfwarFilePath(this WaveSoundBase waveSound)
        {
            string name = ParameterNames.WaveSound.WaveArchiveBinaryFilePathForPartsConvert;

            if (waveSound.Parameters.ContainsKey(name) == false)
            {
                return String.Empty;
            }

            string value = waveSound.Parameters[name].Value.ToString();
            if (value == String.Empty)
            {
                return String.Empty;
            }

            return Sre.GetFilePath(value);
        }

        public static uint Index(this WaveSoundBase waveSound)
        {
            string name = ParameterNames.WaveSound.Index;

            if (waveSound.Parameters.ContainsKey(name) == false)
            {
                return uint.MaxValue;
            }

            return (uint)waveSound.Parameters[name].Value;
        }
    }

    /// <summary>
    /// バンク拡張
    /// </summary>
    public static class SoundSetBankEx
    {
        private static RealtimeEditService Sre
        {
            get
            {
                ApplicationBase app = ApplicationBase.Instance;
                Debug.Assert(app != null, "Application is not FormsApplicationCafe");
                return app.RealtimeEditService;
            }
        }

        ///
        public static string BfbnkFilePath(this SoundSetBankBase bank)
        {
            string name = ParameterNames.SoundSetBank.BinaryFilePathForPartsConvert;

            if (bank.Parameters.ContainsKey(name) == false)
            {
                return null;
            }

            string value = bank.Parameters[name].Value.ToString();
            if (value == String.Empty)
            {
                return String.Empty;
            }

            return Sre.GetFilePath(value);
        }

        ///
        public static string BfwarFilePath(this SoundSetBankBase bank)
        {
            string name = ParameterNames.SoundSetBank.WaveArchiveBinaryFilePathForPartsConvert;

            if (bank.Parameters.ContainsKey(name) == false)
            {
                return null;
            }

            string value = bank.Parameters[name].Value.ToString();
            if (value == String.Empty)
            {
                return String.Empty;
            }

            return Sre.GetFilePath(value);
        }
    }

    /// <summary>
    /// ストリームサウンド拡張
    /// </summary>
    public static class StreamSoundEx
    {
        private static RealtimeEditService Sre
        {
            get
            {
                ApplicationBase app = ApplicationBase.Instance;
                Debug.Assert(app != null, "Application is not FormsApplicationCafe");
                return app.RealtimeEditService;
            }
        }

        public static ushort TrackAllocationFlags(this StreamSoundBase streamSound)
        {
            string name = ParameterNames.StreamSound.TrackAllocationFlags;

            if (streamSound.Parameters.ContainsKey(name) == false)
            {
                return 0;
            }

            return Convert.ToUInt16(streamSound.Parameters[name].Value);
        }

        ///
        public static ushort TotalChannelCount(this StreamSoundBase streamSound)
        {
            string name = ParameterNames.StreamSound.TotalChannelCount;

            if (streamSound.Parameters.ContainsKey(name) == false)
            {
                return 0;
            }

            return Convert.ToUInt16(streamSound.Parameters[name].Value);
        }

        ///
        public static string BfstmFilePath(this StreamSoundBase streamSound)
        {
            string name = ParameterNames.StreamSound.BinaryFilePathForPartsConvert;

            if (streamSound.Parameters.ContainsKey(name) == false)
            {
                return String.Empty;
            }

            string value = streamSound.Parameters[name].Value.ToString();
            if (value == String.Empty)
            {
                return String.Empty;
            }

            return Sre.GetFilePath(value);
        }

        /// <summary>
        ///
        /// </summary>
        public static string BfstpFilePath(this StreamSoundBase streamSound)
        {
            string name = ParameterNames.StreamSound.PrefetchBinaryFilePathForPartsConvert;

            if (streamSound.Parameters.ContainsKey(name) == false)
            {
                return String.Empty;
            }

            string value = streamSound.Parameters[name].Value.ToString();
            if (value == String.Empty)
            {
                return String.Empty;
            }

            return Sre.GetFilePath(value);
        }

        /// <summary>
        ///
        /// </summary>
        public static bool GetHasLoop(this StreamSoundBase streamSound)
        {
            string key = ParameterNames.StreamSound.HasLoop;

            if (streamSound.Parameters.ContainsKey(key) == false)
            {
                streamSound.Parameters.AddValue(key, new BoolParameterValue(false));
            }

            return (bool)streamSound.Parameters[key].Value;
        }

        /// <summary>
        ///
        /// </summary>
        public static uint GetLoopStartFrame(this StreamSoundBase streamSound)
        {
            return GetUIntValue(streamSound, ParameterNames.StreamSound.LoopStartFrame);
        }

        /// <summary>
        ///
        /// </summary>
        public static uint GetLoopEndFrame(this StreamSoundBase streamSound)
        {
            return GetUIntValue(streamSound, ParameterNames.StreamSound.LoopEndFrame);
        }

        /// <summary>
        ///
        /// </summary>
        private static uint GetUIntValue(StreamSoundBase streamSound, string key)
        {
            if (streamSound.Parameters.ContainsKey(key) == false)
            {
                streamSound.Parameters.AddValue(key, new UIntParameterValue(0));
            }

            return (uint)streamSound.Parameters[key].Value;
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class ParameterNames
    {
        /// <summary>
        ///
        /// </summary>
        public class StreamSound
        {
            public const string TrackAllocationFlags = "StreamSound_TrackAllocationFlags";
            public const string TotalChannelCount = "StreamSound_TotalChannelCount";
            public const string ContainerType = "StreamSound_ContainerType";
            public const string BinaryFilePathForPartsConvert = "StreamSound_BinaryFilePathForPartsConvert";
            public const string PrefetchBinaryFilePathForPartsConvert = "StreamSound_PrefetchBinaryFilePathForPartsConvert";
            public const string HasLoop = "StreamSound_HasLoop";
            public const string LoopStartFrame = "StreamSound_LoopStartFrame";
            public const string LoopEndFrame = "StreamSound_LoopEndFrame";
        }

        /// <summary>
        ///
        /// </summary>
        public class WaveSound
        {
            public const string Index = "WaveSound_Index";
            public const string BinaryFilePathForPartsConvert = "WaveSound_BinaryFilePathForPartsConvert";
            public const string WaveArchiveBinaryFilePathForPartsConvert = "WaveSound_WaveArchiveBinaryFilePathForPartsConvert";
        }

        /// <summary>
        ///
        /// </summary>
        public class SequenceSound
        {
            public const string StartOffset = "SequenceSound_StartOffset";
            public const string AllocateTrackFlags = "SequenceSound_AllocateTrackFlags";
            public const string BinaryFilePathForPartsConvert = "SequenceSound_BinaryFilePathForPartsConvert";
            public const string BankBinaryFilePathsForPartsConvert = "SequenceSound_BankBinaryFilePathsForPartsConvert";
            public const string WaveArchiveBinaryFilePathsForPartsConvert = "SequenceSound_WaveArchiveBinaryFilePathsForPartsConvert";
        }

        /// <summary>
        ///
        /// </summary>
        public class SoundSetBank
        {
            public const string BinaryFilePathForPartsConvert = "SoundSetBank_BinaryFilePathForPartsConvert";
            public const string WaveArchiveBinaryFilePathForPartsConvert = "SoundSetBank_WaveArchiveBinaryFilePathForPartsConvert";
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class SndEdit
    {
        /// <summary>
        /// 編集対象にする/しない
        /// </summary>
        public static bool SetSndEdit(IAdapterManipulator adapter, Component targetComponent, string name, object value)
        {
            Debug.Assert(name == ProjectParameterNames.SndEdit, "Name is not SndEdit");

            targetComponent.Parameters[name].Value = value;

            return true;
        }

        /// <summary>
        /// 編集対象の有効化/無効化
        /// </summary>
        public static bool SetSndEditEnabled(IAdapterManipulator adapter, Component targetComponent, string name, object value)
        {
            Debug.Assert(name == ProjectParameterNames.SndEditEnabled, "Name is not SndEditEnabled");
            targetComponent.Parameters[name].Value = value;

            return true;
        }

        /// <summary>
        ///
        /// </summary>
        public static IParameterValue GetSndEditState(Component component, string name, ref bool cancel)
        {
            RealtimeEditService service = Application.RealtimeEditService;
            RealtimeEditService.MonitoringState state = service.GetMonitoringState(component);

            switch (state)
            {
                case RealtimeEditService.MonitoringState.Converting:
                    return new IntParameterValue((int)SndEditState.Converting);

                case RealtimeEditService.MonitoringState.Transferring:
                    return new IntParameterValue((int)SndEditState.Transferring);

                case RealtimeEditService.MonitoringState.Completed:
                    return new IntParameterValue((int)SndEditState.Completed);

                case RealtimeEditService.MonitoringState.Error:
                    return new IntParameterValue((int)SndEditState.Error);

                case RealtimeEditService.MonitoringState.Deleting:
                    return new IntParameterValue((int)SndEditState.Deleting);

                default:
                    return new IntParameterValue((int)SndEditState.Invalid);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public enum SndEditState : int
        {
            Invalid = -1,
            Converting = 0,
            Transferring = 1,
            Completed = 2,
            Error = 3,
            Deleting = 4,
        }

        /// <summary>
        ///
        /// </summary>
        private static FormsApplication Application
        {
            get
            {
                return FormsApplication.Instance;
            }
        }
    }
}
