﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using System.Threading;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Preview.Service
{
    using NintendoWare.SoundFoundation.Conversion;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.FileManagement;
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Sre;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Tool;
    using NintendoWare.SoundMaker.Framework.Utilities;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Preview.Service;
    using NintendoWare.SoundMaker.Windows.Forms;
    using NW4R.ProtocolSound;

    using FR = NintendoWare.SoundMaker.Framework.Resources;

    /// <summary>
    ///
    /// </summary>
    public partial class RealtimeEditService
    {
        private const int MaxGetItemNameCount = 3;

        private static readonly HashAlgorithm hashAlgorithm = new SHA1CryptoServiceProvider();

        private bool enabled = false;
        private SoundProjectService projectService = null;
        private ISoundProjectConvertService projectConvertService = null;
        private BankServiceManager bankServiceManager = null;
        private List<BankService> bankServices = new List<BankService>();

        private FormsBasedInvoker invoker = null;
        private BackgroundPartsConvertService convertService = null;

        private MonitoringManager monitoringManager = null;

        ///
        private Dictionary<Component, HashCodeSet> hashCodeDictionary = new Dictionary<Component, HashCodeSet>();

        private Dictionary<string, string> filePathDictionary = new Dictionary<string, string>();

        private string preprocessedTag = string.Empty;

        private FileWatcherComponentFilePaths fileWatcherComponentFilePaths;
        private FileWatcherBankWaveFilePaths fileWatcherBankWaveFilePaths;
        private List<Component> addEventedBank = new List<Component>();

        private UpdateTimer updateTimer;
        private IAsyncResult asyncResult = null;

        /// <summary>
        ///
        /// </summary>
        public enum ResultTypes
        {
            True,
            False,

            ItemNotFound,

            NotInitialized,
            OutOfMemory,
            NotImplemented,
            CacheOverflow,

            TimeOut,
            SendError,
            RecieveError,
        }

        ///
        public class TargetMachineInformation
        {
            public uint MemoryMax { get; set; }
            public uint MemoryUsage { get; set; }
            public uint ItemCountMax { get; set; }
            public uint ItemCount { get; set; }
            public uint FileCountMax { get; set; }
            public uint FileCount { get; set; }

            public bool OutOfMemory { get; set; }
            public bool ItemOverflow { get; set; }
            public bool FileOverflow { get; set; }
        }

        /// <summary>
        ///
        /// </summary>
        public void SetTargetMachineInformation(uint memoryMax, uint memoryUsage, uint itemCountMax, uint itemCount, uint fileCountMax, uint fileCount, bool isOutOfMemory, bool isItemOverflow, bool isFileOverflow)
        {
            TargetMachineInformation info = new TargetMachineInformation();

            info.MemoryMax = memoryMax;
            info.MemoryUsage = memoryUsage;
            info.ItemCountMax = itemCountMax;
            info.ItemCount = itemCount;
            info.FileCountMax = fileCountMax;
            info.FileCount = fileCount;

            info.OutOfMemory = isOutOfMemory;
            info.ItemOverflow = isItemOverflow;
            info.FileOverflow = isFileOverflow;

            this.invoker.BeginInvoke(() => OnSetTargetMachineInformation(info));
        }

        /// <summary>
        ///
        /// </summary>
        public void BeginItemCache(string name)
        {
        }

        /// <summary>
        ///
        /// </summary>
        public void EndItemCache(string name, ResultTypes result)
        {
            Debug.WriteLine("EndItemCache(" + name + ", " + result + ") in");

            if (result != ResultTypes.True)
            {
                this.invoker.BeginInvoke(() => this.ClearSoundEditInfo(name));
            }

            this.invoker.BeginInvoke(() => this.monitoringManager.Update(name, result == ResultTypes.True));

            // 再転送要求を全て実行します。
            this.invoker.BeginInvoke(ExecuteAllConvertQuery);
        }

        /// <summary>
        /// SoundEditInfoが作成された時に呼ばれます。
        /// </summary>
        public void CreatedSoundEditInfo(SoundEditInfo info)
        {
            if (SoundEditInfoCreated != null)
            {
                SoundEditInfoCreated(info);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private enum AllItemsActions
        {
            None = 0,
            Update = 1,
            Remove = 2,
        }

        private AllItemsActions allItemsAction = AllItemsActions.None;

        public int AllItemsAction
        {
            get
            {
                return (int)this.allItemsAction;
            }
        }

        public void ResetAllItemsAction()
        {
            this.allItemsAction = AllItemsActions.None;
        }

        /// <summary>
        ///
        /// </summary>
        public delegate void MonitoringTargetEventHandler(Component[] components);
        public event MonitoringTargetEventHandler MonitoringTargetStateChanged;

        public delegate void SoundEditInfoEventHandler(SoundEditInfo info);
        public event SoundEditInfoEventHandler SoundEditInfoCreated;

        /// 実機に関する情報です。
        public delegate void TargetMachineInformationChangedHandler(TargetMachineInformation info);
        public event TargetMachineInformationChangedHandler TargetMachineInformationChanged;

        public void RequestTransfer(Component component)
        {
            ConvertedMonitoringTargetInfo(component);
            PushSoundEditInfo(component);
            SetNeedTransfer(component);
        }

        public void RequestConvert(Component component)
        {
            if (component is SoundSetBankBase == true)
            {
                string bankFilePath = (component as SoundSetBankBase).FilePath;
                Bank bank = this.GetBank(bankFilePath);

                if (bank != null)
                {
                    SoundSetBankBase[] soundSetBanks = this.GetSoundSetBanks(bank).Cast<SoundSetBankBase>().ToArray();
                    this.SetModified(bank);
                    foreach (SoundSetBankBase soundSetBank in soundSetBanks)
                    {
                        // 現時点でのハッシュ値を保存します。
                        SetModified(soundSetBank);
                    }

                    this.convertService.Send(soundSetBanks, bank);

                    return;
                }
            }

            if (component is SoundSetItem == true)
            {
                // 現時点でのハッシュ値を保存します。
                this.SetModified(component);
                // コンバートサービスにコンバートが必要なcomponentを渡します。
                SoundSetItem soundSetItem = component as SoundSetItem;
                this.convertService.Send(soundSetItem);
            }
        }

        /// <summary>
        /// 編集対象の状態を表します。
        /// </summary>
        public enum MonitoringState
        {
            None = 0,
            Converting = 1,
            Transferring = 2,
            Completed = 3,
            Error = 4,
            Deleting = 5,
        }

        /// <summary>
        /// 編集対象の状態を取得します。
        /// </summary>
        public MonitoringState GetMonitoringState(Component component)
        {
            return this.monitoringManager.GetMonitoringState(component);
        }

        /// <summary>
        /// 編集対象の状態を設定します。
        /// </summary>
        private void SetMonitoringState(Component component, MonitoringState state)
        {
            this.monitoringManager.SetMonitoringState(component, state);
        }

        private void OnMonitoringTargetStateChanged(Component[] components)
        {
            if (MonitoringTargetStateChanged != null)
            {
                MonitoringTargetStateChanged(components);
            }
        }

        private void UpdateTimerCallback()
        {
            if (this.asyncResult == null || this.asyncResult.IsCompleted == true)
            {
                this.asyncResult = this.invoker.BeginInvoke(() => this.monitoringManager.Execute());
            }
        }

        private enum EditAction { Add, Remove, OldNameRemove, Modify, Disabled, Updating, Removing, Updated, Removed, Transfer, NoAction };
        /// <summary>
        /// 編集対象を管理します。
        /// </summary>
        private class MonitoringManager
        {
            private RealtimeEditService realtimeEditService;
            private MonitoringStatusManager statusManager;
            private MonitoringInfoManager infoManager;
            private List<Component> viewList;

            private ChangedMonitoringTarget changedMonitoringTarget;

            public event MonitoringTargetEventHandler MonitoringTargetStateChanged;

            public event MonitoringTargetEventHandler MonitoringTargetAdded;
            public event MonitoringTargetEventHandler MonitoringTargetRemoved;

            public MonitoringManager(RealtimeEditService service)
            {
                this.realtimeEditService = service;
                this.statusManager = new MonitoringStatusManager();
                this.infoManager = new MonitoringInfoManager(this, this.statusManager);
                this.viewList = new List<Component>();
                this.changedMonitoringTarget = new ChangedMonitoringTarget();

                this.statusManager.MonitoringTargetStateChanged += OnMonitoringTargetStateChanged;
            }

            // 編集対象にします。
            public void Add(Component component)
            {
                if (this.viewList.Contains(component) == false)
                {
                    this.viewList.Add(component);
                }
                this.statusManager.Add(component);
                this.infoManager.Add(component, this.IsConnected());
            }

            // 編集対象から外します。
            public void Remove(Component component)
            {
                this.infoManager.Remove(component, this.IsConnected());
            }

            // 編集対象の編集があれば呼び出します。
            public void Modify(Component component)
            {
                this.statusManager.Add(component);
                this.infoManager.Modify(component, this.IsConnected());
            }

            // 編集対象の名前変更があれば呼び出します。
            public void Rename(Component component, string oldName)
            {
                this.statusManager.Add(component);
                this.infoManager.Rename(component, oldName, this.IsConnected());
            }

            // 編集対象を転送します。
            public void Transfer(Component component)
            {
                this.statusManager.Add(component);
                this.infoManager.Transfer(component, this.IsConnected());
            }

            public void Clear(Component component)
            {
                this.infoManager.Clear(component);
            }

            public void Clear()
            {
                this.statusManager.Clear();
                this.infoManager.Clear();
            }

            public void Initialize()
            {
                this.statusManager.Initialize();
                this.infoManager.Initialize();
                this.changedMonitoringTarget.Clear();
            }

            public void GetChangedMonitoringTargets(List<string> updateList, List<string> removeList, int maxPopCount)
            {
                this.changedMonitoringTarget.GetTargets(updateList, removeList, maxPopCount);
            }

            private void Update()
            {
                this.infoManager.Update();
            }

            public void Update(string name, bool result)
            {
                Component component;

                switch (this.infoManager.Update(name, out component))
                {
                    case EditAction.NoAction:
                        break;

                    case EditAction.Updated:
                        this.statusManager.Update(component, result);
                        OnMonitoringTargetAdded(new Component[] { component });
                        break;

                    case EditAction.Removed:
                        if (component != null)
                        {
                            if (component.IsEnabledSndEdit() == false)
                            {
                                this.statusManager.Remove(component);
                            }
                            else if (component.IsEnabledSndEditEnabled() == false ||
                                     component.IsEnabled == false)
                            {
                                this.statusManager.SetState(component, MonitoringState.None);
                            }
                            OnMonitoringTargetRemoved(new Component[] { component });
                        }
                        break;

                    default:
                        Debug.Assert(false, "Illegal EditAction.");
                        break;
                }
            }

            public void Execute()
            {
                this.Update();
                this.infoManager.Execute(this.changedMonitoringTarget);
            }

            public ConvertResult CheckNeedConvert(Component component)
            {
                if (component is SoundSetBankBase == true)
                {
                    SoundSetBankBase soundSetBank = component as SoundSetBankBase;
                    Bank bank = this.realtimeEditService.GetBank(soundSetBank.FilePath);
                    if (bank != null)
                    {
                        return this.realtimeEditService.CheckNeedConvertSoundSetBankDependsOnBank(bank);
                    }
                }

                return this.realtimeEditService.CheckNeedConvert(component);
            }

            public bool IsModified(Component component)
            {
                return this.realtimeEditService.IsModified(component);
            }

            public bool IsNeedTransfer(Component component)
            {
                return this.realtimeEditService.IsNeedTransfer(component);
            }

            public void RequestTransfer(Component component)
            {
                this.realtimeEditService.RequestTransfer(component);
            }

            public void RequestConvert(Component component)
            {
                this.realtimeEditService.RequestConvert(component);
            }

            public void SetConverted(Component component, bool converted)
            {
                this.infoManager.SetConverted(component, converted);
            }

            public Component[] GetMonitoringTargets()
            {
                Component[] components = this.infoManager.GetMonitoringTargets();
                // 対象 components を this.viewList の並びにする
                // this.viewList にないものは後ろに追加する。
                List<Component> targets = this.viewList.Where(a => components.Contains(a) == true).ToList();
                List<Component> addComponents = components.Where(a => viewList.Contains(a) == false).ToList();
                targets.AddRange(addComponents);
                this.viewList = targets; // 並びを更新

                return this.viewList.ToArray();
            }

            public MonitoringState GetMonitoringState(Component component)
            {
                if (this.IsConnected() == true)
                {
                    return GetMonitoringStatus(component).State;
                }

                return MonitoringState.None;
            }

            public void SetMonitoringState(Component component, MonitoringState state)
            {
                this.statusManager.SetState(component, state);
            }

            private MonitoringStatus GetMonitoringStatus(Component component)
            {
                return this.statusManager.GetMonitoringStatus(component);
            }

            private bool IsConnected()
            {
                return this.realtimeEditService.Enabled;
            }

            private void OnMonitoringTargetStateChanged(Component[] components)
            {
                if (MonitoringTargetStateChanged != null)
                {
                    MonitoringTargetStateChanged(components);
                }
            }

            private void OnMonitoringTargetAdded(Component[] components)
            {
                if (MonitoringTargetAdded != null)
                {
                    MonitoringTargetAdded(components);
                }
            }

            private void OnMonitoringTargetRemoved(Component[] components)
            {
                if (MonitoringTargetRemoved != null)
                {
                    MonitoringTargetRemoved(components);
                }
            }
        }

        /// <summary>
        /// 編集対象のアクションタスクを管理します。
        /// </summary>
        private class MonitoringInfoManager
        {
            MonitoringManager monitoringManager;
            MonitoringStatusManager statusManager;
            Dictionary<Component, MonitoringInfo> dictionary;

            public MonitoringInfoManager(MonitoringManager monitoringManager, MonitoringStatusManager statusManager)
            {
                this.monitoringManager = monitoringManager;
                this.statusManager = statusManager;
                this.dictionary = new Dictionary<Component, MonitoringInfo>();
            }

            public void Add(Component component, bool isConnected)
            {
                MonitoringInfo info = this.GetMonitoringInfo(component);

                if (isConnected == true)
                {
                    if (component.IsEnabled == true)
                    {
                        info.Add(EditAction.Add, component.Name);
                    }

                    MonitoringState state = MonitoringState.None;
                    if (info.Enabled == true)
                    {
                        state = this.EditAction2MonitoringState(info.GetLastEditAction());
                    }
                    this.SetMonitoringState(component, state);
                }
            }

            public void Remove(Component component, bool isConnected)
            {
                MonitoringInfo info = this.GetMonitoringInfo(component);

                if (isConnected == true)
                {
                    info.Add(EditAction.Remove, component.Name);

                    MonitoringState state = MonitoringState.None;
                    if (info.Enabled == true)
                    {
                        state = this.EditAction2MonitoringState(info.GetLastEditAction());
                    }

                    this.SetMonitoringState(component, state);
                }
            }

            public void Modify(Component component, bool isConnected)
            {
                MonitoringInfo info = this.GetMonitoringInfo(component);

                if (isConnected == true)
                {
                    if (component.IsEnabled == true &&
                        component.IsEnabledSndEditEnabled() == true)
                    {
                        info.Add(EditAction.Modify, component.Name);
                    }

                    MonitoringState state = MonitoringState.None;
                    if (info.Enabled == true)
                    {
                        state = this.EditAction2MonitoringState(info.GetLastEditAction());
                    }

                    this.SetMonitoringState(component, state);
                }
            }

            public void Rename(Component component, string oldName, bool isConnected)
            {
                MonitoringInfo info = this.GetMonitoringInfo(component);

                if (isConnected == true)
                {
                    if (component.IsEnabled == true &&
                        component.IsEnabledSndEditEnabled() == true)
                    {
                        info.Add(EditAction.OldNameRemove, oldName);
                        info.Add(EditAction.Add, component.Name);
                    }

                    MonitoringState state = MonitoringState.None;
                    if (info.Enabled == true)
                    {
                        state = this.EditAction2MonitoringState(info.GetLastEditAction());
                    }

                    this.SetMonitoringState(component, state);
                }
            }

            public void Transfer(Component component, bool isConnected)
            {
                MonitoringInfo info = this.GetMonitoringInfo(component);

                if (isConnected == true)
                {
                    if (component.IsEnabledSndEditEnabled() == true &&
                        component.IsEnabled == true)
                    {
                        info.Add(EditAction.Transfer, component.Name);
                    }

                    MonitoringState state = MonitoringState.None;
                    if (info.Enabled == true)
                    {
                        state = this.EditAction2MonitoringState(info.GetLastEditAction());
                    }

                    this.SetMonitoringState(component, state);
                }
            }

            public void SetConverted(Component component, bool converted)
            {
                this.GetMonitoringInfo(component).Converted = converted;
            }

            public void Initialize()
            {
                foreach (MonitoringInfo info in this.GetMonitoringInfos())
                {
                    info.Initialize();
                }
            }

            public void Clear()
            {
                this.dictionary.Clear();
            }

            public void Clear(Component component)
            {
                MonitoringInfo info = this.GetMonitoringInfo(component);
                info.Clear();
            }

            public List<Component> Update()
            {
                List<Component> list = new List<Component>();

                foreach (MonitoringInfo info in this.GetMonitoringInfos())
                {
                    if (info.Update() == true)
                    {
                        switch (info.EditAction)
                        {
                            case EditAction.Remove:
                            case EditAction.OldNameRemove:
                                MonitoringState state = this.EditAction2MonitoringState(info.EditAction);
                                this.SetMonitoringState(info.Target, state);
                                break;

                            default:
                                list.Add(info.Target);
                                break;
                        }
                    }
                }

                return list;
            }

            public EditAction Update(string name, out Component component)
            {
                EditAction editAction;
                component = null;

                foreach (MonitoringInfo info in this.dictionary.Values.ToArray())
                {
                    if (info.Update(name, out editAction) == true)
                    {
                        component = info.Target;
                        return editAction;
                    }
                }

                return EditAction.NoAction;
            }

            public void Execute(ChangedMonitoringTarget changedMonitoringTarget)
            {
                foreach (MonitoringInfo info in this.GetMonitoringInfos())
                {
                    info.Execute(changedMonitoringTarget);
                }
            }

            public ConvertResult CheckNeedConvert(Component component)
            {
                return this.monitoringManager.CheckNeedConvert(component);
            }

            public void RequestTransfer(Component component)
            {
                this.monitoringManager.RequestTransfer(component);
            }

            public void RequestConvert(Component component)
            {
                this.monitoringManager.RequestConvert(component);
            }

            public bool IsModified(Component component)
            {
                return this.monitoringManager.IsModified(component);
            }

            public bool IsNeedTransfer(Component component)
            {
                return this.monitoringManager.IsNeedTransfer(component);
            }

            public Component[] GetMonitoringTargets()
            {
                List<Component> list = new List<Component>();

                foreach (Component component in this.dictionary.Keys.ToArray())
                {
                    if (this.dictionary[component].Enabled == true ||
                        component.IsEnabledSndEdit() == true)
                    {
                        list.Add(component);
                    }
                }

                return list.ToArray();
            }

            public void SetMonitoringState(Component component, MonitoringState state)
            {
                this.statusManager.SetState(component, state);
            }

            private MonitoringInfo GetMonitoringInfo(Component component)
            {
                if (this.dictionary.ContainsKey(component) == false)
                {
                    this.dictionary[component] = new MonitoringInfo(this, component);
                }

                return this.dictionary[component];
            }

            private MonitoringInfo[] GetMonitoringInfos()
            {
                return this.dictionary.Values.ToArray();
            }

            private MonitoringState EditAction2MonitoringState(EditAction editAction)
            {
                MonitoringState state = MonitoringState.None;

                switch (editAction)
                {
                    case EditAction.Removed:
                        state = MonitoringState.None;
                        break;

                    case EditAction.Add:
                    case EditAction.Modify:
                    case EditAction.OldNameRemove:
                        state = MonitoringState.Converting;
                        break;

                    case EditAction.Remove:
                    case EditAction.Disabled:
                    case EditAction.Removing:
                        state = MonitoringState.Deleting;
                        break;

                    case EditAction.Transfer:
                    case EditAction.Updating:
                        state = MonitoringState.Transferring;
                        break;

                    case EditAction.Updated:
                        state = MonitoringState.Completed;
                        break;

                    default:
                        Debug.Assert(false, "Illegal EditAction.");
                        break;
                }

                return state;
            }
        }

        /// <summary>
        /// 編集対象のキュー等を処理します。
        /// </summary>
        private class MonitoringInfo
        {
            private MonitoringInfoManager monitoringInfoManager;
            private MonitoringTargetInfo targetInfo;
            private List<MonitoringTargetInfo> targetInfoQueue;
            private bool tryConverted;

            public MonitoringInfo(MonitoringInfoManager monitoringInfoManager, Component component)
            {
                this.monitoringInfoManager = monitoringInfoManager;
                this.targetInfo = null;
                Target = component;
                this.targetInfoQueue = new List<MonitoringTargetInfo>();
                this.tryConverted = false;
                Converted = false;
            }

            public Component Target
            {
                get; private set;
            }

            public EditAction EditAction
            {
                get
                {
                    Debug.Assert(this.targetInfo != null, "targetInfo is null.");
                    return this.targetInfo.EditAction;
                }
                set
                {
                    Debug.Assert(this.targetInfo != null, "targetInfo is null.");
                    this.targetInfo.EditAction = value;
                }
            }

            public bool Converted
            {
                get; set;
            }

            public bool Enabled
            {
                get
                {
                    if (this.targetInfo == null && this.targetInfoQueue.Count() == 0)
                    {
                        return false;
                    }

                    return true;
                }
            }

            public void Initialize()
            {
                lock (this.targetInfoQueue)
                {
                    this.targetInfo = null;
                    this.targetInfoQueue.Clear();
                    this.tryConverted = false;
                    Converted = false;
                }
            }

            public void Clear()
            {
                this.targetInfo = null;
                this.tryConverted = false;
                Converted = false;
            }

            public void Add(EditAction editAction, string name)
            {
                this.Enqueue(editAction, name);
            }

            public EditAction GetLastEditAction()
            {
                MonitoringTargetInfo lastInfo;

                if (0 < this.targetInfoQueue.Count())
                {
                    lastInfo = this.targetInfoQueue.Last();
                }
                else if (this.targetInfo != null)
                {
                    lastInfo = this.targetInfo;
                }
                else
                {
                    return EditAction.NoAction;
                }

                if (lastInfo.EditAction == EditAction.Add ||
                    lastInfo.EditAction == EditAction.Modify ||
                    lastInfo.EditAction == EditAction.Transfer)
                {
                    if (this.monitoringInfoManager.IsModified(Target) == true)
                    {
                        return lastInfo.EditAction;
                    }
                    else if (this.monitoringInfoManager.IsNeedTransfer(Target) == true)
                    {
                        return EditAction.Updating;
                    }
                    else if (lastInfo.EditAction == EditAction.Add ||
                             lastInfo.EditAction == EditAction.Transfer)
                    {
                        return EditAction.Updating;
                    }
                }

                return lastInfo.EditAction;
            }

            public bool Update()
            {
                if (this.targetInfo == null)
                {
                    MonitoringTargetInfo info = this.Dequeue();
                    if (info != null)
                    {
                        this.targetInfo = info;
                        Converted = false;
                        this.tryConverted = false;
                        return true;
                    }
                }

                return false;
            }

            public void Execute(ChangedMonitoringTarget changedMonitoringTarget)
            {
                if (this.targetInfo != null)
                {
                    if (this.tryConverted == false &&
                        (this.EditAction == EditAction.Add ||
                         this.EditAction == EditAction.Modify ||
                         this.EditAction == EditAction.Transfer))
                    {
                        switch (this.monitoringInfoManager.CheckNeedConvert(Target))
                        {
                            case ConvertResult.OK:
                                this.monitoringInfoManager.RequestConvert(Target);
                                this.monitoringInfoManager.SetMonitoringState(Target, MonitoringState.Converting);
                                break;

                            case ConvertResult.NeedTransfer:
                                this.monitoringInfoManager.RequestTransfer(Target);
                                this.monitoringInfoManager.SetMonitoringState(Target, MonitoringState.Transferring);
                                break;

                            case ConvertResult.NotNeed:
                                Debug.WriteLine("NotNeed");
                                if (this.EditAction == EditAction.Add ||
                                    this.EditAction == EditAction.Transfer)
                                {
                                    Debug.WriteLine("NotNeed Transfer " + this.EditAction);
                                    this.monitoringInfoManager.RequestTransfer(Target);
                                    this.monitoringInfoManager.SetMonitoringState(Target, MonitoringState.Transferring);
                                }
                                else
                                {
                                    Debug.WriteLine("NotNeed !Transfer " + this.EditAction);
                                    this.targetInfo = null;
                                    Converted = false;
                                    this.tryConverted = false;
                                    this.monitoringInfoManager.SetMonitoringState(Target, MonitoringState.Completed);
                                    return;
                                }
                                break;
                        }
                        this.tryConverted = true;
                    }

                    this.targetInfo.Execute(changedMonitoringTarget);
                }
            }

            public bool Update(string name, out EditAction editAction)
            {
                editAction = EditAction.Updated;

                if (this.targetInfo == null)
                {
                    return false;
                }

                if (name != this.targetInfo.Name)
                {
                    return false;
                }

                return this.Update(out editAction);
            }

            private bool Update(out EditAction editAction)
            {
                editAction = EditAction.Updated;

                if (this.targetInfo == null)
                {
                    return false;
                }

                switch (this.targetInfo.EditAction)
                {
                    case EditAction.Updating:
                        editAction = EditAction.Updated;
                        this.targetInfo = null;
                        return true;

                    case EditAction.Removing:
                        editAction = EditAction.Removed;
                        this.targetInfo = null;
                        return true;

                    default:
                        return false;
                }
            }

            private void Enqueue(EditAction editAction, string oldName)
            {
                lock (this.targetInfoQueue)
                {
                    MonitoringTargetInfo info =
                        new MonitoringTargetInfo(this, Target, oldName, editAction);

                    Debug.WriteLine("Enqueue " + editAction);

                    switch (editAction)
                    {
                        case EditAction.Add:
                            this.OptimizeAddEnqueue(info);
                            break;

                        case EditAction.Modify:
                            this.OptimizeModifyEnqueue(info);
                            break;

                        case EditAction.Remove:
                            this.OptimizeRemoveEnqueue(info);
                            break;

                        case EditAction.OldNameRemove:
                            Debug.WriteLine("Enqueue OldNameRemove" + oldName);
                            this.OldNameRemoveEnqueue(info);
                            break;

                        case EditAction.Transfer:
                            this.OptimizeTransferEnqueue(info);
                            break;

                        default:
                            Debug.WriteLine("Illegal EditAction.");
                            break;
                    }
                }
            }

            private void OldNameRemoveEnqueue(MonitoringTargetInfo info)
            {
                if (this.targetInfoQueue.Count() == 0)
                {
                    this.targetInfoQueue.Add(info); // 積む
                }
                else
                {
                    MonitoringTargetInfo last = this.targetInfoQueue.Last();

                    switch (last.EditAction)
                    {
                        case EditAction.Add:
                            this.targetInfoQueue.Remove(last); // １つ前を削除するだけで積まない
                            break;

                        case EditAction.Modify:
                        case EditAction.Remove:
                        case EditAction.Transfer:
                            this.targetInfoQueue.Remove(last);  // 積まれている１つ前を削除
                            OldNameRemoveEnqueue(info);         // 再帰的に行う
                            break;

                        case EditAction.OldNameRemove:
                            // 積まない
                            break;

                        default:
                            Debug.Assert(false, "Illegal EditAction.");
                            break;
                    }
                }
            }

            private void OptimizeAddEnqueue(MonitoringTargetInfo info)
            {
                if (this.targetInfoQueue.Count() == 0)
                {
                    this.targetInfoQueue.Add(info); // 積む
                }
                else
                {
                    MonitoringTargetInfo last = this.targetInfoQueue.Last();

                    switch (last.EditAction)
                    {
                        case EditAction.Add:
                        case EditAction.Modify:
                        case EditAction.Remove:
                        case EditAction.Transfer:
                            this.targetInfoQueue.Remove(last);   // 積まれている１つ前を削除
                            this.OptimizeAddEnqueue(info); // 再帰的に行う
                            break;

                        case EditAction.OldNameRemove:
                            this.targetInfoQueue.Add(info); // 積む
                            break;

                        default:
                            Debug.Assert(false, "Illegal EditAction.");
                            break;
                    }
                }

            }

            private void OptimizeModifyEnqueue(MonitoringTargetInfo info)
            {
                if (this.targetInfoQueue.Count() == 0)
                {
                    this.targetInfoQueue.Add(info); // 積む
                }
                else
                {
                    MonitoringTargetInfo last = this.targetInfoQueue.Last();

                    switch (last.EditAction)
                    {
                        case EditAction.Add:
                            // 積まない
                            break;

                        case EditAction.Modify:
                        case EditAction.Remove:
                        case EditAction.Transfer:
                            this.targetInfoQueue.Remove(last);   // 積まれている１つ前を削除
                            this.OptimizeModifyEnqueue(info); // 再帰的に行う
                            break;

                        case EditAction.OldNameRemove:
                            this.targetInfoQueue.Add(info); // 積む
                            break;

                        default:
                            Debug.Assert(false, "Illegal EditAction.");
                            break;
                    }
                }

            }

            private void OptimizeRemoveEnqueue(MonitoringTargetInfo info)
            {
                if (this.targetInfoQueue.Count() == 0)
                {
                    if (this.targetInfo == null ||
                        this.targetInfo.EditAction != EditAction.Removing)
                    {
                        this.targetInfoQueue.Add(info); // 積む
                    }
                }
                else
                {
                    MonitoringTargetInfo last = this.targetInfoQueue.Last();

                    switch (last.EditAction)
                    {
                        case EditAction.Add:
                            this.targetInfoQueue.Remove(last); // １つ前を削除するだけで積まない
                            break;

                        case EditAction.Modify:
                        case EditAction.Remove:
                        case EditAction.Transfer:
                            this.targetInfoQueue.Remove(last);
                            this.OptimizeRemoveEnqueue(info); // 再帰的に行う
                            break;

                        case EditAction.OldNameRemove:
                            this.targetInfoQueue.Add(info); // 積む
                            break;

                        default:
                            Debug.Assert(false, "Illegal EditAction.");
                            break;
                    }
                }
            }

            private void OptimizeTransferEnqueue(MonitoringTargetInfo info)
            {
                if (this.targetInfoQueue.Count() == 0)
                {
                    this.targetInfoQueue.Add(info); // 積む
                }
                else
                {
                    MonitoringTargetInfo last = this.targetInfoQueue.Last();

                    switch (last.EditAction)
                    {
                        case EditAction.Add:
                            // 積まない
                            break;

                        case EditAction.Modify:
                        case EditAction.Remove:
                        case EditAction.Transfer:
                            this.targetInfoQueue.Remove(last);   // 積まれている１つ前を削除
                            this.OptimizeTransferEnqueue(info);  // 再帰的に行う
                            break;

                        case EditAction.OldNameRemove:
                            this.targetInfoQueue.Add(info); // 積む
                            break;

                        default:
                            Debug.Assert(false, "Illegal EditAction.");
                            break;
                    }
                }

            }

            private MonitoringTargetInfo Dequeue()
            {
                lock (this.targetInfoQueue)
                {
                    MonitoringTargetInfo info = null;

                    if (0 < this.targetInfoQueue.Count())
                    {
                        info = this.targetInfoQueue.First();
                        this.targetInfoQueue.Remove(info);
                        Debug.WriteLine("Dequeue " + info.EditAction);
                    }

                    return info;
                }
            }

            private class MonitoringTargetInfo
            {
                private MonitoringInfo monitoringInfo;

                public Component Target { get; set; }
                public EditAction EditAction { get; set; }
                public string Name { get; private set; }

                public MonitoringTargetInfo(MonitoringInfo info, Component component, string name, EditAction editAction)
                {
                    this.monitoringInfo = info;
                    Target = component;
                    EditAction = editAction;
                    Name = name;
                }

                public void Execute(ChangedMonitoringTarget changedMonitoringTarget)
                {
                    switch (EditAction)
                    {
                        case EditAction.Add:
                        case EditAction.Modify:
                        case EditAction.Transfer:
                            if (this.monitoringInfo.Converted == true)
                            {
                                changedMonitoringTarget.AddUpdateTarget(Name);
                                EditAction = EditAction.Updating;
                                this.monitoringInfo.Converted = false;
                            }
                            break;

                        case EditAction.Remove:
                            changedMonitoringTarget.AddRemoveTarget(Name);
                            EditAction = EditAction.Removing;
                            break;

                        case EditAction.Disabled:
                            changedMonitoringTarget.AddRemoveTarget(Name);
                            EditAction = EditAction.Removing;
                            break;

                        case EditAction.OldNameRemove:
                            changedMonitoringTarget.AddRemoveTarget(Name);
                            EditAction = EditAction.Removing;
                            break;
                    }
                }
            }
        }


        /// <summary>
        /// 編集対象の状態を管理します。
        /// </summary>
        private class MonitoringStatusManager
        {
            private Dictionary<Component, MonitoringStatus> dictionary;

            public event MonitoringTargetEventHandler MonitoringTargetStateChanged;

            public MonitoringStatusManager()
            {
                this.dictionary = new Dictionary<Component, MonitoringStatus>();
            }

            public void Add(Component component)
            {
                if (this.dictionary.ContainsKey(component) == false)
                {
                    MonitoringStatus status = new MonitoringStatus(component);
                    this.dictionary.Add(component, status);
                }
            }

            public void Remove(Component component)
            {
                if (this.dictionary.ContainsKey(component) == true)
                {
                    this.dictionary.Remove(component);
                }
            }

            public void Update(Component component, bool result)
            {
                MonitoringState state = MonitoringState.Completed;

                if (result == false)
                {
                    state = MonitoringState.Error;
                }

                SetState(component, state);
            }

            public void Initialize()
            {
                foreach (Component component in this.dictionary.Keys.ToArray())
                {
                    SetState(component, MonitoringState.None);
                }
            }

            public void Clear()
            {
                this.dictionary.Clear();
            }

            public MonitoringStatus GetMonitoringStatus(Component component)
            {
                if (this.dictionary.ContainsKey(component) == false)
                {
                    Add(component);
                }

                return this.dictionary[component];
            }

            public Component[] GetMonitoringTargets()
            {
                List<Component> list = new List<Component>();

                foreach (Component component in this.dictionary.Keys.ToArray())
                {
                    if (this.dictionary[component].State != MonitoringState.None ||
                        component.IsEnabledSndEdit() == true)
                    {
                        list.Add(component);
                    }
                }

                return list.ToArray();
            }

            public void SetState(Component component, MonitoringState state)
            {
                Debug.WriteLine("state = " + state.ToString());
                GetMonitoringStatus(component).State = state;
                OnMonitoringTargetStateChanged(new Component[] { component });
            }

            private void OnMonitoringTargetStateChanged(Component[] components)
            {
                Debug.WriteLine("MonitoringStatusManager StateChanged");
                if (MonitoringTargetStateChanged != null)
                {
                    MonitoringTargetStateChanged(components);
                }
            }
        }

        private class MonitoringStatus
        {
            private Component component = null;
            private MonitoringState state = MonitoringState.None;

            public MonitoringStatus(Component component)
            {
                this.component = component;
            }

            public MonitoringState State
            {
                get
                {
                    return this.state;
                }
                set
                {
                    if (this.state == MonitoringState.None &&
                        value == MonitoringState.Deleting)
                    {
                        // 何もしない
                    }
                    else if (this.state == MonitoringState.Deleting)
                    {
                        switch (value)
                        {
                            case MonitoringState.None:
                                this.state = value;
                                break;

                            case MonitoringState.Converting:
                            case MonitoringState.Transferring:
                            case MonitoringState.Error:
                                if (component.IsEnabledSndEdit() == true)
                                {
                                    this.state = value;
                                }
                                break;
                        }
                    }
                    else
                    {
                        this.state = value;
                    }
                }
            }
        }

        public class ChangedMonitoringTarget
        {
            object lockObject = new object();

            private enum ChangeState { Update, Remove, }

            List<Tuple<string, ChangeState>> changedList;

            public ChangedMonitoringTarget()
            {
                this.changedList = new List<Tuple<string, ChangeState>>();
            }

            public void Clear()
            {
                lock (lockObject)
                {
                    this.changedList.Clear();
                }
            }

            public void AddUpdateTarget(string name)
            {
                lock (lockObject)
                {
                    this.changedList.Add(new Tuple<string, ChangeState>(name, ChangeState.Update));
                }
            }

            public void AddRemoveTarget(string name)
            {
                lock (lockObject)
                {
                    this.changedList.Add(new Tuple<string, ChangeState>(name, ChangeState.Remove));
                }
            }

            public void GetTargets(List<string> updateList, List<string> removeList, int maxPopCount)
            {
                lock (lockObject)
                {
                    int count = this.changedList.Count();

                    if (0 < count)
                    {
                        if (maxPopCount < count)
                        {
                            count = maxPopCount;
                        }

                        for (int i = 0; i < count; i++)
                        {
                            Tuple<string, ChangeState> changedTarget = this.changedList[i];

                            if (changedTarget.Item2 == ChangeState.Update)
                            {
                                if (removeList.Contains(changedTarget.Item1) == true)
                                {
                                    this.changedList.RemoveRange(0, i);
                                    return;
                                }
                                updateList.Add(changedTarget.Item1);
                            }
                            else
                            {
                                if (updateList.Contains(changedTarget.Item1) == true)
                                {
                                    this.changedList.RemoveRange(0, i);
                                    return;
                                }
                                removeList.Add(changedTarget.Item1);
                            }
                        }

                        this.changedList.RemoveRange(0, count);
                    }
                }
            }
        }


        /// <summary>
        /// 更新、削除されたComponentを取得します。
        /// 内部情報の更新が同時に行われます。
        /// </summary>
        public void PopChangedMonitoringTargets(List<string> updateList, List<string> removeList, int maxPopCount)
        {
            Debug.WriteLine("in PopChangedMonitoringTargets()");

            this.monitoringManager.GetChangedMonitoringTargets(updateList, removeList, maxPopCount);
        }

        /// <summary>
        /// 編集対象の Component を取得します。
        /// </summary>
        public Component[] MonitoringTargets
        {
            get
            {
                return this.monitoringManager.GetMonitoringTargets();
            }
        }

        public void Clear()
        {
            this.convertService.Clear();
            this.filePathDictionary.Clear();
            this.hashCodeDictionary.Clear();
            this.ClearConvertQuery();
            this.ClearSoundEditInfo();
            this.ClearMonitoringTargetInfo();
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual bool AutoEnableMonitoringTarget
        {
            get
            {
                return ApplicationBase.Instance.AppConfiguration.Options.Application.InGame.AutoEnableMonitoringTarget;
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual bool AutoEnableBankBySequenceSound
        {
            get
            {
                return ApplicationBase.Instance.AppConfiguration.Options.Application.InGame.AutoEnableBankBySequenceSound;
            }
        }

        ///
        public event MonitoringTargetEventHandler MonitoringTargetAdded;
        public event MonitoringTargetEventHandler MonitoringTargetRemoved;
        public event MonitoringTargetEventHandler MonitoringTargetRenamed;
        public event EventHandler MonitoringTargetCleared;

        /// <summary>
        /// リアルタイム編集対象に追加します。
        /// </summary>
        private void AddMonitoringTarget(Component component)
        {
            // ファイル監視に登録
            this.FileWatcherServiceAddWithChildren(component);

            // ディクショナリによる編集対象の付加情報管理
            AddMonitoringTargetInfo(component);
        }

        /// <summary>
        /// リアルタイム編集対象から削除します。
        /// 接続中は削除を要求するだけです。
        /// </summary>
        private void RemoveMonitoringTarget(Component component)
        {
            // ファイル監視を解除
            this.FileWatcherServiceRemoveWithChildren(component);

            RemoveMonitoringTargetInfo(component);
        }

        /// <summary>
        /// リアルタイム編集対象を転送します。
        /// </summary>
        public void TransferMonitoringTarget(Component component)
        {
            TransferMonitoringTargetInfo(component);
        }

        /// <summary>
        /// リアルタイム編集対象に含まれているのか調べます。
        /// </summary>
        private bool ContainsMonitoringTarget(Component component)
        {
            return component.IsEnabledSndEdit();
        }

        private void FileWatcherServiceRemoveWithChildren(Component component)
        {
            foreach (Component child in component.Children)
            {
                FileWatcherServiceRemoveWithChildren(child);
            }

            if (component.Parameters.ContainsKey(ProjectParameterNames.FilePath) == true)
            {
                this.fileWatcherComponentFilePaths.Remove(component);
                if (component is SoundSetBankBase == true)
                {
                    this.FileWatcherServiceRemoveBank(component);
                }
            }
        }

        private void FileWatcherServiceRemoveBank(Component component)
        {
            if (component is SoundSetBankBase == true)
            {
                string bankFilePath = (component as SoundSetBankBase).FilePath;

                fileWatcherBankWaveFilePaths.Remove(bankFilePath);
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public RealtimeEditService(FormsBasedInvoker invoker)
        {
            this.monitoringManager = new MonitoringManager(this);
            this.monitoringManager.MonitoringTargetAdded += OnMonitoringTargetAdded;
            this.monitoringManager.MonitoringTargetRemoved += OnMonitoringTargetRemoved;
            this.monitoringManager.MonitoringTargetStateChanged += OnMonitoringTargetStateChanged;

            this.invoker = invoker;
            this.updateTimer = new UpdateTimer(this.UpdateTimerCallback);

            Enabled = false;

            this.convertService = new BackgroundPartsConvertService(invoker);
            this.convertService.Converted += OnConverted;

            SoundEditInfoCreated += OnSoundEditInfoCreated;

            this.fileWatcherComponentFilePaths = new FileWatcherComponentFilePaths(this.OnFileChanged);
            this.fileWatcherBankWaveFilePaths = new FileWatcherBankWaveFilePaths(this.OnWaveFileChanged);
            this.fileWatcherComponentFilePaths.Updated += OnFileWatcherComponentUpdated;
        }

        /// <summary>
        ///
        /// </summary>
        public bool Enabled
        {
            get
            {
                return this.enabled;
            }
            set
            {
                this.enabled = value;

                //
                ClearConvertQuery();
                ClearSoundEditInfo();

                //
                if (value == true)
                {
                    this.monitoringManager.Initialize();
                    this.updateTimer.Start();

                    foreach (Component component in MonitoringTargets)
                    {
                        this.TransferMonitoringTarget(component);
                    }
                }
                else
                {
                    this.updateTimer.Stop();
                    this.monitoringManager.Initialize();
                    SetTargetMachineInformation(0, 0, 0, 0, 0, 0, false, false, false);
                }
            }
        }

        /// <summary>
        /// テンポラリディレクトリを取得します。
        /// </summary>
        public string TemporaryDirectory { get; set; }

        /// <summary>
        ///
        /// </summary>
        public void Attach(SoundProjectService projectService, ISoundProjectConvertService projectConvertService, BankServiceManager bankServiceManager)
        {
            this.SetupTemporaryDirectory();
            this.hashCodeDictionary.Clear();
            ClearConvertQuery();
            ClearSoundEditInfo();

            this.projectService = projectService;
            this.projectConvertService = projectConvertService;
            this.bankServiceManager = bankServiceManager;

            projectService.Opened += OnOpened;
            projectService.Closing += OnClosing;
            projectService.Closed += OnClosed;

            projectService.SoundSetAdded += OnSoundSetAdded;
            projectService.SoundSetRemoved += OnSoundSetRemoved;

            projectService.ComponentsAdded += OnComponentsAdded;
            projectService.ComponentsRemoved += OnComponentsRemoved;

            bankServiceManager.Added += OnBankServiceAdded;
            bankServiceManager.Removed += OnBankServiceRemoved;

            //
            this.convertService.Attach(projectService, projectConvertService);
        }

        /// <summary>
        ///
        /// </summary>
        public void Dispose()
        {
            if (this.projectConvertService.IsConverting == true)
            {
                this.projectConvertService.Cancel();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void Detach()
        {
            Debug.Assert(this.projectService != null, "Not set attach SoundProjectService");

            this.projectService.Opened -= OnOpened;
            this.projectService.Closing -= OnClosing;
            this.projectService.Closed -= OnClosed;

            this.projectService.SoundSetAdded -= OnSoundSetAdded;
            this.projectService.SoundSetRemoved -= OnSoundSetRemoved;

            this.projectService.ComponentsAdded -= OnComponentsAdded;
            this.projectService.ComponentsRemoved -= OnComponentsRemoved;

            this.projectService = null;

            this.bankServiceManager.Added -= OnBankServiceAdded;
            this.bankServiceManager.Removed -= OnBankServiceRemoved;
            this.bankServiceManager = null;

            this.convertService.Detach();

            this.CleanupTemporaryDirectory();
        }

        /// <summary>
        /// 全ての編集対象をコンバートします。
        /// </summary>
        public void ConvertAllMonitoringTargets()
        {
            foreach (Component component in MonitoringTargets)
            {
                this.TransferMonitoringTarget(component);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdateAllMonitoringTargets()
        {
            UpdateMonitoringTargets(MonitoringTargets);
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdateMonitoringTargets(Component[] components)
        {
            if (Enabled == false)
            {
                return;
            }

            //
            List<Component> list = new List<Component>();
            foreach (Component baseComponent in components)
            {
                Component component = baseComponent;

                // ストリームトラックが編集されたのか？
                // コンバート対象は親のストリームになります。
                if (component is StreamSoundTrackBase)
                {
                    component = component.Parent;
                }

                // 編集対象が無効化されているのか？
                if (component.IsEnabledSndEdit() == false ||
                    component.IsEnabledSndEditEnabled() == false ||
                    component.IsEnabled == false)
                {
                    continue;
                }

                list.Add(component);
            }

            //
            foreach (Component component in list)
            {
                this.TransferMonitoringTarget(component);
            }
        }

        /// <summary>
        /// ファイルパスから、名前変更後ファイルパスを取得します。
        /// </summary>
        public string GetFilePath(string filePath)
        {
            if (IsModified(filePath) == false)
            {
                return GetReplacedFilePath(filePath);
            }

            return ReplaceAndCopyFilePath(filePath);
        }


        /// <summary>
        /// ファイル監視でファイルが更新された時に呼ばれます。
        /// </summary>
        private void OnFileChanged(string filePath)
        {
            foreach (Component component in MonitoringTargets)
            {
                if (this.CheckFileChangedWithChildren(component, filePath) == true)
                {
                    if (component is SoundSetBankBase == true)
                    {
                        if (this.bankServiceManager.Contains(filePath) == true)
                        {
                            continue; // 再読み込み用監視処理にまかせる
                        }
                        string[] waveFilePaths = this.GetWaveFilePathsFromBankFilePath(filePath);
                        this.fileWatcherBankWaveFilePaths.Update(filePath, waveFilePaths);
                    }

                    this.ModifyMonitoringTargetInfo(component);
                }
            }
        }

        private bool CheckFileChangedWithChildren(Component component, string filePath)
        {
            if (component.Parameters.ContainsKey(ProjectParameterNames.FilePath) == true)
            {
                string componentFilePath = ((FilePathParameterValue)component.Parameters.
                                            GetValue(ProjectParameterNames.FilePath));
                if (filePath == componentFilePath)
                {
                    return true;
                }
            }

            foreach (Component child in component.Children)
            {
                if (this.CheckFileChangedWithChildren(child, filePath) == true)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// ファイル監視で波形ファイルが更新された時に呼ばれます。
        /// </summary>
        private void OnWaveFileChanged(string waveFilePath)
        {
            List<Component> soundSetBanks = new List<Component>();
            string[] bankFilePaths = this.fileWatcherBankWaveFilePaths.GetBankFilePathsByWaveFilePath(waveFilePath);

            foreach (string bankFilePath in bankFilePaths)
            {
                List<Component> components = GetSoundSetBanks(bankFilePath);
                foreach (Component component in components)
                {
                    if (soundSetBanks.Contains(component) == false)
                    {
                        soundSetBanks.Add(component);
                    }
                }
            }

            foreach (Component component in soundSetBanks)
            {
                this.ModifyMonitoringTargetInfo(component);
            }
        }

        /// <summary>
        /// ターゲットマシン情報が更新された時に呼ばれます。
        /// </summary>
        private void OnSetTargetMachineInformation(TargetMachineInformation info)
        {
            if (TargetMachineInformationChanged != null)
            {
                TargetMachineInformationChanged(info);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnMonitoringTargetAdded(Component[] components)
        {
            if (MonitoringTargetAdded != null)
            {
                MonitoringTargetAdded(components);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnMonitoringTargetRemoved(Component[] components)
        {
            if (MonitoringTargetRemoved != null)
            {
                MonitoringTargetRemoved(components);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnMonitoringTargetRenamed(Component[] components)
        {
            if (MonitoringTargetRenamed != null)
            {
                MonitoringTargetRenamed(components);
            }
        }

        private void FileWatcherServiceAddWithChildren(Component component)
        {
            foreach (Component child in component.Children)
            {
                this.FileWatcherServiceAddWithChildren(child);
            }

            if (component.Parameters.ContainsKey(ProjectParameterNames.FilePath) == true)
            {
                this.fileWatcherComponentFilePaths.Add(component);
                if (component is SoundSetBankBase == true)
                {
                    this.FileWatcherServiceAddBank(component);
                }
            }
        }

        private void FileWatcherServiceAddBank(Component component)
        {
            if (component is SoundSetBankBase == true)
            {
                string bankFilePath = (component as SoundSetBankBase).FilePath;
                string[] waveFilePaths = this.GetWaveFilePathsFromBankFilePath(bankFilePath);

                this.fileWatcherBankWaveFilePaths.Add(bankFilePath, waveFilePaths);
            }
        }

        private string[] GetWaveFilePathsFromBankFilePath(string bankFilePath)
        {
            string[] waveFilePaths = { };

            if (this.bankServiceManager.Contains(bankFilePath) == true)
            {
                waveFilePaths = BankUtility.GetAllWaveFilePath(this.bankServiceManager[bankFilePath].Bank);
            }
            else
            {
                waveFilePaths = BankUtility.GetAllWaveFilePath(bankFilePath);
            }

            return waveFilePaths;
        }

        /// <summary>
        /// 編集対象に含まれているのか調べます。
        /// </summary>
        private bool ContainsMonitoringTargetInfo(Component component)
        {
            return component.IsEnabledSndEdit();
        }

        /// <summary>
        ///
        /// </summary>
        private void AddMonitoringTargetInfo(Component component)
        {
            this.monitoringManager.Add(component);

            this.invoker.Invoke(() => OnMonitoringTargetAdded(new Component[] { component }));
        }

        /// <summary>
        ///
        /// </summary>
        private void RemoveMonitoringTargetInfo(Component component)
        {
            this.monitoringManager.Remove(component);

            this.invoker.Invoke(() => OnMonitoringTargetRemoved(new Component[] { component }));
        }

        /// <summary>
        ///
        /// </summary>
        private void ModifyMonitoringTargetInfo(Component component)
        {
            if (Enabled == true)
            {
                if (component.IsEnabledSndEdit() == false)
                {
                    if (this.AutoEnableMonitoringTarget == true &&
                        component.IsEnabled == true)
                    {
                        this.EnableSndEdit(component);
                    }
                }
                else
                {
                    this.monitoringManager.Modify(component);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void ModifyMonitoringTargetInfo(List<Component> soundSetBanks)
        {
            foreach (Component component in soundSetBanks)
            {
                this.ModifyMonitoringTargetInfo(component);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void RenameMonitoringTargetInfo(Component component, string oldName)
        {
            if (Enabled == true)
            {
                if (component.IsEnabledSndEdit() == false)
                {
                    if (this.AutoEnableMonitoringTarget == true &&
                        component.IsEnabled == true)
                    {
                        this.EnableSndEdit(component);
                    }
                }
                else
                {
                    this.monitoringManager.Rename(component, oldName);
                }
            }

            this.invoker.Invoke(() => OnMonitoringTargetRenamed(new Component[] { component }));
        }

        /// <summary>
        ///
        /// </summary>
        private void TransferMonitoringTargetInfo(Component component)
        {
            if (Enabled == true)
            {
                this.monitoringManager.Transfer(component);
            }
        }

        /// <summary>
        /// 編集対象の情報を"コンバート完了"状態に設定します。
        /// </summary>
        private void ConvertedMonitoringTargetInfo(Component component)
        {
            this.monitoringManager.SetConverted(component, true);
            this.monitoringManager.SetMonitoringState(component, MonitoringState.Transferring);
        }

        private void ConvertedMonitoringTargetInfo(Component[] components)
        {
            foreach (Component component in components)
            {
                this.ConvertedMonitoringTargetInfo(component);
            }
        }

        /// <summary>
        /// 編集対象の情報を全て削除します。
        /// </summary>
        private void ClearMonitoringTargetInfo()
        {
            this.monitoringManager.Clear();

            if (MonitoringTargetCleared != null)
            {
                MonitoringTargetCleared(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// 編集対象にすることができるComponentなのか調べます。
        /// </summary>
        private bool IsMonitoringTarget(Component component)
        {
            return (component is StreamSoundTrackBase ||
                     component is StreamSoundBase ||
                     component is WaveSoundBase ||
                     component is SequenceSoundBase ||
                     component is SoundSetBankBase ||
                     component is Bank ||
                     component is Instrument ||
                     component is KeyRegion ||
                     component is VelocityRegion) ? true : false;
        }

        /// <summary>
        ///
        /// </summary>
        private bool IsMonitoringParameter(Component component, string name)
        {
            if (name == null)
            {
                return true;
            }

            if (component.Parameters.ContainsKey(name) == false)
            {
                return false;
            }

            IParameterValue value = component.Parameters[name];
            if (value.Attributes.Contains(ParameterAttributes.ComputeHash) == false &&
                value.Attributes.Contains(ParameterAttributes.ComputeSoundArchiveHash) == false &&
                value.Attributes.Contains(ParameterAttributes.ComputeSndeditTransferHash) == false)
            {
                //Console.WriteLine( "$Monitoring parameter false");
                return false;
            }

            //Console.WriteLine( "$Monitoring parameter true");
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        private void AddEvent(Component component)
        {
            AddEvent(component, true);
        }

        /// <summary>
        ///
        /// </summary>
        private void AddEvent(Component component, bool recursive)
        {
            if (IsMonitoringTarget(component) == false)
            {
                return;
            }

            if (component is Bank)
            {
                Bank bank = component as Bank;
                if (this.addEventedBank.Contains(bank) == false)
                {
                    this.addEventedBank.Add(bank);
                }
                else
                {
                    return;
                }
            }

            if (component is Bank || component is Instrument || component is KeyRegion ||
                component is StreamSoundBase)
            {
                component.Children.CollectionChanged += OnChildrenChanged;
            }

            component.ParameterValueChanged += OnParameterValueChanged;
            component.NameChanged += OnNameChanged;
#if DEBUG
            IncEventCount( component);
#endif
            if (recursive == true)
            {
                foreach (Component child in component.Children)
                {
                    AddEvent(child, recursive);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void RemoveEvent(Component component)
        {
            RemoveEvent(component, true);
        }

        /// <summary>
        ///
        /// </summary>
        private void RemoveEvent(Component component, bool recursive)
        {
            if (IsMonitoringTarget(component) == false)
            {
                return;
            }

            if (component is Bank)
            {
                Bank bank = component as Bank;
                if (this.addEventedBank.Contains(bank) == true)
                {
                    this.addEventedBank.Remove(bank);
                }
                else
                {
                    return;
                }
            }

            if (component is Bank || component is Instrument || component is KeyRegion ||
                component is StreamSoundBase)
            {
                component.Children.CollectionChanged -= OnChildrenChanged;
            }

            if (recursive == true)
            {
                foreach (Component child in component.Children)
                {
                    RemoveEvent(child, recursive);
                }
            }

            component.ParameterValueChanged -= OnParameterValueChanged;
            component.NameChanged -= OnNameChanged;
#if DEBUG
            DecEventCount( component);
#endif
        }

#if DEBUG
        /// <summary>
        ///
        /// </summary>
        private Dictionary<Component, int> eventDictionary = new Dictionary<Component, int>();

        /// <summary>
        ///
        /// </summary>
        private void IncEventCount( Component component)
        {
            if( this.eventDictionary.ContainsKey( component) == false )
            {
                this.eventDictionary.Add( component, 1);
            }
            else
            {
                this.eventDictionary[component]++;
            }

            int count = this.eventDictionary[component];
            //Debug.WriteLine( String.Format( "++++ Event ++++ {0} : {1}", component.Name, count));
            Debug.Assert( count == 1, "Event count is not 1");
        }

        /// <summary>
        ///
        /// </summary>
        private void DecEventCount( Component component)
        {
            if( this.eventDictionary.ContainsKey( component) == false )
            {
                Debug.Assert( false, "Event not add dictionary");
            }

            this.eventDictionary[component]--;

            int count = this.eventDictionary[component];
            //Debug.WriteLine( String.Format( "---- Event ---- {0} : {1}", component.Name, count));
            Debug.Assert( count == 0, "Event count is not 0");
            if (count <= 0)
            {
                this.eventDictionary.Remove(component);
            }
        }

        private void ShowEventCount()
        {
            foreach( KeyValuePair<Component, int> d in this.eventDictionary )
            {
                Debug.WriteLine( String.Format( "{0} = {1}", d.Key.Name, d.Value));
            }
        }
#endif

        /// <summary>
        /// テンポラリディレクトリの準備を行います。
        /// </summary>
        private void SetupTemporaryDirectory()
        {
            Debug.Assert(TemporaryDirectory == null, "Temporary directory is not null");

            Random random = new Random();
            string directory = null;
            string systemTempDirectory = Path.Combine(Path.GetTempPath(), "NintendoSDK_SoundMaker_Temporary");

            do
            {
                string name = "realtimeEdit" + random.Next(256 * 256 * 256).ToString("x6");
                directory = Path.GetFullPath(Path.Combine(systemTempDirectory, name));
            }
            while (Directory.Exists(directory) == true);

            Directory.CreateDirectory(directory);
            TemporaryDirectory = directory;
        }

        /// <summary>
        /// テンポラリディレクトリの後片付けを行います。
        /// </summary>
        private void CleanupTemporaryDirectory()
        {
            Debug.Assert(TemporaryDirectory != null, "Temporary directory is null");

            try
            {
                Directory.Delete(TemporaryDirectory, true);
            }
            catch
            {
            }
            finally
            {
                TemporaryDirectory = null;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private IEnumerable<SoundSetItem> GetSoundSetItems(SoundProjectService projectService)
        {
            foreach (Sound sound in projectService.Sounds)
            {
                yield return sound;
            }

            foreach (SoundSetBankBase soundSetBank in projectService.SoundSetBanks)
            {
                yield return soundSetBank;
            }
        }

        public Bank GetBank(string filePath)
        {
            Bank bank = null;

            foreach (BankService bankService in this.bankServiceManager.Values)
            {
                BankDocument bankDocument = bankService.BankDocument;
                if (bankDocument.Resource.Key == filePath)
                {
                    bank = bankDocument.Bank;
                    break;
                }
            }

            return bank;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnOpened(object sender, EventArgs e)
        {
        }

        /// <summary>
        ///
        /// </summary>
        private void OnClosing(object sender, EventArgs e)
        {
            this.convertService.Clear();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnClosed(object sender, EventArgs e)
        {
        }

        /// <summary>
        ///
        /// </summary>
        private void OnComponentsAdded(object sender, ComponentEventArgs e)
        {
            Debug.WriteLine("OnComponentsAdded() in");
            foreach (Component component in e.Components)
            {
                if (IsMonitoringTarget(component) == true)
                {
                    Debug.WriteLine(component.Name);
                    AddEvent(component, false);

                    // ファイル監視に登録
                    this.FileWatcherServiceAddWithChildren(component);
                }
            }
            Debug.WriteLine("OnComponentsAdded() out");
        }

        /// <summary>
        ///
        /// </summary>
        private void OnComponentsRemoved(object sender, ComponentEventArgs e)
        {
            Debug.WriteLine("OnComponentsRemoved() in");
            foreach (Component component in e.Components)
            {
                Debug.WriteLine(component.Name);

                if (IsMonitoringTarget(component) == true)
                {
                    if (component.HasSndEdit() == true && component.IsEnabledSndEdit() == true)
                    {
                        DisableSndEdit(component);
                    }
                    RemoveEvent(component, false);
                }
            }
            Debug.WriteLine("OnComponentsRemoved() out");
        }

        private void OnBankComponentsAdded(object sender, ComponentEventArgs e)
        {
            Debug.WriteLine("OnBankComponentsAdded() in");
            foreach (Component component in e.Components)
            {
                if (IsMonitoringTarget(component) == true)
                {
                    Debug.WriteLine(component.Name);
                    if (component is Instrument)
                    {
                        Component bank = component.Parent;
                        AddEvent(bank, false);
                    }
                    AddEvent(component, false);
                }
            }
            Debug.WriteLine("OnBankComponentsAdded() out");
        }

        /// <summary>
        ///
        /// </summary>
        private void OnBankComponentsRemoved(object sender, ComponentEventArgs e)
        {
            Debug.WriteLine("OnBankComponentsRemoved() in");
            foreach (Component component in e.Components)
            {
                Debug.WriteLine(component.Name);

                if (IsMonitoringTarget(component) == true)
                {
                    if (component.HasSndEdit() == true && component.IsEnabledSndEdit() == true)
                    {
                        DisableSndEdit(component);
                    }
                    RemoveEvent(component, false);
                }
            }
            Debug.WriteLine("OnBankComponentsRemoved() out");
        }

        /// <summary>
        ///
        /// </summary>
        private void OnBankReloaded(object sender, EventArgs e)
        {
            if (sender is BankService)
            {
                BankService bankService = sender as BankService;
                string bankFilePath = bankService.BankDocument.Resource.Key;
                string[] waveFilePaths = this.GetWaveFilePathsFromBankFilePath(bankFilePath);
                this.fileWatcherBankWaveFilePaths.Update(bankFilePath, waveFilePaths);

                List<Component> soundSetBanks = new List<Component>();
                List<Component> components = GetSoundSetBanks(bankFilePath);
                foreach (Component component in components)
                {
                    if (soundSetBanks.Contains(component) == false)
                    {
                        soundSetBanks.Add(component);
                    }
                }

                foreach (Component component in soundSetBanks)
                {
                    this.ModifyMonitoringTargetInfo(component);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnSoundSetAdded(object sender, SoundDocumentEventArgs e)
        {
            SoundDocument document = e.Document;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnSoundSetRemoved(object sender, SoundDocumentEventArgs e)
        {
            SoundDocument document = e.Document;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnBankServiceAdded(object sender, BankServiceEventArgs e)
        {
            Debug.WriteLine("OnBankServiceAdded() in");
            if (this.bankServices.Contains(e.BankService) == true)
            {
                Debug.WriteLine("OnBankServiceAdded() out return");
                return;
            }

            e.BankService.Reloaded += OnBankReloaded;
            e.BankService.ComponentsAdded += OnBankComponentsAdded;
            e.BankService.ComponentsRemoved += OnBankComponentsRemoved;
            AddEvent(e.BankService.Bank);
            this.bankServices.Add(e.BankService);

            Debug.WriteLine("OnBankServiceAdded() out");
        }

        /// <summary>
        ///
        /// </summary>
        private void OnBankServiceRemoved(object sender, BankServiceEventArgs e)
        {
            Debug.WriteLine("OnBankServiceRemoved() in");

            if (this.bankServices.Contains(e.BankService) == false)
            {
                Debug.WriteLine("OnBankServiceRemoved() out return");
                return;
            }

            this.bankServices.Remove(e.BankService);
            RemoveEvent(e.BankService.Bank);
            e.BankService.Reloaded -= OnBankReloaded;
            e.BankService.ComponentsAdded -= OnBankComponentsAdded;
            e.BankService.ComponentsRemoved -= OnBankComponentsRemoved;

            Debug.WriteLine("OnBankServiceRemoved() out");
        }

        /// <summary>
        /// コンバートが完了した時に呼ばれます。
        /// </summary>
        private void OnConverted(SoundSetItem[] soundSetItems, Component dependentComponent, bool succeeded, bool canceled)
        {
            Debug.WriteLine("OnConverted in");
            if (Enabled == false)
            {
                return;
            }

            if (soundSetItems == null)
            {
                return;
            }

            //
            if (canceled == false)
            {
                // コンバートが成功したのか？
                if (succeeded == true)
                {
                    Debug.WriteLine("OnConverted success");
                    if (this.projectService != null)
                    {
                        Debug.WriteLine("OnConverted preprocessedTag in");
                        this.preprocessedTag = this.projectService.GetPreprocessedTag();
                        Debug.WriteLine("OnConverted preprocessedTag out");
                    }

                    foreach (SoundSetItem soundSetItem in soundSetItems)
                    {
                        ConvertedMonitoringTargetInfo(soundSetItem);
                        PushSoundEditInfo(soundSetItem);
                        Debug.WriteLine("OnConverted Push");
                    }
                }
                else
                {
                    Debug.WriteLine("OnConverted not success");
                    foreach (SoundSetItem soundSetItem in soundSetItems)
                    {
                        SetMonitoringState(soundSetItem, MonitoringState.Error);
                        ClearModified(soundSetItem); // コンバートエラーなので必ずクリアする
                        this.monitoringManager.Clear(soundSetItem);
                        Debug.WriteLine("OnConverted Error");
                    }
                }
            }
            Debug.WriteLine("OnConverted out");
        }

        /// <summary>
        /// コンバートが成功したか失敗したかを取得します。
        /// </summary>
        private bool IsConvertSuccess(SoundSetItem soundSetItem)
        {
            if (soundSetItem is StreamSoundBase == true)
            {
                string bfstmFilePath = (soundSetItem as StreamSoundBase).BfstmFilePath();
                if (string.IsNullOrEmpty(bfstmFilePath) == true)
                {
                    return false;
                }
            }
            else if (soundSetItem is SequenceSoundBase == true)
            {
                string bfseqFilePath = (soundSetItem as SequenceSoundBase).BfseqFilePath();
                if (string.IsNullOrEmpty(bfseqFilePath) == true)
                {
                    return false;
                }
            }
            else if (soundSetItem is WaveSoundBase == true)
            {
                string bfwsdFilePath = (soundSetItem as WaveSoundBase).BfwsdFilePath();
                if (string.IsNullOrEmpty(bfwsdFilePath) == true)
                {
                    return false;
                }
            }
            else if (soundSetItem is SoundSetBankBase == true)
            {
                string bfbnkFilePath = (soundSetItem as SoundSetBankBase).BfbnkFilePath();
                if (string.IsNullOrEmpty(bfbnkFilePath) == true)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        ///
        /// </summary>
        private bool suspendParameterValueChanged = false;

        public void EnableParameterValueChanged()
        {
            this.suspendParameterValueChanged = false;
        }

        public void DisableParameterValueChanged()
        {
            this.suspendParameterValueChanged = true;
        }

        /// <summary>
        /// サウンドアイテムの名前が変更された時に呼ばれます。
        /// </summary>
        private void OnNameChanged(object sender, NameChangedEventArgs e)
        {
            Component component = sender as Component;

            if (this.IsMonitoringTarget(component) == true)
            {
                if (component is Bank == false &&
                    component is Instrument == false &&
                    component is KeyRegion == false &&
                    component is VelocityRegion == false)
                {
                    RenameMonitoringTargetInfo(component, e.OldName);
                }
            }
        }

        /// <summary>
        /// サウンドアイテムのパラメータが変更された時に呼ばれます。
        /// </summary>
        private void OnParameterValueChanged(object sender, ParameterEventArgs e)
        {
            Debug.WriteLine("OnParameterValueChanged()");

            if (this.suspendParameterValueChanged == true)
            {
                Debug.WriteLine("suspendParameterValueChanged return");
                return;
            }

            //
            Component component = sender as Component;

            // FilePath の変更によるファイル監視対象の変更
            if (e.Key == ProjectParameterNames.FilePath &&
                component.Parameters.ContainsKey(ProjectParameterNames.FilePath) == true)
            {
                if (component is VelocityRegion == true)
                {
                    List<Component> soundSetBanks = this.GetSoundSetBanks(component);
                    if (soundSetBanks.Count > 0)
                    {
                        string bankFilePath = (soundSetBanks[0] as SoundSetBankBase).FilePath;
                        string[] waveFilePaths = this.GetWaveFilePathsFromBankFilePath(bankFilePath);
                        this.fileWatcherBankWaveFilePaths.Update(bankFilePath, waveFilePaths);
                    }
                }
                else
                {
                    this.fileWatcherComponentFilePaths.Update(component);
                }
            }

            if (this.IsMonitoringTarget(component) == true)
            {
                if (this.IsMonitoringParameter(component, e.Key) == true)
                {
                    if (component is Bank ||
                        component is Instrument ||
                        component is KeyRegion ||
                        component is VelocityRegion)
                    {
                        this.ModifyMonitoringTargetInfo(this.GetSoundSetBanks(component));
                    }
                    else
                    {
                        if (component is StreamSoundTrackBase == true)
                        {
                            component = component.Parent;
                        }

                        this.ModifyMonitoringTargetInfo(component);

                        if (e.Key == ProjectParameterNames.SequenceSound.SoundSetBankReferences &&
                            component.IsEnabledSndEdit() == true)
                        {
                            this.CheckAutoEnableBankBySequenceSound(component);
                        }
                    }
                }
                else
                {
                    switch (e.Key)
                    {
                        case ProjectParameterNames.IsEnabled:
                            if (component.IsEnabledSndEdit() == true)
                            {
                                if (component.IsEnabled == false)
                                {
                                    this.RemoveMonitoringTargetInfo(component);
                                }
                                else if (component.IsEnabledSndEditEnabled() == true)
                                {
                                    this.AddMonitoringTargetInfo(component);
                                    this.CheckAutoEnableBankBySequenceSound(component);
                                }
                                else
                                {
                                    this.OnMonitoringTargetStateChanged(new Component[] { component });
                                }
                            }
                            break;

                        case ProjectParameterNames.SndEditEnabled:
                            if (component.IsEnabledSndEdit() == true)
                            {
                                if (component.IsEnabledSndEditEnabled() == false)
                                {
                                    this.RemoveMonitoringTargetInfo(component);
                                }
                                else if (component.IsEnabled == true)
                                {
                                    this.AddMonitoringTargetInfo(component);
                                    this.CheckAutoEnableBankBySequenceSound(component);
                                }
                                else
                                {
                                    this.OnMonitoringTargetStateChanged(new Component[] { component });
                                }
                            }
                            break;

                        case ProjectParameterNames.SndEdit:
                            if (component.IsEnabledSndEdit() == true)
                            {
                                if (component.IsEnabled == true)
                                {
                                    this.AddMonitoringTarget(component);
                                    this.CheckAutoEnableBankBySequenceSound(component);
                                }
                                else
                                {
                                    this.FileWatcherServiceAddWithChildren(component);
                                    this.RemoveMonitoringTargetInfo(component);
                                }

                                if (component.IsEnabledSndEditEnabled() == false)
                                {
                                    SetSndEditEnabled(component, true);
                                }
                            }
                            else
                            {
                                this.RemoveMonitoringTarget(component);
                            }
                            break;
                    }
                }
            }
        }

        private void CheckAutoEnableBankBySequenceSound(Component component)
        {
            if (this.AutoEnableBankBySequenceSound == true &&
                component is SequenceSoundBase == true)
            {
                SequenceSoundBase sequenceSound = component as SequenceSoundBase;

                foreach (ComponentReference bankReference in sequenceSound.SoundSetBankReferences)
                {
                    SoundSetBankBase soundSetBank = this.projectService.SoundSetBanks.Where(c => c.Name == bankReference.TargetName).FirstOrDefault();
                    if (soundSetBank != null &&
                        soundSetBank.IsEnabledSndEdit() == false)
                    {
                        this.EnableSndEdit(soundSetBank);
                    }
                }
            }
        }

        /// <summary>
        /// サウンドアイテムのコレクションが変更された時に呼ばれます。
        /// </summary>
        private void OnChildrenChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            if (Enabled == false)
            {
                return;
            }

            ComponentList componentList = sender as ComponentList;
            if (componentList == null)
            {
                return;
            }

            if (componentList.Owner is KeyRegion ||
                componentList.Owner is Instrument ||
                componentList.Owner is Bank)
            {
                if (e.Action == NotifyCollectionChangedAction.Add ||
                    e.Action == NotifyCollectionChangedAction.Remove ||
                    e.Action == NotifyCollectionChangedAction.Reset)
                {
                    List<Component> soundSetBanks = this.GetSoundSetBanks(componentList.Owner);
                    string bankFilePath = (soundSetBanks[0] as SoundSetBankBase).FilePath;
                    string[] waveFilePaths = this.GetWaveFilePathsFromBankFilePath(bankFilePath);

                    this.fileWatcherBankWaveFilePaths.Update(bankFilePath, waveFilePaths);

                    this.ModifyMonitoringTargetInfo(soundSetBanks);
                }
                else
                {
                    // 何もしない。
                }
            }
            else
            {
                this.ModifyMonitoringTargetInfo(componentList.Owner);
            }
        }

        private List<VelocityRegion> GetVelocityRegion(Component owner, System.Collections.IList list)
        {
            List<VelocityRegion> velocityRegionList = new List<VelocityRegion>();

            if (owner is KeyRegion)
            {
                foreach (VelocityRegion velocityRegion in list)
                {
                    velocityRegionList.Add(velocityRegion);
                }
            }
            else if (owner is Instrument)
            {
                foreach (KeyRegion keyRegion in list)
                {
                    foreach (VelocityRegion velocityRegion in keyRegion.Children)
                    {
                        velocityRegionList.Add(velocityRegion);
                    }
                }
            }

            return velocityRegionList;
        }

        /// <summary>
        ///
        /// </summary>
        private bool valueChanging = false;
        private string parameterName = null;
        private int intValue = 0;

        /// <summary>
        ///
        /// </summary>
        public void Attach(ParameterPanel parameterPanel)
        {
            SoundParameterPanel panel = parameterPanel.SoundParameterPanel;

            panel.ValueChanging += OnValueChanging;
            panel.ValueChanged += OnValueChanged;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnValueChanging(object sender, ValueChangeEventArgs e)
        {
            this.valueChanging = true;

            if (e is IntValueChangingEventArgs)
            {
                IntValueChangingEventArgs args = e as IntValueChangingEventArgs;

                if (args.Name == ProjectParameterNames.Volume)
                {
                    this.parameterName = args.Name;
                    this.intValue = (int)args.Value;

                    UpdateMonitoringTargets(args.Components);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnValueChanged(object sender, ValueChangeEventArgs e)
        {
            this.valueChanging = false;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnSoundEditInfoCreated(SoundEditInfo info)
        {
            if (this.valueChanging == true)
            {
                switch (this.parameterName)
                {
                    case ProjectParameterNames.Volume:
                        info.soundBasicParam.volume = (byte)this.intValue;
                        break;

                    default:
                        //Debug.Assert( false, "No impliment parameter name.");
                        break;
                }
            }
        }

        /// <summary>
        /// コンバートが必要か調べます。（コンバートは実行されません）
        /// </summary>
        public ConvertResult CheckNeedConvert(Component component)
        {
            if (Enabled == false)
            {
                return ConvertResult.Faild;
            }

            if (IsMonitoringTarget(component) == true)
            {
                // ストリームトラックが編集されたのか？
                // コンバート対象は親のストリームになります。
                if (component is StreamSoundTrackBase)
                {
                    component = component.Parent;
                }

                return CheckNeedConvertComponent(component);
            }
            return ConvertResult.Cancel;
        }

        /// <summary>
        /// Bankに依存したSoundSetBankのコンバート
        /// </summary>
        public ConvertResult CheckNeedConvertSoundSetBankDependsOnBank(Component component)
        {
            if (Enabled == false)
            {
                return ConvertResult.Faild;
            }

            if (!(component is Bank) &&
                !(component is Instrument) &&
                !(component is KeyRegion) &&
                !(component is VelocityRegion))
            {
                return ConvertResult.Cancel;
            }

            SoundSetBankBase[] soundSetBanks = GetSoundSetBanks(component)
                .Cast<SoundSetBankBase>()
                .ToArray();
            if (soundSetBanks.Length <= 0)
            {
                return ConvertResult.Cancel;
            }

            // 複数のSoundSetBankが対象だがバンクファイルパスは同一なので
            // SoundSetBankBase[0].FilePathとしても問題無いです。
            string bankFilePath = soundSetBanks[0].FilePath;
            if (this.bankServiceManager.Contains(bankFilePath) == false)
            {
                return ConvertResult.Cancel;
            }

            Bank dependentBank = this.bankServiceManager[bankFilePath].Bank;

            // コンバート中なのか？
            if (this.convertService.IsConverting(soundSetBanks) == true ||
                IsTransferring(soundSetBanks) == true)
            {
                Debug.WriteLine
                    (String.Format("******** Converting {0}", dependentBank.Name));

                return ConvertResult.OK;
            }

            Debug.Write(String.Format("######## {0} is ", dependentBank.Name));
            if (IsModified(dependentBank) == false)
            {
                Debug.WriteLine("clean");
                return ConvertResult.NotNeed;
            }

            Debug.WriteLine("dirty");

            // 複数のSoundSetBankのコンバートを行います。
            if (soundSetBanks.Count() > 0)
            {
                return ConvertResult.OK;
            }
            else
            {
                Debug.WriteLine("Convert target is 0");
                return ConvertResult.Cancel;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public enum ConvertResult
        {
            OK,            // コンバートの要求が受け付けられました。
            Faild,         // コンバートに失敗しました。
            Cancel,        // コンバートの条件を満たしていませんでした。
            NotNeed,       // コンバートの必要がありませんでした。
            NeedTransfer,  // コンバートの必要はありませんでしたが、転送は必要でした。
        }

        /// <summary>
        ///
        /// </summary>
        private ConvertResult CheckNeedConvertComponent(Component component)
        {
            // コンバート、転送中なのか？
            if (this.convertService.IsConverting(component) == true ||
                IsTransferring(component) == true)
            {
                Debug.WriteLine(String.Format("******** {0} {1}",
                                                (IsTransferring(component) == true ?
                                                  "Transferring" : "Converting"),
                                                component.Name));
                return ConvertResult.OK;
            }

            //
            Debug.Write(String.Format("######## {0} is ", component.Name));

            // 編集対象が更新されているのか？
            if (IsModified(component) == false)
            {
                // 転送の必要があるのか？
                if (IsNeedTransfer(component) == true)
                {
                    //
                    Debug.WriteLine("need transfer");

                    // 編集対象の転送が無効なのか？
                    if (component.IsEnabledSndEditEnabled() == false ||
                        component.IsEnabled == false)
                    {
                        return ConvertResult.Cancel;
                    }

                    return ConvertResult.NeedTransfer;
                }
                else
                {
                    Debug.WriteLine("clean");

                    // 編集対象の転送が無効なのか？
                    if (component.IsEnabledSndEditEnabled() == false ||
                        component.IsEnabled == false)
                    {
                        return ConvertResult.Cancel;
                    }

                    return ConvertResult.NotNeed;
                }
            }

            Debug.WriteLine("dirty");

            // 編集対象の転送が無効なのか？
            if (component.IsEnabledSndEditEnabled() == false ||
                component.IsEnabled == false)
            {
                return ConvertResult.Cancel;
            }

            return ConvertResult.OK;
        }

        /// <summary>
        /// 指定Componentに依存するSoundSetBankを取得します。
        /// </summary>
        private List<Component> GetSoundSetBanks(Component component)
        {
            while (component is Bank == false)
            {
                component = component.Parent;
                Debug.Assert(component != null, "Component is not relation Bank.");
            }

            List<Component> soundSetBanks = new List<Component>();

            foreach (BankService bankService in this.bankServiceManager.Values)
            {
                BankDocument bankDocument = bankService.BankDocument;
                if (component == bankDocument.Bank)
                {
                    foreach (SoundSetBankBase soundSetBank in this.projectService.SoundSetBanks)
                    {
                        if (bankDocument.Resource.Key == soundSetBank.FilePath)
                        {
                            soundSetBanks.Add(soundSetBank);
                        }
                    }
                }
            }

            return soundSetBanks;
        }

        /// <summary>
        /// 指定bankファイルパスに依存するSoundSetBankを取得します。
        /// </summary>
        private List<Component> GetSoundSetBanks(string bankFilePath)
        {
            List<Component> components = new List<Component>();

            if (string.IsNullOrEmpty(bankFilePath) == false)
            {
                foreach (SoundSetBankBase soundSetBank in this.projectService.SoundSetBanks)
                {
                    if (bankFilePath == soundSetBank.FilePath)
                    {
                        components.Add(soundSetBank);
                    }
                }
            }

            return components;
        }

        /// <summary>
        /// Componentのハッシュ値を保存するクラスです。
        /// </summary>
        private class HashCodeSet
        {
            private HashCode previousItemHashCode = HashCode.Empty;
            private HashCode previousFileHashCode = HashCode.Empty;

            public HashCode ItemHashCode { get; set; }
            public HashCode FileHashCode { get; set; }
            public HashCode TransferHashCode { get; set; }

            public override bool Equals(object o)
            {
                HashCodeSet hashCodeSet = o as HashCodeSet;

                if (hashCodeSet == null ||
                    hashCodeSet.ItemHashCode != this.ItemHashCode ||
                    hashCodeSet.FileHashCode != this.FileHashCode)
                {
                    return false;
                }
                return true;
            }

            public override int GetHashCode()
            {
                return this.ItemHashCode.GetHashCode() ^ this.FileHashCode.GetHashCode();
            }

            public void SetPreviousHashCode(HashCodeSet hashCodeSet)
            {
                this.previousItemHashCode = hashCodeSet.ItemHashCode;
                this.previousFileHashCode = hashCodeSet.FileHashCode;
            }

            public void CopyFromPrevious()
            {
                ItemHashCode = this.previousItemHashCode;
                FileHashCode = this.previousFileHashCode;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private HashCodeSet GetHashCodeSet(Component component)
        {
            HashCodeSet hashCodeSet = new HashCodeSet();
            Func<IParameterValue, bool> optionFilter = null;

            if (!string.IsNullOrEmpty(this.preprocessedTag))
            {
                optionFilter = parameterValue => string.Equals(this.preprocessedTag, parameterValue.Value);
            }

            hashCodeSet.ItemHashCode = component.GetComponentSoundArchiveHashCode(hashAlgorithm, optionFilter);
            hashCodeSet.FileHashCode = component.GetComponentHashCode(hashAlgorithm, optionFilter);
            hashCodeSet.TransferHashCode = component.GetComponentSndeditTransferHashCode(hashAlgorithm, optionFilter);

            return hashCodeSet;
        }

        /// <summary>
        ///
        /// </summary>
        private void EnableSndEdit(Component component)
        {
            component.Parameters[ProjectParameterNames.SndEdit].Value = true;
        }

        /// <summary>
        ///
        /// </summary>
        private void DisableSndEdit(Component component)
        {
            component.Parameters[ProjectParameterNames.SndEdit].Value = false;
        }

        /// <summary>
        ///
        /// </summary>
        private void SetSndEditEnabled(Component component, bool b)
        {
            component.Parameters[ProjectParameterNames.SndEditEnabled].Value = b;
        }

        /// <summary>
        ///
        /// </summary>
        private bool IsRequireConvert(Component component, Component dependentComponent)
        {
            if (IsModified(component) == true)
            {
                return true;
            }

            if (dependentComponent != null)
            {
                if (IsModified(dependentComponent) == true)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsNeedTransfer(Component component)
        {
            HashCodeSet hashCodeSet = GetHashCodeSet(component);

            if (this.hashCodeDictionary.ContainsKey(component) == false)
            {
                return true;
            }

            HashCodeSet oldHashCodeSet = this.hashCodeDictionary[component];

            if (oldHashCodeSet.TransferHashCode != null)
            {
                //Console.WriteLine( "Old {0}", oldHashCodeSet.TransferHashCode);
            }

            if (hashCodeSet.TransferHashCode != null)
            {
                //Console.WriteLine( "New {0}", hashCodeSet.TransferHashCode);
            }

            return oldHashCodeSet.TransferHashCode != hashCodeSet.TransferHashCode ? true : false;
        }

        /// <summary>
        ///
        /// </summary>
        private void SetNeedTransfer(Component component)
        {
            HashCodeSet newHashCodeSet = GetHashCodeSet(component);
            HashCodeSet hashCodeSet = this.hashCodeDictionary[component];
            hashCodeSet.TransferHashCode = newHashCodeSet.TransferHashCode;
        }

        /// <summary>
        /// Componentが更新されているのは調べます。
        /// </summary>
        public bool IsModified(Component component)
        {
            HashCodeSet hashCodeSet = GetHashCodeSet(component);

            if (this.hashCodeDictionary.ContainsKey(component) == false)
            {
                //this.hashCodeDictionary.Add( component, hashCodeSet);
                return true;
            }

            HashCodeSet oldHashCodeSet = this.hashCodeDictionary[component];
            //Console.WriteLine( "Old {0}", oldHashCodeSet.ItemHashCode);
            //Console.WriteLine( "New {0}", hashCodeSet.ItemHashCode);
            return oldHashCodeSet.Equals(hashCodeSet) == false ? true : false;
        }

        /// <summary>
        /// ハッシュ値を保存します。
        /// Componentの値が編集されたかの比較に使用されます。
        /// </summary>
        private void SetModified(Component component)
        {
            HashCodeSet hashCodeSet = GetHashCodeSet(component);
            if (this.hashCodeDictionary.ContainsKey(component) == false)
            {
                this.hashCodeDictionary.Add(component, hashCodeSet);
            }
            else
            {
                HashCodeSet oldHashCodeSet = this.hashCodeDictionary[component];
                hashCodeSet.SetPreviousHashCode(oldHashCodeSet);
                this.hashCodeDictionary[component] = hashCodeSet;
            }
        }

        /// <summary>
        /// ハッシュ値を元に戻します。
        /// </summary>
        private void CancelModified(Component component)
        {
            if (this.hashCodeDictionary.ContainsKey(component) == false)
            {
                return;
            }

            this.hashCodeDictionary[component].CopyFromPrevious();
        }

        /// <summary>
        /// ハッシュ値をクリアします。
        /// </summary>
        private void ClearModified(Component component)
        {
            if (this.hashCodeDictionary.ContainsKey(component) == true)
            {
                this.hashCodeDictionary.Remove(component);
            }
        }

        /// <summary>
        /// ファイルから、名前変更後ファイルを作成します。
        /// </summary>
        private string ReplaceAndCopyFilePath(string filePath)
        {
            //
            ApplicationBase app = ApplicationBase.Instance;
            Debug.Assert(app != null, "Application is not FormsApplicationCafe");
            string baseDirectory = this.projectService.ProjectInGameEditCacheOutputPath;
            RealtimeEditService s = app.RealtimeEditService;

            string relativeFilePath = PathEx.MakeRelative(filePath, baseDirectory);


            string extension = Path.GetExtension(relativeFilePath);
            Random random = new Random();
            string newFilePath = null;

            do
            {
                string newFileName = ("file" + random.Next(256 * 256 * 256).ToString("x6") +
                                       extension);
                newFilePath = Path.GetFullPath
                    (Path.Combine(this.TemporaryDirectory, newFileName));
                //( Path.Combine( app.RealtimeEditService.TemporaryDirectory, newFileName));
            }
            while (File.Exists(newFilePath) == true);

            CopyFile(filePath, newFilePath);
            Modified(filePath, newFilePath);

            return newFilePath;
        }

        /// <summary>
        /// ファイルが、名前変更後ファイルよりも新しくなっているのか調べます。
        /// </summary>
        private bool IsModified(string origFilePath)
        {
            if (this.filePathDictionary.ContainsKey(origFilePath) == false)
            {
                return true;
            }

            string filePath = this.filePathDictionary[origFilePath];
            Debug.Assert(filePath != null, "File path is null");

            DateTime origTimeStamp = File.GetLastWriteTime(origFilePath);
            DateTime timeStamp = File.GetLastWriteTime(filePath);

            return origTimeStamp > timeStamp ? true : false;
        }

        /// <summary>
        /// ファイル、名前変更後ファイルを記録しておきます。
        /// </summary>
        private void Modified(string origFilePath, string filePath)
        {
            if (this.filePathDictionary.ContainsKey(origFilePath) == false)
            {
                this.filePathDictionary.Add(origFilePath, filePath);
            }
            else
            {
                this.filePathDictionary[origFilePath] = filePath;
            }
        }

        /// <summary>
        /// 名前変更後ファイル名を取得します。
        /// </summary>
        private string GetReplacedFilePath(string origFilePath)
        {
            if (this.filePathDictionary.ContainsKey(origFilePath) == false)
            {
                return null;
            }
            else
            {
                return this.filePathDictionary[origFilePath];
            }
        }

        /// <summary>
        /// ファイルのコピーを行います。
        /// </summary>
        private void CopyFile(string sourceFilePath, string destinationFilePath)
        {
            try
            {
                string directory = Path.GetDirectoryName(destinationFilePath);
                if (Directory.Exists(directory) == false)
                {
                    Directory.CreateDirectory(directory);
                }

                if (File.Exists(destinationFilePath) == true)
                {
                    File.Delete(destinationFilePath);
                }

                File.Copy(sourceFilePath, destinationFilePath);

                //
                string pcDirectoryName = "_forPC";
                string origExtension = Path.GetExtension(sourceFilePath);

                if (String.Compare(origExtension, ".aac", true) == 0)
                {
                    // コピー元の "_forPC"ディレクトリから AACファイルの対となる
                    // ストリームファイルを取得します。
                    string[] filePaths = Directory.GetFiles
                        (Path.Combine(Path.GetDirectoryName(sourceFilePath), pcDirectoryName),
                          Path.ChangeExtension(Path.GetFileName(sourceFilePath), "*"),
                          SearchOption.TopDirectoryOnly);
                    string stmFilePath = filePaths.Length > 0 ? filePaths[0] : null;

                    if (stmFilePath != null)
                    {
                        string stmExtension = Path.GetExtension(stmFilePath);

                        string newDirectory = Path.Combine
                            (Path.GetDirectoryName(destinationFilePath), pcDirectoryName);
                        if (Directory.Exists(newDirectory) == false)
                        {
                            Directory.CreateDirectory(newDirectory);
                        }

                        string newFileName = Path.GetFileName(destinationFilePath);
                        newFileName = Path.ChangeExtension(newFileName, stmExtension);

                        string newFilePath = Path.Combine(newDirectory, newFileName);
                        if (File.Exists(newFilePath) == true)
                        {
                            File.Delete(newFilePath);
                        }

                        File.Copy(stmFilePath, newFilePath);
                    }
                }

                //Debug.WriteLine( String.Format( "Copy file {0} -> {1}", sourceFilePath, destinationFilePath));
                Debug.WriteLine(String.Format("Copy file {0} -> {1}",
                                                Path.GetFileName(sourceFilePath),
                                                Path.GetFileName(destinationFilePath)));

            }

            catch (Exception e)
            {
                Debug.WriteLine(e.Message);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private class ModifiedInfo
        {
            private string name = null;
            private uint dataType = 0;

            public ModifiedInfo(string name, uint dataType)
            {
                this.name = name;
                this.dataType = dataType;
            }

            public string Name
            {
                get
                {
                    return this.name;
                }
            }

            public uint DataType
            {
                get
                {
                    return this.dataType;
                }
            }
        }

        private class FileWatcherComponentFilePaths
        {
            private Dictionary<Component, string> fileWatcherFilePathDictionary;
            private FileChangedEventHandler fileChanged;
            public delegate void FileWatcherComponentFilePathsEventHandler(Component component, string oldFilePath, string newFilePath);
            public event FileWatcherComponentFilePathsEventHandler Updated;

            public FileWatcherComponentFilePaths(FileChangedEventHandler fileChanged)
            {
                this.fileWatcherFilePathDictionary = new Dictionary<Component, string>();
                this.fileChanged = fileChanged;
            }

            public void Add(Component component)
            {
                if (this.fileWatcherFilePathDictionary.ContainsKey(component) == false)
                {
                    string filePath = ((FilePathParameterValue)component.Parameters.
                                       GetValue(ProjectParameterNames.FilePath));
                    this.fileWatcherFilePathDictionary.Add(component, filePath);
                    ApplicationBase.Instance.FileWatcherService.Add(filePath, this.fileChanged);
                }
            }

            public void Remove(Component component)
            {
                if (this.fileWatcherFilePathDictionary.ContainsKey(component) == true)
                {
                    string filePath = ((FilePathParameterValue)component.Parameters.
                                       GetValue(ProjectParameterNames.FilePath));
                    this.fileWatcherFilePathDictionary.Remove(component);
                    ApplicationBase.Instance.FileWatcherService.Remove(filePath, this.fileChanged);
                }
            }

            public void Update(Component component)
            {
                if (this.fileWatcherFilePathDictionary.ContainsKey(component) == true)
                {
                    string oldFilePath = fileWatcherFilePathDictionary[component];
                    string newFilePath = ((FilePathParameterValue)component.Parameters.
                                          GetValue(ProjectParameterNames.FilePath));
                    if (oldFilePath != newFilePath)
                    {
                        ApplicationBase.Instance.FileWatcherService.Remove(oldFilePath, this.fileChanged);
                        ApplicationBase.Instance.FileWatcherService.Add(newFilePath, this.fileChanged);
                        fileWatcherFilePathDictionary[component] = newFilePath;

                        if (this.Updated != null)
                        {
                            this.Updated(component, oldFilePath, newFilePath);
                        }
                    }
                }
            }
        }

        private void OnFileWatcherComponentUpdated(Component component, string oldFilePath, string newFilePath)
        {
            if (component is SoundSetBankBase == true)
            {
                string[] newWaveFilePaths = this.GetWaveFilePathsFromBankFilePath(newFilePath);

                this.fileWatcherBankWaveFilePaths.Update(oldFilePath, newFilePath, newWaveFilePaths);
            }
        }

        private class FileWatcherBankWaveFilePaths
        {
            private Dictionary<string, int> bankFilePathDictionary;
            private Dictionary<string, ReferenceWaveFilePath> waveFilePathsDictionary;
            private FileChangedEventHandler waveFileChanged;

            public FileWatcherBankWaveFilePaths(FileChangedEventHandler waveFileChanged)
            {
                this.bankFilePathDictionary = new Dictionary<string, int>();
                this.waveFilePathsDictionary = new Dictionary<string, ReferenceWaveFilePath>();
                this.waveFileChanged = waveFileChanged;
            }


            public void Add(string bankFilePath, string[] waveFilePaths)
            {
                if (this.bankFilePathDictionary.ContainsKey(bankFilePath) == false)
                {
                    this.bankFilePathDictionary.Add(bankFilePath, 0);
                    this.waveFilePathsDictionary[bankFilePath] = new ReferenceWaveFilePath(this.OnWaveFileChanged);
                }

                this.bankFilePathDictionary[bankFilePath]++;

                if (this.bankFilePathDictionary[bankFilePath] == 1)
                {
                    foreach (string waveFilePath in waveFilePaths)
                    {
                        this.waveFilePathsDictionary[bankFilePath].Add(waveFilePath);
                    }
                }
            }

            public void Remove(string bankFilePath)
            {
                this.bankFilePathDictionary[bankFilePath]--;
                if (this.bankFilePathDictionary[bankFilePath] <= 0)
                {
                    string[] waveFilePaths = waveFilePathsDictionary[bankFilePath].WaveFilePaths;
                    foreach (string waveFilePath in waveFilePaths)
                    {
                        this.waveFilePathsDictionary[bankFilePath].Remove(waveFilePath);
                    }
                    this.waveFilePathsDictionary.Remove(bankFilePath);
                    this.bankFilePathDictionary.Remove(bankFilePath);
                }
            }

            public void Update(string oldFilePath, string newFilePath, string[] newWaveFilePaths)
            {
                if (this.bankFilePathDictionary.ContainsKey(oldFilePath) == true)
                {
                    this.Remove(oldFilePath);
                }

                if (this.bankFilePathDictionary.ContainsKey(newFilePath) == true)
                {
                    this.Update(newFilePath, newWaveFilePaths);
                }
                else
                {
                    this.Add(newFilePath, newWaveFilePaths);
                }
            }

            public void Update(string bankFilePath, string[] newWaveFilePaths)
            {
                if (this.bankFilePathDictionary.ContainsKey(bankFilePath) == true)
                {
                    string[] oldWaveFilePaths = this.waveFilePathsDictionary[bankFilePath].WaveFilePaths;

                    foreach (string waveFilePath in newWaveFilePaths)
                    {
                        this.waveFilePathsDictionary[bankFilePath].Add(waveFilePath);
                    }

                    foreach (string waveFilePath in oldWaveFilePaths)
                    {
                        this.waveFilePathsDictionary[bankFilePath].Remove(waveFilePath);
                    }
                }
            }

            public string[] GetBankFilePathsByWaveFilePath(string waveFilePath)
            {
                List<string> bankFilePaths = new List<string>();

                foreach (string bankFilePath in this.bankFilePathDictionary.Keys.ToArray())
                {
                    if (this.waveFilePathsDictionary[bankFilePath].Contains(waveFilePath) == true)
                    {
                        bankFilePaths.Add(bankFilePath);
                    }
                }

                return bankFilePaths.ToArray();
            }

            private void OnWaveFileChanged(string waveFilePath)
            {
                if (this.waveFileChanged != null)
                {
                    this.waveFileChanged(waveFilePath);
                }
            }

            private class ReferenceWaveFilePath
            {
                private Dictionary<string, int> waveFilePathDictionary;
                private FileChangedEventHandler waveFileChanged;

                public ReferenceWaveFilePath(FileChangedEventHandler waveFileChanged)
                {
                    this.waveFilePathDictionary = new Dictionary<string, int>();
                    this.waveFileChanged = waveFileChanged;
                }


                public string[] WaveFilePaths
                {
                    get { return waveFilePathDictionary.Keys.ToArray(); }
                }

                public void Add(string waveFilePath)
                {
                    if (this.waveFilePathDictionary.ContainsKey(waveFilePath) == false)
                    {
                        this.waveFilePathDictionary.Add(waveFilePath, 0);
                        ApplicationBase.Instance.FileWatcherService.Add(waveFilePath, this.OnWaveFileChanged);
                    }

                    this.waveFilePathDictionary[waveFilePath]++;
                }

                public void Remove(string waveFilePath)
                {
                    if (this.waveFilePathDictionary.ContainsKey(waveFilePath) == true)
                    {
                        this.waveFilePathDictionary[waveFilePath]--;
                        if (this.waveFilePathDictionary[waveFilePath] <= 0)
                        {
                            ApplicationBase.Instance.FileWatcherService.Remove(waveFilePath, this.OnWaveFileChanged);
                            this.waveFilePathDictionary.Remove(waveFilePath);
                        }
                    }
                }

                public bool Contains(string waveFilePath)
                {
                    return this.waveFilePathDictionary.ContainsKey(waveFilePath);
                }

                private void OnWaveFileChanged(string waveFilePath)
                {
                    if (this.waveFileChanged != null)
                    {
                        this.waveFileChanged(waveFilePath);
                    }
                }
            }
        }

        private class UpdateTimer
        {
            private Action action;
            private ManualResetEvent resetEvent = new ManualResetEvent(false);

            private volatile Thread thread = null;

            public UpdateTimer(Action action)
            {
                this.action = action;
            }

            public void Start()
            {
                this.resetEvent.Reset();
                this.thread = new Thread(new ThreadStart(this.ThreadMain));
                this.thread.Name = this.ToString();
                this.thread.IsBackground = true;
                this.thread.Start();
            }

            public void Stop()
            {
                this.resetEvent.Set();
                if (this.thread != null)
                {
                    this.thread.Join();
                    this.thread = null;
                }
            }

            private void ThreadMain()
            {
                while (this.resetEvent.WaitOne(100) == false)
                {
                    this.action();
                }
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class BackgroundPartsConvertService
    {
        private SoundProjectService projectService = null;
        private ISoundProjectConvertService projectConvertService = null;

        private Queue<Task> taskQueue = new Queue<Task>();
        //private ManualResetEvent queuingEvent = new ManualResetEvent( false );

        private FormsBasedInvoker invoker = null;

        /// スレッド
        private volatile Thread thread = null;
        /// 終了フラグ
        private volatile bool wantToExit = false;

        ///
        public delegate void ConvertedEventHandler(SoundSetItem[] soundSetItems, Component dependentComponent, bool succeeded, bool canceled);
        public event ConvertedEventHandler Converted;
        public delegate bool ConvertHandler(SoundSetItem[] soundSetItems, Component dependentComponent);

        private Component convertingDependentComponent = null;
        private SoundSetItem[] convertingSoundSetItems = null;
        private HashSet<SoundSetItem> convertingSoundSetItemHashSet = new HashSet<SoundSetItem>();

        /// <summary>
        ///
        /// </summary>
        public BackgroundPartsConvertService(FormsBasedInvoker invoker)
        {
            this.invoker = invoker;
        }

        /// <summary>
        ///
        /// </summary>
        public void Send(SoundSetItem soundSetItem)
        {
            AddTask(new Task(soundSetItem, null));
        }

        /// <summary>
        ///
        /// </summary>
        public void Send(SoundSetItem[] soundSetItems, Component dependentComponent)
        {
            AddTask(new Task(soundSetItems, dependentComponent));
        }

        /// <summary>
        ///
        /// </summary>
        public void Clear()
        {
            lock (this.taskQueue)
            {
                this.taskQueue.Clear();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void Attach(SoundProjectService projectService, ISoundProjectConvertService projectConvertService)
        {
            this.projectService = projectService;
            this.projectConvertService = projectConvertService;

            projectConvertService.EndConvert += OnEndConvert;

            BeginThread();
        }

        /// <summary>
        ///
        /// </summary>
        public void Detach()
        {
            Debug.Assert(this.projectService != null, "Not set attach SoundProjectService");

            Clear();
            EndThread();

            this.projectConvertService.EndConvert -= OnEndConvert;

            this.projectService = null;
            this.projectConvertService = null;
        }

        /// <summary>
        /// 指定コンポーネントがコンバート中なのかを調べます。
        /// </summary>
        public bool IsConverting(Component component)
        {
            if (component is WaveSoundBase)
            {
                component = component.Parent;
            }

            if (component is SoundSetItem)
            {
                SoundSetItem soundSetItem = component as SoundSetItem;
                return this.convertingSoundSetItemHashSet.Contains(soundSetItem);
            }
            return false;
        }

        /// <summary>
        /// 指定コンポーネントがコンバート中なのかを調べます。
        /// </summary>
        public bool IsConverting(Component[] components)
        {
            foreach (Component component in components)
            {
                if (IsConverting(component) == true)
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnEndConvert(object sender, SoundProjectConvertEventArgs e)
        {
#if DEBUG
            if( this.convertingSoundSetItems != null )
            {
                Debug.WriteLine( "-------- EndConvert --------");
                foreach( SoundSetItem s in this.convertingSoundSetItems )
                {
                    Debug.WriteLine( s.Name);
                }
                Debug.WriteLine( "----------------------------");
            }
#endif

            Component convertedDependentComponent = this.convertingDependentComponent;
            SoundSetItem[] convertedSoundSetItems = this.convertingSoundSetItems;

            this.convertingDependentComponent = null;
            this.convertingSoundSetItems = null;
            this.convertingSoundSetItemHashSet.Clear();

            if (Converted != null)
            {
                Converted(convertedSoundSetItems, convertedDependentComponent,
                           e.Succeeded, e.Canceled);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private bool Convert(SoundSetItem[] soundSetItems, Component dependentComponent)
        {
            Debug.Assert(this.projectConvertService != null, "SoundProjectConvertService is null");

            if (this.projectConvertService.IsConverting == true)
            {
                //Debug.WriteLine( "Already converting other sound set item");
                return false;
            }

            Debug.WriteLine(String.Format("Converting... = {0} ({1})",
                                            soundSetItems[0].Name,
                                            soundSetItems.Length));

            this.convertingDependentComponent = dependentComponent;
            this.convertingSoundSetItems = soundSetItems;

            soundSetItems = ValidSoundSetItems(soundSetItems);

            // コンバート中のアイテムをHashSetに記録します。
            foreach (SoundSetItem soundSetItem in soundSetItems)
            {
                this.convertingSoundSetItemHashSet.Add(soundSetItem);
            }

            //
            this.projectConvertService.ConvertParts(
                this.projectService,
                SoundProjectConvertPartsSettings.CreateForReconvert(
                    string.Empty,
                    soundSetItems,
                    this.projectService.ProjectInGameEditCacheOutputPath,
                    false));

            return true;
        }

        /// <summary>
        ///
        /// </summary>
        private SoundSetItem[] ValidSoundSetItems(SoundSetItem[] soundSetItems)
        {
            HashSet<SoundSetItem> hashSet = new HashSet<SoundSetItem>();

            foreach (SoundSetItem soundSetItem in soundSetItems)
            {
                hashSet.Add(soundSetItem);
            }
            return hashSet.ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        private bool QueryConvert(SoundSetItem[] soundSetItems, Component dependentComponent)
        {
            FormsApplication app = ApplicationBase.Instance as FormsApplication;
            MainWindow mainWindow = app.UIService.MainWindow;

            try
            {
                ConvertHandler handler = new ConvertHandler(Convert);
                bool result = (bool)mainWindow.Invoke
                    (handler, new object[] { soundSetItems, dependentComponent });

                if (result == false)
                {
                    return false;
                }

                this.projectConvertService.Wait();
            }
            catch
            {
                Debug.WriteLine("Convert faild");
                return false;
            }

            Debug.WriteLine(String.Format("Converted = {0} ({1})",
                                           soundSetItems[0].Name,
                                           soundSetItems.Length));
            return true;
        }

        /// <summary>
        /// スレッド
        /// </summary>
        private bool BeginThread()
        {
            try
            {
                this.thread = new Thread(new ThreadStart(ThreadMain));
                this.thread.Name = ToString();
                this.thread.IsBackground = true;
                this.thread.Start();
            }

            catch (Exception)
            {
                return false;
            }
            return true;
        }




        ///
        private void EndThread()
        {
            if (this.thread != null)
            {
                if (this.wantToExit == true)
                {
                    this.wantToExit = true;
                    this.thread.Join(Timeout.Infinite);

                    if (this.thread != null)
                    {
                        this.thread.Abort();
                    }
                }
                else
                {
                    Thread.Sleep(100);
                }

                this.thread = null;
            }
        }

        /// <summary>
        /// スレッドのメインループ
        /// </summary>
        private void ThreadMain()
        {
            while (this.wantToExit == false)
            {
                Task task = GetTask();
                if (task == null)
                {
                    //this.queuingEvent.WaitOne( 50, false);
                }
                else
                {
                    SoundSetItem[] soundSetItems = task.SoundSetItems;
                    Component dependentComponent = task.DependentComponent;
                    if (QueryConvert(soundSetItems, dependentComponent) == false)
                    {
                        // コンバートに失敗したので再コンバートを要求します。
                        Send(soundSetItems, dependentComponent);
                    }
                }

                ///
                Thread.Sleep(10);
            }
        }

        /// <summary>
        /// タスク数の取得
        /// </summary>
        private int GetTaskCount()
        {
            lock (this.taskQueue)
            {
                return this.taskQueue.Count;
            }
        }

        /// <summary>
        /// タスクの追加
        /// </summary>
        private void AddTask(Task task)
        {
            lock (this.taskQueue)
            {
                this.taskQueue.Enqueue(task);
            }
        }

        /// <summary>
        /// タスクの取得
        /// </summary>
        private Task GetTask()
        {
            Task task = null;
            lock (this.taskQueue)
            {
                if (this.taskQueue.Count > 0)
                {
                    task = this.taskQueue.Dequeue();
                    //this.queuingEvent.Reset();
                }
            }
            return task;
        }

        /// <summary>
        ///
        /// </summary>
        private class Task
        {
            private SoundSetItem[] soundSetItems = null;
            private Component dependentComponent = null;

            public Task(SoundSetItem soundSetItem, Component dependentComponent) : this(new SoundSetItem[] { soundSetItem }, dependentComponent)
            {
            }

            public Task(SoundSetItem[] soundSetItems, Component dependentComponent)
            {
                this.soundSetItems = soundSetItems;
                this.dependentComponent = dependentComponent;
            }

            public SoundSetItem[] SoundSetItems
            {
                get
                {
                    return this.soundSetItems;
                }
            }

            public Component DependentComponent
            {
                get
                {
                    return this.dependentComponent;
                }
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class SoundEditInfoPack
    {
        private SoundEditInfo soundEditInfo = null;
        private SndEditPacketHeader.ResultTypes resultType;

        /// <summary>
        ///
        /// </summary>
        public SoundEditInfoPack(SoundEditInfo soundEditInfo, SndEditPacketHeader.ResultTypes resultType)
        {
            this.soundEditInfo = soundEditInfo;
            this.resultType = resultType;
        }

        public SoundEditInfo SoundEditInfo
        {
            get
            {
                return this.soundEditInfo;
            }
        }

        public SndEditPacketHeader.ResultTypes ResultType
        {
            get
            {
                return this.resultType;
            }
        }
    }
}
