﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Framework.Preview.Communications.Tool
{
    using NW4R.ProtocolSound;

    #region ツールシステムパケット
    /// <summary>
    /// ツールシステムパケット
    /// <para>
    /// ツールからビューアに送る、システム関連パケットのベースクラスです。
    /// </para>
    /// </summary>
    public abstract class ToolSystemPacket : ToolPacket
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        protected ToolSystemPacket()
        {
            Debug.Assert(Category == Constants.CATEGORY_SYSTEM, "Category is not system");
        }
    }
    #endregion

    #region プロトコルのバージョン
    /// <summary>
    /// プロトコルのバージョン
    /// <para>
    /// プロトコルのバージョンを確認するためのパケットです。
    /// </para>
    /// </summary>
    public class VersionPacket : ToolSystemPacket
    {
        //---------------------------------------------------------------------
        /// <summary>
        /// 送る
        /// </summary>
        public static void Send()
        {
            if (CommManager.Instance.IsConnecting == false &&
                CommManager.Instance.IsConnected == false)
            {
                return;
            }

            CommManager.Instance.SendPacket(new VersionPacket());
        }

        /// <summary>
        /// 書き出し
        /// </summary>
        public override CommPacketHeader Write(ProtocolSoundWriter writer)
        {
            CommPacketHeader header = CreateHeader();
            header.Write(writer);
            ToolSystemVersion packet = new ToolSystemVersion();
            packet.version = Constants.PROTOCOL_VERSION;
            packet.Write(writer);
            return header;
        }

        /// <summary>
        /// 読み込み
        /// </summary>
        public override void Read(
                                  ProtocolSoundReader reader, CommPacketHeader header)
        {
            Debug.Assert(header.Size == ToolSystemVersion.StructSize, "Unmatch header size");
            ToolSystemVersion packet = new ToolSystemVersion();
            packet.Read(reader);
            if (packet.version != Constants.PROTOCOL_VERSION)
            {
                ShowError("プロトコルのバージョンが違います。{0:x8} {1:x8}",
                          Constants.PROTOCOL_VERSION, packet.version);
            }

            CommManager.Instance.SetState(ConnectionState.Connected);
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// メッセージタイプ
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SYSTEM_VERSION; }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public override ushort Size
        {
            get { return ToolSystemVersion.StructSize; }
        }

        /// <summary>
        /// 文字列変換
        /// </summary>
        public override string ToString()
        {
            return base.ToString() + " " +
                Constants.PROTOCOL_VERSION.ToString("x8");
        }
    }
    #endregion

    //=========================================================================
    #region リセット
    /// <summary>
    /// リセット
    /// <para>
    /// ビューアをリセットするためのパケットです。
    /// </para>
    /// </summary>
    public class ResetPacket : ToolSystemPacket
    {
        //---------------------------------------------------------------------
        /// <summary>
        /// 送る
        /// </summary>
        public static void Send()
        {
            if (!CommManager.Instance.IsConnected) { return; }
            ResetPacket packet = new ResetPacket();
            CommManager.Instance.SendPacket(packet);
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// メッセージタイプ
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SYSTEM_RESET; }
        }
    }
    #endregion

    //=========================================================================
    #region 接続を閉じる
    /// <summary>
    /// 接続を閉じる
    /// <para>
    /// 現在送信しようとしているパケットを送りきった後に接続を閉じるパケットです。
    /// </para>
    /// </summary>
    public class DisconnectPacket : ToolSystemPacket
    {
        //---------------------------------------------------------------------
        /// <summary>
        /// 送る
        /// </summary>
        public static void Send(Control mainControl)
        {
            if (!CommManager.Instance.IsConnected) { return; }
            CommManager.Instance.SendPacket(new DisconnectPacket(mainControl));
        }

        /// <summary>
        /// 書き出し
        /// </summary>
        public override CommPacketHeader Write(ProtocolSoundWriter writer)
        {
            CommPacketHeader header = CreateHeader();
            header.Write(writer);
            ToolSystemSync packet = new ToolSystemSync();
            packet.tagName = 0;
            packet.Write(writer);
            return header;
        }

        /// <summary>
        /// 読み込み
        /// </summary>
        public override void Read(
                                  ProtocolSoundReader reader, CommPacketHeader header)
        {
            Debug.Assert(header.Size == ToolSystemSync.StructSize, "Unmatch header size");
            ToolSystemSync packet = new ToolSystemSync();
            packet.Read(reader);
            Debug.Assert(packet.tagName == 0, "Tag name is not zero");
            _mainControl.BeginInvoke(
                                     new MethodInvoker(CommManager.Instance.Disconnect));
        }

        /// <summary>
        /// 圧縮
        /// </summary>
        public override bool Compress(ToolPacket packet)
        {
            return (packet is DisconnectPacket);
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// メッセージタイプ
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SYSTEM_SYNC; }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public override ushort Size
        {
            get { return ToolSystemSync.StructSize; }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        protected DisconnectPacket(Control mainControl)
        {
            _mainControl = mainControl;
        }

        // メインスレッドコントロール
        private Control _mainControl;
    }
    #endregion
}
