﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;

namespace NintendoWare.SoundMaker.Framework.Preview.Communications.Tool
{
    using NW4R.ProtocolSound;

    /// <summary>
    /// サウンド再生制御パケットの基本クラス
    /// </summary>
    public abstract class PlaybackBasePacket : ToolSoundPacket
    {
        private uint _previewPlayerIndex = 0;   /// 対象プレビュープレイヤーのインデックス

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="previewPlayerIndex">対象プレビュープレイヤーのインデックス</param>
        /// <param name="name">対象サウンドの名前</param>
        public PlaybackBasePacket(uint previewPlayerIndex)
        {
            _previewPlayerIndex = previewPlayerIndex;
        }

        #region ** プロパティのオーバーライド

        /// <summary>
        /// サイズ
        /// </summary>
        public override ushort Size
        {
            get { return NW4R.ProtocolSound.ToolSoundPreviewPlayerID.StructSize; }
        }

        #endregion

        #region ** メソッドのオーバーライド

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override CommPacketHeader Write(ProtocolSoundWriter writer)
        {
            CommPacketHeader header = CreateHeader();
            header.Write(writer);

            ToolSoundPreviewPlayerID packet = new ToolSoundPreviewPlayerID();
            packet.index = _previewPlayerIndex;

            packet.Write(writer);

            return header;
        }

        #endregion
    }

    /// <summary>
    /// サウンド再生パケット
    /// </summary>
    public class PlaySoundPacket : PlaybackBasePacket
    {
        /// <summary>
        /// サウンド再生パケットを送信します。
        /// </summary>
        /// <param name="previewPlayerIndex">対象プレビュープレイヤーのインデックス</param>
        public static void Send(uint previewPlayerIndex)
        {
            if (!CommManager.Instance.IsConnected)
                return;
            SendPacket(new PlaySoundPacket(previewPlayerIndex));
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="previewPlayerIndex">対象プレビュープレイヤーのインデックス</param>
        /// <param name="name">対象サウンドの名前</param>
        public PlaySoundPacket(uint previewPlayerIndex) : base(previewPlayerIndex) { }

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_PLAY; }
        }
    }

    /// <summary>
    /// サウンド一時停止パケット
    /// </summary>
    public class PauseSoundPacket : PlaybackBasePacket
    {
        /// <summary>
        /// サウンド一時停止パケットを送信します。
        /// </summary>
        /// <param name="previewPlayerIndex">対象プレビュープレイヤーのインデックス</param>
        /// <param name="name">対象サウンドの名前</param>
        public static void Send(uint previewPlayerIndex)
        {
            if (!CommManager.Instance.IsConnected)
                return;
            SendPacket(new PauseSoundPacket(previewPlayerIndex));
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="previewPlayerIndex">対象プレビュープレイヤーのインデックス</param>
        /// <param name="name">対象サウンドの名前</param>
        public PauseSoundPacket(uint previewPlayerIndex) : base(previewPlayerIndex) { }

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_PAUSE; }
        }
    }

    /// <summary>
    /// サウンド停止パケット
    /// </summary>
    public class StopSoundPacket : PlaybackBasePacket
    {
        /// <summary>
        /// サウンド停止パケットを送信します。
        /// </summary>
        /// <param name="previewPlayerIndex">対象プレビュープレイヤーのインデックス</param>
        /// <param name="name">対象サウンドの名前</param>
        public static void Send(uint previewPlayerIndex)
        {
            if (!CommManager.Instance.IsConnected)
                return;
            SendPacket(new StopSoundPacket(previewPlayerIndex));
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="previewPlayerIndex">対象プレビュープレイヤーのインデックス</param>
        /// <param name="name">対象サウンドの名前</param>
        public StopSoundPacket(uint previewPlayerIndex) : base(previewPlayerIndex) { }

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_STOP; }
        }
    }

}
