﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;

namespace NintendoWare.SoundMaker.Framework.Preview.Communications.Tool
{
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Error;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Tool;
    using NW4R.ProtocolSound;

    /// <summary>
    /// ツールサウンドパケット
    /// </summary>
    public abstract class ToolSoundPacket : ToolPacket
    {
        /// <summary>
        ///
        /// </summary>
        public ToolSoundPacket()
        {
            Debug.Assert(Category == Constants.CATEGORY_SOUND, "Category is not sound");
        }
    }

    #region ** 同期パケットクラス

    #region ** イベントクラス

    /// <summary>
    ///
    /// </summary>
    public class ReplyEventArgsT<EventArgsType> : EventArgs
    where EventArgsType : EventArgs
    {
        #region ** イベントハンドラ デリゲート

        public delegate void EventHandler(object sender, EventArgsType e);

        #endregion

        #region ** パラメータ

        private uint _errorCode;
        private object _userData;

        #endregion

        public ReplyEventArgsT(uint errorCode, object userData)
        {
            _errorCode = errorCode;
            _userData = userData;
        }

        #region ** プロパティ

        public bool Failed
        {
            get { return (0 != (Constants.SOUND_ERRORFLAG & _errorCode)); }
        }

        public uint ErrorCode
        {
            get { return _errorCode; }
        }

        public object UserData
        {
            get { return _userData; }
        }

        #endregion
    }

    /// <summary>
    ///
    /// </summary>
    public class ReplyEventArgs : ReplyEventArgsT<ReplyEventArgs>
    {
        /// <summary>
        ///
        /// </summary>
        public ReplyEventArgs(uint errorCode, object userData) : base(errorCode, userData)
        {
        }
    }

    #endregion

    /// <summary>
    /// ツールサウンド同期パケット
    /// </summary>
    public abstract class ToolSoundSyncPacketT<EventArgsType> : ToolSoundPacket, IDisposable
        where EventArgsType : ReplyEventArgsT<EventArgsType>
    {
        #region ** パラメータ

        private ReplyEventArgsT<EventArgsType>.EventHandler _handler;
        private object _userData;

        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="handler">結果通知用デリゲート</param>
        /// <param name="userData">ユーザデータ</param>
        public ToolSoundSyncPacketT(ReplyEventArgsT<EventArgsType>.EventHandler handler, object userData)
        {
            _handler = handler;
            _userData = userData;
        }

        #region ** メソッドのオーバーライド

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public sealed override void Read(ProtocolSoundReader reader, CommPacketHeader header)
        {
            // エラーパケットを受け取るためのイベント設定解除
            SoundErrorPacket.PacketRecieved -= OnErrorPacketReceived;

            EventArgsType recvEvent = OnRead(reader, header);
            Debug.Assert(null != recvEvent, "Recive event is null");

            OnReply(recvEvent);
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public sealed override CommPacketHeader Write(ProtocolSoundWriter writer)
        {
            CommPacketHeader header = OnWrite(writer);

            // エラーパケットを受け取るためのイベント設定
            SoundErrorPacket.PacketRecieved += OnErrorPacketReceived;

            // 返信イベントを設定する
            SetHandler();

            return header;
        }

        #endregion

        #region ** IDisposable の実装

        void IDisposable.Dispose()
        {
            ClearHandler();
        }

        #endregion

        #region ** プロパティ

        protected object UserData
        {
            get { return _userData; }
        }

        #endregion

        #region ** イベント

        public event ReplyEventArgsT<EventArgsType>.EventHandler ReplyRecieved;

        #endregion

        #region ** イベントハンドラ

        protected virtual CommPacketHeader OnWrite(ProtocolSoundWriter writer) { return base.Write(writer); }
        protected virtual EventArgsType OnRead(ProtocolSoundReader reader, CommPacketHeader header) { return null; }
        protected virtual EventArgsType OnError(ErrorPacketRecievedEventArgs e) { return null; }

        protected virtual void OnReply(EventArgsType e)
        {
            try
            {
                if (null != ReplyRecieved)
                {
                    ReplyRecieved(this, e);
                }
            }
            finally
            {
                ClearHandler();
            }
        }

        private void OnErrorPacketReceived(object sender, ErrorPacketRecievedEventArgs e)
        {
            // エラーパケットを受け取るためのイベント設定解除
            SoundErrorPacket.PacketRecieved -= OnErrorPacketReceived;

            EventArgsType errorEvent = OnError(e);

            if (null != errorEvent)
            {
                OnReply(errorEvent);
            }
        }

        #endregion

        #region ** メソッド

        private void SetHandler()
        {
            if (null != _handler)
            {
                ReplyRecieved -= _handler;
                ReplyRecieved += _handler;
            }
        }

        private void ClearHandler()
        {
            if (null != _handler)
            {
                ReplyRecieved -= _handler;
                _handler = null;
            }

        }

        #endregion
    }

    public abstract class ToolSoundSyncPacket : ToolSoundSyncPacketT<ReplyEventArgs>
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="handler">結果通知用デリゲート</param>
        /// <param name="userData">ユーザデータ</param>
        public ToolSoundSyncPacket(ReplyEventArgs.EventHandler handler, object userData) : base(handler, userData) { }
    }

    #endregion

    /// <summary>
    /// サウンドアーカイブを読み込むパケットです。
    /// </summary>
    public class LoadSarSoundPacket : ToolSoundSyncPacket
    {
        #region ** パラメータ

        private string _soundArchiveFileName;

        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="modelFileName">サウンドアーカイブのファイルパス</param>
        /// <param name="handler">結果通知用デリゲート</param>
        /// <param name="userData">ユーザデータ</param>
        public LoadSarSoundPacket(string soundArchiveFileName, ReplyEventArgs.EventHandler handler, object userData)
            : base(handler, userData)
        {
            _soundArchiveFileName = soundArchiveFileName;
        }

        #region ** プロパティのオーバーライド

        /// <summary>
        /// パケットタイプ
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_LOADSAR; }
        }

        /// <summary>
        /// データサイズ
        /// </summary>
        public override ushort Size
        {
            get { return (ushort)(ToolSoundLoadSar.StructSize); }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// アニメーションプレイヤー作成パケットを送信します。
        /// </summary>
        public static bool Send(string soundArchiveFilePath)
        {
            return Send(soundArchiveFilePath, null);
        }
        public static bool Send(string soundArchiveFilePath, ReplyEventArgs.EventHandler handler)
        {
            return Send(soundArchiveFilePath, handler, null);
        }
        public static bool Send(string soundArchiveFilePath, ReplyEventArgs.EventHandler handler, object userData)
        {
            //if( !CommManager.Instance.IsConnected ) { return false; }

            SendPacket(new LoadSarSoundPacket(soundArchiveFilePath, handler, userData));
            return true;
        }

        #endregion

        #region ** イベントハンドラのオーバーライド

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        protected override CommPacketHeader OnWrite(ProtocolSoundWriter writer)
        {
            Debug.Assert(null != writer, "Writer is null");

            // パケットヘッダを出力する
            CommPacketHeader header = CreateHeader();
            header.Write(writer);

            // パラメータを出力する
            ToolSoundLoadSar parameter = new ToolSoundLoadSar();
            EncodeFilePath(_soundArchiveFileName, parameter.pathName);

            parameter.Write(writer);

            return header;
        }

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        protected override ReplyEventArgs OnRead(ProtocolSoundReader reader, CommPacketHeader header)
        {
            ToolSoundLoadSarReply reply = new ToolSoundLoadSarReply();
            reply.Read(reader);

            ShowMessage("SoundArchive loaded. \"{0}\" ID 0x{1:x8}",
                         Path.GetFileNameWithoutExtension(_soundArchiveFileName), reply.id);

            return new ReplyEventArgs(Constants.SOUND_S_OK, UserData);
        }

        protected override ReplyEventArgs OnError(ErrorPacketRecievedEventArgs e)
        {
            return new ReplyEventArgs(e.ErrorPacket.ErrorInfo.errorCode, UserData);
        }

        #endregion
    }

    ///
    public class CloseSarSoundPacket : ToolSoundSyncPacket
    {
        public CloseSarSoundPacket(ReplyEventArgs.EventHandler handler, object userData)
            : base(handler, userData)
        {
        }

        ///
        public static bool Send()
        {
            if (!(CommManager.Instance.IsConnecting || CommManager.Instance.IsConnected)) { return false; }

            bool replied = false;
            bool success = false;

            CloseSarSoundPacket packet = new CloseSarSoundPacket
                (delegate (object sender, ReplyEventArgs e)
                       {
                           replied = true;
                           success = !e.Failed;
                       },
                  null);

            CommManager.Instance.SendPacket(packet);

            //
            Stopwatch sw = new Stopwatch();
            sw.Start();
            while (replied == false)
            {
                System.Threading.Thread.Sleep(1);
                if (sw.ElapsedMilliseconds > 10 * 1000)
                {
                    return false;
                }
            }
            return success;
        }

        ///
        protected override CommPacketHeader OnWrite(ProtocolSoundWriter writer)
        {
            Debug.Assert(null != writer, "Writer is null");

            CommPacketHeader header = CreateHeader();
            header.Write(writer);
            return header;
        }

        ///
        protected override ReplyEventArgs OnRead(ProtocolSoundReader reader, CommPacketHeader header)
        {
            ToolSoundCloseSarReply reply = new ToolSoundCloseSarReply();
            reply.Read(reader);

            ShowMessage(string.Format("SoundArchive closed. Result 0x{0:x8}",
                                        reply.result));

            return new ReplyEventArgs
                (reply.result != 0 ? Constants.SOUND_S_OK : Constants.SOUND_E_FAILED
                  , UserData);
        }

        ///
        protected override ReplyEventArgs OnError(ErrorPacketRecievedEventArgs e)
        {
            return new ReplyEventArgs(e.ErrorPacket.ErrorInfo.errorCode, UserData);
        }

        /// <summary>
        /// メッセージタイプ
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_CLOSESAR; }
        }
    }

    /// <summary>
    /// 強制的にサウンドモードに切り替えるパケットです。
    /// </summary>
    public class FocusSoundPacket : ToolSoundPacket
    {
        public static FocusSoundPacket Send()
        {
            if (!CommManager.Instance.IsConnected) { return null; }
            FocusSoundPacket packet = new FocusSoundPacket();
            CommManager.Instance.SendPacket(packet);
            return packet;
        }
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_FOCUS; }
        }
    }

    /// <summary>
    /// 呼びかけて返信を要求するパケットです。
    /// 返信があると、ReadFlag が true になります。
    /// </summary>
    public class EchoSoundPacket : ToolSoundPacket
    {
        private bool readFlag;

        public static EchoSoundPacket Send()
        {
            if (!CommManager.Instance.IsConnected)
            {
                return null;
            }
            EchoSoundPacket packet = new EchoSoundPacket();
            SendPacket(packet);
            return packet;
        }
        public override void Read(ProtocolSoundReader reader, CommPacketHeader header)
        {
            this.readFlag = true;
        }
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_ECHO; }
        }

        public bool ReadFlag
        {
            get
            {
                return this.readFlag;
            }
        }
    }

    /// <summary>
    /// 全サウンドを停止するパケットです。
    /// </summary>
    public class AllStopSoundPacket : ToolSoundPacket
    {

        public static void Send()
        {
            if (!CommManager.Instance.IsConnected)
            {
                return;
            }
            SendPacket(new AllStopSoundPacket());
        }

        /// <summary>
        /// パケットタイプ
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_ALLSTOP; }
        }
    }

    /// <summary>
    /// サウンドシステム設定パケット
    /// </summary>
    public class SetSoundSystemParametersPacket : ToolSoundPacket
    {
        private ToolSoundOutputMode _outputMode = ToolSoundOutputMode.OutputMode_Invalid;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="outputMode">出力モード</param>
        public SetSoundSystemParametersPacket(ToolSoundOutputMode outputMode)
        {
            _outputMode = outputMode;
        }

        #region ** プロパティのオーバーライド

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_SET_SYSTEMPARAMETERS; }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public override ushort Size
        {
            get
            {
                return (ushort)ToolSoundSystemParameters.StructSize;
            }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// サウンド設定パケットを送信します。
        /// </summary>
        /// <param name="outputMode">出力モード</param>
        public static void Send(ToolSoundOutputMode outputMode)
        {
            if (!CommManager.Instance.IsConnected) { return; }
            SendPacket(new SetSoundSystemParametersPacket(outputMode));
        }

        #endregion

        #region ** メソッドのオーバーライド

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public override void Read(ProtocolSoundReader reader, CommPacketHeader header)
        {
            // Write のみ
            Debug.Assert(false, "Don't call this method");
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override CommPacketHeader Write(ProtocolSoundWriter writer)
        {
            Debug.Assert(null != writer, "Writer is null");

            // パケットヘッダを出力する
            CommPacketHeader header = CreateHeader();
            header.Write(writer);

            // パラメータを出力する
            ToolSoundSystemParameters parameters = new ToolSoundSystemParameters();
            parameters.outputMode = _outputMode;
            parameters.Write(writer);

            return header;
        }

        #endregion
    }
}
