﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Preview.Communications.Ctrl
{
    using System;
    using System.Diagnostics;
    using System.Text;
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Sre;

    /// <summary>
    ///
    /// </summary>
    public abstract class CtrlPacket : CommPacket
    {
        protected CtrlPacket()
        {
            Debug.Assert(!Error, "Error is true");
        }

        /// <summary>
        /// 返信フラグ
        /// </summary>
        public override bool Reply
        {
            get
            {
                return ((PacketType & Constants.SRE_MESSAGE_REPLY_FLAG) != 0);
            }
        }

        /// <summary>
        /// エラーフラグ
        /// </summary>
        public override bool Error
        {
            get
            {
                return false;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public override ushort PacketType
        {
            get
            {
                Debug.Assert(false, "Not impliment");
                return 0;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public override CommPacketHeader CreateHeader()
        {
            CtrlPacketHeader header = null;
            header = new CtrlPacketHeader(PacketType | Constants.SRE_CATEGORY_SNDCTRL, Size);

            return header;
        }

        /// <summary>
        ///
        /// </summary>
        protected uint Alignment4(uint value)
        {
            return ((value + 3) / 4) * 4;
        }

        /// <summary>
        ///
        /// </summary>
        protected ItemNamePack GetItemNamePack(string name)
        {
            int length = ActualItemNameSize(name);
            int nameLength = name.Length + 1;
            int paddingLength = length - nameLength;

            ItemNamePack pack = new ItemNamePack();
            pack.NameArray = Encoding.GetEncoding(932).GetBytes(name);
            pack.PaddingLength = (uint)(paddingLength + 1); // +1は'\0'の分
            pack.Length = length;
            return pack;
        }

        /// <summary>
        ///
        /// </summary>
        protected void WritePack(ProtocolSoundWriter writer, ItemNamePack pack)
        {
            writer.Write(pack.NameArray.Length);
            writer.Write(pack.NameArray);

            for (int loop = 0; loop < pack.PaddingLength; loop++)
            {
                writer.Write((byte)0);
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected int ActualItemNameSize(string name)
        {
            int nameLength = name.Length + 1; // '\0'の分 + 1
            return (nameLength / 4 + ((nameLength % 4) != 0 ? 1 : 0)) * 4;
        }

        /// <summary>
        ///
        /// </summary>
        protected class ItemNamePack
        {
            public byte[] NameArray { get; set; }
            public uint PaddingLength { get; set; }
            public int Length { get; set; }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class CtrlPacketHeader : SndEditPacketHeader
    {
        public CtrlPacketHeader(ProtocolSoundReader reader)
            : base(reader)
        {
        }

        public CtrlPacketHeader(uint packetType, uint size)
            : base(packetType, size)
        {
        }
    }

#if true
#if false
    /// <summary>
    ///
    /// </summary>
    public class CtrlPacketCafe : CtrlPacket
    {
        public override ushort PacketType
        {
            get
            {
                Debug.Assert(false, "Not impliment");
                return 0;
            }
        }

        public override CommPacketHeader CreateHeader()
        {
            CtrlPacketHeader header = null;
            header = new CtrlPacketHeader(PacketType | Constants.SRE_CATEGORY_SNDCTRL, Size);

            return header;
        }

        protected ItemNamePack GetItemNamePack(string name)
        {
            int length = ActualItemNameSize(name);
            int nameLength = name.Length + 1;
            int paddingLength = length - nameLength;

            ItemNamePack pack = new ItemNamePack();
            pack.NameArray = Encoding.GetEncoding(932).GetBytes(name);
            pack.PaddingLength = (uint)(paddingLength + 1); // +1は'\0'の分
            pack.Length = length;
            return pack;
        }

        protected void WritePack(ProtocolSoundWriter writer, ItemNamePack pack)
        {
            writer.Write(pack.NameArray.Length);
            writer.Write(pack.NameArray);

            for (int loop = 0; loop < pack.PaddingLength; loop++)
            {
                writer.Write((byte)0);
            }
        }

        protected int ActualItemNameSize(string name)
        {
            int nameLength = name.Length + 1; // '\0'の分 + 1
            return (nameLength / 4 + ((nameLength % 4) != 0 ? 1 : 0)) * 4;
        }

        protected class ItemNamePack
        {
            public byte[] NameArray { get; set; }
            public uint PaddingLength { get; set; }
            public int Length { get; set; }
        }
    }
#endif

    /// <summary>
    /// 再生要求パケットクラス
    /// </summary>
    public class PlaySoundPacket : CtrlPacket
    {
        private int soundHandleIndex;
        private string soundName;

        public PlaySoundPacket(int soundHandleIndex, string soundName)
        {
            this.soundHandleIndex = soundHandleIndex;
            this.soundName = soundName;
        }

        public static PlaySoundPacket Send(int soundHandleIndex, string soundName)
        {
            var manager = CommManager.Instance;
            if (manager.IsConnectedSndEdit == false)
            {
                return null;
            }

            PlaySoundPacket packet = new PlaySoundPacket(soundHandleIndex, soundName);
            manager.SendPacketByCtrlConnection(packet);
            return packet;
        }

        public override ushort PacketType
        {
            get { return Constants.SNDCTRL_PLAY_SOUND; }
        }

        public override ushort Size
        {
            get
            {
                return (ushort)(
                    4 + 4 + this.ActualItemNameSize(this.soundName
                    ));
            }
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override CommPacketHeader Write(ProtocolSoundWriter writer)
        {
            Debug.Assert(null != writer, "Writer is null");

            // パケットヘッダを出力する
            CommPacketHeader header = CreateHeader();
            header.Write(writer);

            writer.Write((UInt32)this.soundHandleIndex);
            this.WritePack(writer, this.GetItemNamePack(this.soundName));

            return header;
        }
    }

    /// <summary>
    /// 停止要求パケットクラス
    /// </summary>
    public class StopSoundPacket : CtrlPacket
    {
        private int soundHandleIndex;

        public StopSoundPacket(int soundHandleIndex)
        {
            this.soundHandleIndex = soundHandleIndex;
        }

        public static StopSoundPacket Send(int soundHandleIndex)
        {
            var manager = CommManager.Instance;
            if (manager.IsConnectedSndEdit == false)
            {
                return null;
            }

            StopSoundPacket packet = new StopSoundPacket(soundHandleIndex);
            manager.SendPacketByCtrlConnection(packet);
            return packet;
        }

        public override ushort PacketType
        {
            get { return Constants.SNDCTRL_STOP_SOUND; }
        }

        public override ushort Size
        {
            get
            {
                return 4;
            }
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override CommPacketHeader Write(ProtocolSoundWriter writer)
        {
            Debug.Assert(null != writer, "Writer is null");

            // パケットヘッダを出力する
            CommPacketHeader header = CreateHeader();
            header.Write(writer);

            writer.Write((UInt32)this.soundHandleIndex);

            return header;
        }
    }

    /// <summary>
    /// 一時停止要求パケットクラス
    /// </summary>
    public class PauseSoundPacket : CtrlPacket
    {
        private int soundHandleIndex;

        public PauseSoundPacket(int soundHandleIndex)
        {
            this.soundHandleIndex = soundHandleIndex;
        }

        public static PauseSoundPacket Send(int soundHandleIndex)
        {
            var manager = CommManager.Instance;
            if (manager.IsConnectedSndEdit == false)
            {
                return null;
            }

            PauseSoundPacket packet = new PauseSoundPacket(soundHandleIndex);
            manager.SendPacketByCtrlConnection(packet);
            return packet;
        }

        public override ushort PacketType
        {
            get { return Constants.SNDCTRL_PAUSE_SOUND; }
        }

        public override ushort Size
        {
            get
            {
                return 4;
            }
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override CommPacketHeader Write(ProtocolSoundWriter writer)
        {
            Debug.Assert(null != writer, "Writer is null");

            // パケットヘッダを出力する
            CommPacketHeader header = CreateHeader();
            header.Write(writer);

            writer.Write((UInt32)this.soundHandleIndex);

            return header;
        }
    }

#if false
    public class UpdateSoundHandleInfoPacket : CtrlPacketCafe
    {
        private static PreviewPlayerInformationCollectionCafe _informations = null;
        private static bool _received = true;

        private enum SoundHandleState
        {
            Disabled = 0,
            Stopped = 1,
            Playing = 2,
            Paused = 3,
        }

        public static PreviewPlayerInformationCollectionCafe Target
        {
            get { return _informations; }
            set { _informations = value; }
        }

        public static bool Received
        {
            get { return _received; }
            set { _received = value; }
        }

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.SNDCTRL_UPDATE_SOUND_HANDLE_INFO; }
        }

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public override void Read(ProtocolSoundReader reader, CommPacketHeader header)
        {
            _received = true;

            uint count = reader.ReadUInt32();

            // プレビュープレイヤー情報を解析する
            for (uint i = 0; i < count; i++)
            {
                SoundHandleState state = (SoundHandleState)reader.ReadUInt32();

                if (_informations == null || !_informations.Contains(i)) { continue; }

                _informations[i].State = ToPreviewPlayerState(state);

            }
        }

        private PreviewPlayerState ToPreviewPlayerState(SoundHandleState state)
        {
            switch (state)
            {
                case SoundHandleState.Stopped:
                    return PreviewPlayerState.Stopped;

                case SoundHandleState.Playing:
                    return PreviewPlayerState.Playing;

                case SoundHandleState.Paused:
                    return PreviewPlayerState.Paused;

                case SoundHandleState.Disabled:
                    return PreviewPlayerState.Disabled;
            }

            return PreviewPlayerState.Disabled;
        }
    }
#endif
#endif
}
