﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;

namespace NintendoWare.SoundMaker.Framework.Preview.Communications
{
    public abstract class CommPacket
    {
        /// <summary>
        /// ヘッダの作成
        /// </summary>
        public virtual CommPacketHeader CreateHeader()
        {
            return new CommPacketHeader(PacketType, Size);
        }

        /// <summary>
        /// 読み込み
        /// </summary>
        public virtual void Read(
                                 ProtocolSoundReader reader, CommPacketHeader header)
        {
            // デフォルトの動作は内容無しのパケットを想定
            Debug.Assert(header.Size == 0, "Header size is not zero");
        }

        /// <summary>
        /// 書き出し
        /// </summary>
        public virtual CommPacketHeader Write(ProtocolSoundWriter writer)
        {
            // デフォルトの動作は内容無しのパケットを想定
            Debug.Assert(Size == 0, "Packet size is not zero");
            CommPacketHeader header = CreateHeader();
            header.Write(writer);
            return header;
        }

        /// <summary>
        /// 文字列変換
        /// </summary>
        public override string ToString()
        {
            return this.GetType().Name + " " + CreateHeader().ToString();
        }

        //---------------------------------------------------------------------
        // プロパティ
        //---------------------------------------------------------------------
        /// <summary>
        /// メッセージタイプ
        /// </summary>
        public abstract ushort PacketType { get; }

        /// <summary>
        /// パケットのカテゴリ
        /// </summary>
        public ushort Category
        {
            get
            {
                return (ushort)(PacketType & Constants.HEADER_CATEGORY_MASK);
            }
        }

        /// <summary>
        /// パケットのカテゴリ内のコマンド
        /// </summary>
        public ushort Command
        {
            get
            {
                return (ushort)(PacketType & Constants.HEADER_COMMAND_MASK);
            }
        }

        /// <summary>
        /// 返信フラグ
        /// </summary>
        public virtual bool Reply
        {
            get
            {
                return ((PacketType & Constants.HEADER_REPLY_FLAG) != 0);
            }
        }

        /// <summary>
        /// エラーフラグ
        /// </summary>
        public virtual bool Error
        {
            get { return ((PacketType & Constants.HEADER_ERROR_FLAG) != 0); }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public virtual ushort Size { get { return 0; } }

        //---------------------------------------------------------------------
        /// <summary>
        /// メッセージ
        /// </summary>
        protected static void ShowMessage(string format, params object[] args)
        {
            CommManager.Instance.Console.WriteMessage(format, args);
        }

#if false
        /// <summary>
        /// メッセージをリソースから取得します。
        /// </summary>
        protected static string GetMessageByResource( string id)
        {
            return NintendoWare.AnimSoundMaker.Framework.AnimeSoundApplication.MessageService.Message( id);
        }
#endif

        /// <summary>
        /// エラー
        /// </summary>
        protected static void ShowError(string format, params object[] args)
        {
            throw new ApplicationException(string.Format(format, args));
        }

        //---------------------------------------------------------------------
        // 非公開メンバ
        //---------------------------------------------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        protected CommPacket()
        {
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// ファイルパスのエンコード
        /// </summary>
        protected void EncodeFilePath(string filePath, byte[] byteArray)
        {
            if (Encoding.Default.GetByteCount(filePath) > byteArray.Length)
            {
                ShowError("ファイルパスが長すぎます", filePath);
            }
            Encoding.Default.GetBytes(
                                      filePath, 0, filePath.Length, byteArray, 0);
        }
    }
}
