﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Preset
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;

    public class BookmarkAddedEventArgs : EventArgs
    {
        public BookmarkAddedEventArgs(BookmarkComponent bookmark) { this.Bookmark = bookmark; }
        public BookmarkComponent Bookmark { get; set; }
    }

    public delegate void BookmarkAddedEventHandler(object sender, BookmarkAddedEventArgs e);

    public class BookmarkService
    {
        private BookmarkConfiguration bookmarkConfiguration = null;

        public event EventHandler BookmarkChanged;
        public event BookmarkAddedEventHandler BookmarkAdded;

        public BookmarkService(BookmarkConfiguration config)
        {
            this.bookmarkConfiguration = config;
            this.BookmarkRoot = new BookmarkFolder(MessageResource.BookmarkRootFolderLabel);
            this.BookmarkRoot.Expanded = true;
        }

        protected BookmarkService()
        {
        }

        public BookmarkFolder BookmarkRoot { get; set; }
        public string ProjectFilePath { get; set; }
        public string BookmarkFilePath
        {
            get
            {
                return ProjectFilePath + ".bookmark";
            }
        }

        public void Clear()
        {
            if (this.BookmarkRoot != null)
            {
                this.BookmarkRoot.Children.Clear();
                this.OnChangedBookmark(this, EventArgs.Empty);
            }
        }

        public virtual void Add(Component component)
        {
            if (component != null)
            {
                this.Add(component, false);
            }
        }

        public void Add(Component component, bool top)
        {
            this.Add(this.BookmarkRoot.Children, component, top);
        }

        public void Add(ComponentList componentList, Component component, bool top)
        {
            if (component == null) return;

            SoundSet soundSet = this.GetSoundSet(component);
            string name = GetDefaultName(soundSet, component);
            Bookmark bookmark = new Bookmark(ApplicationBase.Instance.ProjectService.ComponentDictionary, name, soundSet, component);
            if (top == true && 0 < componentList.Count)
            {
                componentList.Insert(0, bookmark);
            }
            else
            {
                componentList.Add(bookmark);
            }
            this.OnBookmarkAdded(this, new BookmarkAddedEventArgs(bookmark));
        }

        public void Add(string name, string kind, string keyword, string[] parameters, string[] soundsets, string[] filters)
        {
            BookmarkFind bookmarkFind = new BookmarkFind(name, kind, keyword, parameters, soundsets, filters);
            ComponentList componentList = this.BookmarkRoot.Children;
            componentList.Add(bookmarkFind);
            this.OnBookmarkAdded(this, new BookmarkAddedEventArgs(bookmarkFind));
        }

        public void CreateBookmarkFolder(BookmarkComponent component)
        {
            BookmarkNameSettingDialog dialog = new BookmarkNameSettingDialog();
            dialog.Text = MessageResource.BookmarkCreateFolder_DialogTitle;
            dialog.NameLabelText = MessageResource.BookmarkCreateFolder_DialogNameLabel;

            if (component == null)
            {
                component = this.BookmarkRoot;
            }

            if (component is BookmarkFolder == true)
            {
                if (dialog.ShowDialog() == DialogResult.OK)
                {
                    BookmarkFolder folder = component as BookmarkFolder;
                    BookmarkFolder newFolder = new BookmarkFolder(dialog.BookmarkName);
                    folder.Children.Add(newFolder);
                }
            }
            else if ((component is Bookmark == true || component is BookmarkFind == true) &&
                     component.Parent != null)
            {
                BookmarkFolder folder = component.Parent as BookmarkFolder;
                if (dialog.ShowDialog() == DialogResult.OK)
                {
                    BookmarkFolder newFolder = new BookmarkFolder(dialog.BookmarkName);
                    folder.Children.Insert(component, newFolder);
                }
            }

            this.OnChangedBookmark(this, EventArgs.Empty);
        }

        public void RenameBookmark(BookmarkComponent component)
        {
            BookmarkNameSettingDialog dialog = new BookmarkNameSettingDialog();
            dialog.Text = MessageResource.BookmarkRename_DialogTitle;
            dialog.NameLabelText = MessageResource.BookmarkRename_DialogNameLabel;
            dialog.BookmarkName = component.Name;

            if (component is BookmarkFolder)
            {
                dialog.NameLabelText = MessageResource.BookmarkCreateFolder_DialogNameLabel;
            }

            if (dialog.ShowDialog() == DialogResult.OK)
            {
                component.Name = dialog.BookmarkName;
            }

            this.OnChangedBookmark(this, EventArgs.Empty);
        }

        public void Delete(BookmarkComponent component)
        {
            this.DeleteBookmark(this.BookmarkRoot, component);

            this.OnChangedBookmark(this, EventArgs.Empty);
        }

        //
        public void ComponentsAdded(Component[] components)
        {
            foreach (Component component in components)
            {
                this.ComponentAdded(this.BookmarkRoot, component);
            }

            this.OnChangedBookmark(this, EventArgs.Empty);
        }

        public void ComponentsRemoved(Component[] components)
        {
            foreach (Component component in components)
            {
                this.ComponentRemoved(this.BookmarkRoot, component);
            }

            this.OnChangedBookmark(this, EventArgs.Empty);
        }

        public void ComponentNameChanged(string oldName, string newName)
        {
            if (string.IsNullOrEmpty(oldName) == false &&
                string.IsNullOrEmpty(newName) == false)
            {
                this.ComponentNameChanged(this.BookmarkRoot, oldName, newName);
            }
        }

        public string GetDefaultName(Bookmark bookmark)
        {
            if (bookmark.SoundSet == null || bookmark.Target == null)
            {
                return bookmark.Name;
            }

            return this.GetDefaultName(bookmark.SoundSet, bookmark.Target);
        }

        public void UpdateTarget()
        {
            XmlBookmarkConfiguration config = this.GetBookmarkConfiguration();
            this.SetBookmarkConfiguration(config);
            this.OnChangedBookmark(this, EventArgs.Empty);
        }

        public void Save()
        {
            this.bookmarkConfiguration.BookmarkList = this.GetBookmarkConfiguration();
            this.bookmarkConfiguration.Save(this.BookmarkFilePath);
        }

        public void Load()
        {
            this.bookmarkConfiguration.Load(this.BookmarkFilePath);
            this.SetBookmarkConfiguration(this.bookmarkConfiguration.BookmarkList);
        }

        protected void SetBookmarkConfiguration(XmlBookmarkConfiguration config)
        {
            this.BookmarkRoot.Children.Clear();

            if (config != null && config.Bookmarks != null && config.Bookmarks.Length == 1)
            {
                XmlBookmark xmlRoot = config.Bookmarks[0];
                this.BookmarkRoot = this.TranslateXmlBookmarkToComponentBookmarkFolder(xmlRoot);
            }
        }

        protected XmlBookmarkConfiguration GetBookmarkConfiguration()
        {
            XmlBookmarkConfiguration config = new XmlBookmarkConfiguration();
            XmlBookmark root = this.TranslateComponentBookmarkFolderToXmlBookmark(this.BookmarkRoot);
            config.Bookmarks = new XmlBookmark[] { root };

            return config;
        }

        private BookmarkFolder TranslateXmlBookmarkToComponentBookmarkFolder(XmlBookmark xmlFolder)
        {
            BookmarkFolder bookmarkFolder = null;

            if (xmlFolder != null)
            {
                bookmarkFolder = new BookmarkFolder(xmlFolder.Name);
                bookmarkFolder.Expanded = xmlFolder.Expanded;

                if (xmlFolder.Bookmarks != null)
                {
                    foreach (XmlBookmark xmlBookmark in xmlFolder.Bookmarks)
                    {
                        if (xmlBookmark.Type == "Folder")
                        {
                            BookmarkFolder folder =
                                this.TranslateXmlBookmarkToComponentBookmarkFolder(xmlBookmark);
                            bookmarkFolder.Children.Add(folder);
                        }
                        else if (xmlBookmark.Type == "Find")
                        {
                            BookmarkFind bookmarkFind = new BookmarkFind(xmlBookmark.Name,
                                                                         xmlBookmark.FindKind,
                                                                         xmlBookmark.FindKeyword,
                                                                         xmlBookmark.FindParameters,
                                                                         xmlBookmark.FindSoundSets,
                                                                         xmlBookmark.FindFilters);
                            bookmarkFolder.Children.Add(bookmarkFind);
                        }
                        else
                        {
                            Bookmark bookmark = new Bookmark(ApplicationBase.Instance.ProjectService.ComponentDictionary,
                                                             xmlBookmark.Name,
                                                             xmlBookmark.SoundSetName,
                                                             xmlBookmark.Type,
                                                             xmlBookmark.TargetName,
                                                             xmlBookmark.DateTime);
                            bookmarkFolder.Children.Add(bookmark);
                        }
                    }
                }
            }

            return bookmarkFolder;
        }

        private XmlBookmark TranslateComponentBookmarkFolderToXmlBookmark(BookmarkFolder folder)
        {
            XmlBookmark xmlBookmarkFolder = null;

            if (folder != null)
            {
                xmlBookmarkFolder = new XmlBookmark();
                xmlBookmarkFolder.Name = folder.Name;
                xmlBookmarkFolder.Type = "Folder";
                xmlBookmarkFolder.Expanded = folder.Expanded;
                List<XmlBookmark> bookmarks = new List<XmlBookmark>();

                foreach (BookmarkComponent node in folder.Children)
                {
                    if (node is BookmarkFolder == true)
                    {
                        XmlBookmark xmlFolder = this.TranslateComponentBookmarkFolderToXmlBookmark(node as BookmarkFolder);
                        if (xmlFolder != null)
                        {
                            bookmarks.Add(xmlFolder);
                        }
                    }
                    else if (node is Bookmark == true)
                    {
                        Bookmark bookmark = node as Bookmark;
                        XmlBookmark xmlBookmark = new XmlBookmark();
                        xmlBookmark.Name = bookmark.Name;
                        xmlBookmark.SoundSetName = bookmark.SoundSetName;
                        xmlBookmark.Type = bookmark.Type;

                        DateTime dateTime;
                        if (DateTime.TryParse(bookmark.DateTime, out dateTime) == true)
                        {
                            xmlBookmark.DateTime = dateTime.ToBinary().ToString();
                        }

                        xmlBookmark.Bookmarks = null;
                        if (bookmark.Target == null ||
                            bookmark.Target is WaveSoundSetBase ||
                            bookmark.Target is SequenceSoundSetBase ||
                            bookmark.Target is SoundSetBankBase)
                        {
                            xmlBookmark.TargetName = bookmark.TargetName;
                        }
                        bookmarks.Add(xmlBookmark);
                    }
                    else if (node is BookmarkFind == true)
                    {
                        BookmarkFind bookmarkFind = node as BookmarkFind;
                        XmlBookmark xmlBookmark = new XmlBookmark();
                        xmlBookmark.Name = bookmarkFind.Name;
                        xmlBookmark.Type = "Find";
                        xmlBookmark.FindKind = bookmarkFind.FindKind;
                        xmlBookmark.FindKeyword = bookmarkFind.FindKeyword;
                        xmlBookmark.FindParameters = bookmarkFind.FindParameters;
                        xmlBookmark.FindSoundSets = bookmarkFind.FindSoundSets;
                        xmlBookmark.FindFilters = bookmarkFind.FindFilters;
                        bookmarks.Add(xmlBookmark);
                    }
                }

                xmlBookmarkFolder.Bookmarks = bookmarks.ToArray();
            }

            return xmlBookmarkFolder;
        }

        private string GetDefaultName(SoundSet soundset, Component component)
        {
            string type = GetTypeText(component);
            string name = soundset.Name + "/" + type;

            if (component is WaveSoundSetBase == true ||
                component is SequenceSoundSetBase == true ||
                component is SoundSetBankBase == true)
            {
                return name + "/" + component.Name;
            }

            return name;
        }

        private string GetTypeText(Component component)
        {
            if (component is StreamSoundPack == true)
            {
                return MessageResource.ProjectCtrl_StreamSound;
            }
            else if (component is WaveSoundSetPack == true)
            {
                return MessageResource.ProjectCtrl_WaveSoundSet;
            }
            else if (component is WaveSoundSetBase == true)
            {
                return MessageResource.ProjectCtrl_WaveSoundSet;
            }
            else if (component is SequenceSoundPack == true)
            {
                return MessageResource.ProjectCtrl_SequenceSound;
            }
            else if (component is SequenceSoundSetPack == true)
            {
                return MessageResource.ProjectCtrl_SequenceSoundSet;
            }
            else if (component is SequenceSoundSetBase == true)
            {
                return MessageResource.ProjectCtrl_SequenceSoundSet;
            }
            else if (component is SoundSetBankPack == true)
            {
                return MessageResource.ProjectCtrl_Bank;
            }
            else if (component is SoundSetBankBase == true)
            {
                return MessageResource.ProjectCtrl_Bank;
            }
            else if (component is WaveArchivePack == true)
            {
                return MessageResource.ProjectCtrl_WaveArchive;
            }
            else if (component is GroupPack == true)
            {
                return MessageResource.ProjectCtrl_Group;
            }
            else if (component is PlayerPack == true)
            {
                return MessageResource.ProjectCtrl_Player;
            }
            else return string.Empty;
        }

        //
        private void ComponentAdded(BookmarkFolder folder, Component target)
        {
            foreach (Component component in folder.Children)
            {
                if (component is BookmarkFolder)
                {
                    this.ComponentAdded(component as BookmarkFolder, target);
                }
                else if (component is Bookmark)
                {
                    Bookmark bookmark = component as Bookmark;
                    if (bookmark.TargetName == target.Name)
                    {
                        bookmark.Target = target;
                    }
                }
            }
        }

        private void ComponentRemoved(BookmarkFolder folder, Component target)
        {
            foreach (Component component in folder.Children)
            {
                if (component is BookmarkFolder)
                {
                    this.ComponentRemoved(component as BookmarkFolder, target);
                }
                else if (component is Bookmark)
                {
                    Bookmark bookmark = component as Bookmark;
                    if (bookmark.Target == target)
                    {
                        bookmark.Target = null;
                    }
                }
            }
        }

        //
        private SoundSet GetSoundSet(Component component)
        {
            if (component == null)
            {
                return null;
            }

            if (component is SoundSet == true)
            {
                return component as SoundSet;
            }

            return this.GetSoundSet(component.Parent);
        }

        private bool DeleteBookmark(BookmarkFolder bookmarkFolder, Component component)
        {
            foreach (Component bookmark in bookmarkFolder.Children)
            {
                if (bookmark == component)
                {
                    bookmarkFolder.Children.Remove(bookmark);
                    return true;
                }
                else if (bookmark is BookmarkFolder == true)
                {
                    BookmarkFolder folder = bookmark as BookmarkFolder;
                    if (this.DeleteBookmark(folder, component) == true)
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        private void ComponentNameChanged(BookmarkFolder folder, string oldName, string newName)
        {
            foreach (Component component in folder.Children)
            {
                if (component is BookmarkFolder)
                {
                    this.ComponentNameChanged(component as BookmarkFolder, oldName, newName);
                }
                else if (component is Bookmark)
                {
                    Bookmark bookmark = component as Bookmark;
                    if (bookmark.Target != null &&
                        bookmark.Target.Name == newName &&
                        bookmark.TargetName == oldName)
                    {
                        bookmark.TargetName = newName;
                    }
                    else if (bookmark.SoundSet != null &&
                             bookmark.SoundSet.Name == newName &&
                             bookmark.SoundSetName == oldName)
                    {
                        bookmark.SoundSetName = newName;
                    }
                }
            }
        }

        private void OnBookmarkAdded(object sender, BookmarkAddedEventArgs e)
        {
            if (this.BookmarkAdded != null)
            {
                this.BookmarkAdded(sender, e);
            }
        }

        private void OnChangedBookmark(object sender, EventArgs e)
        {
            if (this.BookmarkChanged != null)
            {
                this.BookmarkChanged(sender, e);
            }
        }
    }
}
