﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace NintendoWare.SoundMaker.Framework.Preset
{
    using NintendoWare.SoundFoundation.Projects;

    public class Bookmark : BookmarkComponent
    {
        public const string TypeStringWaveSoundSet = "WaveSoundSet";
        public const string TypeStringSequenceSoundSet = "SequenceSoundSet";
        public const string TypeStringSoundSetBank = "SoundSetBank";

        public const string TypeStringStreamSoundPack = "StreamSoundPack";
        public const string TypeStringWaveSoundSetPack = "WaveSoundSetPack";
        public const string TypeStringSequenceSoundPack = "SequenceSoundPack";
        public const string TypeStringSequenceSoundSetPack = "SequenceSoundSetPack";
        public const string TypeStringSoundSetBankPack = "SoundSetBankPack";
        public const string TypeStringWaveArchivePack = "WaveArchivePack";
        public const string TypeStringGroupPack = "GroupPack";
        public const string TypeStringPlayerPack = "PlayerPack";

        private ComponentDictionary componentDictionary;

        public Bookmark(ComponentDictionary componentDictionary,
                        string name, string soundSetName,
                        string type, string targetName, string dateTime)
        {
            this.componentDictionary = componentDictionary;
            this.Name = name;
            this.SoundSetName = soundSetName;
            this.Type = type;
            this.TargetName = targetName;

            long result;
            if (Int64.TryParse(dateTime, out result) == true)
            {
                DateTime time = System.DateTime.FromBinary(result);
                this.DateTime = time.ToString("d") + time.ToString(" HH:mm");
            }

            if (componentDictionary != null)
            {
                Component[] components;

                if (componentDictionary.Contains(soundSetName) == true)
                {
                    components = componentDictionary[soundSetName];
                    foreach (Component component in components)
                    {
                        if (component is SoundSet == true)
                        {
                            this.SoundSet = component as SoundSet;
                            break;
                        }
                    }
                }

                if (IsComponentPack(type) == true)
                {
                    this.Target = this.GetComponentPack(this.SoundSet, type);
                    this.TargetName = string.Empty;
                }
                else if (componentDictionary.Contains(targetName) == true)
                {
                    components = componentDictionary[targetName];
                    foreach (Component component in components)
                    {
                        if (this.GetTypeString(component) == type)
                        {
                            this.Target = component;
                            break;
                        }
                    }
                }
            }
        }

        public Bookmark(ComponentDictionary componentDictionary, string name, SoundSet soundSet, Component target)
        {
            this.componentDictionary = componentDictionary;
            this.Name = name;
            this.SoundSet = soundSet;
            this.SoundSetName = soundSet.Name;
            this.Target = target;
            this.Type = this.GetTypeString(this.Target);
            DateTime dateTime = System.DateTime.Now;
            this.DateTime = dateTime.ToString("d") + dateTime.ToString(" HH:mm");

            if (IsComponentPack(target) == false)
            {
                this.TargetName = target.Name;
            }
        }

        public SoundSet SoundSet { get; set; }
        public Component Target { get; set; }
        public string SoundSetName { get; set; }
        public string Type { get; set; }
        public string TargetName { get; set; }
        public string DateTime { get; set; }

        public bool ValidateTarget()
        {
            if (this.Target == null) return false;
            if (this.componentDictionary.Contains(this.Target.Name) == false) return false;

            return true;
        }

        private bool IsComponentPack(string type)
        {
            if (type == TypeStringStreamSoundPack) return true;
            else if (type == TypeStringWaveSoundSetPack) return true;
            else if (type == TypeStringSequenceSoundPack) return true;
            else if (type == TypeStringSequenceSoundSetPack) return true;
            else if (type == TypeStringSoundSetBankPack) return true;
            else if (type == TypeStringWaveArchivePack) return true;
            else if (type == TypeStringGroupPack) return true;
            else if (type == TypeStringPlayerPack) return true;
            else return false;
        }

        private bool IsComponentPack(Component component)
        {
            if (component is StreamSoundPack) return true;
            else if (component is WaveSoundSetPack) return true;
            else if (component is SequenceSoundPack) return true;
            else if (component is SequenceSoundSetPack) return true;
            else if (component is SoundSetBankPack) return true;
            else if (component is WaveArchivePack) return true;
            else if (component is GroupPack) return true;
            else if (component is PlayerPack) return true;
            else return false;
        }

        private string GetTypeString(Component component)
        {
            if (component is StreamSoundPack) return TypeStringStreamSoundPack;
            else if (component is WaveSoundSetPack) return TypeStringWaveSoundSetPack;
            else if (component is WaveSoundSetBase) return TypeStringWaveSoundSet;
            else if (component is SequenceSoundPack) return TypeStringSequenceSoundPack;
            else if (component is SequenceSoundSetPack) return TypeStringSequenceSoundSetPack;
            else if (component is SequenceSoundSetBase) return TypeStringSequenceSoundSet;
            else if (component is SoundSetBankPack) return TypeStringSoundSetBankPack;
            else if (component is SoundSetBankBase) return TypeStringSoundSetBank;
            else if (component is WaveArchivePack) return TypeStringWaveArchivePack;
            else if (component is GroupPack) return TypeStringGroupPack;
            else if (component is PlayerPack) return TypeStringPlayerPack;
            else return string.Empty;
        }

        private Component GetComponentPack(SoundSet soundSet, string type)
        {
            if (soundSet != null)
            {
                foreach (Component child in SoundSet.Children)
                {
                    if (this.GetTypeString(child) == type)
                    {
                        return child;
                    }
                }
            }

            return null;
        }
    }
}
