﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.FileManagement
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;

    public delegate void FileChangedEventHandler(string filePath);

    public class FileWatcherService
    {
        /// <summary>
        /// ファイル状態のポーリング間隔(ミリ秒)です。
        /// </summary>
        private const int Interval = 250;

        /// <summary>
        /// ファイルの内容の変化だけを監視したい場合に使用するオブジェクト群。
        /// </summary>
        private readonly WatchData _watchDataForChangeOnly;

        /// <summary>
        /// ファイルの内容の変化に加えてリネームと削除を監視したい場合に使用するオブジェクト群。
        /// </summary>
        private readonly WatchData _watchDataForChangeAndRename;

        private Timer updateFilesTimer = null;
        private bool delayAdd = false;

        public FileWatcherService()
        {
            _watchDataForChangeOnly = new WatchData(watchChangeOnly: true);
            _watchDataForChangeAndRename = new WatchData(watchChangeOnly: false);

            this.updateFilesTimer = new Timer();
            this.updateFilesTimer.Tick += this.CheckUpdateFiles;
            this.updateFilesTimer.Interval = Interval;
            this.updateFilesTimer.Enabled = Design.IsDesignMode == false ? true : false;
        }

        public bool Enabled
        {
            get { return this.updateFilesTimer.Enabled; }
            set { this.updateFilesTimer.Enabled = value; }
        }

        private IEnumerable<WatchData> WatchDatas
        {
            get
            {
                yield return _watchDataForChangeOnly;
                yield return _watchDataForChangeAndRename;
            }
        }

        private WatchData GetWatchData(bool watchChangeOnly)
        {
            return watchChangeOnly ? _watchDataForChangeOnly : _watchDataForChangeAndRename;
        }

        /// <summary>
        /// ファイル監視を登録します。
        /// <para>
        /// <paramref name="watchChangeOnly"/> が true の場合、ファイルの内容の変化のみ監視します。指定のファイルが存在しない場合は監視を開始しません。
        /// </para>
        /// <para>
        /// <paramref name="watchChangeOnly"/> が false の場合は、ファイルの内容の変化に加えてリネームと削除も監視します。また指定のファイルが存在しなくても監視を開始します。
        /// </para>
        /// </summary>
        /// <param name="filePath"></param>
        /// <param name="fileChanged"></param>
        /// <param name="watchChangeOnly">ファイルのリネームと削除も監視したい場合は false を指定します。</param>
        public void Add(string filePath, FileChangedEventHandler fileChanged, bool watchChangeOnly = true)
        {
            if (filePath == null)
            {
                throw new ArgumentNullException(nameof(filePath));
            }

            if (fileChanged == null)
            {
                throw new ArgumentNullException(nameof(fileChanged));
            }

            var watchData = this.GetWatchData(watchChangeOnly);

            watchData.Add(filePath, fileChanged, this.delayAdd);
        }

        /// <summary>
        /// ファイル監視の登録を削除します。
        /// <para>
        /// <paramref name="watchChangeOnly"/> には <see cref="Add"/> に指定したのと同じ値を指定します。
        /// </para>
        /// </summary>
        /// <param name="filePath"></param>
        /// <param name="fileChanged"></param>
        /// <param name="watchChangeOnly"></param>
        public void Remove(string filePath, FileChangedEventHandler fileChanged, bool watchChangeOnly = true)
        {
            if (filePath == null)
            {
                throw new ArgumentNullException(nameof(filePath));
            }

            if (fileChanged == null)
            {
                throw new ArgumentNullException(nameof(fileChanged));
            }

            var watchData = this.GetWatchData(watchChangeOnly);

            watchData.Remove(filePath, fileChanged);
        }

        public void CheckClear()
        {
            this.WatchDatas.ForEach(it => it.CheckClear());
        }

        public void BeginDelayAdd()
        {
            Debug.Assert(this.delayAdd == false);

            this.delayAdd = true;

            this.WatchDatas.ForEach(it => it.ClearDelayAdd());
        }

        public void EndDelayAdd()
        {
            Debug.Assert(this.delayAdd == true);

            this.delayAdd = false;

            this.WatchDatas.ForEach(it => it.ApplyDelayAdd());
        }

        public void CancelDelayAdd()
        {
            this.WatchDatas.ForEach(it => it.ClearDelayAdd());

            this.delayAdd = false;
        }

        private void CheckUpdateFiles(object sender, EventArgs e)
        {
            this.WatchDatas.ForEach(it => it.CheckUpdateFiles());
        }

        /// <summary>
        /// ファイル監視に使用するオブジェクト群をまとめたクラスです。
        /// </summary>
        private class WatchData
        {
            private readonly FileSystemWatcherManager _fileSystemWatcherManager;
            private readonly Dictionary<string, FileChangedEventHandler> _fileChangedEventHandlerDictionary = new Dictionary<string, FileChangedEventHandler>();
            private readonly List<Tuple<string, FileChangedEventHandler>> _delayAddList = new List<Tuple<string, FileChangedEventHandler>>();

            public WatchData(bool watchChangeOnly)
            {
                _fileSystemWatcherManager = new FileSystemWatcherManager(watchChangeOnly);
            }

            public void Add(string filePath, FileChangedEventHandler fileChanged, bool delayAdd)
            {
                if (delayAdd == true)
                {
                    _delayAddList.Add(Tuple.Create(filePath, fileChanged));
                }
                else
                {
                    // ファイルの変化だけを監視したい場合、ファイルが存在しないときは監視をしません。
                    if (_fileSystemWatcherManager.WatchChangeOnly == true)
                    {
                        if ( File.Exists(filePath) == false)
                        {
                            return;
                        }
                    }

                    if (_fileChangedEventHandlerDictionary.ContainsKey(filePath) == false)
                    {
                        _fileChangedEventHandlerDictionary.Add(filePath, null);
                    }

                    _fileChangedEventHandlerDictionary[filePath] += fileChanged;
                    _fileSystemWatcherManager.Add(filePath);
                }
            }

            public void Remove(string filePath, FileChangedEventHandler fileChanged)
            {
                if (_fileChangedEventHandlerDictionary.ContainsKey(filePath) == true)
                {
                    _fileChangedEventHandlerDictionary[filePath] -= fileChanged;
                    if (_fileChangedEventHandlerDictionary[filePath] == null)
                    {
                        _fileChangedEventHandlerDictionary.Remove(filePath);
                        _fileSystemWatcherManager.Remove(filePath);
                    }
                }
                else
                {
                    _delayAddList.Remove(Tuple.Create(filePath, fileChanged));
                }
            }

            public void ApplyDelayAdd()
            {
                foreach (Tuple<string, FileChangedEventHandler> tuple in _delayAddList)
                {
                    this.Add(tuple.Item1, tuple.Item2, delayAdd: false);
                }

                _delayAddList.Clear();
            }

            public void ClearDelayAdd()
            {
                _delayAddList.Clear();
            }

            public void CheckClear()
            {
                string[] updateFilePaths = _fileSystemWatcherManager.GetFilePaths();
            }

            public void CheckUpdateFiles()
            {
                string[] updateFilePaths = _fileSystemWatcherManager.GetFilePaths();
                foreach (string filePath in updateFilePaths)
                {
                    if (_fileChangedEventHandlerDictionary.ContainsKey(filePath) == true)
                    {
                        _fileChangedEventHandlerDictionary[filePath]?.Invoke(filePath);
                    }
                }
            }
        }
    }

    /// デバッグ用？
    internal class Design
    {
        public static bool IsDesignMode
        {
            get
            {
                bool returnFlag = false;

#if DEBUG
                if (System.ComponentModel.LicenseManager.UsageMode == System.ComponentModel.LicenseUsageMode.Designtime)
                {
                    returnFlag = true;
                }
                else if (Process.GetCurrentProcess().ProcessName.ToUpper().Equals("DEVENV"))
                {
                    returnFlag = true;
                }
#endif

                return returnFlag;
            }
        }
    }
}
