﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Xml;
    using System.Xml.Schema;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Resources;
    using Schemas = NintendoWare.SoundMaker.Framework.Configurations.Schemas;

    /// <summary>
    /// サウンドプロジェクト設定を格納します。
    /// </summary>
    public abstract class SoundProjectConfiguration : ConfigurationBase
    {
        protected const string NameSpaceOfWindowConfiguration = "NintendoWare.SoundMaker.WindowConfiguration";
        protected const string NameSpaceOfUriConfiguration = "NintendoWare.SoundMaker.UriConfiguration";
        protected const string NameSpaceOfSoundProjectConfiguration = "NintendoWare.SoundMaker.SoundProjectConfiguration";
        protected const string ElementPrefixOfWindowConfiguration = "nwsw";
        protected const string ElementPrefixOfUriConfiguration = "nwsu";
        protected const string ElementPrefixOfSoundProjectConfiguration = "nwsp";

        // コンフィギュレーション情報
        private Schemas.XmlMainWindow _xmlMainWindow;
        private IDictionary<string, Schemas.XmlDocumentView> _xmlDocumentViews;
        private Schemas.XmlOutput _xmlOutput;
        private Schemas.XmlConvert _xmlConvert;
        private IDictionary<string, Schemas.XmlUri> _xmlUris = new Dictionary<string, Schemas.XmlUri>();
        private Schemas.XmlSoundProjectConfiguration _xmlSoundProjectConfiguration;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="platform">対象プラットフォーム。</param>
        /// <param name="version">コンフィギュレーションファイルのバージョン。</param>
        public SoundProjectConfiguration(string platform, string version) : base(platform, version) { }

        /// <summary>
        /// メインウィンドウ設定を取得します。
        /// </summary>
        public Schemas.XmlMainWindow MainWindow
        {
            get { return _xmlMainWindow; }
        }

        /// <summary>
        /// 全ての DocumentView 設定を取得します。
        /// </summary>
        public IDictionary<string, Schemas.XmlDocumentView> DocumentViews
        {
            get { return _xmlDocumentViews; }
        }

        /// <summary>
        /// 出力設定を取得します。
        /// </summary>
        public Schemas.XmlOutput Output
        {
            get { return _xmlOutput; }
        }

        /// <summary>
        /// Convert設定を取得します。
        /// </summary>
        public Schemas.XmlConvert Convert
        {
            get { return _xmlConvert; }
        }

        /// <summary>
        /// 全ての URI 設定を取得します。
        /// </summary>
        public IDictionary<string, Schemas.XmlUri> Uris
        {
            get { return _xmlUris; }
        }

        /// <summary>
        /// SoundProject 設定を取得します。
        /// </summary>
        public Schemas.XmlSoundProjectConfiguration SoundProject
        {
            get { return _xmlSoundProjectConfiguration; }
        }

        /// <summary>
        /// リストのデフォルト設定を読み込みます。
        /// </summary>
        /// <returns>リスト設定。</returns>
        public Schemas.XmlLists LoadListsDefault()
        {
            Schemas.XmlLists lists = LoadListsDefaultInternal();

            foreach (Schemas.XmlList list in lists.List)
            {
                switch (list.Name)
                {
                    case ListTraits.ListName_StreamSound:
                    case ListTraits.ListName_WaveSoundSetItem:
                    case ListTraits.ListName_SequenceSoundSetItem:
                    case ListTraits.ListName_SequenceSound:
                        Schemas.XmlListColumn[] listColumn = list.ListColumns.ListColumn;
                        list.ListColumns.ListColumn = this.CreateXmlListColumnUserParameters(listColumn);
                        break;
                }
            }

            return lists;
        }

        /// <summary>
        /// リストのデフォルト設定を読み込みます。
        /// </summary>
        /// <returns>リスト設定。</returns>
        public virtual Schemas.XmlLists LoadListsDefaultInternal()
        {
            return ReadConfigurationPart<Schemas.XmlLists>(
                                FileResourcesProxy.SoundProjectConfigurationListsDefault,
                                CreateReaderSettings());
        }

        /// <summary>
        /// SoundProjectDocumentView のデフォルト設定を読み込みます。
        /// </summary>
        /// <returns>サウンドプロジェクトドキュメント設定。</returns>
        public Schemas.XmlSoundProjectDocumentView LoadSoundProjectDocumentViewDefault()
        {
            Schemas.XmlSoundProjectDocumentView xmlView =
                        ReadConfigurationPart<Schemas.XmlSoundProjectDocumentView>(
                                FileResourcesProxy.SoundProjectConfigurationSoundProjectDocumentViewDefault,
                                CreateReaderSettings());

            xmlView.ListDefaults.List = LoadListsDefault().List;

            return xmlView;
        }

        /// <summary>
        /// SoundSetDocumentView のデフォルト設定を読み込みます。
        /// </summary>
        /// <returns>サウンドセットドキュメント設定。</returns>
        public Schemas.XmlSoundSetDocumentView LoadSoundSetDocumentViewDefault()
        {
            return ReadConfigurationPart<Schemas.XmlSoundSetDocumentView>(
                                FileResourcesProxy.SoundProjectConfigurationSoundSetDocumentViewDefault,
                                CreateReaderSettings());
        }

        /// <summary>
        /// BankDocumentView のデフォルト設定を読み込みます。
        /// </summary>
        /// <returns>バンクドキュメント設定。</returns>
        public Schemas.XmlBankDocumentView LoadBankDocumentViewDefault()
        {
            return ReadConfigurationPart<Schemas.XmlBankDocumentView>(
                                FileResourcesProxy.SoundProjectConfigurationBankDocumentViewDefault,
                                CreateReaderSettings());
        }

        /// <summary>
        /// デフォルト設定を読み込みます。
        /// </summary>
        /// <param name="settings">XmlReader 設定。</param>
        /// <param name="forced">現在の設定を全て破棄する場合は true、読み込み済みの設定を残す場合は false。</param>
        protected override void LoadDefaultsInternal(XmlReaderSettings settings, bool forced)
        {
            if (forced || null == _xmlMainWindow)
            {
                _xmlMainWindow = ReadConfigurationPart<Schemas.XmlMainWindow>(
                                    FileResourcesProxy.SoundProjectConfigurationMainWindowDefault, settings);
            }

            if (forced || null == _xmlDocumentViews)
            {
                _xmlDocumentViews = new Dictionary<string, Schemas.XmlDocumentView>();
            }

            if (forced || null == _xmlOutput)
            {
                _xmlOutput = ReadConfigurationPart<Schemas.XmlOutput>(
                                FileResourcesProxy.SoundProjectConfigurationOutputDefault, settings);
            }

            if (forced || null == _xmlConvert)
            {
                _xmlConvert = ReadConfigurationPart<Schemas.XmlConvert>(
                                FileResourcesProxy.SoundProjectConfigurationConvertDefault, settings);
            }

            if (forced || null == _xmlUris)
            {
                _xmlUris = new Dictionary<string, Schemas.XmlUri>();
            }

            if (forced || null == _xmlSoundProjectConfiguration)
            {
                _xmlSoundProjectConfiguration = ReadConfigurationPart<Schemas.XmlSoundProjectConfiguration>(
                                     FileResourcesProxy.SoundProjectConfigurationDefault, settings);
            }
        }

        /// <summary>
        /// スキーマセットを作成します。
        /// </summary>
        /// <returns>スキーマセット。</returns>
        protected override XmlSchemaSet CreateSchemaSet()
        {
            XmlSchemaSet schemas = base.CreateSchemaSet();
            schemas.Add(XmlSchema.Read(new StringReader(FileResourcesProxy.SchemaSoundProjectConfiguration), null));
            schemas.Add(XmlSchema.Read(new StringReader(FileResourcesProxy.SchemaUriConfiguration), null));

            return schemas;
        }

        /// <summary>
        /// コンフィギュレーションを読み込めるかどうかを調べます。
        /// </summary>
        /// <param name="readingInfo">読み込もうとしているコンフィギュレーションの情報を指定します。</param>
        /// <param name="readableInfo">読み込み可能なコンフィギュレーションの情報を指定します。</param>
        /// <returns>読み込める場合は true、読み込めない場合は false。</returns>
        protected override bool CanLoadConfiguration(ConfigurationRoot readingInfo, ConfigurationRoot readableInfo)
        {
            return readingInfo.Platform == readableInfo.Platform &&
                readingInfo.Version == readableInfo.Version;
        }

        /// <summary>
        /// コンフィギュレーション要素を読み込みます。
        /// </summary>
        /// <param name="xmlElement">コンフィギュレーション要素。</param>
        /// <param name="basePath">基準パス。</param>
        protected override void ReadConfigurationElement(XmlElement xmlElement, string basePath)
        {
            if (null == xmlElement) { throw new ArgumentNullException("xmlElement"); }
            if (null == basePath) { throw new ArgumentNullException("basePath"); }

            switch (xmlElement.LocalName)
            {
                case "Windows":

                    XmlNodeList nodes = xmlElement.GetElementsByTagName("MainWindow",
                                                                         NameSpaceOfWindowConfiguration);
                    if (0 == nodes.Count) { break; }

                    _xmlMainWindow = ReadConfigurationPart<Schemas.XmlMainWindow>(nodes[0].OuterXml);

                    break;

                case "DocumentViews":
                    _xmlDocumentViews = ReadConfigurationPart<Schemas.XmlDocumentViews>(
                                                    xmlElement.OuterXml).CreateDictionary(basePath);
                    break;

                case "Output":
                    _xmlOutput = ReadConfigurationPart<Schemas.XmlOutput>(xmlElement.OuterXml);
                    break;

                case "Convert":
                    _xmlConvert = ReadConfigurationPart<Schemas.XmlConvert>(xmlElement.OuterXml);
                    break;

                case "Uris":
                    _xmlUris = ReadConfigurationPart<Schemas.XmlUris>(xmlElement.OuterXml).CreateDictionary();
                    break;

                case "SoundProjectConfiguration":
                    _xmlSoundProjectConfiguration = ReadConfigurationPart<Schemas.XmlSoundProjectConfiguration>(xmlElement.OuterXml);
                    break;
            }
        }

        /// <summary>
        /// コンフィギュレーション要素を書き出します。
        /// </summary>
        /// <param name="reader">XmlReader。</param>
        /// <param name="basePath">基準パス。</param>
        protected override void WriteConfigurationElements(XmlWriter writer, string basePath)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }
            if (null == basePath) { throw new ArgumentNullException("basePath"); }

            writer.WriteStartElement("nwsw", "Windows", null);
            WriteConfigurationPart<Schemas.XmlMainWindow>(writer, _xmlMainWindow);
            writer.WriteEndElement();

            WriteDocumentViews(writer, basePath);

            WriteConfigurationPart<Schemas.XmlOutput>(writer, _xmlOutput);

            WriteConfigurationPart<Schemas.XmlConvert>(writer, _xmlConvert);

            WriteConfigurationPart<Schemas.XmlUris>(writer,
                                    new Schemas.XmlUris() { Uri = _xmlUris.Values.ToArray() });
            if (ApplicationBase.Instance.RealtimeEditService == null)
            {
                if (_xmlSoundProjectConfiguration != null &&
                    _xmlSoundProjectConfiguration.SndEditList != null)
                {
                    if (_xmlSoundProjectConfiguration.SndEditList.SndEdit == null)
                    {
                        _xmlSoundProjectConfiguration.SndEditList = null;
                    }
                }
            }

            WriteConfigurationPart<Schemas.XmlSoundProjectConfiguration>(writer, _xmlSoundProjectConfiguration);
        }

        /// <summary>
        /// コンフィギュレーションルートを書き出します。
        /// </summary>
        /// <param name="writer">XmlWriter。</param>
        protected override void WriteStartConfigurationRoot(XmlWriter writer)
        {
            base.WriteStartConfigurationRoot(writer);

            writer.WriteAttributeString("xmlns", ElementPrefixOfWindowConfiguration,
                                         null, NameSpaceOfWindowConfiguration);
            writer.WriteAttributeString("xmlns", ElementPrefixOfUriConfiguration,
                                         null, NameSpaceOfUriConfiguration);
            writer.WriteAttributeString("xmlns", ElementPrefixOfSoundProjectConfiguration,
                                         null, NameSpaceOfSoundProjectConfiguration);
        }

        /// <summary>
        /// 読み込んだコンフィギュレーションを検証します。
        /// </summary>
        /// <returns>読み込んだ結果が正常な場合は true、それ以外の場合は false。</returns>
        protected override bool Validate()
        {
            // MainWindow
            if (null == _xmlMainWindow) { return false; }

            IDictionary<string, Schemas.XmlTab> xmlTabs = _xmlMainWindow.CreateTabDictionary();
            if (2 > xmlTabs.Count) { return false; }
            if (!xmlTabs.ContainsKey("Main")) { return false; }
            if (!xmlTabs.ContainsKey("Sub")) { return false; }

            // Output
            if (null == _xmlOutput) { return false; }
            if (null == _xmlOutput.HtmlInstrumentList) { return false; }
            if (null == _xmlOutput.HtmlInstrumentList.Lists) { return false; }
            if (null == _xmlOutput.HtmlInstrumentList.Lists.List) { return false; }
            if (0 == _xmlOutput.HtmlInstrumentList.Lists.List.Length) { return false; }

            if (null == _xmlConvert) { return false; }

            if (null == _xmlSoundProjectConfiguration.SndEditList)
            {
                _xmlSoundProjectConfiguration.SndEditList = new Schemas.XmlSndEditList();
            }

            return true;
        }

        private void WriteDocumentViews(XmlWriter writer, string basePath)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }
            if (null == basePath) { throw new ArgumentNullException("basePath"); }

            List<Schemas.XmlDocumentView> xmlViews = new List<Schemas.XmlDocumentView>();

            foreach (Schemas.XmlDocumentView view in _xmlDocumentViews.Values)
            {
                view.FilePath = PathEx.MakeRelative(view.FilePath, basePath);
            }

            Schemas.XmlDocumentViews xmlDocumentViews = new Schemas.XmlDocumentViews()
            {
                Items = _xmlDocumentViews.Values.ToArray()
            };

            WriteConfigurationPart<Schemas.XmlDocumentViews>(writer, xmlDocumentViews);

            // ★仮対応。相対パスをフルパスに戻すためのコード。
            _xmlDocumentViews = xmlDocumentViews.CreateDictionary(basePath);
        }

        //
        private Schemas.XmlListColumn[] CreateXmlListColumnUserParameters(Schemas.XmlListColumn[] listColumns)
        {
            List<Schemas.XmlListColumn> listColumnList = new List<Schemas.XmlListColumn>();

            foreach (Schemas.XmlListColumn listColumn in listColumns)
            {
                listColumnList.Add(listColumn);
            }

            if (ApplicationBase.Instance.ProjectService.Project != null)
            {
                List<UserParameterStructureSetting> settings =
                    ApplicationBase.Instance.ProjectService.Project.UserDataStructureSettings.Settings;

                int i = 0;
                foreach (UserParameterStructureSetting setting in settings)
                {
                    int j = 0;
                    foreach (UserParameterStructure structure in setting.Structures)
                    {
                        Schemas.XmlListColumn xmlListColumn = new Schemas.XmlListColumn();
                        xmlListColumn.Name = ListTraits.ColumnName_UserParameters[i, j];
                        xmlListColumn.Visible = true;
                        xmlListColumn.LengthSpecified = true;
                        if (structure.StructureType == StructureTypes.Boolean)
                        {
                            xmlListColumn.Length = 32;
                        }
                        else
                        {
                            xmlListColumn.Length = 48;
                        }
                        listColumnList.Add(xmlListColumn);
                        j++;
                    }
                    i++;
                }
            }

            return listColumnList.ToArray();
        }
    }
}
