﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;

namespace NintendoWare.SoundMaker.Framework.Configurations
{
    public abstract class PresetConfiguration : ConfigurationBase
    {
        protected const string NameSpaceOfPresetConfiguration = "NintendoWare.SoundMaker.PresetConfiguration";
        protected const string ElementPrefixOfPresetConfiguration = "nwsp";
        protected const string ElementNameOfPresets = "Presets";
        protected const string ElementNameOfPreset = "Preset";
        protected const string ElementNameOfPresetItems = "PresetItems";
        protected const string ElementNameOfPresetItem = "PresetItem";
        protected const string AttributeNameOfName = "Name";

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="platform">対象プラットフォーム。</param>
        /// <param name="version">コンフィギュレーションファイルのバージョン。</param>
        public PresetConfiguration(string platform, string version) : base(platform, version) { }

        /// <summary>
        /// デフォルト設定を読み込みます。
        /// </summary>
        /// <param name="settings">XmlReader 設定。</param>
        /// <param name="forced">現在の設定を全て破棄する場合は true、読み込み済みの設定を残す場合は false。</param>
        protected override void LoadDefaultsInternal(XmlReaderSettings settings, bool forced)
        {
        }

        /// <summary>
        /// コンフィギュレーション要素を読み込みます。
        /// </summary>
        /// <param name="xmlElement">コンフィギュレーション要素。</param>
        /// <param name="basePath">基準パス。</param>
        protected sealed override void ReadConfigurationElement(XmlElement xmlElement, string basePath)
        {
            if (null == xmlElement) { throw new ArgumentNullException("xmlElement"); }

            switch (xmlElement.LocalName)
            {
                case ElementNameOfPresets:
                    ReadPresets(xmlElement);
                    break;
            }
        }

        /// <summary>
        /// コンフィギュレーション要素を書き出します。
        /// </summary>
        /// <param name="reader">XmlReader。</param>
        /// <param name="basePath">基準パス。</param>
        protected sealed override void WriteConfigurationElements(XmlWriter writer, string basePath)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            writer.WriteStartElement(ElementPrefixOfPresetConfiguration, ElementNameOfPresets, null);

            WritePresets(writer);

            writer.WriteEndElement();
        }

        /// <summary>
        /// コンフィギュレーションルートを書き出します。
        /// </summary>
        /// <param name="writer">XmlWriter。</param>
        protected override void WriteStartConfigurationRoot(XmlWriter writer)
        {
            base.WriteStartConfigurationRoot(writer);

            writer.WriteAttributeString("xmlns", ElementPrefixOfPresetConfiguration, null, NameSpaceOfPresetConfiguration);
        }

        /// <summary>
        /// プリセットアイテムデータを読み込みます。
        /// </summary>
        /// <param name="presetName">プリセット名。</param>
        /// <param name="presetItemName">プリセットアイテム名。</param>
        /// <param name="xml">PresetItem 要素の内容。</param>
        protected abstract void ReadPresetItemData(string presetName, string presetItemName, string xml);

        /// <summary>
        /// 全てのプリセットを書き出します。
        /// </summary>
        /// <param name="writer">XmlWriter。</param>
        protected abstract void WritePresets(XmlWriter writer);

        /// <summary>
        /// Presets 要素を読み込みます。
        /// </summary>
        /// <param name="xmlElement">Presets 要素。</param>
        private void ReadPresets(XmlElement xmlElement)
        {
            if (null == xmlElement) { throw new ArgumentNullException("xmlElement"); }

            foreach (XmlElement xmlChild in xmlElement.GetElementsByTagName(ElementNameOfPreset,
                                                                              NameSpaceOfPresetConfiguration))
            {
                ReadPreset(xmlChild);
            }
        }

        /// <summary>
        /// プリセット要素を読み込みます。
        /// </summary>
        /// <param name="xmlElement">Preset 要素。</param>
        private void ReadPreset(XmlElement xmlElement)
        {
            if (null == xmlElement) { throw new ArgumentNullException("xmlElement"); }

            string name = xmlElement.GetAttribute(AttributeNameOfName);
            if (null == name || 0 == name.Length) { return; }

            XmlNodeList xmlChildren = xmlElement.GetElementsByTagName(ElementNameOfPresetItems,
                                                                       NameSpaceOfPresetConfiguration);
            if (0 == xmlChildren.Count) { return; }

            ReadPresetItems(name, xmlChildren[0] as XmlElement);
        }

        /// <summary>
        /// プリセットアイテム要素を読み込みます。
        /// </summary>
        /// <param name="presetName">プリセット名。</param>
        /// <param name="xmlElement">PresetItems 要素。</param>
        private void ReadPresetItems(string presetName, XmlElement xmlElement)
        {
            if (null == xmlElement) { throw new ArgumentNullException("xmlElement"); }

            foreach (XmlElement xmlChild in xmlElement.GetElementsByTagName(ElementNameOfPresetItem,
                                                                              NameSpaceOfPresetConfiguration))
            {
                ReadPresetItem(presetName, xmlChild);
            }
        }

        /// <summary>
        /// プリセットアイテム要素を読み込みます。
        /// </summary>
        /// <param name="presetName">プリセット名。</param>
        /// <param name="xmlElement">PresetItem 要素。</param>
        private void ReadPresetItem(string presetName, XmlElement xmlElement)
        {
            if (null == xmlElement) { throw new ArgumentNullException("xmlElement"); }

            string presetItemName = xmlElement.GetAttribute(AttributeNameOfName);
            if (null == presetItemName || 0 == presetItemName.Length) { return; }

            ReadPresetItemData(presetName, presetItemName, xmlElement.InnerXml);
        }
    }
}
