﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using NintendoWare.SoundFoundation.CommandHandlers;
using NintendoWare.SoundFoundation.Commands;
using NintendoWare.SoundFoundation.Operations;
using NintendoWare.SoundFoundation.Projects;

namespace NintendoWare.SoundMaker.Framework.CommandHandlers
{
    /// <summary>
    /// コンポーネントを追加します。
    /// </summary>
    public abstract class AddComponentHandler : CommandHandler
    {
        private string _commandID;
        private bool _insert;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="commandID">対象コマンドID。</param>
        /// <param name="insert">挿入する場合は false、追加する場合は false。</param>
        /// <param name="queryTarget">対象の問い合わせ先。</param>
        protected AddComponentHandler(string commandID, bool insert, IQueryCommandParameter queryTarget)
            : base(queryTarget)
        {
            if (null == commandID) { throw new ArgumentNullException("commandID"); }
            if (null == queryTarget) { throw new ArgumentNullException("queryTarget"); }
            _commandID = commandID;
            _insert = insert;
        }

        /// <summary>
        /// コンポーネントを追加します。
        /// </summary>
        /// <param name="document">追加先のドキュメント。</param>
        /// <param name="parentItem">追加先のコンポーネント。</param>
        /// <param name="insertTargetItem">挿入位置のコンポーネント。</param>
        /// <param name="newItem">追加するアイテム。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        public static bool Execute(SoundDocument document, Component parentItem,
                                   Component insertTargetItem, Component newItem)
        {
            if (null == document) { throw new ArgumentNullException("document"); }
            if (null == parentItem) { throw new ArgumentNullException("parentItem"); }
            if (null == newItem) { throw new ArgumentNullException("newItem"); }

            Operation operation = new InsertComponentOperation(parentItem, insertTargetItem, newItem);
            if (!operation.Execute()) { return false; }

            document.OperationHistory.AddOperation(operation);

            return true;
        }

        /// <summary>
        /// コマンドハンドラが実行可能なコマンドを取得します。
        /// </summary>
        public override string TargetCommandID
        {
            get { return _commandID; }
        }

        /// <summary>
        /// アプリケーションを取得します。
        /// </summary>
        protected ApplicationBase Application
        {
            get { return ApplicationBase.Instance; }
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            if (null == GetTargetDocument(parameters)) { return CommandStatus.SupportedAndVisible; }
            if (null == GetInsertParentComponent(parameters)) { return CommandStatus.SupportedAndVisible; }
            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// 対象プロジェクトサービスを取得します。
        /// </summary>
        protected ComponentService GetTargetComponentService(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            if (!parameters.ContainsParameter(CommandParameterNames.TargetComponentService))
            {
                return null;
            }

            return parameters.GetParameter(CommandParameterNames.TargetComponentService)
                   as ComponentService;
        }

        /// <summary>
        /// 対象サウンドセットドキュメントを取得します。
        /// </summary>
        protected SoundDocument GetTargetDocument(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            if (!parameters.ContainsParameter(CommandParameterNames.TargetDocuments))
            {
                return null;
            }

            return (parameters.GetParameter(CommandParameterNames.TargetDocuments)
                     as IEnumerable<SoundDocument>).FirstOrDefault();
        }

        /// <summary>
        /// 挿入先親コンポーネントを取得します。
        /// </summary>
        protected virtual Component GetInsertParentComponent(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            if (!parameters.ContainsParameter(CommandParameterNames.InsertParentComponent))
            {
                return null;
            }

            return parameters.GetParameter(CommandParameterNames.InsertParentComponent)
                   as Component;
        }

        /// <summary>
        /// 挿入位置のコンポーネントを取得します。
        /// </summary>
        protected virtual Component GetInsertTargetComponent(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            if (!_insert) { return null; }

            if (!parameters.ContainsParameter(CommandParameterNames.InsertTargetComponent))
            {
                return null;
            }

            return parameters.GetParameter(CommandParameterNames.InsertTargetComponent)
                   as Component;
        }
    }
}
