﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using NintendoWare.SoundFoundation.Commands;
using NintendoWare.SoundFoundation.Operations;
using NintendoWare.SoundFoundation.Projects;

namespace NintendoWare.SoundMaker.Framework.CommandHandlers
{
    /// <summary>
    /// バンクにアイテムを追加します。
    /// </summary>
    public abstract class AddBankItemHandler : AddComponentHandler
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="commandID">対象コマンドID。</param>
        /// <param name="insert">挿入する場合は false、追加する場合は false。</param>
        /// <param name="queryTarget">対象の問い合わせ先。</param>
        protected AddBankItemHandler(string commandID, bool insert, IQueryCommandParameter queryParameter)
            : base(commandID, insert, queryParameter) { }

        /// <summary>
        /// バンクにアイテムを追加します。
        /// </summary>
        /// <param name="document">追加先のバンクドキュメント。</param>
        /// <param name="parentItem">追加先のコンポーネント。</param>
        /// <param name="insertTargetItem">挿入位置のコンポーネント。</param>
        /// <param name="newItem">追加するバンクアイテム。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        public static bool Execute(BankDocument document, Component parentItem,
                                    Component insertTargetItem, BankItem newItem)
        {
            if (null == document) { throw new ArgumentNullException("document"); }
            if (null == parentItem) { throw new ArgumentNullException("parentItem"); }
            if (null == newItem) { throw new ArgumentNullException("newItem"); }

            if (!(parentItem is Bank) && !(parentItem is BankItem))
            {
                throw new ArgumentException("parentItem is invalid.");
            }

            Operation operation = new InsertComponentOperation(parentItem, insertTargetItem, newItem);
            if (!operation.Execute()) { return false; }

            document.OperationHistory.AddOperation(operation);

            return true;
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            BankDocument targetDocument = GetTargetDocument(parameters);

            if (null == targetDocument) { return CommandStatus.SupportedAndVisible; }
            if (targetDocument.IsReadOnly == true)
            {
                return CommandStatus.SupportedAndVisible;
            }

            CommandStatus status = base.QueryStatusInternal(command, parameters);
            if (status.IsEnabled()) { return status; }

            return (GetInsertParentComponent(parameters) != null) ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// 対象バンクドキュメントを取得します。
        /// </summary>
        protected new BankDocument GetTargetDocument(IQueryCommandParameter parameters)
        {
            return base.GetTargetDocument(parameters) as BankDocument;
        }

        /// <summary>
        /// バンクにアイテムを追加します。
        /// </summary>
        /// <param name="newItem">追加するバンクアイテム。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected virtual bool Execute(IQueryCommandParameter parameters, BankItem newItem)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }
            if (null == newItem) { throw new ArgumentNullException("newItem"); }

            BankDocument targetDocument = GetTargetDocument(parameters);
            Component targetComponent = GetInsertParentComponent(parameters);

            if ((targetComponent != targetDocument.Bank) &&
                 (!(targetComponent is BankItem) ||
                   (targetComponent as BankItem).Bank != targetDocument.Bank))
            {
                throw new ArgumentException("target component is invalid.");
            }

            return Execute(targetDocument, targetComponent,
                            GetInsertTargetComponent(parameters), newItem);
        }
    }
}
